/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode filter:
 * Copyright (C) Thomas Oestreich - June 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-astat
 *
 * <refsect2>
 * <para>
 * Determines the maximum audio scale value that can safely be applied
 * (that is, does not lead to clipping of audio samples).
 * At EOS (end-of-stream), it generates an element message named
 * <classname>&quot;astat&quot;</classname>.
 * The message's structure contains only one field,
 * #gdouble <classname>&quot;scale&quot;</classname>, which is then the
 * value to use for the <link linkend="GstVolume--volume">volume</link>
 * property in the <link linkend="GstVolume">volume</link> element.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * <para>
 * Transcode astat filter [Thomas Oestreich]
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <string.h>


#define GST_TYPE_ASTAT \
  (gst_astat_get_type())
#define GST_ASTAT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_ASTAT,GstAstat))
#define GST_ASTAT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_ASTAT,GstAstatClass))
#define GST_IS_ASTAT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_ASTAT))
#define GST_IS_ASTAT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_ASTAT))


typedef struct _GstAstat GstAstat;
typedef struct _GstAstatClass GstAstatClass;

struct _GstAstat
{
  GstBaseTransform parent;

  gint min, max;
};


struct _GstAstatClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (astat_debug);
#define GST_CAT_DEFAULT astat_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_SHIFT
      /* FILL ME */
};

#define DEFAULT_SHIFT      0

static GstStaticPadTemplate gst_astat_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-raw-int, "
        "rate = (int) [ 1, MAX ], "
        "channels = (int) [ 1, MAX ], "
        "endianness = (int) BYTE_ORDER, "
        "width = (int) { 16 }, "
        "depth = (int) { 16 }, " "signed = (boolean) true")
    );

static GstStaticPadTemplate gst_astat_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-raw-int, "
        "rate = (int) [ 1, MAX ], "
        "channels = (int) [ 1, MAX ], "
        "endianness = (int) BYTE_ORDER, "
        "width = (int) { 16 }, "
        "depth = (int) { 16 }, " "signed = (boolean) true")
    );

static gboolean gst_astat_sink_event (GstBaseTransform * btrans,
    GstEvent * event);
static GstFlowReturn gst_astat_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_astat_start (GstBaseTransform * btrans);
static gboolean gst_astat_stop (GstBaseTransform * btrans);

static void gst_astat_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_astat_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstAstat, gst_astat, GstBaseTransform,
    GST_TYPE_BASE_TRANSFORM);


static void
gst_astat_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Astat",
      "Filter/Analyzer/Audio", "Audio statistics plugin",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Thomas Oestreich");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_astat_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_astat_src_template));
}

static void
gst_astat_class_init (GstAstatClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (astat_debug, "astat", 0, "astat");

  gobject_class->set_property = gst_astat_set_property;
  gobject_class->get_property = gst_astat_get_property;

  trans_class->event = GST_DEBUG_FUNCPTR (gst_astat_sink_event);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_astat_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_astat_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_astat_stop);
}

static void
gst_astat_init (GstAstat * filter, GstAstatClass * g_class)
{

}

static gboolean
gst_astat_sink_event (GstBaseTransform * btrans, GstEvent * event)
{
  GstAstat *filter;

  filter = GST_ASTAT (btrans);

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_EOS:{
      gdouble scale;
      GstMessage *m;

      scale = MAX (filter->max, ABS (filter->min));
      if (scale)
        scale = (double) (-G_MININT16) / scale;
      else
        scale = 1.0;
      m = gst_message_new_element (GST_OBJECT (filter),
          gst_structure_new ("astat", "scale", G_TYPE_DOUBLE, scale, NULL));
      gst_element_post_message (GST_ELEMENT (filter), m);
      break;
    }
    default:
      break;
  }

  return GST_BASE_TRANSFORM_CLASS (parent_class)->event (btrans, event);
}


static GstFlowReturn
gst_astat_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstAstat *filter;
  gint16 *src;
  guint y;

  filter = GST_ASTAT (btrans);

  src = (gint16 *) GST_BUFFER_DATA (in);

  for (y = 0; y < GST_BUFFER_SIZE (in) >> 1; y++) {
    if (*src > filter->max)
      filter->max = *src;
    else if (*src < filter->min)
      filter->min = *src;
  }

  return GST_FLOW_OK;
}


static gboolean
gst_astat_start (GstBaseTransform * btrans)
{
  GstAstat *filter;

  filter = GST_ASTAT (btrans);

  filter->max = filter->min = 0;

  return TRUE;
}

static gboolean
gst_astat_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_astat_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstAstat *src;

  g_return_if_fail (GST_IS_ASTAT (object));
  src = GST_ASTAT (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_astat_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstAstat *src;

  g_return_if_fail (GST_IS_ASTAT (object));
  src = GST_ASTAT (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
