/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * mplayer hqdn3d filter
 *   Copyright (C) 2003 Daniel Moreno <comac@comac.darktech.org>
 *
 * mplayer/transcode denoise3d filter
 *   Copyright (C) 2003 Daniel Moreno <comac@comac.darktech.org>
 *   Copyright (C) 2003 Erik Slagter <erik@oldconomy.org> (GPL) 2003
 *      transcode optimizations
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-hqdn3d
 *
 * <refsect2>
 * <para>
 * This filter aims to reduce image noise  producing  smooth  images  and  making
 * still images really still (This should enhance compressibility).
 * </para>
 * <para>
 * Depending on <link linkend="GstHqdn3d--high-quality">hiqh-quality</link> setting,
 * it runs either mplayer's hqdn3d filter algorithm or the denoise3d algorithm,
 * which is cruder and simpler, but also faster.
 * </para>
 * <para>
 * Leaving or setting any of the other parameters to 0 will make the filter
 * use the indicated internal defaults.  Setting some parameters will
 * also have this value 'trickle' into the other parameters as follows:
 * <itemizedlist>
 * <listitem>
 * luma_spatial --> luma_temp, chroma_spatial, chroma_temp
 * </listitem>
 * <listitem>
 * chroma_spatial --> chroma_temp
 * </listitem>
 * <listitem>
 * luma_temp --> chroma_temp
 * </listitem>
 * </itemizedlist>
 * In addition, setting a components (both) parameters to a negative value will not
 * have the filter applied to this component.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * Mplayer denoise3d and hqdn3d filter [Daniel Moreno]
 * </listitem>
 * <listitem>
 * Also used in transcode; denoise3d filter somewhat optimized [Erik Slagter]
 * </listitem>
 * <listitem>
 * Also available in avidemux (MPlayer hqdn3d, MPlayer denoise3d)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-mencoder.h"

#include <math.h>


#define GST_TYPE_HQDN3D \
  (gst_hqdn3d_get_type())
#define GST_HQDN3D(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_HQDN3D,GstHqdn3d))
#define GST_HQDN3D_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_HQDN3D,GstHqdn3dClass))
#define GST_IS_HQDN3D(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_HQDN3D))
#define GST_IS_HQDN3D_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_HQDN3D))

typedef struct _GstHqdn3d GstHqdn3d;
typedef struct _GstHqdn3dClass GstHqdn3dClass;

/* number of coefficients for each 'dimension' */
#define COEFS_SIZE             512*16

/* function that will do the actual denoising */
typedef void (*GstDenoiseFunction) (guint8 * frame, guint * lineant,
    gushort ** frameprevptr,
    gint w, gint h, gint * horizontal, gint * vertical, gint * temporal);


struct _GstHqdn3d
{
  GstVideoFilter videofilter;

  gint width, height;

  /* user denoise parameters */
  gdouble set_luma_spatial, set_luma_temp, set_chroma_spatial, set_chroma_temp;
  /* internal denoise parameters */
  gdouble luma_spatial, luma_temp, chroma_spatial, chroma_temp;
  gboolean quality;

  /* coefs used in denoising */
    gint (*coefs)[COEFS_SIZE];
  /* stores the previous line temporarily (within a plane) */
  guint *line;
  /* store the previous *plane*, so several required per frame */
  gushort *frame[3];
  /* performs actual denoising depending on quality setting */
  GstDenoiseFunction denoise_func;
};

struct _GstHqdn3dClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (hqdn3d_debug);
#define GST_CAT_DEFAULT hqdn3d_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_LUMA_SPATIAL,
  PROP_CHROMA_SPATIAL,
  PROP_LUMA_TEMP,
  PROP_CHROMA_TEMP,
  PROP_HIGH_QUALITY
      /* FILL ME */
};

#define DEFAULT_LUMA_SPATIAL    4.0
#define DEFAULT_LUMA_TEMP       6.0
#define DEFAULT_CHROMA_SPATIAL  3.0
#define DEFAULT_CHROMA_TEMP   \
  (DEFAULT_LUMA_TEMP * DEFAULT_CHROMA_SPATIAL / DEFAULT_LUMA_SPATIAL)
#define HQDN3D_MAX_PARAM        255
#define DEFAULT_HIGH_QUALITY    TRUE

/* use these names rather than hardcoded indexes for the coefficient array below */
enum
{
  COEFS_LS_INDEX,
  COEFS_LT_INDEX,
  COEFS_CS_INDEX,
  COEFS_CT_INDEX,
  COEFS_LAST_INDEX
};

static GstStaticPadTemplate gst_hqdn3d_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_hqdn3d_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static gboolean gst_hqdn3d_hook_caps (GstHqdn3d * btrans, GstCaps * incaps,
    GstCaps * outcaps);
static GstFlowReturn gst_hqdn3d_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_hqdn3d_start (GstBaseTransform * btrans);
static gboolean gst_hqdn3d_stop (GstBaseTransform * btrans);

static void gst_hqdn3d_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_hqdn3d_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

static void gst_hqdn3d_finalize (GObject * object);

GST_BOILERPLATE (GstHqdn3d, gst_hqdn3d, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstHqdn3d, gst_hqdn3d,
    gst_hqdn3d_hook_caps);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_hqdn3d);

static void
gst_hqdn3d_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Hqdn3d",
      "Filter/Effect/Video", "High Quality 3D Denoiser",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n"
      "Daniel Moreno, Erik Slagter");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_hqdn3d_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_hqdn3d_src_template));
}

static void
gst_hqdn3d_class_init (GstHqdn3dClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (hqdn3d_debug, "hqdn3d", 0, "hqdn3d");

  gobject_class->finalize = GST_DEBUG_FUNCPTR (gst_hqdn3d_finalize);

  gobject_class->set_property = gst_hqdn3d_set_property;
  gobject_class->get_property = gst_hqdn3d_get_property;

  g_object_class_install_property (gobject_class, PROP_LUMA_SPATIAL,
      g_param_spec_double ("luma-spatial", "Luma Spatial",
          "Spatial Luma Strength (0: use default)",
          -1, HQDN3D_MAX_PARAM, DEFAULT_LUMA_SPATIAL,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_SPATIAL,
      g_param_spec_double ("chroma-spatial", "Chroma Spatial",
          "Spatial Chroma Strength (0: use default)",
          -1, HQDN3D_MAX_PARAM, DEFAULT_CHROMA_SPATIAL,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LUMA_TEMP,
      g_param_spec_double ("luma-temp", "Luma Temporal",
          "Temporal Luma Strength (0: use default)",
          -1, HQDN3D_MAX_PARAM, DEFAULT_LUMA_TEMP,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_TEMP,
      g_param_spec_double ("chroma-temp", "Chroma Temporal",
          "Temporal Chroma Strength (0: use default)",
          -1, HQDN3D_MAX_PARAM, DEFAULT_CHROMA_TEMP,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_HIGH_QUALITY,
      g_param_spec_boolean ("high-quality", "High Quality",
          "High Quality Denoising (hqdn3d versus denoise3d)",
          DEFAULT_HIGH_QUALITY, G_PARAM_READWRITE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_hqdn3d_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_hqdn3d_get_unit_size);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_hqdn3d_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_hqdn3d_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_hqdn3d_stop);
}

static void
gst_hqdn3d_init (GstHqdn3d * filter, GstHqdn3dClass * g_class)
{
  filter->line = NULL;
  filter->frame[0] = filter->frame[1] = filter->frame[2] = NULL;

  /* does not depend on any stream specific property,
   * so can be allocated here */
  filter->coefs = g_malloc (sizeof (*(filter->coefs)) * COEFS_LAST_INDEX);

  filter->quality = DEFAULT_HIGH_QUALITY;
}

static void
gst_hqdn3d_finalize (GObject * object)
{
  GstHqdn3d *filter = GST_HQDN3D (object);

  g_free (filter->coefs);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gst_hqdn3d_free (GstHqdn3d * filter)
{
  g_free (filter->line);
  filter->line = NULL;
  /* hm, just too small to do loop */
  g_free (filter->frame[0]);
  g_free (filter->frame[1]);
  g_free (filter->frame[2]);
  filter->frame[0] = filter->frame[1] = filter->frame[2] = NULL;
}

static void
gst_hqdn3d_alloc (GstHqdn3d * filter)
{
  filter->line = g_malloc (filter->width * sizeof (guint));
  /* frame will be allocated at the actual transformation time when needed */
}

static gboolean
gst_hqdn3d_hook_caps (GstHqdn3d * filter, GstCaps * incaps, GstCaps * outcaps)
{
  gst_hqdn3d_free (filter);
  gst_hqdn3d_alloc (filter);

  return TRUE;
}

static void
gst_hqdn3d_precalc_coefs (gint * coefs, double Dist25)
{
  int i;
  double Gamma, Simil, C;

  Gamma = log (0.25) / log (1.0 - Dist25 / 255.0 - 0.00001);

  for (i = -256 * 16; i < 256 * 16; i++) {
    Simil = 1.0 - (i > 0 ? i : -i) / (16 * 255.0);
    C = pow (Simil, Gamma) * 65536.0 * (double) i / 16.0;
    coefs[16 * 256 + i] = (C < 0) ? (C - 0.5) : (C + 0.5);
  }
}

static inline guint
LowPassMul (guint PrevMul, guint CurrMul, gint * Coef)
{
//    int dMul= (PrevMul&0xFFFFFF)-(CurrMul&0xFFFFFF);
  gint dMul = PrevMul - CurrMul;
  gint d = ((dMul + 0x10007FF) >> 12);
  return CurrMul + Coef[d];
}

static void
gst_hqdn3d_denoise (guint8 * Frame, guint * LineAnt, gushort ** FrameAntPtr,
    gint W, gint H, gint * Horizontal, gint * Vertical, gint * Temporal)
{
  gint X, Y;
  gint sLineOffs = 0;
  guint PixelAnt;
  guint PixelDst;
  gushort *FrameAnt = (*FrameAntPtr);

  if (!FrameAnt) {
    (*FrameAntPtr) = FrameAnt = g_malloc (W * H * sizeof (gushort));
    for (Y = 0; Y < H; Y++) {
      gushort *dst = &FrameAnt[Y * W];
      guint8 *src = Frame + Y * W;
      for (X = 0; X < W; X++)
        dst[X] = src[X] << 8;
    }
  }

  /* First pixel has no left nor top neighbour. Only previous frame */
  LineAnt[0] = PixelAnt = Frame[0] << 16;
  PixelDst = LowPassMul (FrameAnt[0] << 8, PixelAnt, Temporal);
  FrameAnt[0] = ((PixelDst + 0x1000007F) >> 8);
  Frame[0] = ((PixelDst + 0x10007FFF) >> 16);

  /* First line has no top neighbour. Only left one for each pixel and
   * last frame
   */
  for (X = 1; X < W; X++) {
    LineAnt[X] = PixelAnt = LowPassMul (PixelAnt, Frame[X] << 16, Horizontal);
    PixelDst = LowPassMul (FrameAnt[X] << 8, PixelAnt, Temporal);
    FrameAnt[X] = ((PixelDst + 0x1000007F) >> 8);
    Frame[X] = ((PixelDst + 0x10007FFF) >> 16);
  }

  for (Y = 1; Y < H; Y++) {
    guint PixelAnt;
    gushort *LinePrev = &FrameAnt[Y * W];
    sLineOffs += W;
    /* First pixel on each line doesn't have previous pixel */
    PixelAnt = Frame[sLineOffs] << 16;
    LineAnt[0] = LowPassMul (LineAnt[0], PixelAnt, Vertical);
    PixelDst = LowPassMul (LinePrev[0] << 8, LineAnt[0], Temporal);
    LinePrev[0] = ((PixelDst + 0x1000007F) >> 8);
    Frame[sLineOffs] = ((PixelDst + 0x10007FFF) >> 16);

    for (X = 1; X < W; X++) {
      gint PixelDst;
      /* The rest are normal */
      PixelAnt = LowPassMul (PixelAnt, Frame[sLineOffs + X] << 16, Horizontal);
      LineAnt[X] = LowPassMul (LineAnt[X], PixelAnt, Vertical);
      PixelDst = LowPassMul (LinePrev[X] << 8, LineAnt[X], Temporal);
      LinePrev[X] = ((PixelDst + 0x1000007F) >> 8);
      Frame[sLineOffs + X] = ((PixelDst + 0x10007FFF) >> 16);
    }
  }
}

/* denoise3d variant */

static void
gst_denoise3d_precalc_coefs (gint * ct, double dist25)
{
  int i;
  double gamma, simil, c;

  gamma = log (0.25) / log (1.0 - dist25 / 255.0);

  for (i = -256; i <= 255; i++) {
    simil = 1.0 - (double) ABS (i) / 255.0;
    c = pow (simil, gamma) * (double) i;
    ct[256 + i] = (int) ((c < 0) ? (c - 0.5) : (c + 0.5));
  }
}

#define LowPass(prev, curr, coef) (curr + coef[(gint) prev - (gint) curr])

static void
gst_denoise3d_denoise (guint8 * frame, guint * lineant, gushort ** frameprevptr,
    gint w, gint h, gint * horizontal, gint * vertical, gint * temporal)
{
  gint x, y;
  guint8 pixelant;
  guint8 *lineantptr;
  guint8 *frameprev = (guint8 *) (*frameprevptr);

  if (!*frameprevptr)
    *frameprevptr = g_memdup (frame, w * h);
  frameprev = (guint8 *) * frameprevptr;

  lineantptr = (guint8 *) lineant;

  horizontal += 256;
  vertical += 256;
  temporal += 256;

  /* First pixel has no left nor top neighbour, only previous frame */
  *lineantptr = pixelant = *frame;
  *frame = *frameprev = LowPass (*frameprev, *lineantptr, temporal);
  frame++;
  frameprev++;
  lineantptr++;

  /* First line has no top neighbour, only left one for each pixel and last frame */
  for (x = 1; x < w; x++) {
    pixelant = LowPass (pixelant, *frame, horizontal);
    *lineantptr = pixelant;
    *frame = *frameprev = LowPass (*frameprev, *lineantptr, temporal);
    frame++;
    frameprev++;
    lineantptr++;
  }

  for (y = 1; y < h; y++) {
    lineantptr = (guint8 *) lineant;

    /* First pixel on each line doesn't have previous pixel */
    pixelant = *frame;
    *lineantptr = LowPass (*lineantptr, pixelant, vertical);
    *frame = *frameprev = LowPass (*frameprev, *lineantptr, temporal);
    frame++;
    frameprev++;
    lineantptr++;

    for (x = 1; x < w; x++) {
      /* The rest is normal */
      pixelant = LowPass (pixelant, *frame, horizontal);
      *lineantptr = LowPass (*lineantptr, pixelant, vertical);
      *frame = *frameprev = LowPass (*frameprev, *lineantptr, temporal);
      frame++;
      frameprev++;
      lineantptr++;
    }
  }
}


static GstFlowReturn
gst_hqdn3d_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstHqdn3d *filter;
  guint8 *src, *dest;
  GstFlowReturn ret = GST_FLOW_OK;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_HQDN3D (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);
  dest = (guint8 *) GST_BUFFER_DATA (in);

  gint width = filter->width;
  gint height = filter->height;

  /* first run it on the luma plane */
  if (G_LIKELY (filter->luma_spatial > 0 && filter->luma_temp > 0))
    filter->denoise_func (src, filter->line, &(filter->frame[0]),
        GST_VIDEO_I420_Y_ROWSTRIDE (width), height,
        filter->coefs[COEFS_LS_INDEX], filter->coefs[COEFS_LS_INDEX],
        filter->coefs[COEFS_LT_INDEX]);
  /* then on the chroma planes */
  if (G_LIKELY (filter->chroma_spatial > 0 && filter->chroma_temp > 0)) {
    filter->denoise_func (src + GST_VIDEO_I420_U_OFFSET (width, height),
        filter->line, &(filter->frame[1]),
        GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2,
        filter->coefs[COEFS_CS_INDEX], filter->coefs[COEFS_CS_INDEX],
        filter->coefs[COEFS_CT_INDEX]);
    filter->denoise_func (src + GST_VIDEO_I420_V_OFFSET (width, height),
        filter->line, &(filter->frame[2]),
        GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2,
        filter->coefs[COEFS_CS_INDEX], filter->coefs[COEFS_CS_INDEX],
        filter->coefs[COEFS_CT_INDEX]);
  }

  return ret;
}

static void
gst_hqdn3d_set_params (GstHqdn3d * filter)
{
  /* recalculate only the needed params */

  filter->luma_spatial = filter->set_luma_spatial;
  if (!filter->luma_spatial)
    filter->luma_spatial = DEFAULT_LUMA_SPATIAL;

  filter->chroma_spatial = filter->set_chroma_spatial;
  if (!filter->chroma_spatial)
    filter->chroma_spatial = DEFAULT_CHROMA_SPATIAL * filter->luma_spatial /
        DEFAULT_LUMA_SPATIAL;

  filter->luma_temp = filter->set_luma_temp;
  if (!filter->luma_temp)
    filter->luma_temp = DEFAULT_LUMA_TEMP * filter->luma_spatial /
        DEFAULT_LUMA_SPATIAL;

  filter->chroma_temp = filter->set_chroma_temp;
  if (!filter->chroma_temp)
    filter->chroma_temp = filter->luma_temp * filter->chroma_spatial /
        filter->luma_spatial;

}

static void
gst_hqdn3d_update_props (GstHqdn3d * filter)
{
  void (*calc_coefs) (gint *, double);

  /* are we hqdn3d or denoise3d */
  if (filter->quality) {
    calc_coefs = gst_hqdn3d_precalc_coefs;
  } else {
    calc_coefs = gst_denoise3d_precalc_coefs;
  }

  /* and pre-calculated based on the configured properties */
  gst_hqdn3d_set_params (filter);
  calc_coefs (filter->coefs[COEFS_LS_INDEX], filter->luma_spatial);
  calc_coefs (filter->coefs[COEFS_CS_INDEX], filter->chroma_spatial);
  calc_coefs (filter->coefs[COEFS_LT_INDEX], filter->luma_temp);
  calc_coefs (filter->coefs[COEFS_CT_INDEX], filter->chroma_temp);
}

static gboolean
gst_hqdn3d_start (GstBaseTransform * btrans)
{
  GstHqdn3d *filter = GST_HQDN3D (btrans);

  /* let's not change this mid-flight */
  if (filter->quality) {
    filter->denoise_func = gst_hqdn3d_denoise;
  } else {
    filter->denoise_func = gst_denoise3d_denoise;
  }

  gst_hqdn3d_update_props (filter);

  GST_DEBUG_OBJECT (filter,
      "starting with luma=%f, luma-temp=%f, chroma=%f, chroma-temp=%f",
      filter->luma_spatial, filter->luma_temp,
      filter->chroma_spatial, filter->chroma_temp);

  return TRUE;
}

static gboolean
gst_hqdn3d_stop (GstBaseTransform * btrans)
{
  GstHqdn3d *filter = GST_HQDN3D (btrans);

  gst_hqdn3d_free (filter);

  return TRUE;
}

static void
gst_hqdn3d_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstHqdn3d *src;
  gboolean changed = FALSE;
  gdouble tmp;

  g_return_if_fail (GST_IS_HQDN3D (object));
  src = GST_HQDN3D (object);

  switch (prop_id) {
    case PROP_LUMA_SPATIAL:
      tmp = g_value_get_double (value);
      if (tmp != src->set_luma_spatial) {
        src->set_luma_spatial = tmp;
        changed = TRUE;
      }
      break;
    case PROP_CHROMA_SPATIAL:
      tmp = g_value_get_double (value);
      if (tmp != src->set_chroma_spatial) {
        src->set_chroma_spatial = tmp;
        changed = TRUE;
      }
      break;
    case PROP_LUMA_TEMP:
      tmp = g_value_get_double (value);
      if (tmp != src->set_luma_temp) {
        src->set_luma_temp = tmp;
        changed = TRUE;
      }
      break;
    case PROP_CHROMA_TEMP:
      tmp = g_value_get_double (value);
      if (tmp != src->set_chroma_temp) {
        src->set_chroma_temp = tmp;
        changed = TRUE;
      }
      break;
    case PROP_HIGH_QUALITY:
      src->quality = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  if (changed)
    gst_hqdn3d_update_props (src);
}

static void
gst_hqdn3d_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstHqdn3d *src;

  g_return_if_fail (GST_IS_HQDN3D (object));
  src = GST_HQDN3D (object);

  switch (prop_id) {
    case PROP_LUMA_SPATIAL:
      g_value_set_double (value, src->luma_spatial);
      break;
    case PROP_CHROMA_SPATIAL:
      g_value_set_double (value, src->chroma_spatial);
      break;
    case PROP_LUMA_TEMP:
      g_value_set_double (value, src->luma_temp);
      break;
    case PROP_CHROMA_TEMP:
      g_value_set_double (value, src->chroma_temp);
      break;
    case PROP_HIGH_QUALITY:
      g_value_set_boolean (value, src->quality);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
