/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-shift
 *
 * <refsect2>
 * <para>
 * Delay passing buffers by amount indicated by
 * <link linkend="GstShift--delay">delay</link> (in microseconds).
 * That is, add <link linkend="GstShift--delay">delay</link>
 * to passing buffers' timestamp.
 * </para>
 * <para>
 * Passing segment events are also shifted accordingly.
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-entrans.h"

#include <string.h>


#define GST_TYPE_SHIFT \
  (gst_shift_get_type())
#define GST_SHIFT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_SHIFT,GstShift))
#define GST_SHIFT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_SHIFT,GstShiftClass))
#define GST_IS_SHIFT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_SHIFT))
#define GST_IS_SHIFT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_SHIFT))


typedef struct _GstShift GstShift;
typedef struct _GstShiftClass GstShiftClass;

struct _GstShift
{
  GstBaseTransform parent;

  gint width, height;

  /* properties */
  guint delay;
};


struct _GstShiftClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (shift_debug);
#define GST_CAT_DEFAULT shift_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_DELAY
      /* FILL ME */
};

#define DEFAULT_DELAY    0

static GstStaticPadTemplate gst_shift_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static GstStaticPadTemplate gst_shift_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static gboolean gst_shift_event (GstBaseTransform * btrans, GstEvent * event);

static GstFlowReturn gst_shift_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_shift_start (GstBaseTransform * btrans);
static gboolean gst_shift_stop (GstBaseTransform * btrans);

static void gst_shift_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_shift_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstShift, gst_shift, GstBaseTransform,
    GST_TYPE_BASE_TRANSFORM);

static void
gst_shift_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Shift",
      "Generic", "Shift buffers in time, thus re-syncing",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_shift_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_shift_src_template));
}

static void
gst_shift_class_init (GstShiftClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (shift_debug, "shift", 0, "shift");

  gobject_class->set_property = gst_shift_set_property;
  gobject_class->get_property = gst_shift_get_property;

  g_object_class_install_property (gobject_class, PROP_DELAY,
      g_param_spec_uint ("delay", "Delay",
          "Microseconds to delay stream by adding it to buffer timestamps",
          0, G_MAXUINT, DEFAULT_DELAY, G_PARAM_READWRITE));

  /* not setting passthrough assures (metadata) writable buffers */
  trans_class->event = GST_DEBUG_FUNCPTR (gst_shift_event);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_shift_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_shift_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_shift_stop);
}

static void
gst_shift_init (GstShift * filter, GstShiftClass * g_class)
{
  filter->delay = DEFAULT_DELAY;
}

static gboolean
gst_shift_event (GstBaseTransform * btrans, GstEvent * event)
{
  gboolean ret = TRUE;
  GstShift *filter;

  filter = GST_SHIFT (btrans);

  /* as in identity element */
  if (GST_EVENT_TYPE (event) == GST_EVENT_NEWSEGMENT) {
    GstEvent *news;
    GstFormat format;
    gboolean update;
    gdouble rate;
    gint64 start, stop, pos;

    gst_event_parse_new_segment (event, &update, &rate, &format,
        &start, &stop, &pos);

    /* shift the segment times */
    if (GST_CLOCK_TIME_IS_VALID (start))
      start += filter->delay;
    if (GST_CLOCK_TIME_IS_VALID (stop))
      stop += filter->delay;
    if (GST_CLOCK_TIME_IS_VALID (pos))
      pos += filter->delay;
    news = gst_event_new_new_segment (update, rate, format, start, stop, pos);

    if (!(gst_pad_event_default (btrans->sinkpad, news)))
      return FALSE;
  }

  GST_BASE_TRANSFORM_CLASS (parent_class)->event (btrans, event);

  if (GST_EVENT_TYPE (event) == GST_EVENT_NEWSEGMENT) {
    /* eat up segments */
    ret = FALSE;
  }

  return ret;
}

static GstFlowReturn
gst_shift_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstShift *filter;

  filter = GST_SHIFT (btrans);

  GST_BUFFER_TIMESTAMP (in) += filter->delay * GST_USECOND;

  return GST_FLOW_OK;
}


static gboolean
gst_shift_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_shift_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_shift_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstShift *src;

  g_return_if_fail (GST_IS_SHIFT (object));
  src = GST_SHIFT (object);

  switch (prop_id) {
    case PROP_DELAY:
      src->delay = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_shift_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstShift *src;

  g_return_if_fail (GST_IS_SHIFT (object));
  src = GST_SHIFT (object);

  switch (prop_id) {
    case PROP_DELAY:
      g_value_set_uint (value, src->delay);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
