//*****************************************************************************
//                               PrcNgSpice.cpp                               *
//                              ----------------                              *
//  Started     : 07/05/2004                                                  *
//  Last Update : 09/01/2008                                                  *
//  Copyright   : (C) 2004 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "process/PrcNgSpice.hpp"

//*****************************************************************************
// Constructor.

PrcNgSpice::PrcNgSpice( void ) : PrcSimrBase( )
{
  // Set the simulation engine type
  m_eSimrType = eSIMR_NGSPICE;

  // Set the simulator binary file name if it can be found
  bSetBinary( wxT("ngspice") );
}

//*****************************************************************************
// Destructor.

PrcNgSpice::~PrcNgSpice( )
{
}

//*****************************************************************************
// Parse a OP command and put the data in a simulation object.
//
// Argument List :
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdOP( wxString & rosCmd, Simulation & roSim )
{
  // ??? Not yet implemented

  return( FALSE );
}

//*****************************************************************************
// Parse a IC command and put the data in a simulation object.
//
// Argument List :
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdIC( wxString & rosCmd, Simulation & roSim )
{
  // ??? Not yet implemented

  return( FALSE );
}

//*****************************************************************************
// Parse a DC command and put the data in a simulation object.
//
// Argument List :
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdDC( wxString & rosCmd, Simulation & roSim )
{
  bool          bRtnValue=TRUE;
  CmdNgSpiceDC  tCmd_DC;
  wxString      os1;

  // Parse the DC command
  tCmd_DC = rosCmd;
  if( ! tCmd_DC.bParse( ) )                      return( FALSE );

  // Set the sweep type: source or temperature
  if( tCmd_DC.m_osSource == wxT("TEMP") )
    roSim.bSetSwpScale( 1 );
  else
  {
    os1 = roSim.rosGetCpnt( tCmd_DC.m_osSource );
    if( !os1.IsEmpty( ) && roSim.bSetSigSrc( os1 ) )
         roSim.bSetSwpScale( 0 );
    else                                         bRtnValue = FALSE;
  }

  // Set the sweep start, stop and step values
  if( ! roSim.bSetSwpStart( tCmd_DC.m_fStart ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop ( tCmd_DC.m_fStop  ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep ( tCmd_DC.m_fStep  ) ) bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Parse a AC command and put the data in a simulation object.
//
// Argument List :
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdAC( wxString & rosCmd, Simulation & roSim )
{
  bool          bRtnValue=TRUE;
  CmdNgSpiceAC  tCmd_AC;

  // Parse the AC command
  tCmd_AC = rosCmd;
  if( ! tCmd_AC.bParse( ) )                            return( FALSE );

  // Extract the sweep start, stop and step values
  if( ! roSim.bSetSwpStart( tCmd_AC.m_fStart ) )       bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop ( tCmd_AC.m_fStop  ) )       bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep ( tCmd_AC.m_fStep  ) )       bRtnValue = FALSE;

  // Set the sweep scale type
  if( ! roSim.bSetSwpScale( (int) tCmd_AC.m_eScale ) ) bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Parse a TRANSIENT command and put the data in a simulation object.
//
// Argument List :
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdTR( wxString & rosCmd, Simulation & roSim )
{
  bool          bRtnValue=TRUE;
  CmdNgSpiceTR  tCmd_TR;

  // Parse the TR command
  tCmd_TR = rosCmd;
  if( ! tCmd_TR.bParse( ) )                            return( FALSE );

  // Extract the sweep start, stop and step values
  if( ! roSim.bSetSwpStart( tCmd_TR.m_fStart ) )       bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop ( tCmd_TR.m_fStop  ) )       bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep ( tCmd_TR.m_fStep  ) )       bRtnValue = FALSE;

  // Set the initial conditions
  if( ! roSim.bSetSwpScale( (int) tCmd_TR.m_eInitC ) ) bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Parse a PRINT command and put the data in a simulation object.
//
// Argument List :
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdPR( wxString & rosCmd, Simulation & roSim )
{
  bool          bRtnValue=TRUE;
  CmdNgSpicePR  tCmd_PR;
  wxString      os1;
  size_t        sz1;

  // Parse the PR command
  tCmd_PR = rosCmd;
  if( ! tCmd_PR.bParse( ) ) return( FALSE );

  // Extract the analysis type
  roSim.bSetAnaType( tCmd_PR.m_eAnaType );

  // Extract the parameter specifier
  for( sz1=(size_t)ePAR_FST; sz1<=(size_t)ePAR_LST; sz1++ )
    roSim.bSetOutPar( (eParType) sz1, tCmd_PR.m_bParms[ sz1 ] );

  // Extract the complex part if the analysis type is AC
  if( roSim.eGetAnaType( ) == eANA_AC )
  {
    for( sz1=(size_t)eCPX_FST; sz1<=(size_t)eCPX_LST; sz1++ )
      roSim.bSetOutCpx( (eCpxType) sz1, tCmd_PR.m_bCpxPrts[ sz1 ] );
  }

  // Extract the test nodes
  for( sz1=0; sz1<tCmd_PR.m_osaNodes.GetCount( ); sz1++ )
    if( ! roSim.bAddTstNode( tCmd_PR.m_osaNodes[ sz1 ] ) )
      bRtnValue = FALSE;

  // Extract test components
  for( sz1=0; sz1<tCmd_PR.m_osaCpnts.GetCount( ); sz1++ )
  {
    os1 = tCmd_PR.m_osaCpnts[ sz1 ];

    if( bMatchCpnt( roSim, os1 ) )
      if( ! roSim.bAddTstCpnt( os1 ) )
        continue;

    bRtnValue = FALSE;
  }

  return( bRtnValue );
}

//*****************************************************************************
// Match a component to a set of node labels. Component voltages cannot be
// specified using the component label in the NG-Spice print statement. The two
// node labels connected to the component must be specified. When parsing the
// print statement the component label must be derived from the node labels.
// This may not work where components are connected in parallel. The nodes are
// specified in the form "<Node1>,<Node2>" eg. "1,2".
//
// Argument List :
//   rosToNodes - The nodes to match, also used to return the component label
//
// Return Values :
//   TRUE  - Success (a component with the specified nodes was found)
//   FALSE - Failure

bool  PrcNgSpice::bMatchCpnt( Simulation & roSim, wxString & rosToNodes )
{
  const wxArrayString & roasCpnts=roSim.roasGetCpnts( );
  wxArrayString  oasNodes;
  Component  oCpnt;
  wxString   os1;
  size_t     sz1;

  // Argument validity checks
  if( rosToNodes.IsEmpty( ) )            return( FALSE );
  if( roasCpnts.GetCount( ) <= 0 )       return( FALSE );
  if( rosToNodes.Freq( wxT(',') ) != 1 ) return( FALSE );

  // Extract the node labels
  oasNodes.Add( rosToNodes.BeforeFirst( wxT(',') ) );
  oasNodes.Add( rosToNodes.AfterLast( wxT(',') ) );

  // Attempt to match the nodes with a component
  for( sz1=0; sz1<roasCpnts.GetCount( ); sz1++ )
  {
    os1 = roasCpnts.Item( sz1 );
    oCpnt.bSetDefn( os1 );
    if( oCpnt.roasGetNodes( ) == oasNodes ) break;
    oCpnt.Clear( );
  }

  // Was a match found?
  if( ! oCpnt.bIsOk( ) )                 return( FALSE );

  // A match was found
  rosToNodes = oCpnt.rosGetName( );
  return( TRUE );
}

//*****************************************************************************
// Create the simulator command sequence for a DC analysis.
//
// Argument List :
//   roSim - Simulation object

void  PrcNgSpice::MakeCmdsDC( Simulation & roSim )
{
  CmdNgSpicePR  tCmd_PR;
  CmdNgSpiceDC  tCmd_DC;
  wxString      osCmdWIDTH;
  int           i1;

  // Clear the field list
  m_osFieldLst.Empty( );

  // Create the DC command
  tCmd_DC.m_fStart = roSim.fGetSwpStart( );
  tCmd_DC.m_fStop  = roSim.fGetSwpStop ( );
  tCmd_DC.m_fStep  = roSim.fGetSwpStep ( );
  if( roSim.iGetSwpScale( ) == 0 )
       tCmd_DC.m_osSource = roSim.rosGetSigSrc( ).BeforeFirst( wxT(' ') );
  else tCmd_DC.m_osSource = wxT("TEMP");
  tCmd_DC.bFormat( );

  // Create the PRINT command
  tCmd_PR.m_eAnaType = eANA_DC;
  tCmd_PR.m_osaNodes = roSim.roasGetTstNodes( );
  tCmd_PR.m_osaCpnts = roSim.roasGetTstCpnts( );
  for( i1=(int) ePAR_FST; i1<=(int) ePAR_LST; i1++ )
    tCmd_PR.m_bParms  [ i1 ] = roSim.bGetOutPar( (eParType) i1 );
  for( i1=(int) eCPX_FST; i1<=(int) eCPX_LST; i1++ )
    tCmd_PR.m_bCpxPrts[ i1 ] = roSim.bGetOutCpx( (eCpxType) i1 );
  tCmd_PR.bFormat( );
  MakeFieldLst( tCmd_PR, roSim );

  // Create the WIDTH command
  osCmdWIDTH = rosMakeCmdWIDTH( tCmd_PR );

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdWIDTH );
  roSim.bAddSimCmd( tCmd_PR );
  roSim.bAddSimCmd( tCmd_DC );
}

//*****************************************************************************
// Create the simulator command sequence for a AC analysis.
//
// Argument List :
//   roSim - Simulation object

void  PrcNgSpice::MakeCmdsAC( Simulation & roSim )
{
  CmdNgSpicePR  tCmd_PR;
  CmdNgSpiceAC  tCmd_AC;
  wxString      osCmdWIDTH;
  wxString      osCmdOP;
  int           i1;

  // Clear the field list
  m_osFieldLst.Empty( );

  // Create the AC command
  tCmd_AC.m_fStart = roSim.fGetSwpStart( );
  tCmd_AC.m_fStop  = roSim.fGetSwpStop( );
  tCmd_AC.m_fStep  = roSim.fGetSwpStep( );
  tCmd_AC.m_eScale = (eScaleType) roSim.iGetSwpScale( );
  tCmd_AC.bFormat( );

  // Create the PRINT command
  tCmd_PR.m_eAnaType = eANA_AC;
  tCmd_PR.m_osaNodes = roSim.roasGetTstNodes( );
  tCmd_PR.m_osaCpnts = roSim.roasGetTstCpnts( );
  for( i1=(int) ePAR_FST; i1<=(int) ePAR_LST; i1++ )
    tCmd_PR.m_bParms  [ i1 ] = roSim.bGetOutPar( (eParType) i1 );
  for( i1=(int) eCPX_FST; i1<=(int) eCPX_LST; i1++ )
    tCmd_PR.m_bCpxPrts[ i1 ] = roSim.bGetOutCpx( (eCpxType) i1 );
  tCmd_PR.bFormat( );
  MakeFieldLst( tCmd_PR, roSim );

  // Create the WIDTH command
  osCmdWIDTH = rosMakeCmdWIDTH( tCmd_PR );

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdWIDTH );
  roSim.bAddSimCmd( tCmd_PR );
  roSim.bAddSimCmd( osCmdOP );
  roSim.bAddSimCmd( tCmd_AC );
}

//*****************************************************************************
// Create the simulator command sequence for a transient analysis.
//
// Argument List :
//   roSim - Simulation object

void  PrcNgSpice::MakeCmdsTR( Simulation & roSim )
{
  CmdNgSpicePR  tCmd_PR;
  CmdNgSpiceTR  tCmd_TR;
  wxString      osCmdIC;
  wxString      osCmdWIDTH;
  int           i1;
  size_t        sz1;

  // Clear the field list
  m_osFieldLst.Empty( );

  // Create initial conditions command
  i1 = roSim.iGetSwpScale( );
  if( i1==eINITC_COLD || i1==eINITC_UICS )
  {
    const wxArrayString & roas1 = roSim.roasGetNodeLbls( );
    if( roas1.GetCount( ) > 0 )
    {
      osCmdIC << wxT(".IC");
      for( sz1=0; sz1<roas1.GetCount( ); sz1++ )
        osCmdIC << wxT(" V(") << roas1.Item( sz1 ) << wxT(")=0.0");
    }
  }

  // Create the TR command
  tCmd_TR.m_fStart = roSim.fGetSwpStart( );
  tCmd_TR.m_fStop  = roSim.fGetSwpStop( );
  tCmd_TR.m_fStep  = roSim.fGetSwpStep( );
  tCmd_TR.m_eInitC = (eInitCType) roSim.iGetSwpScale( );
  tCmd_TR.bFormat( );

  // Create the PRINT commands
  tCmd_PR.m_eAnaType = eANA_TR;
  tCmd_PR.m_osaNodes = roSim.roasGetTstNodes( );
  tCmd_PR.m_osaCpnts = roSim.roasGetTstCpnts( );
  for( i1=(int) ePAR_FST; i1<=(int) ePAR_LST; i1++ )
    tCmd_PR.m_bParms  [ i1 ] = roSim.bGetOutPar( (eParType) i1 );
  for( i1=(int) eCPX_FST; i1<=(int) eCPX_LST; i1++ )
    tCmd_PR.m_bCpxPrts[ i1 ] = roSim.bGetOutCpx( (eCpxType) i1 );
  tCmd_PR.bFormat( );
  MakeFieldLst( tCmd_PR, roSim );

  // Create the WIDTH command
  osCmdWIDTH = rosMakeCmdWIDTH( tCmd_PR );

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdIC );
  roSim.bAddSimCmd( osCmdWIDTH );
  roSim.bAddSimCmd( tCmd_PR );
  roSim.bAddSimCmd( tCmd_TR );
}

//*****************************************************************************
// Convert a list of component labels to the corresponding list of node pairs.
//
// Argument List:
//   rtCmd_PR - A PRINT command structure
//   roSim    - A Simulation object

void  PrcNgSpice::MakeFieldLst( CmdNgSpicePR & rtCmd_PR, Simulation & roSim )
{
  m_osFieldLst = rtCmd_PR;
  m_osFieldLst = m_osFieldLst.AfterFirst( wxT(' ') );
  m_osFieldLst = m_osFieldLst.AfterFirst( wxT(' ') );

  Convert2Nodes( rtCmd_PR.m_osaCpnts, roSim );

  rtCmd_PR.bFormat( );
}

//*****************************************************************************
// Convert a list of component labels to the corresponding list of node pairs.
//
// Argument List:
//   roasCpnts - A list of component labels
//   roSim     - A Simulation object

void  PrcNgSpice::Convert2Nodes( wxArrayString & roasCpnts, Simulation & roSim )
{
  Component  oCpnt;
  wxString   os1;
  size_t     sz1;

  for( sz1=0; sz1<roasCpnts.GetCount( ); sz1++ )
  {
    os1 = roasCpnts.Item( sz1 );
    os1 = roSim.rosGetCpnt( os1.c_str( ) );
    if( os1.IsEmpty( ) )                         continue;
    if( ! oCpnt.bSetDefn( os1.c_str( ) ) )       continue;
    if( oCpnt.roasGetNodes( ).GetCount( ) != 2 ) continue;
    os1.Empty( );
    os1 << oCpnt.roasGetNodes( ).Item( 0 ) << wxT(',')
        << oCpnt.roasGetNodes( ).Item( 1 );
    roasCpnts.Item( sz1 ) = os1;
  }
}

//*****************************************************************************
// Generate the simulator commands to be executed based on the simulation
// specifications and the NetList description.
//
// Argument List :
//   roSim - The net list and simulation specification
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bMakeCmds( Simulation & roSim )
{
  switch( roSim.eGetAnaType( ) )
  {
    case eANA_DC : MakeCmdsDC( roSim ); break;
    case eANA_AC : MakeCmdsAC( roSim ); break;
    case eANA_TR : MakeCmdsTR( roSim ); break;
    default :      return( FALSE );
  }

  return( ! roSim.roasGetSimCmds( ).IsEmpty( ) );
}

//*****************************************************************************
// Create a WIDTH command based on a PRINT command.
// By default raw output from NG-Spice uses an 80 column output width which
// allows about two dependant variables to be displayed. To display more
// information a .WIDTH statement must be added to increase the number columns
// in the output.
//
// Argument List :
//   rosCmdPR - The Print command
//
// Return Values :
//   Success - A .WIDTH command
//   Failure - An empty string

wxString & PrcNgSpice::rosMakeCmdWIDTH( wxString & rosCmdPR )
{
  static  wxString  osCmdWIDTH;
  wxStringTokenizer  ostk1;
  wxString  os1;
  int  i1=0;

  osCmdWIDTH.Empty( );

  ostk1 = rosCmdPR;
  if( ostk1.GetNextToken( ).StartsWith( wxT(".PR") ) )
  {
    os1 = ostk1.GetNextToken( );

    if(      os1.StartsWith( wxT("DC") ) )
      i1 = 8 + 16 + ostk1.CountTokens( ) * 16;
    else if( os1.StartsWith( wxT("AC") ) )
      i1 = 8 + 2 * 16 + ostk1.CountTokens( ) * 16;
    else if( os1.StartsWith( wxT("TR") ) )
      i1 = 8 + 16 + ostk1.CountTokens( ) * 16;

    if( i1 > 80 ) osCmdWIDTH << wxT(".WIDTH OUT=") << i1;
  }

  return( osCmdWIDTH );
}

//*****************************************************************************
// Format the contents of the results file so that gWave can read it ie. a
// header line at the top containing parameter names followed by the columns of
// data.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtResults( void )
{
  bool  bRtnValue;

  m_osErrMsg.Empty( );

  // Attempt to open the results file
  if( ! roGetResultsFile( ).FileExists( ) )
  {
    m_osErrMsg << wxT("Results file doesn't exist : \n\n")
               << roGetResultsFile( ).GetFullPath( );
    return( FALSE );
  }
  if( ! m_oFileResults.Open( roGetResultsFile( ).GetFullPath( ) ) )
  {
    m_osErrMsg << wxT("Results file couldn't be opened : \n\n")
               << roGetResultsFile( ).GetFullPath( );
    return( FALSE );
  }

  // Format the simulation results
  bRtnValue = bFmtGeneric( );

  // Need last line to end with a '\n' or text controls will not load it
  if( m_oFileResults.GetLastLine( ).Last( ) != wxT('\n') )
    m_oFileResults.GetLastLine( ) << wxT('\n');

  m_oFileResults.Write( );  // Save the changes to disk
  m_oFileResults.Close( );  // Close the file

  return( bRtnValue );
}

//*****************************************************************************
// This function is a generic results file formatter which works in most
// circumstance.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtGeneric( void )
{
  wxString  os1;
  size_t    sz1, sz2;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  // Find the beginning of the data area (ie. the 1st line starting with "Index")
  for( sz1=0; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = m_oFileResults.GetLine( sz1 );
    if( os1.StartsWith( wxT("Index") ) ) break;
  }
  if( sz1 >= (m_oFileResults.GetLineCount( )-1) )
  {
    m_osErrMsg << wxT("Couldn't find the data section in the results file.");
    return( FALSE );
  }

  // Delete everything before the data area
  sz2 = sz1;
  for( sz1=0; sz1<sz2; sz1++ )             m_oFileResults.RemoveLine( 0 );
  if( m_oFileResults.GetLineCount( ) > 0 ) m_oFileResults.RemoveLine( 1 );

  // Format the column header line
  bFmtColumnHdr( );

  // Delete lines other than data lines
  for( sz1=1; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = m_oFileResults.GetLine( sz1 );
    if( ! wxIsdigit( os1.GetChar( 0 ) ) )
    {
      m_oFileResults.RemoveLine( sz1 );
      sz1--;
    }
  }

  // Format the data lines
  bFmtDataLines( );

  return( TRUE );
}

//*****************************************************************************
// Format the column header in the results file.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtColumnHdr( void )
{
  wxStringTokenizer  ostk1;
  wxString  osLine, os1;
  int  i1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  // Get the column header line
  ostk1 = m_oFileResults.GetFirstLine( );
  if( ostk1.CountTokens( ) < 3 )     return( FALSE );

  // Dispose of the first field (ie. "Index")
  ostk1.GetNextToken( );

  // Extract the operator field and replace the NG-Spice field identifiers
  os1 << ostk1.GetNextToken( ) << wxT(' ') << m_osFieldLst;
  ostk1.SetString( os1 );

  // Format each field
// osLine = wxT("# "); // ??? 06/08/2005
  osLine = wxT('#'); // gWave breaks if there's a space after initial '#'
  while( ostk1.HasMoreTokens( ) )
  {
    if( osLine.Length( ) > 2 )
    { // Pad the column with spaces to the required width
      i1 = NGSPICE_COL_WD - (osLine.Length( ) % NGSPICE_COL_WD) + 1;
      osLine.Append( wxT(' '), i1 );
    }

    // Add the next label to the line
    osLine << ostk1.GetNextToken( );
  }

  osLine.Trim( ); // Remove trailing space characters

  m_oFileResults.GetFirstLine( ) = osLine;

  return( TRUE );
}

//*****************************************************************************
// Format a data lines in the results file.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtDataLines( void )
{
  wxStringTokenizer  ostk1;
  wxString  osLine, os1;
  size_t    sz1;
  double    f1;
  int       i1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  for( sz1=1; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    // Get a line of data
    osLine = m_oFileResults.GetLine( sz1 );

    // Tokenize the string
    ostk1.SetString( osLine );
    if( ostk1.CountTokens( ) < 3 )   return( FALSE );

    // Dispose of the first field
    ostk1.GetNextToken( );

    // Format each field
    osLine.Empty( );
    while( ostk1.HasMoreTokens( ) )
    {
      // Add the next parameter to the line
      os1 = ostk1.GetNextToken( );
      if( os1.Last( ) == wxT(',') )
      {
        os1.Last( ) = wxT(' ');
        ostk1.GetNextToken( );
      }
      if( ! ConvertType::bStrToFlt( os1, &f1 ) ) f1  =      -9.999e+99;
      if( ! ConvertType::bFltToStr( f1, os1 ) )  os1 = wxT("-9.999e+99");
      osLine << os1;

      // Pad the column with spaces to the required width
      i1 = NGSPICE_COL_WD - (osLine.Length( ) % NGSPICE_COL_WD);
      osLine.Append( wxT(' '), i1 );
    }

    m_oFileResults.GetLine( sz1 ) = osLine;

    osLine.Trim( ); // Remove trailing space characters
  }

  return( TRUE );
}

//*****************************************************************************
// Format any phase values contained in the results file (this is only
// pertinent for AC analysis). It involves removing the discontinuity at 360
// degree in the data.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtPhaseData( void )
{
  // ??? (02/02/2006) Not yet implemented

  return( TRUE );
}

//*****************************************************************************
// Parse the simulation commands strings found in a Simulation object and load
// the values back into the Simulation object.
//
// Argument List :
//   roSim - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseSim( Simulation & roSim )
{
  wxString  os1;
  size_t    sz1;

  // Set the default state of some of the simulation object attributes
  roSim.bSetOutPar( ePAR_VLT,   TRUE );
  roSim.bSetOutCpx( eCPX_MAG,   TRUE );
  roSim.bSetOutCpx( eCPX_MAGDB, TRUE );

  // Go through the simulation commands and attempt to parse them
  const wxArrayString & roasCmds=roSim.roasGetSimCmds( );
  for( sz1=0; sz1<roasCmds.GetCount( ); sz1++ )
  {
    os1 = roasCmds.Item( sz1 );

    if( bParseCmdOP( os1, roSim ) ) continue;
    if( bParseCmdIC( os1, roSim ) ) continue;
    if( bParseCmdDC( os1, roSim ) ) continue;
    if( bParseCmdAC( os1, roSim ) ) continue;
    if( bParseCmdTR( os1, roSim ) ) continue;
    if( bParseCmdPR( os1, roSim ) ) continue;
  }

  return( TRUE );
}

//*****************************************************************************
// Create the simulation commands from the values found in a Simulation object
// and load the command string back into the Simulation object.
//
// Argument List :
//   roSim - The simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bMakeSim( Simulation & roSim )
{
  wxFileName  ofn1;
  wxString  os1;
  int       i1;

  // Clear error attributes
  m_osErrMsg.Empty( );

  // Only execute simulation if it isn't already running
  if( bIsExec( ) )
  {
    m_osErrMsg = wxT("Simulation already running");
    return( FALSE );
  }

  // Create the simulator commands
  if( ! bMakeCmds( roSim ) )
  {
    m_osErrMsg = wxT("Couldn't create simulator commands");
    return( FALSE );
  }

  // Save circuit description and SPICE commands to file
  ofn1 = roSim.rofnGetLoadFile( );
  if( ! roSim.bSaveFile( ofn1.GetFullPath( ) ) )
  {
    m_osErrMsg = wxT("Simulation couldn't be saved to file");
    return( FALSE );
  }

  // Create the results file name
  os1 = roSim.rofnGetLoadFile( ).GetPath( ) + wxT('/')
      + roSim.rofnGetLoadFile( ).GetName( );
  i1 = os1.Find( wxT(".gspiceui") );
  if( i1 > 0 ) os1 = os1.Truncate( (size_t) i1 );
  os1 << wxT('.') << rofnGetBinary( ).GetName( );
  switch( roSim.eGetAnaType( ) )
  {
    case eANA_DC : os1 << wxT(".dc"); break;
    case eANA_AC : os1 << wxT(".ac"); break;
    case eANA_DI : os1 << wxT(".di"); break;
    case eANA_NO : os1 << wxT(".no"); break;
    case eANA_PZ : os1 << wxT(".pz"); break;
    case eANA_SE : os1 << wxT(".se"); break;
    case eANA_TR : os1 << wxT(".tr"); break;
    case eANA_TF : os1 << wxT(".tf"); break;
    default :      return( FALSE );
  }
  if( ! bSetResultsFile( os1.c_str( ) ) )
  {
    m_osErrMsg = wxT("Couldn't set results file name");
    return( FALSE );
  }

  // Construct the command line to execute the simulation
  os1 = wxT("-n -b ") + roSim.rofnGetSaveFile( ).GetFullPath( );
  if( ! bSetArgLst( os1.c_str( ) ) )
  {
    m_osErrMsg = wxT("Couldn't set argument list");
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
