//*****************************************************************************
//                               PnlGnuCapDC.cpp                              *
//                              -----------------                             *
// Started     : 18/08/2003                                                   *
// Last Update : 05/10/2007                                                   *
// Copyright   : (C) 2003 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapDC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapDC, PnlAnaBase )

  EVT_CHOICE  ( ID_CHO_SIGSRCCPNT, PnlGnuCapDC::OnSigSrcCpnt )
  EVT_RADIOBOX( ID_RBX_STEPSCALE,  PnlGnuCapDC::OnStepScale  )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlGnuCapDC::PnlGnuCapDC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrType( eSIMR_GNUCAP );
  bSetAnaType ( eANA_DC );

  Create( );        // Create the analysis panel
  InitSwpUnits( );  // Initialize the sweep units
  bClear( );        // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlGnuCapDC::~PnlGnuCapDC( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlGnuCapDC::Create( void )
{
  int  ix, iy;

  // Set the sweep parameters labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 201 );
  m_oSbxSwpPars.SetLabel( wxT(" DC Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Value") );
  m_oPnlStop   .bSetName( wxT("Stop Value") );

  // Create and add the scale radio butwons
  wxString  osScale[ 3 ] = { wxT("Lin  "), wxT("Log "), wxT("Dec ") };
  m_oRbxStepScale.Create( this, ID_RBX_STEPSCALE, wxT(" Step Scale "),
                          wxPoint( 13, 123 ), wxDefaultSize, 3, osScale, 3 );
  bSetStepScale( eSCALE_LIN );

  // Create and add input source
  new wxStaticText( this, ID_UNUSED, wxT("Signal Source"),
                    wxPoint( 18, 176 ) );
  m_oChoSigSrcCpnt.Create( this, ID_CHO_SIGSRCCPNT,
                           wxPoint( 144, 171 ), wxSize( 110, 29 ) );

  PnlAnaBase::CreateTemp( );  // Create the analysis temperature controls
}

//*****************************************************************************
// Initialize the sweep parameter units.

void  PnlGnuCapDC::InitSwpUnits( void )
{
  wxString              os1;
  ChoUnits::eUnitsType  eUnits;

  // Determine the sweep parameters
  os1 = m_oChoSigSrcCpnt.GetStringSelection( );
  if( os1.IsEmpty( ) ) os1 = wxT('?');
  switch( wxToupper( os1.GetChar( 0 ) ) )
  {
    case wxT('C') :  // Units of capacitance
      os1 = wxT("Capacitance");
      eUnits = ChoUnits::eUNITS_CAP;
      break;

    case wxT('L') :  // Units of inductance
      os1 = wxT("Inductance");
      eUnits = ChoUnits::eUNITS_IND;
      break;

    case wxT('R') :  // Units of resistance
      os1 = wxT("Resistance");
      eUnits = ChoUnits::eUNITS_RES;
      break;

    case wxT('I') :  // Units of current
      os1 = wxT("Current");
      eUnits = ChoUnits::eUNITS_CURR;
      break;

    case wxT('V') :  // Units of voltage
      os1 = wxT("Voltage");
      eUnits = ChoUnits::eUNITS_VOLT;
      break;

    default :        // No units
      os1 = wxT("Value");
      eUnits = ChoUnits::eUNITS_NONE;
  }

  // Set the sweep parameters
  m_oPnlStart.bSetName( wxString( wxT("Start ") ) + os1 );
  m_oPnlStop .bSetName( wxString( wxT("Stop ")  ) + os1 );
  m_oPnlStart.bSetUnitsType( eUnits );
  m_oPnlStop .bSetUnitsType( eUnits );
  m_oPnlStep .bSetUnitsType( eUnits );
}

//*****************************************************************************
// Initialize the step scale.

void  PnlGnuCapDC::InitStepScale( void )
{
  switch( m_oRbxStepScale.GetSelection( ) )
  {
    case eSCALE_LIN :  // Increment the step value by adding constant value
      m_oPnlStep.bSetName( wxT("Step Increment") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_FLT );
      m_oPnlStep.bSetParms( 10.0, 0.0, 1000.0, 0.01, 100.0 );
      m_oPnlStep.bShowUnits( TRUE );
      break;

    case eSCALE_LOG :  // Increase the step by a constant multiplier
      m_oPnlStep.bSetName( wxT("Step Multiplier") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_FLT );
      m_oPnlStep.bSetParms( 1.1, 0.0, 100.0, 0.01, 10.0 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    case eSCALE_DEC :  // Specify a fixed number of steps per decade
      m_oPnlStep.bSetName( wxT("Steps / Decade") );
      m_oPnlStep.bSetVarType( SpinCtrl::eVAR_INT );
      m_oPnlStep.bSetParms( 10, 1, 1000, 1, 100 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    default :
      break;
  }
}

//*****************************************************************************
// Set the state of the step scale radio box.
//
// Argument List:
//   eScale - The enumerated scale specifier
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  PnlGnuCapDC::bSetStepScale( eScaleType eScale )
{
  // Argument validity checks
  if( eScale<eSCALE_FST || eScale>eSCALE_LST )          return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxStepScale.GetCount( ) < (uint) eScale + 1 ) return( FALSE );
#else
  if( m_oRbxStepScale.GetCount( ) < (int)  eScale + 1 ) return( FALSE );
#endif

  // Set appropraite radio button
  m_oRbxStepScale.SetSelection( (int) eScale );

  // Initialize the step scale
  InitStepScale( );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapDC::bClear( void )
{
  bool  bRtnValue=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )       bRtnValue = FALSE;

  // Set default step scale type and sweep values
  if( ! bSetStepScale( eSCALE_LIN ) ) bRtnValue = FALSE;
  m_oPnlStart.bSetValue( (float)   0.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)  10.0 );

  // Set input source default values
  m_oChoSigSrcCpnt.Clear( );
  m_oChoSigSrcCpnt.Append( wxT("None") );
  m_oChoSigSrcCpnt.SetSelection( 0 );

  return( bRtnValue );
}

//*****************************************************************************
// Load information from a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapDC::bLoad( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  // Load the components into the source choice box
  LoadSigSrcCpnts( roSim, wxT("VIRLC") );

  // Don't go any further if the analysis type doesn't match
  if( roSim.eGetAnaType( ) != eGetAnaType( ) )               return( TRUE );

  // Set the source component
  if( ! PnlAnaBase::bSetSigSrc( roSim ) )                    bRtnValue = FALSE;

  // Initialize the sweep parameter units
  InitSwpUnits( );

  // Set the step scale
  if( ! bSetStepScale( (eScaleType) roSim.iGetSwpScale() ) ) bRtnValue = FALSE;

  // Set the parameters to derive
  m_oCbxVoltage.SetValue( roSim.bGetOutPar( ePAR_VLT ) );
  m_oCbxCurrent.SetValue( roSim.bGetOutPar( ePAR_CUR ) );
  m_oCbxPower  .SetValue( roSim.bGetOutPar( ePAR_PWR ) );
  m_oCbxResist .SetValue( roSim.bGetOutPar( ePAR_RES ) );

  // Perform any base class load tasks
  if( ! PnlAnaBase::bLoad( roSim ) )                         bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapDC::bSave( Simulation & roSim )
{
  wxString  os1;

  // Execute base class save tasks
  PnlAnaBase::bSave( roSim );

  // Modify the component to be used as the sweep source
  if( m_oChoSigSrcCpnt.GetSelection( ) > 0 )
  {
    os1 = roSim.rosGetSigSrc( ).BeforeLast( wxT(' ') );
    os1 << wxT(' ') << roSim.rosGetSwpStart( );
    if( ! roSim.bSetSigSrc( os1 ) )
      SetErrMsg( wxT("Source component couldn't be set.") );
  }

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Step scale radio box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapDC::OnStepScale( wxCommandEvent & roEvtCmd )
{
  InitStepScale( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapDC::OnSigSrcCpnt( wxCommandEvent & roEvtCmd )
{
  InitSwpUnits( );
}

//*****************************************************************************
