#include "pos2img.h"
#include "osl/record/usi.h"
#include "osl/record/kanjiPrint.h"
#include "osl/pieceStand.h"
#include <boost/program_options.hpp>
#include <boost/filesystem.hpp>
#include <boost/regex.hpp>
#include <iostream>
#include <string>
#include <sstream>
#include <vector>
#include <ctime>
#include <cassert>

/*
 * The encoding of this file must be UTF-8 since ImageMagic requires so.
 */

std::string 
kanjiNumber(const int n)
{
  switch(n)
  {
    case 1: return "一";
    case 2: return "二";
    case 3: return "三";
    case 4: return "四";
    case 5: return "五";
    case 6: return "六";
    case 7: return "七";
    case 8: return "八";
    case 9: return "九";
    case 10: return "十";
    case 11: return "十一";
    case 12: return "十二";
    case 13: return "十三";
    case 14: return "十四";
    case 15: return "十五";
    case 16: return "十六";
    case 17: return "十七";
    case 18: return "十八";
    default: assert(false); return "";
  }
  assert(false);
  return "";
}

std::string 
kanji(const PtypeO& ptypeO)
{
  switch(getPtype(ptypeO))
  {
    case PAWN:    return "歩";
    case LANCE:   return "香";
    case KNIGHT:  return "桂";
    case SILVER:  return "銀";
    case BISHOP:  return "角";
    case ROOK:    return "飛";
    case PPAWN:   return "と";
    case PLANCE:  return "杏";
    case PKNIGHT: return "圭";
    case PSILVER: return "全";
    case PBISHOP: return "馬";
    case PROOK:   return "龍";
    case GOLD:    return "金";
    case KING:
      if (getOwner(ptypeO) == BLACK)            
        return "玉";
      else
        return "王";
    default:      assert(false); return "";
  }
  assert(false);
  return "";
}

/* ===== PieceImage ===== */

ImagePtr
PieceImage::image(const PtypeO& ptypeO) const
{
  const std::string& kanji_piece = kanji(ptypeO);
  ImagePtr piece = image(kanji_piece);
  if (getOwner(ptypeO) == WHITE)
    piece->rotate(180);
  return piece;
}

ImagePtr
PieceImage::image(const std::string& s) const
{
  // 大きな画像で描画し縮小する（文字つぶれ対策）
  const static double a_scale  = 3.0;
  const double an_image_width  = cell_width*a_scale;
  const double an_image_height = an_image_width*scale;

  ImagePtr piece(
    new Magick::Image(Magick::Geometry(an_image_width, an_image_height),
                      Magick::Color("white")));
  piece->strokeAntiAlias(true);
  piece->strokeColor("black");
  piece->strokeWidth(0.1);
  piece->font("VLGothic"); // VLGothic or DejimaMincho
  const int font_size = cell_width * a_scale / dpi * 72.0; /// 1 point = 1/72 inchi
  piece->fontPointsize(font_size);
  piece->density(Magick::Geometry(dpi, dpi));
  piece->annotate(s, Magick::CenterGravity);
  piece->resize(Magick::Geometry(cell_width*0.83, cell_height*0.83));
  return piece;
}

/* ===== BoardImage ===== */

std::string
BoardImage::blackStand() const
{
  std::ostringstream os;
  os << "▲先手　";
  for (unsigned int i=0; i<PieceStand::order.size(); ++i)
  {
    const Ptype ptype = PieceStand::order[i];
    const int count = state.countPiecesOnStand(BLACK, ptype);
    if (count > 0)
    {
      os << kanji(newPtypeO(BLACK, ptype));
      if (count > 1)
	os << kanjiNumber(count);
    }
  }
  return os.str();
}

std::string
BoardImage::whiteStand() const
{
  std::ostringstream os;
  os << "△後手　";
  for (unsigned int i=0; i<PieceStand::order.size(); ++i)
  {
    const Ptype ptype = PieceStand::order[i];
    const int count = state.countPiecesOnStand(WHITE, ptype);
    if (count > 0)
    {
      os << kanji(newPtypeO(WHITE, ptype));
      if (count > 1)
	os << kanjiNumber(count);
    }
  }
  return os.str();
}

void
BoardImage::draw()
{
  image.reset(new Magick::Image(Magick::Geometry(image_width, image_height), 
                                Magick::Color("white")));

  image->strokeAntiAlias(true);
  image->strokeColor("black");
  image->fillColor("white");

  // frame
  image->strokeWidth(2);
  image->draw(Magick::DrawableRectangle(board_origin_x, board_origin_y,
                                        board_origin_x + board_width,
                                        board_origin_y + board_hegiht));
  image->strokeWidth(1);
  // vertical lines
  for (int i=1; i<9; ++i)
  {
    const int x0 = board_origin_x + cell_width*i;
    const int y0 = board_origin_y;
    const int x1 = x0;
    const int y1 = board_origin_y + board_hegiht;
    image->draw(Magick::DrawableLine(x0, y0, x1, y1));
  }
  // horizontal lines
  for (int i=1; i<9; ++i)
  {
    const int x0 = board_origin_x;
    const int y0 = board_origin_y + cell_height*i;
    const int x1 = x0 + board_width;
    const int y1 = y0;
    image->draw(Magick::DrawableLine(x0, y0, x1, y1));
  }
  // pieces on the board
  for (int i=0; i<40; ++i)
  {
    const Piece p = state.getPieceOf(i);
    const Position pos = p.position();
    if (!pos.isOnBoard())
      continue;
    const double x0 = board_origin_x + cell_width*(9-pos.x());// + cell_width*0.5;
    const double y0 = board_origin_y + cell_height*(pos.y()-1);// + cell_height*0.5;
    PieceImage pieceImage(cell_width, scale);
    ImagePtr piece = pieceImage.image(p.ptypeO());
    image->composite(*piece, x0+2.5, y0+2.5*scale, Magick::OverCompositeOp);
  }
  // pieces on stands
  const std::string& black_stands = blackStand();
  for (int i=0; i<black_stands.size()/3; ++i)
  {
    // UTF-8: each string is of 3-byte size
    const std::string s(black_stands.substr(i*3, 3));
    PieceImage pieceImage(cell_width*0.8, scale);
    ImagePtr piece = pieceImage.image(s);
    const double x0 = board_origin_x + cell_width*10;
    const double y0 = board_origin_y + piece->size().height()*i;
    image->composite(*piece, x0, y0, Magick::OverCompositeOp);
  }
  const std::string& white_stands = whiteStand();
  for (int i=0; i<white_stands.size()/3; ++i)
  {
    // UTF-8: each string is of 3-byte size
    const std::string s(white_stands.substr(i*3, 3));
    PieceImage pieceImage(cell_width*0.8, scale);
    ImagePtr piece = pieceImage.image(s);
    const double x0 = board_origin_x - cell_width;
    const double y0 = board_origin_y + board_hegiht - piece->size().height()*(i+1);
    piece->rotate(180);
    image->composite(*piece, x0, y0, Magick::OverCompositeOp);
  }
}

void
BoardImage::write(const std::string& file_name) const
{
  image->write(file_name);
}

/* ===== MISCS ===== */

/**
 * See if an sfen contains possible characters for the escaped format.
 * return true if possible sfen characters; otherwise, false
 */
bool 
ProcessBoard::isPossibleSfenCharacters(const std::string& sfen) const
{
  if (sfen.empty())
    return false;

  static const boost::regex e("^[PLNSGBRKplnsgbrk1-9bw\\._@\\-]+$", boost::regex::perl);
  if (!boost::regex_match(sfen, e))
    return false;
  return true;
}

std::string
ProcessBoard::eliminateSfen(const std::string& sfen) const
{
  const std::string without_sfen(sfen);
  std::string::size_type loc = without_sfen.find("sfen ", 0);
  if (loc == 0)
    return without_sfen.substr(5, without_sfen.size()-5);
  return without_sfen;
}

/**
 * return false if the file does not need to be overwritten; true if the
 * file should be generated or overwritten. 
 */

bool
ProcessBoard::isOld(const bf::path& file) const
{
  const static long month = 3600*24*31;
  if (!bf::exists(file))
    return true; // there is no file
  const std::time_t file_time = bf::last_write_time(file);
  std::time_t now;
  std::time(&now);
  if (file_time + month < now) // too old
    return true;
  return false;
}

/**
 * Generate a png file for the board specified with board_id. The board_id
 * should be an escaped format.
 * For example, "lnsgkgsnl_1r5b1_ppppppppp_9_9_9_PPPPPPPPP_1B5R1_LNSGKGSNL.b.-"
 * gives "lnsgkgsnl_1r5b1_ppppppppp_9_9_9_PPPPPPPPP_1B5R1_LNSGKGSNL.b.-.png".
 */
std::string 
ProcessBoard::generate(const std::string& board_id) const
{
  if (!isPossibleSfenCharacters(board_id))
    return "";

  try
  {
    const std::string file_name = board_id + ".png";
    const bf::path file_path = dir / file_name;
    if (!isOld(file_path))
      return file_name;

    std::string unescaped(board_id);
    record::usi::unescape(unescaped);

    SimpleState state;
    record::usi::parse("sfen " + unescaped, state);

    BoardImage image(state);
    image.draw();
    image.write(file_path.file_string());
    return file_name;
  }
  catch (record::usi::ParseError& e)
  {
    return "";
  }
  catch (std::exception& e)
  {
    std::cerr << "ERROR: Failed to process " << board_id << std::endl;
    std::cerr << e.what() << std::endl;
  }
}

#if 0
int main(int argc, char* argv[])
{
  namespace bp = boost::program_options;
  std::string a_dir(".");

  bp::options_description generic;
  generic.add_options()
    ("help,h", "Show help message")
    ("dir,d",  bp::value<std::string>(&a_dir)->default_value("."), 
               "Directory to put out files")
    ;
  bp::options_description hidden;
  hidden.add_options()
    ("sfen",   bp::value<std::vector<std::string> >(),
               "SFEN strings")
    ;

  bp::options_description command_line_options;
  command_line_options.add(generic).add(hidden);
  
  bp::options_description visible("Allowed options");
  visible.add(generic);

  bp::positional_options_description command_line_args;
  command_line_args.add("sfen", -1);

  bp::variables_map vm;
  try
  {
    bp::store(bp::command_line_parser(argc, argv).
              options(command_line_options).positional(command_line_args).run(), 
              vm);
    bp::notify(vm);
    if (vm.count("help") || argc < 2)
    {
      std::cout << visible << std::endl;
      return 0;
    }
  } 
  catch (std::exception& e)
  {
    std::cerr << e.what() << std::endl;
    std::cout << command_line_options << std::endl;
    return 1;
  }

  const bf::path dir = bf::system_complete(bf::path(a_dir));
  if (!bf::exists(dir))
  {
    std::cerr << "ERROR: Directory not found: " << dir << std::endl;
    std::cout << command_line_options << std::endl;
    return 1;
  }

  if (!vm.count("sfen"))
  {
    std::cerr << visible << std::endl;
    return 1;
  }

  std::vector<std::string> sfens = vm["sfen"].as<std::vector<std::string> >();
  for (std::vector<std::string>::const_iterator each = sfens.begin();
       each != sfens.end(); ++each)
  {
    const std::string board_id(each->c_str());
    processBoard(board_id, dir);
  }

  return 0;
}
#endif

/* vim: set ts=2 sw=2 fenc=utf-8 ff=unix: */
/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
