//  Gnomoradio - roboradio/audio/mp3.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "mp3.h"

extern "C" {
#include <fcntl.h>
#include "../mp3/esd-audio.h"
#include "../mp3/mpg123.h"
}

using namespace Glib;
using namespace std;

Roboradio::Audio::Mp3::Mp3 (const ustring &file)
	: filename(file)
{
}

Roboradio::Audio::Mp3::~Mp3 ()
{
	stop_prober();
}

void Roboradio::Audio::Mp3::play ()
{
	mpg123_play_file(filename.c_str());
	start_prober();
}

void Roboradio::Audio::Mp3::stop ()
{
	mpg123_stop();
	stop_prober();
}

void Roboradio::Audio::Mp3::pause ()
{
	mpg123_pause(true);
	stop_prober();
}

void Roboradio::Audio::Mp3::unpause ()
{
	mpg123_pause(false);
	start_prober();
}

void Roboradio::Audio::Mp3::seek (unsigned int position)
{
	mpg123_seek(position);
}

int Roboradio::Audio::Mp3::get_position ()
{
	int t = get_time();
	if (t >= 0)
		return t / 1000;
	else
		return -1;
}

static bool copy_data (map<ustring,ustring> &info, const char *key, const char *value)
{
	if (!key || !value)
		return false;
	
	try {
		ustring v = string(value);
		if (!v.validate())
			return false;
		info.insert(make_pair(ustring(key), v));
	} catch (...) {
		return false;
	}

	return true;
}

map<ustring,ustring> Roboradio::Audio::Mp3::get_info (unsigned int &length)
{
	map<ustring,ustring> info;

	char *title = 0;

	char *fn = g_strdup(filename.c_str());
	int int_len;
	get_song_info(fn, &title, &int_len);
	length = int_len / 1000;
	if (!read_id_tag(info) || info.find("title")->second == "") {
		info.erase("title");
		if (title && title[0]
		    && copy_data(info, "title", title)) {
		} else {
			ustring::size_type p = filename.rfind("/");
			if (p != ustring::npos)
				copy_data(info, "title", filename.substr(p + 1).c_str());
		}
	}
	g_free(title);
	g_free(fn);

	return info;
}

bool Roboradio::Audio::Mp3::read_id_tag (map<ustring,ustring> &info)
{
	id3_t *id3;
	struct id3v1tag_t id3v1tag;
	struct id3tag_t tag;
	FILE *file;
	
	file = fopen(filename.c_str(), "rb");
	if (!file)
		return false;
	
	/* Try ID3v2 tag first */
	fseek(file, 0, SEEK_SET);
	id3 = id3_open_fp(file, O_RDONLY);
	if (id3) {
		mpg123_get_id3v2(id3, &tag);
		id3_close (id3);
	} else if ((fseek(file, -1 * sizeof(id3v1tag), SEEK_END) == 0) &&
            (fread(&id3v1tag, 1, sizeof(id3v1tag), file) == sizeof(id3v1tag)) &&
	    (strncmp(id3v1tag.tag, "TAG", 3) == 0)) {
		/* Try reading ID3v1 tag. */
		mpg123_id3v1_to_id3v2(&id3v1tag, &tag);
	} else {
		/* Failed to read any sort of tag */
		fclose(file);
		return false;
	}
	
	/* Copy data from tag into our info struct */
	copy_data(info, "title", tag.title);
	copy_data(info, "artist", tag.artist);
	copy_data(info, "album", tag.album);
	copy_data(info, "year", tag.year);
	copy_data(info, "comment", tag.comment);
	//copy_data(info, "track", tag.track);
	
	/* Clean up */
	fclose(file);
	return true;
}
