/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-pool.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <dbus/dbus-glib-lowlevel.h>
#include "gdm-pool.h"
#include "gdm-marshal.h"

enum {
        DEVICE_ADDED,
        DEVICE_REMOVED,
        DEVICE_PROPERTY_CHANGED,
        LAST_SIGNAL
};

static GObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = { 0 };

struct _GdmPoolPrivate
{
        LibHalContext *hal_ctx;
        GHashTable *devices;
};

G_DEFINE_TYPE (GdmPool, gdm_pool, G_TYPE_OBJECT);

static void
gdm_pool_finalize (GdmPool *pool)
{
        if (G_OBJECT_CLASS (parent_class)->finalize)
                (* G_OBJECT_CLASS (parent_class)->finalize) (G_OBJECT (pool));
}

static void
gdm_pool_class_init (GdmPoolClass *klass)
{
        GObjectClass *obj_class = (GObjectClass *) klass;

        parent_class = g_type_class_peek_parent (klass);

        obj_class->finalize = (GObjectFinalizeFunc) gdm_pool_finalize;

        signals[DEVICE_ADDED] =
                g_signal_new ("device_added",
                              G_TYPE_FROM_CLASS (klass),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmPoolClass, device_added),
                              NULL, NULL,
                              g_cclosure_marshal_VOID__OBJECT,
                              G_TYPE_NONE, 1,
                              GDM_TYPE_DEVICE);

        signals[DEVICE_REMOVED] =
                g_signal_new ("device_removed",
                              G_TYPE_FROM_CLASS (klass),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmPoolClass, device_removed),
                              NULL, NULL,
                              g_cclosure_marshal_VOID__OBJECT,
                              G_TYPE_NONE, 1,
                              GDM_TYPE_DEVICE);

        signals[DEVICE_PROPERTY_CHANGED] =
                g_signal_new ("device_property_changed",
                              G_TYPE_FROM_CLASS (klass),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmPoolClass, device_property_changed),
                              NULL, NULL,
                              gdm_marshal_VOID__OBJECT_STRING,
                              G_TYPE_NONE, 2,
                              GDM_TYPE_DEVICE,
                              G_TYPE_STRING);

}

static void
gdm_pool_init (GdmPool *pool)
{
        pool->priv = g_new0 (GdmPoolPrivate, 1);
        pool->priv->hal_ctx = NULL;
}

static void
gdm_pool_add_device_by_udi (GdmPool *pool, const char *udi)
{
        GdmDevice *device;
        device = gdm_device_new_from_udi (pool->priv->hal_ctx, udi);
        g_hash_table_insert (pool->priv->devices, g_strdup (udi), device);

        g_signal_emit (pool, signals[DEVICE_ADDED], 0, device);
}

static void
_hal_device_added (LibHalContext *hal_ctx, const char *udi)
{
        GdmPool *pool;

        /*g_debug ("_hal_device_added, udi='%s'\n", udi);*/

        pool = GDM_POOL (libhal_ctx_get_user_data (hal_ctx));
        gdm_pool_add_device_by_udi (pool, udi);
}

static void
_hal_device_removed (LibHalContext *hal_ctx, const char *udi)
{
        GdmPool *pool;
        GdmDevice *device;

        /*g_debug ("_hal_device_removed, udi='%s'\n", udi);*/

        pool = GDM_POOL (libhal_ctx_get_user_data (hal_ctx));
        if ((device = gdm_pool_get_device_by_udi (pool, udi)) != NULL) {
                g_signal_emit (pool, signals[DEVICE_REMOVED], 0, device);
                g_hash_table_remove (pool->priv->devices, udi);
                
        } else {
                g_warning ("unknown device to remove, udi='%s'", udi);
        }
}

static void
_hal_property_modified (LibHalContext *ctx,
                        const char *udi,
                        const char *key,
                        dbus_bool_t is_removed,
                        dbus_bool_t is_added)
{
        GdmPool *pool;
        GdmDevice *device;
        
        /*g_debug ("*** _hal_property_modified, udi=%s, key=%s is_removed=%s, is_added=%s\n", 
                 udi, key,
                 is_removed ? "true" : "false",
                 is_added ? "true" : "false");*/

        pool = GDM_POOL (libhal_ctx_get_user_data (ctx));

        device = gdm_pool_get_device_by_udi (pool, udi);
        if (device != NULL) {
                gdm_device_hal_property_changed (device, key);
                g_signal_emit (pool, signals[DEVICE_PROPERTY_CHANGED], 0, device, key);
        } else {
                g_warning ("unknown device with property modified, udi='%s'", udi);
        }
}

LibHalContext *
gdm_pool_get_hal_ctx (GdmPool *pool)
{
        return pool->priv->hal_ctx;
}

GdmPool *
gdm_pool_new (void)
{
        int i;
        char **devices;
        int num_devices;
        GdmPool *pool;
        LibHalContext *hal_ctx;
        DBusError error;
        DBusConnection *dbus_connection;


        pool = NULL;
        
        dbus_error_init (&error);
        dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
        if (dbus_error_is_set (&error)) {
                g_warning ("Cannot connect to system bus: %s : %s", error.name, error.message);
                dbus_error_free (&error);
                goto out;
        }

        hal_ctx = libhal_ctx_new ();
        if (hal_ctx == NULL) {
                g_warning ("Failed to get libhal context");
                goto out;
        }

        dbus_connection_setup_with_g_main (dbus_connection, NULL);
        libhal_ctx_set_dbus_connection (hal_ctx, dbus_connection);

        if (!libhal_ctx_init (hal_ctx, &error)) {
                g_warning ("Failed to initialize libhal context: %s : %s", error.name, error.message);
                dbus_error_free (&error);
                goto out;
        }

        pool = GDM_POOL (g_object_new (GDM_TYPE_POOL, NULL));
        pool->priv->hal_ctx = hal_ctx;
        pool->priv->devices = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);

        libhal_device_property_watch_all (hal_ctx, NULL);
        libhal_ctx_set_device_added (hal_ctx, _hal_device_added);
        libhal_ctx_set_device_removed (hal_ctx, _hal_device_removed);
        libhal_ctx_set_device_property_modified (hal_ctx, _hal_property_modified);
        libhal_ctx_set_user_data (hal_ctx, pool);

        devices = libhal_get_all_devices (pool->priv->hal_ctx, &num_devices, NULL);
        if (devices != NULL) {
                for (i = 0; i < num_devices; i++) {
                        char *device_udi;
                        device_udi = devices[i];
                        gdm_pool_add_device_by_udi (pool, device_udi);
                }
                libhal_free_string_array (devices);
        }
out:
        return pool;
}

GdmDevice *
gdm_pool_get_device_by_udi (GdmPool *pool, const char *udi)
{
        return g_hash_table_lookup (pool->priv->devices, udi);
}


GdmDevice *
gdm_pool_get_parent_device (GdmPool *pool, GdmDevice *device)
{
        GdmDevice *parent;
        const char  *parent_udi;

        parent = NULL;
        parent_udi = gdm_device_get_parent_udi (device);
        if (parent_udi == NULL)
                goto out;

        parent = gdm_pool_get_device_by_udi (pool, parent_udi);

out:
        return parent;
}


struct pool_visitor_data_t {
        GdmPool *pool;
        GdmPoolVisitorFunc func;
        gpointer user_data;
        GdmDevice *parent_device;
};

static void
pool_visit_hfunc (gpointer key, gpointer value, gpointer user_data)
{
        struct pool_visitor_data_t *pvd;
        GdmDevice *device;
        const char *parent_udi;

        pvd = user_data;
        device = GDM_DEVICE (value);

        parent_udi = gdm_device_get_parent_udi (device);

        if (parent_udi != NULL) {
                if (g_ascii_strcasecmp (parent_udi, gdm_device_get_udi (pvd->parent_device)) == 0) {
                        struct pool_visitor_data_t pvd2;

                        pvd->func (pvd->pool, device, pvd->parent_device, pvd->user_data);

                        pvd2.pool = pvd->pool;
                        pvd2.func = pvd->func;
                        pvd2.user_data = pvd->user_data;
                        pvd2.parent_device = device;
                        /* yes, this is probably very inefficient... */
                        g_hash_table_foreach (pvd->pool->priv->devices, pool_visit_hfunc, &pvd2);
                }
        }
}

void 
gdm_pool_visit (GdmPool *pool, GdmPoolVisitorFunc func, gpointer user_data)
{
        GdmDevice *computer;

        computer = gdm_pool_get_device_by_udi (pool, "/org/freedesktop/Hal/devices/computer");
        if (computer != NULL) {
                struct pool_visitor_data_t pvd;

                func (pool, computer, NULL, user_data);
                pvd.pool = pool;
                pvd.func = func;
                pvd.user_data = user_data;
                pvd.parent_device = computer;
                g_hash_table_foreach (pool->priv->devices, pool_visit_hfunc, &pvd);
        }

}


