/**
 * GMyth Library
 *
 * @file gmyth/gmyth_stringlist.c
 * 
 * @brief <p> This component contains functions for dealing with the stringlist
 * format of the mythprotocol.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Hallyson Luiz de Morais Melo <hallyson.melo@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_stringlist.h"

#include "gmyth_debug.h"

static void     gmyth_string_list_class_init(GMythStringListClass * klass);
static void     gmyth_string_list_init(GMythStringList * object);

static void     gmyth_string_list_dispose(GObject * object);
static void     gmyth_string_list_finalize(GObject * object);

G_DEFINE_TYPE(GMythStringList, gmyth_string_list, G_TYPE_OBJECT)
    static void     gmyth_string_list_class_init(GMythStringListClass *
                                                 klass)
{
    GObjectClass   *gobject_class;

    gobject_class = (GObjectClass *) klass;

    gobject_class->dispose = gmyth_string_list_dispose;
    gobject_class->finalize = gmyth_string_list_finalize;
}

static void
gmyth_string_list_init(GMythStringList * gmyth_string_list)
{
    gmyth_string_list->glist = NULL;
}

static void
gmyth_string_list_dispose(GObject * object)
{
    GMythStringList *gmyth_string_list = GMYTH_STRING_LIST(object);

    gmyth_string_list_clear_all(gmyth_string_list);

    G_OBJECT_CLASS(gmyth_string_list_parent_class)->dispose(object);
}

static void
gmyth_string_list_finalize(GObject * object)
{
    // GMythStringList *gmyth_string_list = GMYTH_STRING_LIST(object);

    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_string_list_parent_class)->finalize(object);
}

/** Creates a new instance of GStringList.
 * 
 * @return a new instance of GStringList.
 */
GMythStringList *
gmyth_string_list_new()
{
    GMythStringList *gmyth_string_list =
        GMYTH_STRING_LIST(g_object_new(GMYTH_STRING_LIST_TYPE, NULL));

    return gmyth_string_list;
}

/** Appends a guint64 to the string list.
 * 
 * @param strlist The GMythStringList instance.
 * @param value The guint64 to be appended.
 * 
 * @return The appended guint64 converted to a GString object.
 */
GString        *
gmyth_string_list_append_int(GMythStringList * strlist, const gint value)
{
    GString        *value_str = g_string_new("");

    g_string_printf(value_str, "%d", value);

    strlist->glist = g_list_append(strlist->glist, value_str);

    return value_str;
}


/** Appends a gdouble to the string list.
 *
 * @param strlist The GMythStringList instance.
 * @param value The gdouble to be appended.
 *
 * @return The appended gdouble converted to a GString object.
 */
GString        *
gmyth_string_list_append_float(GMythStringList * strlist, const gdouble value)
{
    GString        *value_str = g_string_new("");
    g_string_printf(value_str, "%f", value);
    strlist->glist = g_list_append(strlist->glist, value_str);
    return value_str;
}


/** Appends a guint64 to the string list.
 * 
 * @param strlist The GMythStringList instance.
 * @param value The guint64 to be appended.
 * 
 * @return The appended guint64 converted to a GString object.
 */
GString        *
gmyth_string_list_append_uint64(GMythStringList * strlist,
                                const guint64 value)
{
    GString        *tmp_str1 = g_string_new("");
    GString        *tmp_str2 = g_string_new("");

    gmyth_debug("value = %llu.\n", value);

    gulong          l2 = ((guint64) value & 0xffffffff);
    gulong          l1 = ((guint64) value >> 32);

    /*
     * high order part of guint64 value 
     */
    g_string_printf(tmp_str1, "%lu", l1);

    gmyth_debug("[%s] uint64 (high) = %s\n", __FUNCTION__, tmp_str1->str);

    strlist->glist = g_list_append(strlist->glist, tmp_str1);

    /*
     * low order part of guint64 value 
     */
    g_string_printf(tmp_str2, "%lu", l2);

    gmyth_debug("[%s] uint64 (low) = %s\n", __FUNCTION__, tmp_str2->str);

    strlist->glist = g_list_append(strlist->glist, tmp_str2);

    return tmp_str2;
}

/** Appends a gint64 to the string list.
 * 
 * @param strlist The GMythStringList instance.
 * @param value The gint64 to be appended.
 * 
 * @return The appended gint64 converted to a GString object.
 */
GString        *
gmyth_string_list_append_int64(GMythStringList * strlist,
                               const gint64 value)
{
    GString        *tmp_str1 = g_string_new("");
    GString        *tmp_str2 = g_string_new("");

    gmyth_debug("value = %lld.\n", value);

    glong           l2 = ((gint64) value & 0xffffffff);
    glong           l1 = ((gint64) value >> 32);

    /*
     * high order part of gint64 value 
     */
    g_string_printf(tmp_str1, "%ld", l1);

    gmyth_debug("[%s] int64 (high) = %s\n", __FUNCTION__, tmp_str1->str);

    strlist->glist = g_list_append(strlist->glist, tmp_str1);

    /*
     * low order part of gint64 value 
     */
    g_string_printf(tmp_str2, "%ld", l2);

    gmyth_debug("[%s] int64 (low) = %s\n", __FUNCTION__, tmp_str2->str);

    strlist->glist = g_list_append(strlist->glist, tmp_str2);

    return tmp_str2;
}

/** Appends a char array to the string list.
 * 
 * @param strlist The GMythStringList instance.
 * @param value The char array to be appended.
 * 
 * @return The appended char array converted to a GString object.
 */
GString        *
gmyth_string_list_append_char_array(GMythStringList * strlist,
                                    const gchar * value)
{
    GString        *tmp_str = NULL;

    g_return_val_if_fail(strlist != NULL, NULL);

    tmp_str = g_string_new(value);

    strlist->glist = g_list_append(strlist->glist, tmp_str);

    return tmp_str;
}

/** Appends a string to the string list.
 * 
 * @param strlist The GMythStringList instance.
 * @param value The string to be appended.
 * 
 * @return The appended string itself. 
 */
GString        *
gmyth_string_list_append_string(GMythStringList * strlist, GString * value)
{
    g_return_val_if_fail(strlist != NULL, NULL);

    if (value != NULL) {
        strlist->glist =
            g_list_append(strlist->glist, g_string_new(value->str));
    } else {
        strlist->glist = g_list_append(strlist->glist, NULL);
    }

    return value;
}

/** Gets an integer value from the string list at the given position.
 * 
 * @param strlist The GMythStringList instance.
 * @param index the integer position in the list, starting with zero.
 * @return The integer value.
 */
gint
gmyth_string_list_get_int(GMythStringList * strlist, const gint index)
{
    // TODO: Create static method check_index()
    GString        *tmp_str = NULL;

    g_return_val_if_fail(strlist != NULL, 0);

    tmp_str = (GString *) g_list_nth_data(strlist->glist, index);

    if (NULL == tmp_str || NULL == tmp_str->str
        || strlen(tmp_str->str) <= 0)
        return 0;

    return (gint) (             /* 0x00000000ffffffffL & (gint64) */
                      g_ascii_strtoull(tmp_str->str, NULL, 10));
}

/** Gets a guint64 value from the string list at the given position.
 * According to the Mythtv protocol, the 64 bits value is formed by
 * two strings.
 * 
 * @param strlist The GMythStringList instance.
 * @param index the index of the first string forming the 64 bits value. 
 * Index starts with zero.
 * @return The guint64 value.
 */
guint64
gmyth_string_list_get_uint64(GMythStringList * strlist, const gint index)
{
    // TODO: Create static method check_index()
    guint64         ret_value = 0;
    guint64         l2 = 0;

    g_return_val_if_fail(strlist != NULL, 0);

    const GString  *tmp_str1 =
        (GString *) g_list_nth_data(strlist->glist, index);
    const GString  *tmp_str2 =
        (GString *) g_list_nth_data(strlist->glist, index + 1);

    if (tmp_str1 != NULL)
        gmyth_debug("[%s] seek high bytes = %s\n", __FUNCTION__,
                    tmp_str1->str);
    if (tmp_str2 == NULL || strlen(tmp_str2->str) > 0) {
    } else {
        gmyth_debug("[%s] seek low bytes = %s\n", __FUNCTION__,
                    tmp_str2->str);
    }

    guint64         l1 = ((guint64) g_ascii_strtoull(tmp_str1->str, NULL, 10)   /* & 
                                                                                 * 0xffffffff 
                                                                                 */
        );

    if (tmp_str2 != NULL && tmp_str2->str != NULL
        && strlen(tmp_str2->str) > 0) {
        l2 = ((guint64) g_ascii_strtoull(tmp_str2->str, NULL, 10)
              /*
               * & 0xffffffff 
               */
            );
    } else {
        l2 = l1;
        l1 = 0;
    }

    gmyth_debug("[%s]\t[l1 == %llu, l2 == %llu]\n", __FUNCTION__, l1, l2);

    ret_value =
        ((guint64) (l2) /* & 0xffffffff */ ) | ((guint64) l1 << 32);

    gmyth_debug("[%s] returning uint64 value = %llu\n", __FUNCTION__,
                ret_value);

    return ret_value;
}

/** Gets a gint64 value from the string list at the given position.
 * According to the Mythtv protocol, the 64 bits value is formed by
 * two strings.
 * 
 * @param strlist The GMythStringList instance.
 * @param index the index of the first string forming the 64 bits value. 
 * Index starts with zero.
 * @return The gint64 value.
 */
gint64
gmyth_string_list_get_int64(GMythStringList * strlist, const gint index)
{
    // TODO: Create static method check_index()
    gint64          ret_value = 0;
    gint64          l2 = 0;

    g_return_val_if_fail(strlist != NULL, 0);

    const GString  *tmp_str1 =
        (GString *) g_list_nth_data(strlist->glist, index);
    const GString  *tmp_str2 =
        (GString *) g_list_nth_data(strlist->glist, index + 1);

    if (tmp_str1 != NULL)
        gmyth_debug("[%s] seek high bytes = %s\n", __FUNCTION__,
                    tmp_str1->str);
    if (tmp_str2 == NULL || strlen(tmp_str2->str) > 0) {
    } else {
        gmyth_debug("[%s] seek low bytes = %s\n", __FUNCTION__,
                    tmp_str2->str);
    }

    gint64          l1 = ((guint64) g_ascii_strtoull(tmp_str1->str, NULL, 10)   /* & 
                                                                                 * 0xffffffff 
                                                                                 */
        );

    if (tmp_str2 != NULL && tmp_str2->str != NULL
        && strlen(tmp_str2->str) > 0) {
        l2 = ((gint64) g_ascii_strtoull(tmp_str2->str, NULL, 10)
              /*
               * & 0xffffffff 
               */
            );
    } else {
        l2 = l1;
        l1 = 0;
    }

    gmyth_debug("[%s]\t[l1 == %lld, l2 == %lld]\n", __FUNCTION__, l1, l2);

    ret_value = ((gint64) (l2) /* & 0xffffffff */ ) | ((gint64) l1 << 32);

    gmyth_debug("[%s] returning int64 value = %lld\n", __FUNCTION__,
                ret_value);

    return ret_value;
}


/** Gets a string from the string list at the given position. The GString must be deallocated.
 * 
 * @param strlist The GMythStringList instance.
 * @param index the string position in the list, starting with zero.
 * @return A pointer to the string data.
 */
GString        *
gmyth_string_list_get_string(GMythStringList * strlist, const gint index)
{
    GString        *ret;

    if (!strlist || !(strlist->glist)) {
        gmyth_debug("%s received Null arguments", __FUNCTION__);
        return NULL;
    }

    ret = (GString *) g_list_nth_data(strlist->glist, index);

    return g_string_new(ret->str);
}


static void
gmyth_string_list_clear_element(GString * str_elem, void *data_aux)
{
    if (str_elem != NULL)
        g_string_free(str_elem, TRUE);
}

/** Removes all strings from the string list.
 * 
 * @param strlist The GMythStringList instance.
 */
void
gmyth_string_list_clear_all(GMythStringList * strlist)
{
    if (strlist != NULL && strlist->glist) {
        g_list_foreach(strlist->glist,
                       (GFunc) gmyth_string_list_clear_element, NULL);
        g_list_free(strlist->glist);
        strlist->glist = NULL;
    }
}

/** Retrieves the number of elements in the string list.
 * 
 * @param strlist The GMythStringList instance.
 * @return the string list length.
 */
gint
gmyth_string_list_length(GMythStringList * strlist)
{
    if (!(strlist != NULL && strlist->glist != NULL))
        return 0;

    return g_list_length(strlist->glist);
}
