-----------------------------------------------------------------------------
-- |
-- Module      :  Distribution.Simple.Build.Macros
-- Copyright   :  Simon Marlow 2008
--
-- Maintainer  :  cabal-devel@haskell.org
-- Portability :  portable
--
-- Generate cabal_macros.h - CPP macros for package version testing
--
-- When using CPP you get
--
-- > MIN_VERSION_<package>(A,B,C)
--
-- for each /package/ in @build-depends@, which is true if the version of
-- /package/ in use is @>= A.B.C@, using the normal ordering on version
-- numbers.
--
module Distribution.Simple.Build.Macros (
    generate
  ) where

import Distribution.Package
         ( PackageIdentifier(PackageIdentifier) )
import Distribution.Version
         ( Version(versionBranch) )
import Distribution.PackageDescription
         ( PackageDescription )
import Distribution.Simple.LocalBuildInfo
        ( LocalBuildInfo, externalPackageDeps )
import Distribution.Text
         ( display )

-- ------------------------------------------------------------
-- * Generate cabal_macros.h
-- ------------------------------------------------------------

generate :: PackageDescription -> LocalBuildInfo -> String
generate _pkg_descr lbi = concat $
  "/* DO NOT EDIT: This file is automatically generated by Cabal */\n\n" :
  [ concat
    ["/* package ",display pkgid," */\n"
    ,"#define MIN_VERSION_",pkgname,"(major1,major2,minor) \\\n"
    ,"  (major1) <  ",major1," || \\\n"
    ,"  (major1) == ",major1," && (major2) <  ",major2," || \\\n"
    ,"  (major1) == ",major1," && (major2) == ",major2," && (minor) <= ",minor
    ,"\n\n"
    ]
  | (_, pkgid@(PackageIdentifier name version)) <- externalPackageDeps lbi
  , let (major1:major2:minor:_) = map show (versionBranch version ++ repeat 0)
        pkgname = map fixchar (display name)
  ]
  where fixchar '-' = '_'
        fixchar c   = c

