/**
 *  Copyright (C) 2002-2011  The FreeCol Team
 *
 *  This file is part of FreeCol.
 *
 *  FreeCol is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  FreeCol is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with FreeCol.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sf.freecol.client.gui.menu;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.KeyStroke;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import net.sf.freecol.client.FreeColClient;
import net.sf.freecol.client.gui.Canvas;
import net.sf.freecol.client.gui.GUI;
import net.sf.freecol.client.gui.i18n.Messages;
import net.sf.freecol.client.gui.panel.ChoiceDialog;
import net.sf.freecol.client.gui.panel.ChoiceItem;
import net.sf.freecol.client.gui.panel.MonarchPanel;
import net.sf.freecol.client.gui.panel.StatisticsPanel;
import net.sf.freecol.client.gui.panel.VictoryPanel;
import net.sf.freecol.common.model.Colony;
import net.sf.freecol.common.model.FoundingFather;
import net.sf.freecol.common.model.Game;
import net.sf.freecol.common.model.GameOptions;
import net.sf.freecol.common.model.Map;
import net.sf.freecol.common.model.Monarch;
import net.sf.freecol.common.model.Monarch.MonarchAction;
import net.sf.freecol.common.model.Player;
import net.sf.freecol.common.model.StringTemplate;
import net.sf.freecol.common.model.Tile;
import net.sf.freecol.common.model.Unit;
import net.sf.freecol.common.model.Unit.UnitState;
import net.sf.freecol.common.resources.ImageResource;
import net.sf.freecol.common.resources.Resource;
import net.sf.freecol.common.resources.ResourceManager;
import net.sf.freecol.server.FreeColServer;
import net.sf.freecol.server.ai.AIUnit;
import net.sf.freecol.server.control.InGameController;
import net.sf.freecol.server.model.ServerPlayer;


public class DebugMenu extends JMenu {

    private FreeColClient freeColClient;

    private final Canvas canvas;

    private final GUI gui;


    public DebugMenu(FreeColClient fcc) {
        super(Messages.message("menuBar.debug"));

        this.freeColClient = fcc;

        gui = freeColClient.getGUI();
        canvas = freeColClient.getCanvas();

        buildDebugMenu();
    }

    private void buildDebugMenu() {
        final Game game = freeColClient.getGame();
        final FreeColServer server = freeColClient.getFreeColServer();
        final Game serverGame = (server == null) ? null : server.getGame();
        final Player player = freeColClient.getMyPlayer();
        final Player serverPlayer = (server == null) ? null
            : (Player) serverGame.getFreeColGameObject(player.getId());

        this.setOpaque(false);
        this.setMnemonic(KeyEvent.VK_D);
        add(this);

        final JMenu debugFixMenu = new JMenu("Fixes");
        debugFixMenu.setOpaque(false);
        debugFixMenu.setMnemonic(KeyEvent.VK_F);
        this.add(debugFixMenu);

        final JMenuItem crossBug
            = new JCheckBoxMenuItem("Fix \"not enough crosses\"-bug");
        crossBug.setOpaque(false);
        crossBug.setMnemonic(KeyEvent.VK_B);
        debugFixMenu.add(crossBug);
        crossBug.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    player.updateImmigrationRequired();
                    Iterator<Player> pi = serverGame.getPlayerIterator();
                    while (pi.hasNext()) {
                        pi.next().updateImmigrationRequired();
                    }
                }
            });
        crossBug.setEnabled(server != null);

        this.addSeparator();

        final JCheckBoxMenuItem sc
            = new JCheckBoxMenuItem(Messages.message("menuBar.debug.showCoordinates"),
                                    gui.displayCoordinates);
        sc.setOpaque(false);
        sc.setMnemonic(KeyEvent.VK_S);
        this.add(sc);
        sc.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    gui.displayCoordinates = ((JCheckBoxMenuItem) e.getSource()).isSelected();
                    canvas.refresh();
                }
            });
        sc.setEnabled(true);

        final JCheckBoxMenuItem dam
            = new JCheckBoxMenuItem("Display AI-missions",
                                    gui.debugShowMission);
        final JCheckBoxMenuItem dami
            = new JCheckBoxMenuItem("Additional AI-mission info",
                                    gui.debugShowMissionInfo);
        dam.setOpaque(false);
        dam.setMnemonic(KeyEvent.VK_M);
        this.add(dam);
        dam.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    gui.debugShowMission = ((JCheckBoxMenuItem) e.getSource()).isSelected();
                    dami.setEnabled(gui.debugShowMission);
                    canvas.refresh();
                }
            });
        dam.setEnabled(true);

        dami.setOpaque(false);
        dami.setMnemonic(KeyEvent.VK_I);
        this.add(dami);
        dami.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    gui.debugShowMissionInfo = ((JCheckBoxMenuItem) e.getSource()).isSelected();
                    canvas.refresh();
                }
            });
        dami.setEnabled(gui.debugShowMission);

        final JMenuItem reveal
            = new JCheckBoxMenuItem(Messages.message("menuBar.debug.revealEntireMap"));
        reveal.setOpaque(false);
        reveal.setMnemonic(KeyEvent.VK_R);
        this.add(reveal);
        reveal.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    server.revealMapForAllPlayers();
                    reveal.setEnabled(false);
                    game.getSpecification()
                        .getBooleanOption(GameOptions.FOG_OF_WAR)
                        .setValue(false);
                }
            });
        reveal.setEnabled(server != null);

        final JMenu cvpMenu
            = new JMenu(Messages.message("menuBar.debug.showColonyValue"));
        cvpMenu.setOpaque(false);
        ButtonGroup bg = new ButtonGroup();
        final JRadioButtonMenuItem cv1
            = new JRadioButtonMenuItem("Do not display",
                                       !gui.displayColonyValue);
        cv1.setOpaque(false);
        cv1.setMnemonic(KeyEvent.VK_C);
        cvpMenu.add(cv1);
        bg.add(cv1);
        cv1.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    gui.displayColonyValue = false;
                    gui.displayColonyValuePlayer = null;
                    canvas.refresh();
                }
            });
        this.add(cvpMenu);

        final JRadioButtonMenuItem cv3
            = new JRadioButtonMenuItem("Common outpost value",
                gui.displayColonyValue && gui.displayColonyValuePlayer == null);
        cv3.setOpaque(false);
        cv3.setMnemonic(KeyEvent.VK_C);
        cvpMenu.add(cv3);
        bg.add(cv3);
        cv3.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    gui.displayColonyValue = true;
                    gui.displayColonyValuePlayer = null;
                    canvas.refresh();
                }
            });
        this.add(cvpMenu);

        cvpMenu.addSeparator();

        Iterator<Player> it = game.getPlayerIterator();
        while (it.hasNext()) {
            final Player p = it.next();
            if (p.isEuropean() && p.canBuildColonies()) {
                final JRadioButtonMenuItem cv2
                    = new JRadioButtonMenuItem(Messages.message(p.getNationName()),
                        gui.displayColonyValue && gui.displayColonyValuePlayer == p);
                cv2.setOpaque(false);
                cv2.setMnemonic(KeyEvent.VK_C);
                cvpMenu.add(cv2);
                bg.add(cv2);
                cv2.addActionListener(new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
                            gui.displayColonyValue = true;
                            gui.displayColonyValuePlayer = p;
                            canvas.refresh();
                        }
                    });
            }
        }

        this.addSeparator();

        final JMenuItem skipTurns = new JMenuItem("Skip turns");
        skipTurns.setOpaque(false);
        skipTurns.setMnemonic(KeyEvent.VK_S);
        this.add(skipTurns);
        skipTurns.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    InGameController igc = server.getInGameController();
                    boolean isSkipping = igc.getSkippedTurns() > 0;
                    if (isSkipping) {
                        igc.setSkippedTurns(0);
                        return;
                    }

                    String response = canvas.showInputDialog(null,
                                                             StringTemplate.name("How many turns should be skipped:"),
                                                             Integer.toString(10), "ok", "cancel", true);
                    if (response == null) return;
                    int skip;
                    try {
                        skip = Integer.parseInt(response);
                    } catch (NumberFormatException nfe) {
                        skip = -1;
                    }
                    if (skip > 0) freeColClient.skipTurns(skip);
                }
            });
        this.addChangeListener(new ChangeListener() {
                public void stateChanged(ChangeEvent e) {
                    boolean skipping = server.getInGameController()
                        .getSkippedTurns() > 0;
                    skipTurns.setText((skipping) ? "Stop skipping"
                                      : "Skip turns");
                }
            });
        skipTurns.setEnabled(server != null);

        final String fatherTitle
            = Messages.message("menuBar.debug.addFoundingFather");
        final JMenuItem addFather = new JMenuItem(fatherTitle);
        addFather.setOpaque(false);
        addFather.setMnemonic(KeyEvent.VK_F);
        this.add(addFather);
        addFather.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    List<ChoiceItem<FoundingFather>> fathers
                        = new ArrayList<ChoiceItem<FoundingFather>>();
                    for (FoundingFather father : game.getSpecification().getFoundingFathers()) {
                        if (!player.hasFather(father)) {
                            fathers.add(new ChoiceItem<FoundingFather>(Messages.message(father.getNameKey()),
                                                                       father));
                        }
                    }
                    ChoiceDialog<FoundingFather> choiceDialog =
                        new ChoiceDialog<FoundingFather>(canvas, fatherTitle,
                                                         "Cancel", fathers);
                    FoundingFather father
                        = canvas.showFreeColDialog(choiceDialog, null);
                    if (father != null) {
                        InGameController igc = server.getInGameController();
                        igc.addFoundingFather((ServerPlayer) serverPlayer,
                                              father);
                    }
                }
            });
        addFather.setEnabled(server != null);

        final String monarchTitle
            = Messages.message("menuBar.debug.runMonarch");
        final JMenuItem runMonarch = new JMenuItem(monarchTitle);
        runMonarch.setOpaque(false);
        runMonarch.setMnemonic(KeyEvent.VK_F);
        this.add(runMonarch);
        runMonarch.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    List<ChoiceItem<MonarchAction>> actions
                        = new ArrayList<ChoiceItem<MonarchAction>>();
                    for (MonarchAction action : MonarchAction.values()) {
                        actions.add(new ChoiceItem<MonarchAction>(action));
                    }
                    ChoiceDialog<MonarchAction> choiceDialog
                        = new ChoiceDialog<MonarchAction>(canvas, monarchTitle,
                                                          "Cancel", actions);
                    MonarchAction action
                        = canvas.showFreeColDialog(choiceDialog, null);
                    server.getInGameController()
                        .setMonarchAction(serverPlayer, action);
                }
            });
        runMonarch.setEnabled(server != null);

        final JMenuItem addGold = new JMenuItem(Messages.message("menuBar.debug.addGold"));
        addGold.setOpaque(false);
        this.add(addGold);
        addGold.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    String response
                        = canvas.showInputDialog(null, StringTemplate.key("menuBar.debug.addGold"),
                                                 Integer.toString(1000),
                                                 "ok", "cancel", true);
                    int gold = Integer.parseInt(response);
                    player.modifyGold(gold);
                    serverPlayer.modifyGold(gold);
                }
            });
        addGold.setEnabled(server != null);

        final String immigrationTitle
            = Messages.message("menuBar.debug.addImmigration");
        final JMenuItem addCrosses = new JMenuItem(immigrationTitle);
        addCrosses.setOpaque(false);
        // addCrosses.setMnemonic(KeyEvent.VK_????);
        this.add(addCrosses);
        addCrosses.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    String response
                        = canvas.showInputDialog(null,
                                                 StringTemplate.key("menuBar.debug.addImmigration"),
                                                 Integer.toString(100),
                                                 "ok", "cancel", true);
                    int crosses = Integer.parseInt(response);
                    player.incrementImmigration(crosses);
                    serverPlayer.incrementImmigration(crosses);
                }
            });
        addCrosses.setEnabled(server != null);

        final JMenuItem giveBells
            = new JMenuItem(Messages.message("menuBar.debug.addLiberty"));
        giveBells.setOpaque(false);
        giveBells.setMnemonic(KeyEvent.VK_B);
        this.add(giveBells);
        giveBells.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    for (Colony c : player.getColonies()) {
                        c.addLiberty(100);
                        Colony sc = (Colony) serverGame.getFreeColGameObject(c.getId());
                            sc.addLiberty(100);
                        }
                    }
                });
        giveBells.setEnabled(server != null);

        // random number generator
        final JMenuItem rng = new JMenuItem("Step random number generator");
        rng.setOpaque(false);
        rng.setMnemonic(KeyEvent.VK_X);
        this.add(rng);
        rng.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    InGameController igc = server.getInGameController();
                    boolean more = true;
                    while (more) {
                        int val = igc.stepRandom();
                        more = canvas.showConfirmDialog(null,
                                                        StringTemplate.template("menuBar.debug.stepRandomNumberGenerator")
                                                        .addAmount("%value%", val),
                                                        "more", "ok");
                    }
                }
            });
        rng.setEnabled(server != null);

        this.addSeparator();

        final JMenu panelMenu = new JMenu("Display panels");
        panelMenu.setOpaque(false);

        final JMenuItem monarchPanel = new JMenuItem("Display Monarch panel");
        monarchPanel.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    canvas.showFreeColDialog(new MonarchPanel(canvas, Monarch.MonarchAction.RAISE_TAX));
                }
            });
        panelMenu.add(monarchPanel);

        final JMenuItem victoryPanel = new JMenuItem("Display Victory panel");
        victoryPanel.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    canvas.showPanel(new VictoryPanel(canvas));
                }
            });
        panelMenu.add(victoryPanel);

        for (final Canvas.EventType eventType : Canvas.EventType.values()) {
            final JMenuItem mItem = new JMenuItem("Display " + eventType + " panel");
            mItem.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        canvas.showEventPanel(eventType);
                    }
                });
            panelMenu.add(mItem);
        }
        this.add(panelMenu);

        final JMenuItem europeStatus = new JMenuItem("Display Europe Status");
        europeStatus.setOpaque(false);
        europeStatus.setMnemonic(KeyEvent.VK_E);
        this.add(europeStatus);
        europeStatus.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    net.sf.freecol.server.ai.AIMain aiMain = server.getAIMain();
                    StringBuilder sb = new StringBuilder();
                    for (Player tp : game.getPlayers()) {
                        final Player p = (Player) serverGame.getFreeColGameObject(tp.getId());
                        if (p.getEurope() == null) {
                            continue;
                        }
                        List<Unit> inEurope = new ArrayList<Unit>();
                        List<Unit> toEurope = new ArrayList<Unit>();
                        List<Unit> toAmerica = new ArrayList<Unit>();
                        LinkedHashMap<String,List<Unit>> units = new LinkedHashMap<String, List<Unit>>();
                        units.put("To Europe", toEurope);
                        units.put("In Europe", inEurope);
                        units.put("To America", toAmerica);

                        sb.append("\n==");
                        sb.append(Messages.message(p.getNationName()));
                        sb.append("==\n");

                        for(Unit u : p.getEurope().getUnitList()){
                            if (u.getState() == UnitState.TO_AMERICA) {
                                toAmerica.add(u);
                                continue;
                            }
                            if (u.getState() == UnitState.TO_EUROPE) {
                                toEurope.add(u);
                                continue;
                            }
                            inEurope.add(u);
                        }

                        for (String label : units.keySet()) {
                            List<Unit> list = units.get(label);
                            if (list.size() > 0){
                                sb.append("\n->" + label + "\n");
                                for (Unit u : list) {
                                    sb.append('\n');
                                    sb.append(Messages.message(Messages.getLabel(u)));
                                    if (u.isUnderRepair()) {
                                        sb.append(" (Repairing)");
                                    } else {
                                        sb.append("    ");
                                        AIUnit aiu = aiMain.getAIUnit(u);
                                        if (aiu.getMission() == null) {
                                            sb.append(" (None)");
                                        } else{
                                            sb.append(aiu.getMission().toString().replaceAll("\n", "    \n"));
                                        }
                                    }
                                }
                                sb.append('\n');
                            }
                        }
                    }
                    canvas.showInformationMessage(sb.toString());
                }
            });
        europeStatus.setEnabled(server != null);

        final JMenuItem useAI = new JMenuItem("Use AI");
        useAI.setOpaque(false);
        useAI.setMnemonic(KeyEvent.VK_A);
        useAI.setAccelerator(KeyStroke.getKeyStroke('A',
            Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()
                                                    | InputEvent.ALT_MASK));
        this.add(useAI);
        useAI.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    net.sf.freecol.server.ai.AIMain aiMain = server.getAIMain();
                    net.sf.freecol.server.ai.AIPlayer ap = aiMain.getAIPlayer(player);
                    ap.setDebuggingConnection(freeColClient.getClient().getConnection());
                    ap.startWorking();
                    freeColClient.getConnectController().reconnect();
                }
            });
        useAI.setEnabled(server != null);

        this.addSeparator();

        final JMenuItem compareMaps = new JMenuItem(Messages.message("menuBar.debug.compareMaps"));
        compareMaps.setOpaque(false);
        compareMaps.setMnemonic(KeyEvent.VK_C);
        compareMaps.setAccelerator(KeyStroke.getKeyStroke('C',
            Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()
                                                          | InputEvent.ALT_MASK));
        this.add(compareMaps);
        compareMaps.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    boolean problemDetected = false;
                    Map serverMap = serverGame.getMap();
                    for (Tile t: serverMap.getAllTiles()) {
                        if (serverPlayer.canSee(t)) {
                            Iterator<Unit> unitIterator = t.getUnitIterator();
                            while (unitIterator.hasNext()) {
                                Unit u = unitIterator.next();
                                if (u.isVisibleTo(serverPlayer)) {
                                    if (game.getFreeColGameObject(u.getId()) == null) {
                                        System.out.println("Unsynchronization detected: Unit missing on client-side");
                                        System.out.println(Messages.message(Messages.getLabel(u))
                                                           + "(" + u.getId() + "). Position: "
                                                           + u.getTile().getPosition());
                                        try {
                                            System.out.println("Possible unit on client-side: "
                                                               + game.getMap().getTile(u.getTile().getPosition())
                                                               .getFirstUnit().getId());
                                        } catch (NullPointerException npe) {
                                        }
                                        System.out.println();
                                        problemDetected = true;
                                    } else {
                                        Unit clientSideUnit = (Unit) game.getFreeColGameObject(u.getId());
                                        if (clientSideUnit.getTile() != null
                                            && !clientSideUnit.getTile().getId().equals(u.getTile().getId())) {
                                            System.out.println("Unsynchronization detected: Unit located on different tiles");
                                            System.out.println("Server: " + Messages.message(Messages.getLabel(u))
                                                               + "(" + u.getId() + "). Position: "
                                                               + u.getTile().getPosition());
                                            System.out.println("Client: "
                                                               + Messages.message(Messages.getLabel(clientSideUnit))
                                                               + "(" + clientSideUnit.getId() + "). Position: "
                                                               + clientSideUnit.getTile().getPosition());
                                            System.out.println();
                                            problemDetected = true;
                                        }
                                    }
                                }
                            }
                        }
                    }

                    if (problemDetected) {
                        canvas.showInformationMessage("menuBar.debug.compareMaps.problem");
                    } else {
                        canvas.showInformationMessage("menuBar.debug.compareMaps.checkComplete");
                    }
                }
            });
        compareMaps.setEnabled(server != null);

        // statistics
        final JMenuItem statistics = new JMenuItem("Statistics");
        statistics.setOpaque(false);
        statistics.setMnemonic(KeyEvent.VK_I);
        this.add(statistics);
        statistics.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    canvas.showPanel(new StatisticsPanel(canvas));
                }
            });
        statistics.setEnabled(true);

        // garbage collector
        final JMenuItem gc
            = new JMenuItem(Messages.message("menuBar.debug.memoryManager.gc"));
        gc.setOpaque(false);
        gc.setMnemonic(KeyEvent.VK_G);
        this.add(gc);
        gc.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    System.gc();
                }
            });
        gc.setEnabled(true);

        this.addSeparator();

        final JMenuItem showResourceKeys = new JMenuItem("Show resource keys");
        showResourceKeys.setOpaque(false);
        this.add(showResourceKeys);
        showResourceKeys.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    java.util.Map<String, Resource> resources = ResourceManager.getResources();
                    List<String> keys = new ArrayList<String>(resources.keySet());
                    Collections.sort(keys);
                    StringBuilder builder = new StringBuilder();
                    for (String key : keys) {
                        builder.append(key);
                        Resource resource = resources.get(key);
                        if (resource instanceof ImageResource) {
                            builder.append(" (");
                            builder.append(((ImageResource) resource).getCount());
                            builder.append(")");
                        }
                        builder.append("\n");
                    }
                    canvas.showInformationMessage(builder.toString());
                }
            });
        showResourceKeys.setEnabled(true);

        /*
          TODO: how do we force the ResourceManager to reload images?
        final JMenuItem loadResource = new JMenuItem("Reload images");
        loadResource.setOpaque(false);
        this.add(loadResource);
        loadResource.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    try {
                        canvas.getImageLibrary().init();
                    } catch (Exception ex) {
                        System.out.println("Failed to reload images.");
                    }
                }
            });
        */
    }
}
