/*
  Copyright 2005, 2006, 2007 David Cad�, Damien Stehl�.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; see the file COPYING.  If not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  This program implements ideas from the paper "Floating-point LLL Revisited", 
  by Phong Nguyen and Damien Stehl�, in the Proceedings of Eurocrypt'2005, 
  Springer-Verlag; and was partly inspired by Shoup's NTL library: 
  http://www.shoup.net/ntl/

*/

#ifndef NR_CPP
#define NR_CPP
#include <mpfr.h>
#include "nr.h"



#include <iostream>
#include <cstdlib>
#include <cmath>
#include <gmp.h>



#include "dpe.h"

using namespace std;


template<class Z> inline int Z_NR<Z>::cmp(Z_NR<Z>& m)
{
  Z_NR<Z> x;
  x.sub(*this,m);
  return x.sgn();
}

template<class Z> Z_NR<Z>::Z_NR()
{
}
template<class Z> Z_NR<Z>::~Z_NR()
{
}
template<class Z> inline void Z_NR<Z>::print()
{
  cout << data;
}
template<class Z> inline void Z_NR<Z>::printerr()
{
  cerr << data;
}
template<class Z> inline void Z_NR<Z>::read()
{
  cin >> data;
}
template<class Z> inline void Z_NR<Z>::set(Z_NR<Z>& s)
{
  data=s.GetData();
}
template<class Z> inline void Z_NR<Z>::set(unsigned long int s)
{
  data=static_cast<Z> (s);
}
template<class Z> inline void Z_NR<Z>::set(Z& s)
{
  data=s;
}

template<class Z> inline double Z_NR<Z>::get_d_2exp(signed long int* expo)
{
  return data->get_d_2exp(expo);
}
template<class Z> inline Z& Z_NR<Z>::GetData()
{
  return data;
}



/**********************
 * int specialization *
 **********************/
inline int sizeinbase2(long int data)
{
  long int y=abs(data);
  int resul=0;
  if (y==0) resul=1;
  else{
    while (y>1){
      resul++;
      y>>=1;  //y /=2;
    }
  }    
  return resul;
}

template<> inline double Z_NR<long int>::get_d_2exp(signed long int* expo)
{ 
  if (data==0) *expo = 0;
  else *expo = sizeinbase2 (data);
  return ldexp( (double) data, -*expo);
}

template<> inline int Z_NR<long int>::sgn()
{
  if (data>0) return 1;
  if (data==0) return 0;
  return -1;
}


template<> template<> inline void Z_NR<long int>::submul(Z_NR<long int>& a,Z_NR<long int>& b)
{
  data-=a.GetData()*b.GetData();
}
template<> template<> inline void Z_NR<long int>::submul(Z_NR<long int>& a,Z_NR<mpz_t>& b)
{
  data-=a.GetData()*mpz_get_si(b.GetData());
}
template<> inline void Z_NR<long int>::mul_ui(Z_NR<long int>& a,long unsigned int b)
{
  data=a.GetData()*b;
}
template<> inline void Z_NR<long int>::mul_si(Z_NR<long int>& a,long signed int b)
{
  data=a.GetData()*b;
}
template<> inline void Z_NR<long int>::sub(Z_NR<long int>& a,Z_NR<long int>& b)
{
  data=a.GetData()-b.GetData();
}
template<> inline void Z_NR<long int>::add(Z_NR<long int>& a,Z_NR<long int>& b)
{
  data=a.GetData()+b.GetData();
}

template<> template<> inline void Z_NR<long int>::mul(Z_NR<long int>& a,Z_NR<long int>& b)
{
  data=a.GetData()*b.GetData();
}
template<> template<> inline void Z_NR<long int>::mul(Z_NR<long int>& a,Z_NR<mpz_t>& b)
{
  data=a.GetData()*mpz_get_si(b.GetData());
}
template<> inline void Z_NR<long int>::mul_2exp(Z_NR<long int>& a,long int b)
{
  data=(long int)ldexp((double)a.GetData(),b);
}
template<> inline void Z_NR<long int>::div_2exp(Z_NR<long int>& a,long int b)
{
  data=(long int)ldexp((double)a.GetData(),-b);
}
template<> inline void Z_NR<long int>::addmul_ui(Z_NR<long int>& a,unsigned long int b)
{
  data+=a.GetData()*b;
}
template<> inline void Z_NR<long int>::submul_ui(Z_NR<long int>& a,unsigned long int b)
{
  data-=a.GetData()*b;
}
template<> inline void Z_NR<long int>::randb(int bits)
{
  data=rand()%(1<<bits);
}
template<> inline void Z_NR<long int>::randm(Z_NR<long int>& max)
{
  data=rand()%max.GetData();
}
template<> inline void Z_NR<long int>::add_ui(Z_NR<long int>& a,unsigned int b)
{
  data=a.GetData()+b;
}
template<> inline signed long int Z_NR<long int>::get_si()
{
  return data;
}


/**********************
 * mpz specialization *
 **********************/

template<> Z_NR<mpz_t>::Z_NR()
{
  mpz_init(data);
}
template<> Z_NR<mpz_t>::~Z_NR()
{
  mpz_clear(data);
}
template<> inline void Z_NR<mpz_t>::print()
{
  mpz_out_str(stdout,10,data);
  fflush(stdout);
}
template<> inline void Z_NR<mpz_t>::printerr()
{
  mpz_out_str(stderr,10,data);
  fflush(stderr);
}
template<> inline void Z_NR<mpz_t>::read()
{
  mpz_inp_str(data,stdin,0);
}
template<> inline void Z_NR<mpz_t>::set(Z_NR<mpz_t>& d)
{
  mpz_set(data,d.GetData());
}
template<> inline void Z_NR<mpz_t>::set(mpz_t &d)
{
  mpz_set(data,d);
}
template<> inline void Z_NR<mpz_t>::set(unsigned long int d)
{
  mpz_set_ui(data,d);
}
template<> inline void Z_NR<mpz_t>::set_si(signed long int d)
{
  mpz_set_si(data,d);
}
template<> inline double Z_NR<mpz_t>::get_d_2exp(signed long int* expo)
{
  return mpz_get_d_2exp(expo,data);
}
template<> inline int Z_NR<mpz_t>::sgn()
{
  return mpz_sgn(data);
}

template<> inline void Z_NR<mpz_t>::abs(Z_NR<mpz_t>& x)
{
  return mpz_abs(data,x.data);
}
template<> template<> inline void Z_NR<mpz_t>::mul(Z_NR<mpz_t>& a,Z_NR<mpz_t>& b)
{
  mpz_mul(GetData(),a.GetData(),b.GetData());
}
template<> template<> inline void Z_NR<mpz_t>::mul(Z_NR<mpz_t>& a,Z_NR<long int>& b)
{
  mpz_mul_si(GetData(),a.GetData(),b.GetData());
}

template<> inline void Z_NR<mpz_t>::add(Z_NR<mpz_t>& a,Z_NR<mpz_t>& b)
{
  mpz_add(data,a.GetData(),b.GetData());
}
template<> inline void Z_NR<mpz_t>::mul_si(Z_NR<mpz_t>& a,signed long int b)
{
  mpz_mul_si(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::mul_si(Z_NR<mpz_t>& a,Z_NR<long int>&b)
{
  mpz_mul_si(data,a.GetData(),b.GetData());  
}
template<> inline void Z_NR<mpz_t>::mul_ui(Z_NR<mpz_t>& a,unsigned long int b)
{
  mpz_mul_ui(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::mul_2exp(Z_NR<mpz_t>& a, long int b)
{
  mpz_mul_2exp(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::div_2exp(Z_NR<mpz_t>& a, long int b)
{
  mpz_div_2exp(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::addmul_ui(Z_NR<mpz_t>& a,unsigned long int b)
{
  mpz_addmul_ui(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::submul_ui(Z_NR<mpz_t>& a,unsigned long int b)
{
  mpz_submul_ui(data,a.GetData(),b);
}

template<> template<> inline void Z_NR<mpz_t>::addmul(Z_NR<mpz_t>& a,Z_NR<long int>& b)
{
  if (b.sgn()>0)
    mpz_addmul_ui(data,a.GetData(),b.GetData());
  else
    mpz_submul_ui(data,a.GetData(),-(b.GetData()));
}
template<> template<> inline void Z_NR<mpz_t>::submul(Z_NR<mpz_t>& a,Z_NR<long int>& b)
{
  if (b.sgn()>0)
    mpz_submul_ui(data,a.GetData(),b.GetData());
  else
    mpz_addmul_ui(data,a.GetData(),-(b.GetData()));
}
template<> template<> inline void Z_NR<mpz_t>::addmul(Z_NR<mpz_t>& a,Z_NR<mpz_t>& b)
{
  mpz_addmul(data,a.GetData(),b.GetData());
}
template<> template<> inline void Z_NR<mpz_t>::submul(Z_NR<mpz_t>& a,Z_NR<mpz_t>& b)
{
  mpz_submul(data,a.GetData(),b.GetData());
}


template<> inline void Z_NR<mpz_t>::sub(Z_NR<mpz_t>& a,Z_NR<mpz_t>& b)
{
  mpz_sub(data,a.GetData(),b.GetData());
}
template<> inline void Z_NR<mpz_t>::randb(int bits)
{
  static bool init=false;
  static gmp_randstate_t state;
  if (!init)
    {
      gmp_randinit_default(state);
      init=true;
    }
  mpz_urandomb(data,state,bits);
}
template<> inline void Z_NR<mpz_t>::randm(Z_NR<mpz_t>& max)
{
  static bool init=false;
  static gmp_randstate_t state;
  if (!init)
    {
      gmp_randinit_default(state);
      init=true;
    }
  mpz_urandomm(data,state,max.GetData());
}
template<> inline void Z_NR<mpz_t>::add_ui(Z_NR<mpz_t>& a,unsigned int b)
{
  mpz_add_ui(data,a.GetData(),b);
}
template<> inline signed long int Z_NR<mpz_t>::get_si()
{
  return mpz_get_si(data);
}


/*********
 * FP_NR *
 *********/
template<class F> FP_NR<F>::FP_NR()
{
}
template<class F> FP_NR<F>::~FP_NR()
{
}
template<class F> inline void FP_NR<F>::print()
{
  cout << data;
}
template<class F> inline void FP_NR<F>::printerr()
{
  cerr << data;
}
template<class F> inline void FP_NR<F>::set(FP_NR<F>& s)
{
  data=s.GetData();
}
template<class F> inline void FP_NR<F>::set(unsigned int s)
{
  data=static_cast<F> ( s);
}
template<class F>inline  void FP_NR<F>::set(double s)
{
  data=static_cast<F> (s);
}
template<class F> inline double FP_NR<F>::get()
{
  return static_cast<double>(data);
}
template<class F> inline signed long int FP_NR<F>::get_si()
{
  return static_cast<signed long int>(data);
}
template<class F> inline F& FP_NR<F>::GetData()
{
  return data;
}
/*************************
 * double specialization *
 *************************/
template<> inline void FP_NR<double>::neg(FP_NR<double>& b)
{
  data=-b.GetData();
}
template<> inline void FP_NR<double>::mul(FP_NR<double>& b, FP_NR<double>& c)
{
  data= b.GetData()*c.GetData();
}
template<> inline void FP_NR<double>::add(FP_NR<double>& b, FP_NR<double>& c)
{
  data= b.GetData()+c.GetData();
}
template<> inline void FP_NR<double>::sub(FP_NR<double>& b, FP_NR<double>& c)
{
  data= b.GetData()-c.GetData();
}
template<> inline void FP_NR<double>::div(FP_NR<double>& b, FP_NR<double>& c)
{
  data= b.GetData()/c.GetData();
}
template<> inline void FP_NR<double>::div_2ui(FP_NR<double>& b, unsigned int c)
{
  data=b.GetData()/(1<<c);
}
template<> inline void FP_NR<double>::mul_2ui(FP_NR<double>& b, unsigned int c)
{
  data=b.GetData()*(1<<c);
}
template<> inline void FP_NR<double>::abs(FP_NR<double>& b)
{
  data=b.GetData();
  if (data<0) data=-data;
}

template<> inline int  FP_NR<double>::cmp(FP_NR<double>& b)
{
  if (data<=b.GetData())
    return -1;
  else
    return 1;
}
template<> inline int  FP_NR<double>::cmp(double b)
{
  if (data<=b)
    return -1;
  else
    return 1;
}
template<> inline int FP_NR<double>::sgn()
{
  if (data>0)
    return 1;
  if (data==0)
    return 0;
  return -1;
}
template<> inline void FP_NR<double>::rnd(FP_NR<double>& b)
{
  data=rint(b.GetData());
}

template<> inline int FP_NR<double>::zero_p()
{
  return (data==0);
}

template<> inline void FP_NR<double>::set_nan()
{
  data=NAN;//0.0/0.0;
}
template<> inline int FP_NR<double>::is_nan()
{
  return (data!=data);
}
template<> inline void FP_NR<double>::sqrt(FP_NR<double>& s)
{
  data=std::sqrt(s.GetData());
}

template<> inline int FP_NR<double>::exp()
{
  return ilogb(data)+1;
}


/*************
 * DPE spec. *
 *************/
template<> FP_NR<dpe_t>::FP_NR()
{
  dpe_init(data);
}
template<> FP_NR<dpe_t>::~FP_NR()
{
  dpe_clear(data);
}
template<> inline int FP_NR<dpe_t>::exp()
{
  return DPE_EXP(data);
}
template<> inline void FP_NR<dpe_t>::print()
{
  dpe_out_str(stdout,10,data);
  fflush(stdout);
}
template<> inline void FP_NR<dpe_t>::printerr()
{
  dpe_out_str(stderr,10,data);
  fflush(stderr);
}
template<> inline void FP_NR<dpe_t>::set(FP_NR<dpe_t>& f)
{
  dpe_set(data,f.GetData());
}
template<> inline void FP_NR<dpe_t>::set(double d)
{
  dpe_set_d(data,d);
}
template<> inline double FP_NR<dpe_t>::get()
{
  return dpe_get_d(data);
}
template<> inline void FP_NR<dpe_t>::set(unsigned int s)
{
  dpe_set_d(data,static_cast<double>(s));
}
template<> inline signed long int FP_NR<dpe_t>::get_si()
{
  return dpe_get_si(data);
}
template<> inline void FP_NR<dpe_t>::neg(FP_NR<dpe_t>& a)
{
  dpe_neg(data,a.GetData());
}
template<> inline void FP_NR<dpe_t>::mul(FP_NR<dpe_t>& a,FP_NR<dpe_t>& b)
{
  dpe_mul(data,a.GetData(),b.GetData());
}
template<> inline void FP_NR<dpe_t>::add(FP_NR<dpe_t>& a,FP_NR<dpe_t>& b)
{
  dpe_add(data,a.GetData(),b.GetData());
}
template<> inline void FP_NR<dpe_t>::sub(FP_NR<dpe_t>& a,FP_NR<dpe_t>& b)
{
  dpe_sub(data,a.GetData(),b.GetData());
}
template<> inline void FP_NR<dpe_t>::div(FP_NR<dpe_t>& a,FP_NR<dpe_t>& b)
{
  dpe_div(data,a.GetData(),b.GetData());
}
template<> inline void FP_NR<dpe_t>::div_2ui(FP_NR<dpe_t>& a,unsigned int b)
{
  dpe_div_2exp(data,a.GetData(),b);
}
template<> inline void FP_NR<dpe_t>::mul_2ui(FP_NR<dpe_t>& a,unsigned int b)
{
  dpe_mul_2exp(data,a.GetData(),b);
}
template<> inline void FP_NR<dpe_t>::abs(FP_NR<dpe_t>& a)
{
  dpe_abs(data,a.GetData());
}
template<> inline int FP_NR<dpe_t>::cmp(FP_NR<dpe_t>& a)
{
  return dpe_cmp(data,a.GetData());
}
template<> inline int FP_NR<dpe_t>::cmp(double a)
{
  return dpe_cmp_d(data,a);
}
template<> inline int FP_NR<dpe_t>::sgn()
{
  return cmp(0.0);
}
template<> inline void FP_NR<dpe_t>::rnd(FP_NR<dpe_t>& a)
{
  dpe_round(data,a.GetData());
}
template<> inline void FP_NR<dpe_t>::sqrt(FP_NR<dpe_t>& a)
{
  dpe_sqrt(data,a.GetData());
}
template<> inline int FP_NR<dpe_t>::zero_p()
{
  return dpe_zero_p(data);
}
template<> inline void FP_NR<dpe_t>::set_nan()
{
  dpe_set_d(data,NAN/*0.0/0.0*/);
}
template<> inline int FP_NR<dpe_t>::is_nan()
{
  return (DPE_MANT(data)!=DPE_MANT(data));
}



/*************
 * MPFR spec. *
 *************/
FP_NR<mpfr_t>::FP_NR()
{
  mpfr_init(data);
}
FP_NR<mpfr_t>::~FP_NR()
{
  mpfr_clear(data);
}
inline mpfr_t& FP_NR<mpfr_t>::GetData()
{
  return data;
}
inline int FP_NR<mpfr_t>::exp()
{
  return mpfr_get_exp(data);
}
inline void FP_NR<mpfr_t>::print()
{
  mpfr_out_str(stdout,10,10,data,GMP_RNDN);
  fflush(stdout);
}
inline void FP_NR<mpfr_t>::printerr()
{
  mpfr_out_str(stderr,10,5,data,GMP_RNDN);
  fflush(stderr);
}
inline void FP_NR<mpfr_t>::set(FP_NR<mpfr_t>& f)
{
  mpfr_set(data,f.GetData(),GMP_RNDN);
}
inline void FP_NR<mpfr_t>::set(double d)
{
  mpfr_set_d(data,d,GMP_RNDN);
}
inline double FP_NR<mpfr_t>::get()
{
  return mpfr_get_d(data,GMP_RNDN);
}
inline void FP_NR<mpfr_t>::set(unsigned int s)
{
  mpfr_set_ui(data,s,GMP_RNDN);
}
inline signed long int FP_NR<mpfr_t>::get_si()
{
  return mpfr_get_si(data,GMP_RNDN);
}
inline void FP_NR<mpfr_t>::neg(FP_NR<mpfr_t>& a)
{
  mpfr_neg(data,a.GetData(),GMP_RNDN);
}
inline void FP_NR<mpfr_t>::mul(FP_NR<mpfr_t>& a,FP_NR<mpfr_t>& b)
{
  mpfr_mul(data,a.GetData(),b.GetData(),GMP_RNDN);
}
inline void FP_NR<mpfr_t>::add(FP_NR<mpfr_t>& a,FP_NR<mpfr_t>& b)
{
  mpfr_add(data,a.GetData(),b.GetData(),GMP_RNDN);
}
inline void FP_NR<mpfr_t>::sub(FP_NR<mpfr_t>& a,FP_NR<mpfr_t>& b)
{
  mpfr_sub(data,a.GetData(),b.GetData(),GMP_RNDN);
}
inline void FP_NR<mpfr_t>::div(FP_NR<mpfr_t>& a,FP_NR<mpfr_t>& b)
{
  mpfr_div(data,a.GetData(),b.GetData(),GMP_RNDN);
}
inline void FP_NR<mpfr_t>::div_2ui(FP_NR<mpfr_t>& a,unsigned int b)
{
  mpfr_div_2ui(data,a.GetData(),b,GMP_RNDN);
}
inline void FP_NR<mpfr_t>::mul_2ui(FP_NR<mpfr_t>& a,unsigned int b)
{
  mpfr_mul_2exp(data,a.GetData(),b,GMP_RNDN);
}
inline void FP_NR<mpfr_t>::abs(FP_NR<mpfr_t>& a)
{
  mpfr_abs(data,a.GetData(),GMP_RNDN);
}
inline int FP_NR<mpfr_t>::cmp(FP_NR<mpfr_t>& a)
{
  return mpfr_cmp(data,a.GetData());
}
inline int FP_NR<mpfr_t>::cmp(double a)
{
  return mpfr_cmp_d(data,a);
}
inline int FP_NR<mpfr_t>::sgn()
{
  return mpfr_sgn(data);
}
inline void FP_NR<mpfr_t>::rnd(FP_NR<mpfr_t>& a)
{
  mpfr_round(data,a.GetData());
}
inline void FP_NR<mpfr_t>::sqrt(FP_NR<mpfr_t>& a)
{
  mpfr_sqrt(data,a.GetData(),GMP_RNDN);
}
inline int FP_NR<mpfr_t>::zero_p()
{
  return mpfr_zero_p(data);
}
inline void FP_NR<mpfr_t>::set_nan()
{
  mpfr_set_nan(data);
}
inline int FP_NR<mpfr_t>::is_nan()
{
  return mpfr_nan_p(data);
}

#endif
