/* -*- mode: c++; coding: utf-8; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; show-trailing-whitespace: t  -*-

  This file is part of the Feel library

  Author(s): Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
       Date: 2012-02-05

  Copyright (C) 2012 Université Joseph Fourier (Grenoble I)

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
   \file dar.cpp
   \author Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
   \date 2012-02-05
 */
/** include predefined feel command line options */
#include <feel/options.hpp>

/** include linear algebra backend */
#include <feel/feelalg/backend.hpp>

/** include function space class */
#include <feel/feeldiscr/functionspace.hpp>

/** include gmsh mesh importer */
#include <feel/feelfilters/gmsh.hpp>

/** include exporter factory class */
#include <feel/feelfilters/exporter.hpp>

/** include  the header for the variational formulation language (vf) aka FEEL++ */
#include <feel/feelvf/vf.hpp>

/** use Feel namespace */
using namespace Feel;
using namespace Feel::vf;


/**
 * This routine returns the list of options using the
 * boost::program_options library. The data returned is typically used
 * as an argument of a Feel::Application subclass.
 *
 * \return the list of options
 */
inline
po::options_description
makeOptions()
{
    po::options_description DARoptions("DAR options");
    DARoptions.add_options()
        ("hsize", po::value<double>()->default_value( 0.5 ), "mesh size")
        ("shape", Feel::po::value<std::string>()->default_value( "hypercube" ), "shape of the domain (either simplex or hypercube)")
        ("bx", Feel::po::value<double>()->default_value( 1.0 ), "convection X component")
        ("by", Feel::po::value<double>()->default_value( 0.0 ), "convection Y component")
        ("bz", Feel::po::value<double>()->default_value( 0.0 ), "convection Z component")
        ("epsilon", po::value<double>()->default_value( 1 ), "diffusion coefficient")
        ("mu", po::value<double>()->default_value( 1 ), "reaction coefficient")
        ("weakdir", po::value<int>()->default_value( 1 ), "use weak Dirichlet condition" )
        ("penaldir", Feel::po::value<double>()->default_value( 10 ),
         "penalisation parameter for the weak boundary Dirichlet formulation")
        ("stab", po::value<int>()->default_value( 1 ), "use stabilisation=1, no stabilisation=0" )
        ("stabcoeff", Feel::po::value<double>()->default_value( 2.5e-2 ),
         "stabilisation coefficient")
        ;
    return DARoptions.add( Feel::feel_options() );
}

/**
 * This routine defines some information about the application like
 * authors, version, or name of the application. The data returned is
 * typically used as an argument of a Feel::Application subclass.
 *
 * \return some data about the application.
 */
inline
AboutData
makeAbout()
{
    AboutData about( "DAR" ,
                     "DAR" ,
                     "0.2",
                     "nD(n=1,2,3) DAR on simplices or simplex products",
                     Feel::AboutData::License_GPL,
                     "Copyright (c) 2008-2009 Universite Joseph Fourier");

    about.addAuthor("Christophe Prud'homme", "developer", "christophe.prudhomme@ujf-grenoble.fr", "");
    return about;

}


/**
 * \class DAR
 *
 * DAR Solver using continuous approximation spaces
 * solve \f$ -\Delta u = f\f$ on \f$\Omega\f$ and \f$u= g\f$ on \f$\Gamma\f$
 *
 * \tparam Dim the geometric dimension of the problem (e.g. Dim=1, 2 or 3)
 */
template<int Dim>
class DAR
    :
    public Simget
{
    typedef Simget super;
public:

    //! Polynomial order \f$P_2\f$
    static const uint16_type Order = 2;

    //! numerical type is double
    typedef double value_type;

    //! geometry entities type composing the mesh, here Simplex in Dimension Dim of Order 1
    typedef Simplex<Dim> convex_type;
    //! mesh type
    typedef Mesh<convex_type> mesh_type;
    //! mesh shared_ptr<> type
    typedef boost::shared_ptr<mesh_type> mesh_ptrtype;

    //! the basis type of our approximation space
    typedef bases<Lagrange<Order,Scalar> > basis_type;

    //! the approximation function space type
    typedef FunctionSpace<mesh_type, basis_type> space_type;
    //! the approximation function space type (shared_ptr<> type)
    typedef boost::shared_ptr<space_type> space_ptrtype;
    //! an element type of the approximation function space
    typedef typename space_type::element_type element_type;

    //! the exporter factory type
    typedef Exporter<mesh_type> export_type;
    //! the exporter factory (shared_ptr<> type)
    typedef boost::shared_ptr<export_type> export_ptrtype;

    /**
     * Constructor
     */
    DAR( po::variables_map const& vm, AboutData const& about )
        :
        super( vm, about ),
        meshSize( this->vm()["hsize"].template as<double>() ),
        shape( this->vm()["shape"].template as<std::string>() )
    {
    }

    void run();

    void run( const double* X, unsigned long P, double* Y, unsigned long N );

private:

    //! mesh characteristic size
    double meshSize;

    //! shape of the domain
    std::string shape;
}; // DAR

template<int Dim> const uint16_type DAR<Dim>::Order;

template<int Dim>
void
DAR<Dim>::run()
{
    std::cout << "------------------------------------------------------------\n";
    std::cout << "Execute DAR<" << Dim << ">\n";
    std::vector<double> X( 2 );
    X[0] = meshSize;
    if ( shape == "hypercube" )
        X[1] = 1;
    else // default is simplex
        X[1] = 0;
    std::vector<double> Y( 3 );
    run( X.data(), X.size(), Y.data(), Y.size() );
}
template<int Dim>
void
DAR<Dim>::run( const double* X, unsigned long P, double* Y, unsigned long N )
{
    using namespace Feel::vf;
    using  vf::N;

    if ( X[1] == 0 ) shape = "simplex";
    if ( X[1] == 1 ) shape = "hypercube";

    if ( !this->vm().count( "nochdir" ) )
        Environment::changeRepository( boost::format( "doc/tutorial/%1%/%2%-%3%/P%4%/h_%5%/" )
                                       % this->about().appName()
                                       % shape
                                       % Dim
                                       % Order
                                       % meshSize );

    mesh_ptrtype mesh = createGMSHMesh( _mesh=new mesh_type,
                                        _desc=domain( _name=(boost::format( "%1%-%2%" ) % shape % Dim).str() ,
                                                      _usenames=true,
                                                      _shape=shape,
                                                      _dim=Dim,
                                                      _h=X[0] ),
                                        _update=MESH_RENUMBER|MESH_UPDATE_EDGES|MESH_UPDATE_FACES|MESH_CHECK,
                                        _partitions=this->comm().size());

    /**
     * The function space and some associated elements(functions) are then defined
     */
    /** \code */
    space_ptrtype Xh = space_type::New( mesh );
    element_type u( Xh, "u" );
    element_type v( Xh, "v" );
    element_type gproj( Xh, "v" );
    /** \endcode */

    bool weak_dirichlet = this->vm()["weakdir"].template as<int>();
    value_type penaldir = this->vm()["penaldir"].template as<double>();
    bool stab = this->vm()["stab"].template as<int>();
    value_type stabcoeff = this->vm()["stabcoeff"].template as<double>();
    value_type epsilon = this->vm()["epsilon"].template as<double>();
    value_type mu = this->vm()["mu"].template as<double>();
    value_type bx = this->vm()["bx"].template as<double>();
    value_type by = this->vm()["by"].template as<double>();

    std::cout << "[DAR] hsize = " << X[0] << "\n";
    std::cout << "[DAR] bx = " << bx << "\n";
    std::cout << "[DAR] by = " << by << "\n";
    std::cout << "[DAR] mu = " << mu << "\n";
    std::cout << "[DAR] epsilon = " << epsilon << "\n";
    std::cout << "[DAR] bccoeff = " << penaldir << "\n";
    std::cout << "[DAR] bctype = " << weak_dirichlet << "\n";
    std::cout << "[DAR] stab = " << stab << "\n";
    std::cout << "[DAR] stabcoeff = " << stabcoeff << "\n";



    /** define \f$g\f$ the expression of the exact solution and
     * \f$f\f$ the expression of the right hand side such that \f$g\f$
     * is the exact solution
     */
    /** \code */
    //# marker1 #
    value_type pi = M_PI;
    //! deduce from expression the type of g (thanks to keyword 'auto')
    auto g = sin(pi*Px())*cos(pi*Py())*cos(pi*Pz());
    gproj = vf::project( Xh, elements(mesh), g );
    auto grad_g = vec( +pi*cos(pi*Px())*cos(pi*Py())*cos(pi*Pz()),
                       -pi*sin(pi*Px())*sin(pi*Py())*cos(pi*Pz()) );
    auto beta = vec(cst(bx),cst(by));
    //! deduce from expression the type of f (thanks to keyword 'auto')
    auto f = (pi*pi*Dim*epsilon*g+trans(grad_g)*beta + mu*g);
    //# endmarker1 #
    /** \endcode */





    /**
     * Construction of the right hand side. F is the vector that holds
     * the algebraic representation of the right habd side of the
     * problem
     */
    /** \code */
    //# marker2 #
    auto F = backend(_vm=this->vm())->newVector( Xh );
    form1( _test=Xh, _vector=F, _init=true ) =
        integrate( _range=elements(mesh), _expr=f*id(v) );
    //# endmarker2 #
    if ( weak_dirichlet )
    {
        //# marker41 #
#if 1
        form1( _test=Xh, _vector=F ) +=
            integrate( _range=boundaryfaces(mesh),
                       _expr=g*(-epsilon*grad(v)*vf::N()+penaldir*id(v)/hFace()) );
#endif
        //# endmarker41 #
    }
    /** \endcode */

    /**
     * create the matrix that will hold the algebraic representation
     * of the left hand side
     */
    //# marker3 #
    /** \code */
    size_type pattern=Pattern::COUPLED;
    if ( stab )
        pattern = Pattern::COUPLED|Pattern::EXTENDED;
    auto D = backend()->newMatrix( _test=Xh, _trial=Xh, _pattern=pattern  );
    /** \endcode */

    //! assemble $\int_\Omega \epsilon \nabla u \cdot \nabla v$
    /** \code */
    form2( _test=Xh, _trial=Xh, _matrix=D) =
        integrate( _range=elements(mesh),
                   _expr=(epsilon*gradt(u)*trans(grad(v))+(gradt(u)*beta)*id(v)+mu*idt(u)*id(v) ) );
    /** \endcode */
    //# endmarker3 #

    if ( stab )
    {
        // define the stabilisation coefficient expression
        auto stab_coeff = (stabcoeff*abs(bx*Nx()+by*Ny())* vf::pow(hFace(),2.0));
        form2( _test=Xh, _trial=Xh, _matrix=D) +=
            integrate( _range=internalfaces(mesh),
                       _expr=stab_coeff*(trans(jumpt(gradt(u)))*jump(grad(v))));
    }

    if ( weak_dirichlet )
        {
            /** weak dirichlet conditions treatment for the boundaries marked 1 and 3
             * -# assemble \f$\int_{\partial \Omega} -\nabla u \cdot \mathbf{n} v\f$
             * -# assemble \f$\int_{\partial \Omega} -\nabla v \cdot \mathbf{n} u\f$
             * -# assemble \f$\int_{\partial \Omega} \frac{\gamma}{h} u v\f$
             */
            /** \code */
            //# marker10 #
#if 1
            form2( _test=Xh, _trial=Xh, _matrix=D ) +=
                integrate( _range=boundaryfaces(mesh),
                           _expr= ( -(epsilon*gradt(u)*vf::N())*id(v)
                                    -(epsilon*grad(v)*vf::N())*idt(u)
                                    +penaldir*id(v)*idt(u)/hFace() ) );
#else
            form2( _test=Xh, _trial=Xh, _matrix=D ) +=
                integrate( _range=boundaryfaces(mesh),
                           _expr= ( -(epsilon*gradt(u)*vf::N())*id(v)));


#endif
            //# endmarker10 #
            /** \endcode */
        }
    else
        {
            /** strong(algebraic) dirichlet conditions treatment for the boundaries marked 1 and 3
             * -# first close the matrix (the matrix must be closed first before any manipulation )
             * -# modify the matrix by cancelling out the rows and columns of D that are associated with the Dirichlet dof
             */
            /** \code */
            //# marker5 #
#if 1
            form2( _test=Xh, _trial=Xh, _matrix=D ) +=
                on( _range=boundaryfaces(mesh),_element=u, _rhs=F, _expr=g );
#endif
            //# endmarker5 #
            /** \endcode */

        }
    /** \endcode */

    //! solve the system
    /** \code */
	//# marker6 #
    backend(_rebuild=true,_vm=this->vm())->solve( _matrix=D, _solution=u, _rhs=F );
	//# endmarker6 #
    /** \endcode */

    //! compute the \f$L_2$ norm of the error
    /** \code */
    //# marker7 #
    double L2error2 =integrate(_range=elements(mesh),
                               _expr=(idv(u)-g)*(idv(u)-g) ).evaluate()(0,0);
    double L2error =   math::sqrt( L2error2 );


    Log() << "||error||_L2=" << L2error << "\n";
    std::cout << "||error||_L2=" << L2error << "\n";
    //# endmarker7 #
    /** \endcode */

    //! save the results
    /** \code */
    //! project the exact solution
    element_type e( Xh, "e" );
    e = vf::project( Xh, elements(mesh), g );

    export_ptrtype exporter( export_type::New( this->vm(),
                                               (boost::format( "%1%-%2%-%3%" )
                                                % this->about().appName()
                                                % shape
                                                % Dim).str() ) );
    if ( exporter->doExport() )
    {
        Log() << "exportResults starts\n";

        exporter->step(0)->setMesh( mesh );

        exporter->step(0)->add( "u", u );
        exporter->step(0)->add( "g", e );

        exporter->save();
        Log() << "exportResults done\n";
    }
    /** \endcode */
} // DAR::run

/**
 * main function: entry point of the program
 */
int
main( int argc, char** argv )
{
    Environment env( argc, argv );
    /**
     * create an application
     */
    /** \code */
    Application app( argc, argv, makeAbout(), makeOptions() );
    if ( app.vm().count( "help" ) )
    {
        std::cout << app.optionsDescription() << "\n";
        return 0;
    }
    /** \endcode */

    /**
     * register the simgets
     */
    /** \code */
    app.add( new DAR<2>( app.vm(), app.about() ) );
    /** \endcode */

    /**
     * run the application
     */
    /** \code */
    app.run();
    /** \endcode */
}






