/*
 * $Id: sig_gen_bus.c,v 1.8 2011-01-15 13:39:08 vrsieh Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include "compiler.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fixme.h"

#include "sig_gen_bus.h"

#define NEED_CONFIG_SPACE
#define NEED_INTA
#undef NEED_X

int
bus_(main_type_addr)(
	struct bus_(main) *b,
	void *s,
	unsigned int type,
	bus_addr_t addr
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, bus_addr_t);
		void *func_s;

		if (nr == b->member_count) {
			/* Not found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		func = b->member[nr].f->type_addr;
		func_s = b->member[nr].s;
		if (func
		 && func(func_s, type, addr) == 0) {
			return 0;
		}
	}
}

int
bus_(main_read_data)(
	struct bus_(main) *b,
	void *s,
	unsigned int bs,
	bus_data_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, bus_data_t *);
		void *func_s;

		if (nr == b->member_count) {
			/* Not found. */
			*valp = (bus_data_t) -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		func = b->member[nr].f->read_data;
		func_s = b->member[nr].s;
		if (func
		 && func(func_s, bs, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(main_write_data)(
	struct bus_(main) *b,
	void *s,
	unsigned int bs,
	bus_data_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, bus_data_t);
		void *func_s;

		if (nr == b->member_count) {
			/* Not found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		func = b->member[nr].f->write_data;
		func_s = b->member[nr].s;
		if (func
		 && func(func_s, bs, val) == 0) {
			return 0;
		}
	}
}

#ifdef NEED_CONFIG_SPACE
int
bus_(main_c0r)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t *);
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = (bus_data_t) -1;
			return 1;
		}

		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}

		func = b->member[nr].f->c0r;
		if (func
		 && func(b->member[nr].s, addr, bs, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(main_c0w)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t);
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}

		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}

		func = b->member[nr].f->c0w;
		if (func
		 && func(b->member[nr].s, addr, bs, val) == 0) {
			return 0;
		}
	}
}

int
bus_(c1r)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t *);
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}

		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}

		func = b->member[nr].f->c1r;
		if (func
		 && func(b->member[nr].s, addr, bs, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(c1w)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t);
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}

		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}

		func = b->member[nr].f->c1w;
		if (func
		 && func(b->member[nr].s, addr, bs, val) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_CONFIG_SPACE*/

static inline __attribute__((always_inline)) unsigned int
bus_(_io_hash)(bus_addr_t port, unsigned int bs)
{
	unsigned int val;

	val = port / sizeof(bus_addr_t);
	val ^= bs;
	val %= BUS_HASH_SIZE;

	return val;
}

static inline __attribute__((always_inline)) int
bus_(_ior_lookup)(
	struct bus_(main) *b,
	bus_addr_t port,
	unsigned int bs,
	int (**f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t *valp),
	void **s
)
{
	unsigned int hash;
	struct bus_(main_io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (unlikely(! m)) {
			/* Not found. */
			return 0;
		}
		if (likely(m->type == BUS_(TYPE_IOR)
		 && m->port == port
		 && m->bs == bs)) {
			/* Found. */
			*f = m->ior;
			*s = m->s;
			return 1;
		}
	}
}

static void
bus_(_ior_add)(
	struct bus_(main) *b,
	bus_addr_t port,
	unsigned int bs,
	int (*f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t *valp),
	void *s
)
{
	unsigned int hash;
	struct bus_(main_io) *m;

	m = b->io_lru_last;

	/* Remove from LRU list. */
	m->lru_prev->lru_next = 0;
	b->io_lru_last = m->lru_prev;

	if (m->port != -1) {
		/* Remove from HASH list. */
		hash = bus_(_io_hash)(m->port, m->bs);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->io_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->io_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_IOR);
	m->port = port;
	m->bs = bs;
	m->ior = f;
	m->s = s;

	/* Add to HASH list. */
	hash = bus_(_io_hash)(port, bs);
	m->hash_prev = 0;
	m->hash_next = b->io_hash_first[hash];
	b->io_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->io_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->io_lru_first;
	b->io_lru_first = m;
	m->lru_next->lru_prev = m;
}

int
bus_(ior_info)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*info_f)(void *, bus_addr_t, unsigned int,
			int (**)(void *, bus_addr_t, unsigned int, bus_data_t *),
			void **);
		void *info_s;

		if (nr == b->member_count) {
			/* No info found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		info_f = b->member[nr].f->ior_info;
		info_s = b->member[nr].s;
		if (info_f
		 && info_f(info_s, port, bs, cfp, csp) == 0) {
			return 0;
		}
	}
}

void
bus_(ior_info_flush)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t port,
	unsigned int bs
)
{
	unsigned int hash;
	struct bus_(main_io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/* Not found. */
			return;
		}
		if (m->type == BUS_(TYPE_IOR)
		 && port == m->port
		 && (bs == m->bs
		  || bs == 0)) {
			/* Found. */
			/* Remove from LRU list. */
			if (m->lru_prev) {
				m->lru_prev->lru_next = m->lru_next;
			} else {
				b->io_lru_first = m->lru_next;
			}
			if (m->lru_next) {
				m->lru_next->lru_prev = m->lru_prev;
			} else {
				b->io_lru_last = m->lru_prev;
			}

			/* Remove from HASH list. */
			hash = bus_(_io_hash)(port, bs);
			if (m->hash_prev) {
				m->hash_prev->hash_next = m->hash_next;
			} else {
				b->io_hash_first[hash] = m->hash_next;
			}
			if (m->hash_next) {
				m->hash_next->hash_prev = m->hash_prev;
			} else {
				b->io_hash_last[hash] = m->hash_prev;
			}

			/* Remove info. */
			m->port = -1;
			m->bs = -1;

			/* Don't add empty entry to HASH list. */

			/* Add to LRU list. */
			m->lru_prev = b->io_lru_last;
			m->lru_next = 0;
			m->lru_prev->lru_next = m;
			b->io_lru_last = m;
			return;
		}
	}
}

static int
bus_(_ior_dummy)(
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	*valp = (bus_data_t) -1;
	return 1;
}

int
bus_(ior)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	int (*cf)(void *, bus_addr_t, unsigned int, bus_data_t *);
	void *cs;
	unsigned int nr;

	assert(! (port & (sizeof(bus_data_t) - 1)));

	if (bus_(_ior_lookup)(b, port, bs, &cf, &cs)) {
		return (*cf)(cs, port, bs, valp);
	}

	if (bus_(ior_info)(b, s, port, bs, &cf, &cs) == 0) {
		bus_(_ior_add)(b, port, bs, cf, cs);
		return cf(cs, port, bs, valp);
	}

#if 0
	fprintf(stderr, "INFO: No IOR info for 0x%04x 0x%x\n", port, bs);
#endif

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			break;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		cf = b->member[nr].f->ior;
		cs = b->member[nr].s;
		if (cf
		 && cf(cs, port, bs, valp) == 0) {
			bus_(_ior_add)(b, port, bs, cf, cs);
			return 0;
		}
	}

	cf = bus_(_ior_dummy);
	cs = 0;
	bus_(_ior_add)(b, port, bs, cf, cs);
	return 1;
}

static inline __attribute__((always_inline)) int
bus_(_iow_lookup)(
	struct bus_(main) *b,
	bus_addr_t port,
	unsigned int bs,
	int (**f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t val),
	void **s
)
{
	unsigned int hash;
	struct bus_(main_io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (unlikely(! m)) {
			/* Not found. */
			return 0;
		}
		if (likely(m->type == BUS_(TYPE_IOW)
		 && m->port == port
		 && m->bs == bs)) {
			/* Found. */
			*f = m->iow;
			*s = m->s;
			return 1;
		}
	}
}

static void
bus_(_iow_add)(
	struct bus_(main) *b,
	bus_addr_t port,
	unsigned int bs,
	int (*f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t val),
	void *s
)
{
	unsigned int hash;
	struct bus_(main_io) *m;

	m = b->io_lru_last;

	/* Remove from LRU list. */
	m->lru_prev->lru_next = 0;
	b->io_lru_last = m->lru_prev;

	if (m->port != -1) {
		/* Remove from HASH list. */
		hash = bus_(_io_hash)(m->port, m->bs);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->io_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->io_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_IOW);
	m->port = port;
	m->bs = bs;
	m->iow = f;
	m->s = s;

	/* Add to HASH list. */
	hash = bus_(_io_hash)(port, bs);
	m->hash_prev = 0;
	m->hash_next = b->io_hash_first[hash];
	b->io_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->io_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->io_lru_first;
	b->io_lru_first = m;
	m->lru_next->lru_prev = m;
}

int
bus_(iow_info)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*info_f)(void *, bus_addr_t, unsigned int,
			int (**)(void *, bus_addr_t, unsigned int, bus_data_t),
			void **);
		void *info_s;

		if (nr == b->member_count) {
			/* No info found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		info_f = b->member[nr].f->iow_info;
		info_s = b->member[nr].s;
		if (info_f
		 && info_f(info_s, port, bs, cfp, csp) == 0) {
			return 0;
		}
	}
}

void
bus_(iow_info_flush)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t port,
	unsigned int bs
)
{
	unsigned int hash;
	struct bus_(main_io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/* Not found. */
			return;
		}
		if (m->type == BUS_(TYPE_IOW)
		 && port == m->port
		 && (bs == m->bs
		  || bs == 0)) {
			/* Found. */
			/* Remove from LRU list. */
			if (m->lru_prev) {
				m->lru_prev->lru_next = m->lru_next;
			} else {
				b->io_lru_first = m->lru_next;
			}
			if (m->lru_next) {
				m->lru_next->lru_prev = m->lru_prev;
			} else {
				b->io_lru_last = m->lru_prev;
			}

			/* Remove from HASH list. */
			hash = bus_(_io_hash)(port, bs);
			if (m->hash_prev) {
				m->hash_prev->hash_next = m->hash_next;
			} else {
				b->io_hash_first[hash] = m->hash_next;
			}
			if (m->hash_next) {
				m->hash_next->hash_prev = m->hash_prev;
			} else {
				b->io_hash_last[hash] = m->hash_prev;
			}

			/* Remove info. */
			m->port = -1;
			m->bs = -1;

			/* Don't add empty entry to HASH list. */

			/* Add to LRU list. */
			m->lru_prev = b->io_lru_last;
			m->lru_next = 0;
			m->lru_prev->lru_next = m;
			b->io_lru_last = m;
			return;
		}
	}
}

static int
bus_(_iow_dummy)(
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	return 1;
}

int
bus_(iow)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	int (*cf)(void *, bus_addr_t, unsigned int, bus_data_t);
	void *cs;
	unsigned int nr;

	assert((port & (sizeof(bus_data_t) - 1)) == 0);

	if (bus_(_iow_lookup)(b, port, bs, &cf, &cs)) {
		return cf(cs, port, bs, val);
	}

	if (bus_(iow_info)(b, s, port, bs, &cf, &cs) == 0) {
		bus_(_iow_add)(b, port, bs, cf, cs);
		return cf(cs, port, bs, val);
	}

#if 0
	fprintf(stderr, "INFO: No IOW info for 0x%04x 0x%x\n", port, bs);
#endif

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			break;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		cf = b->member[nr].f->iow;
		cs = b->member[nr].s;
		if (cf
		 && cf(cs, port, bs, val) == 0) {
			bus_(_iow_add)(b, port, bs, cf, cs);
			return 0;
		}
	}

	cf = bus_(_iow_dummy);
	cs = 0;
	bus_(_iow_add)(b, port, bs, cf, cs);
	return 1;
}

static inline unsigned int
bus_(_map_hash)(bus_addr_t addr)
{
	return (addr >> 12) % BUS_HASH_SIZE;
}

static void
bus_(_map_flush)(
	struct bus_(main) *b,
	struct bus_(main_map) *m
)
{
	unsigned int hash;

	/* Remove from LRU list. */
	if (m->lru_prev) {
		m->lru_prev->lru_next = m->lru_next;
	} else {
		b->map_lru_first = m->lru_next;
	}
	if (m->lru_next) {
		m->lru_next->lru_prev = m->lru_prev;
	} else {
		b->map_lru_last = m->lru_prev;
	}

	/* Remove from HASH list. */
	hash = bus_(_map_hash)(m->addr);
	if (m->hash_prev) {
		m->hash_prev->hash_next = m->hash_next;
	} else {
		b->map_hash_first[hash] = m->hash_next;
	}
	if (m->hash_next) {
		m->hash_next->hash_prev = m->hash_prev;
	} else {
		b->map_hash_last[hash] = m->hash_prev;
	}

	/* Unmap page. */
	m->addr = -1;

	/* Don't add empty entry to HASH list. */

	/* Add to LRU list. */
	m->lru_prev = b->map_lru_last;
	m->lru_next = 0;
	m->lru_prev->lru_next = m;
	b->map_lru_last = m;
}

static inline __attribute__((always_inline)) struct bus_(main_map) *
bus_(_map_r_lookup)(
	struct bus_(main) *b,
	bus_addr_t addr
)
{
	unsigned int hash;
	struct bus_(main_map) *m;

	addr &= ~0xfff;

	hash = bus_(_map_hash)(addr);

	for (m = b->map_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/*
			 * Not Found
			 */
			return NULL;
		}
		if (m->type == BUS_(TYPE_MR)
		 && m->addr == addr) {
			/*
			 * Found
			 */
			/* Don't update LRU list. */
			/* Costs too much... */
			return m;
		}
	}
}

static inline __attribute__((always_inline)) struct bus_(main_map) *
bus_(_map_r_add)(
	struct bus_(main) *b,
	bus_addr_t addr
)
{
	char *haddr;
	unsigned int hash;
	struct bus_(main_map) *m;

	addr &= ~0xfff;

	if (bus_(map_r_check)(b, b, addr)
	 || bus_(map_r)(b, b, addr, &haddr)) {
		haddr = NULL;
	}

	/* Get entry from LRU list. */
	m = b->map_lru_last;

	/* Remove from LRU list. */
	assert(m->lru_prev);
	m->lru_prev->lru_next = 0;
	b->map_lru_last = m->lru_prev;

	if (m->addr != -1) {
		/* Remove from HASH list. */
		hash = bus_(_map_hash)(m->addr);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->map_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->map_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_MR);
	m->addr = addr;
	m->mr = (bus_data_t *) haddr;

	/* Add to HASH list. */
	hash = bus_(_map_hash)(addr);
	m->hash_prev = 0;
	m->hash_next = b->map_hash_first[hash];
	b->map_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->map_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->map_lru_first;
	b->map_lru_first = m;
	m->lru_next->lru_prev = m;

	return m;
}

int
bus_(mr)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_map) *m;
	const bus_data_t *from;
	unsigned int nr;
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t *);

	m = bus_(_map_r_lookup)(b, addr);
	if (! m) {
		m = bus_(_map_r_add)(b, addr);
	}
	if (m->mr) {
		from = (const bus_data_t *) m->mr
				+ (addr & 0xfff) / sizeof(bus_data_t);

		/* No need to obey `bs'. */
		*valp = *from;

		return 0;
	}

	/*
	 * Not mappable. Call function...
	 */
	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = (bus_data_t) -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->mr;
		if (func
		 && func(b->member[nr].s, addr, bs, valp) == 0) {
			return 0;
		}
	}
}

static inline __attribute__((always_inline)) struct bus_(main_map) *
bus_(_map_w_lookup)(
	struct bus_(main) *b,
	bus_addr_t addr
)
{
	unsigned int hash;
	struct bus_(main_map) *m;

	addr &= ~0xfff;

	hash = bus_(_map_hash)(addr);

	for (m = b->map_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/*
			 * Not Found
			 */
			return NULL;
		}
		if (m->type == BUS_(TYPE_MW)
		 && m->addr == addr) {
			/*
			 * Found
			 */
			/* Don't update LRU list. */
			/* Costs too much... */
			return m;
		}
	}
}

static inline __attribute__((always_inline)) struct bus_(main_map) *
bus_(_map_w_add)(
	struct bus_(main) *b,
	bus_addr_t addr
)
{
	char *haddr;
	unsigned int hash;
	struct bus_(main_map) *m;

	addr &= ~0xfff;

	if (bus_(map_w_check)(b, b, addr)
	 || bus_(map_w)(b, b, addr, &haddr)) {
		haddr = NULL;
	}

	/* Get entry from LRU list. */
	m = b->map_lru_last;

	/* Remove from LRU list. */
	assert(m->lru_prev);
	m->lru_prev->lru_next = 0;
	b->map_lru_last = m->lru_prev;

	if (m->addr != -1) {
		/* Remove from HASH list. */
		hash = bus_(_map_hash)(m->addr);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->map_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->map_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_MW);
	m->addr = addr;
	m->mw = (bus_data_t *) haddr;

	/* Add to HASH list. */
	hash = bus_(_map_hash)(addr);
	m->hash_prev = 0;
	m->hash_next = b->map_hash_first[hash];
	b->map_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->map_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->map_lru_first;
	b->map_lru_first = m;
	m->lru_next->lru_prev = m;

	return m;
}

int
bus_(mw)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_map) *m;
	bus_data_t *to;
	unsigned int nr;
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t);

	m = bus_(_map_w_lookup)(b, addr);
	if (! m) {
		m = bus_(_map_w_add)(b, addr);
	}
	if (m->mw) {
		to = (bus_data_t *) m->mw + (addr & 0xfff) / sizeof(bus_data_t);

		if ((sizeof(bus_data_t) == 2 && bs == 0x03)
		 || (sizeof(bus_data_t) == 4 && bs == 0x0f)
		 || (sizeof(bus_data_t) == 8 && bs == 0xff)) {
			/* Shortcut... */
			*to = val;

		} else {
			/*
			 * Instead of using uint8_t, uint16_t, ... we
			 * would like to always use bus_data_t. Unfortunately
			 * GCC produces warnings.
			 */
			if ((bs >> 0) & 1) {
				*to &= ~((uint8_t) 0xff << 0);
				*to |= val & ((uint8_t) 0xff << 0);
			}
			if (2 <= sizeof(bus_data_t)
			 && (bs >> 1) & 1) {
				*to &= ~((uint16_t) 0xff << 8);
				*to |= val & ((uint16_t) 0xff << 8);
			}
			if (3 <= sizeof(bus_data_t)
			 && (bs >> 2) & 1) {
				*to &= ~((uint32_t) 0xff << 16);
				*to |= val & ((uint32_t) 0xff << 16);
			}
			if (4 <= sizeof(bus_data_t)
			 && (bs >> 3) & 1) {
				*to &= ~((uint32_t) 0xff << 24);
				*to |= val & ((uint32_t) 0xff << 24);
			}
			if (5 <= sizeof(bus_data_t)
			 && (bs >> 4) & 1) {
				*to &= ~((uint64_t) 0xff << 32);
				*to |= val & ((uint64_t) 0xff << 32);
			}
			if (6 <= sizeof(bus_data_t)
			 && (bs >> 5) & 1) {
				*to &= ~((uint64_t) 0xff << 40);
				*to |= val & ((uint64_t) 0xff << 40);
			}
			if (7 <= sizeof(bus_data_t)
			 && (bs >> 6) & 1) {
				*to &= ~((uint64_t) 0xff << 48);
				*to |= val & ((uint64_t) 0xff << 48);
			}
			if (8 <= sizeof(bus_data_t)
			 && (bs >> 7) & 1) {
				*to &= ~((uint64_t) 0xff << 56);
				*to |= val & ((uint64_t) 0xff << 56);
			}
		}
		return 0;
	}

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->mw;
		if (func
		 && func(b->member[nr].s, addr, bs, val) == 0) {
			return 0;
		}
	}
}

#ifdef NEED_X
#if 0 /* Not used, yet. */
static inline __attribute__((always_inline)) struct bus_(main_map) *
bus_(_map_x_lookup)(
	struct bus_(main) *b,
	unsigned int state,
	unsigned long addr
)
{
	unsigned int hash;
	struct bus_(main_map) *m;

	addr &= ~0xfff;

	hash = bus_(_map_hash)(state, addr);

	for (m = b->map_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/*
			 * Not Found
			 */
			return (struct bus_(main_map) *) 0;
		}
		if (m->type == BUS_(TYPE_MX)
		 && m->state == state
		 && m->addr == addr) {
			/*
			 * Found
			 */
			/* Don't update LRU list. */
			/* Costs too much... */
			return m;
		}
	}
}

static __attribute__((always_inline)) struct bus_(main_map) *
bus_(_map_x_add)(
	struct bus_(main) *b,
	unsigned int state,
	unsigned long addr
)
{
	char *haddr;
	unsigned int hash;
	struct bus_(main_map) *m;

	addr &= ~0xfff;

	if (bus_(map_x)(b, b, state, addr, &haddr) != 0) {
		haddr = NULL;
	}

	/* Get entry from LRU list. */
	m = b->map_lru_last;

	/* Remove from LRU list. */
	assert(m->lru_prev);
	m->lru_prev->lru_next = 0;
	b->map_lru_last = m->lru_prev;

	if (m->addr != -1) {
		/* Remove from HASH list. */
		hash = bus_(_map_hash(m->state, m->addr);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->map_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->map_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_MX);
	m->state = state;
	m->addr = addr;
	m->mx = (uint32_t *) haddr;

	/* Add to HASH list. */
	hash = bus_(_map_hash)(state, addr);
	m->hash_prev = 0;
	m->hash_next = b->map_hash_first[hash];
	b->map_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->map_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->map_lru_first;
	b->map_lru_first = m;
	m->lru_next->lru_prev = m;

	return m;
}
#endif /* Not used, yet. */
#endif /* NEED_X */

int
bus_(map_r_check)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t pa
)
{
	unsigned int hash;
	struct bus_(main_map) *m;
	unsigned int nr;
	int (*func)(void *, bus_addr_t);

	hash = bus_(_map_hash)(pa);

	for (m = b->map_hash_first[hash]; m; m = m->hash_next) {
		if (m->type == BUS_(TYPE_MW)
		 && m->addr == pa) {
			bus_(_map_flush)(b, m);
		}
	}

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_r_check;
		if (func
		 && func(b->member[nr].s, pa)) {
			return 1;
		}
	}
}

int
bus_(map_r)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t pa,
	char **haddr_p
)
{
	unsigned int nr;
	int (*func)(void *, bus_addr_t, char **);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_r;
		if (func
		 && func(b->member[nr].s, pa, haddr_p) == 0) {
			return 0;
		}
	}
}

int
bus_(map_w_check)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t pa
)
{
	unsigned int hash;
	struct bus_(main_map) *m;
	unsigned int nr;
	int (*func)(void *, bus_addr_t);

	hash = bus_(_map_hash)(pa);

	for (m = b->map_hash_first[hash]; m; m = m->hash_next) {
		if (m->type == BUS_(TYPE_MR)
		 && m->addr == pa) {
			bus_(_map_flush)(b, m);
		}
	}

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_w_check;
		if (func
		 && func(b->member[nr].s, pa)) {
			return 1;
		}
	}
}

int
bus_(map_w)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t pa,
	char **haddr_p
)
{
	unsigned int nr;
	int (*func)(void *, bus_addr_t, char **);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_w;
		if (func
		 && func(b->member[nr].s, pa, haddr_p) == 0) {
			return 0;
		}
	}
}

#ifdef NEED_X
int
bus_(map_x_check)(
	struct bus_(main) *b,
	void *s,
	unsigned int state,
	bus_addr_t pa
)
{
	unsigned int hash;
	struct bus_(main_map) *m;
	unsigned int nr;
	int (*func)(void *, unsigned int, bus_addr_t);

	hash = bus_(_map_hash)(state, pa);

	for (m = b->map_hash_first[hash]; m; m = m->hash_next) {
		if (m->type == BUS_(TYPE_MW)
		 && m->state == state
		 && m->addr == pa) {
			bus_(_map_flush)(b, m);
		}
	}

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_x_check;
		if (func
		 && func(b->member[nr].s, state, pa)) {
			return 1;
		}
	}
}

int
bus_(map_x)(
	struct bus_(main) *b,
	void *s,
	unsigned int state,
	bus_addr_t pa,
	char **haddr_p
)
{
	unsigned int nr;
	int (*func)(void *, unsigned int, bus_addr_t, char **);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_x;
		if (func
		 && func(b->member[nr].s, state, pa, haddr_p) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_X */

int
bus_(inta_addr)(
	struct bus_(main) *b,
	void *s
)
{
	unsigned int nr;
	int (*func)(void *);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->inta_addr;
		if (func
		 && func(b->member[nr].s) == 0) {
			return 0;
		}
	}
}

int
bus_(inta_data)(
	struct bus_(main) *b,
	void *s,
	uint8_t *valp
)
{
	unsigned int nr;
	int (*func)(void *, uint8_t *);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->inta_data;
		if (func
		 && func(b->member[nr].s, valp) == 0) {
			return 0;
		}
	}
}

void
bus_(unmap)(
	struct bus_(main) *b,
	void *s,
	bus_addr_t pa,
	bus_addr_t len
)
{
	unsigned int i;
	struct bus_(main_map) *m;
	unsigned int nr;
	void (*func)(void *, bus_addr_t, bus_addr_t);

	assert((pa & 0xfff) == 0x000);
	len = (len + 0xfff) & ~(bus_addr_t) 0xfff;

	for (i = 0; i < sizeof(b->map) / sizeof(b->map[0]); i++) {
		m = &b->map[i];

		if (m->addr == -1
		 || m->addr < pa
		 || pa + len - 1 < m->addr) {
			continue;
		}

		bus_(_map_flush)(b, m);
	}

	for (nr = 0; nr < b->member_count; nr++) {
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->unmap;
		if (func) {
			func(b->member[nr].s, pa, len);
		}
	}
}

void
bus_(main_connect)(
	struct bus_(main) *b,
	void *s,
	const struct bus_(main_funcs) *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_PCI_BUS_MAIN);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

static int
bus_(main_s0_type_addr)(
	void *_f,
	unsigned int type,
	bus_addr_t addr
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_type_addr)(f->s1, f, type, addr);
}

static int
bus_(main_s0_read_data)(
	void *_f,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_read_data)(f->s1, f, bs, valp);
}

static int
bus_(main_s0_write_data)(
	void *_f,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_write_data)(f->s1, f, bs, val);
}

#ifdef NEED_CONFIG_SPACE
static int
bus_(main_s0_c0r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_c0r)(f->s1, f, addr, bs, valp);
}

static int
bus_(main_s0_c0w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_c0w)(f->s1, f, addr, bs, val);
}

static int
bus_(main_s0_c1r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(c1r)(f->s1, f, addr, bs, valp);
}

static int
bus_(main_s0_c1w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(c1w)(f->s1, f, addr, bs, val);
}
#endif /* NEED_CONFIG_SPACE */

static int
bus_(main_s0_ior)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(ior)(f->s1, f, port, bs, valp);
}

static int
bus_(main_s0_iow)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(iow)(f->s1, f, port, bs, val);
}

static int
bus_(main_s0_ior_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(ior_info)(f->s1, f, port, bs, cfp, csp);
}

static int
bus_(main_s0_iow_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(iow_info)(f->s1, f, port, bs, cfp, csp);
}

static void
bus_(main_s0_ior_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(main_merge) *f = _f;

	bus_(ior_info_flush)(f->s1, f, port, bs);
}

static void
bus_(main_s0_iow_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(main_merge) *f = _f;

	bus_(iow_info_flush)(f->s1, f, port, bs);
}

static int
bus_(main_s0_mr)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(mr)(f->s1, f, addr, bs, valp);
}

static int
bus_(main_s0_mw)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(mw)(f->s1, f, addr, bs, val);
}

static int
bus_(main_s0_map_r_check)(
	void *_f,
	bus_addr_t pa
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_r_check)(f->s1, f, pa);
}

static int
bus_(main_s0_map_r)(
	void *_f,
	bus_addr_t pa,
	char **haddr_p
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_r)(f->s1, f, pa, haddr_p);
}

static int
bus_(main_s0_map_w_check)(
	void *_f,
	bus_addr_t pa
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_w_check)(f->s1, f, pa);
}

static int
bus_(main_s0_map_w)(
	void *_f,
	bus_addr_t pa,
	char **haddr_p
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_w)(f->s1, f, pa, haddr_p);
}

#ifdef NEED_X
static int
bus_(main_s0_map_x)(
	void *_f,
	unsigned int state,
	bus_addr_t pa,
	char **haddr_p
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_x)(f->s1, f, state, pa, haddr_p);
}

static int
bus_(main_s0_map_x_check)(
	void *_f,
	unsigned int state,
	bus_addr_t pa
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_x_check)(f->s1, f, state, pa);
}
#endif /* NEED_X */

#ifdef NEED_INTA
static int
bus_(main_s0_inta_addr)(
	void *_f
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(inta_addr)(f->s1, f);
}

static int
bus_(main_s0_inta_data)(
	void *_f,
	uint8_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(inta_data)(f->s1, f, valp);
}
#endif /* NEED_INTA */

static void
bus_(main_s0_unmap)(
	void *_f,
	bus_addr_t pa,
	bus_addr_t len
)
{
	struct bus_(main_merge) *f = _f;

	bus_(unmap)(f->s1, f, pa, len);
}

static int
bus_(main_s1_type_addr)(
	void *_f,
	unsigned int type,
	bus_addr_t addr
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_type_addr)(f->s0, f, type, addr);
}

static int
bus_(main_s1_read_data)(
	void *_f,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_read_data)(f->s0, f, bs, valp);
}

static int
bus_(main_s1_write_data)(
	void *_f,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_write_data)(f->s0, f, bs, val);
}

#ifdef NEED_CONFIG_SPACE
static int
bus_(main_s1_c0r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_c0r)(f->s0, f, addr, bs, valp);
}

static int
bus_(main_s1_c0w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(main_c0w)(f->s0, f, addr, bs, val);
}

static int
bus_(main_s1_c1r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(c1r)(f->s0, f, addr, bs, valp);
}

static int
bus_(main_s1_c1w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(c1w)(f->s0, f, addr, bs, val);
}
#endif /* NEED_CONFIG_SPACE */

static int
bus_(main_s1_ior)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(ior)(f->s0, f, port, bs, valp);
}

static int
bus_(main_s1_iow)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(iow)(f->s0, f, port, bs, val);
}

static int
bus_(main_s1_ior_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(ior_info)(f->s0, f, port, bs, cfp, csp);
}

static int
bus_(main_s1_iow_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(iow_info)(f->s0, f, port, bs, cfp, csp);
}

static void
bus_(main_s1_ior_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(ior_info_flush)(f->s0, f, port, bs);
}

static void
bus_(main_s1_iow_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(iow_info_flush)(f->s0, f, port, bs);
}

static int
bus_(main_s1_mr)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(mr)(f->s0, f, addr, bs, valp);
}

static int
bus_(main_s1_mw)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(mw)(f->s0, f, addr, bs, val);
}

static int
bus_(main_s1_map_r_check)(
	void *_f,
	bus_addr_t pa
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_r_check)(f->s0, f, pa);
}

static int
bus_(main_s1_map_r)(
	void *_f,
	bus_addr_t pa,
	char **haddr_p
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_r)(f->s0, f, pa, haddr_p);
}

static int
bus_(main_s1_map_w_check)(
	void *_f,
	bus_addr_t pa
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_w_check)(f->s0, f, pa);
}

static int
bus_(main_s1_map_w)(
	void *_f,
	bus_addr_t pa,
	char **haddr_p
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_w)(f->s0, f, pa, haddr_p);
}

#ifdef NEED_X
static int
bus_(main_s1_map_x_check)(
	void *_f,
	unsigned int state,
	bus_addr_t pa
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_x_check)(f->s0, f, state, pa);
}

static int
bus_(main_s1_map_x)(
	void *_f,
	unsigned int state,
	bus_addr_t pa,
	char **haddr_p
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(map_x)(f->s0, f, state, pa, haddr_p);
}
#endif /* NEED_X */

#ifdef NEED_INTA
static int
bus_(main_s1_inta_addr)(
	void *_f
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(inta_addr)(f->s0, f);
}

static int
bus_(main_s1_inta_data)(
	void *_f,
	uint8_t *valp
)
{
	struct bus_(main_merge) *f = _f;

	return bus_(inta_data)(f->s0, f, valp);
}
#endif /* NEED_INTA */

static void
bus_(main_s1_unmap)(
	void *_f,
	bus_addr_t pa,
	bus_addr_t len
)
{
	struct bus_(main_merge) *f = _f;

	bus_(unmap)(f->s0, f, pa, len);
}

struct bus_(main_merge) *
bus_(main_merge)(
	struct bus_(main) *s0,
	struct bus_(main) *s1
)
{
	static const struct bus_(main_funcs) s0_funcs = {
		.type_addr = bus_(main_s0_type_addr),
		.read_data = bus_(main_s0_read_data),
		.write_data = bus_(main_s0_write_data),
#ifdef NEED_CONFIG_SPACE
		.c0r = bus_(main_s0_c0r),
		.c0w = bus_(main_s0_c0w),
		.c1r = bus_(main_s0_c1r),
		.c1w = bus_(main_s0_c1w),
#endif /* NEED_CONFIG_SPACE */
		.ior = bus_(main_s0_ior),
		.iow = bus_(main_s0_iow),
		.ior_info = bus_(main_s0_ior_info),
		.iow_info = bus_(main_s0_iow_info),
		.ior_info_flush = bus_(main_s0_ior_info_flush),
		.iow_info_flush = bus_(main_s0_iow_info_flush),
		.mr = bus_(main_s0_mr),
		.mw = bus_(main_s0_mw),
		.map_r_check = bus_(main_s0_map_r_check),
		.map_r = bus_(main_s0_map_r),
		.map_w_check = bus_(main_s0_map_w_check),
		.map_w = bus_(main_s0_map_w),
#ifdef NEED_X
		.map_x_check = bus_(main_s0_map_x_check),
		.map_x = bus_(main_s0_map_x),
#endif /* NEED_X */
#ifdef NEED_INTA
		.inta_addr = bus_(main_s0_inta_addr),
		.inta_data = bus_(main_s0_inta_data),
#endif /* NEED_INTA */
		.unmap = bus_(main_s0_unmap),
	};
	static const struct bus_(main_funcs) s1_funcs = {
		.type_addr = bus_(main_s1_type_addr),
		.read_data = bus_(main_s1_read_data),
		.write_data = bus_(main_s1_write_data),
#ifdef NEED_CONFIG_SPACE
		.c0r = bus_(main_s1_c0r),
		.c0w = bus_(main_s1_c0w),
		.c1r = bus_(main_s1_c1r),
		.c1w = bus_(main_s1_c1w),
#endif /* NEED_CONFIG_SPACE */
		.ior = bus_(main_s1_ior),
		.iow = bus_(main_s1_iow),
		.ior_info = bus_(main_s1_ior_info),
		.iow_info = bus_(main_s1_iow_info),
		.ior_info_flush = bus_(main_s1_ior_info_flush),
		.iow_info_flush = bus_(main_s1_iow_info_flush),
		.mr = bus_(main_s1_mr),
		.mw = bus_(main_s1_mw),
		.map_r_check = bus_(main_s1_map_r_check),
		.map_r = bus_(main_s1_map_r),
		.map_w_check = bus_(main_s1_map_w_check),
		.map_w = bus_(main_s1_map_w),
#ifdef NEED_X
		.map_x_check = bus_(main_s1_map_x_check),
		.map_x = bus_(main_s1_map_x),
#endif /* NEED_X */
#ifdef NEED_INTA
		.inta_addr = bus_(main_s1_inta_addr),
		.inta_data = bus_(main_s1_inta_data),
#endif /* NEED_INTA */
		.unmap = bus_(main_s1_unmap),
	};
	struct bus_(main_merge) *m;

	m = malloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	bus_(main_connect)(s0, m, &s0_funcs);
	m->s1 = s1;
	bus_(main_connect)(s1, m, &s1_funcs);

	return m;
}

void
bus_(main_split)(struct bus_(main_merge) *m)
{
	fixme();
}

struct bus_(main) *
bus_(main_create)(const char *name)
{
	struct bus_(main) *b;
	struct bus_(main_io) *io;
	struct bus_(main_map) *m;
	unsigned int i;

	b = malloc(sizeof(*b));
	assert(b);

	b->type = SIG_GEN_PCI_BUS_MAIN;
	b->member_count = 0;

	b->io_lru_first = 0;
	b->io_lru_last = 0;
	for (i = 0; i < sizeof(b->io_hash_first) / sizeof(b->io_hash_first[0]); i++) {
		b->io_hash_first[i] = NULL;
		b->io_hash_last[i] = NULL;
	}
	for (i = 0; i < sizeof(b->io) / sizeof(b->io[0]); i++) {
		io = &b->io[i];
		io->port = -1;
		io->bs = -1;

		/* Don't add empty entry to HASH list. */

		/* Add to LRU list. */
		io->lru_prev = b->io_lru_last;
		io->lru_next = 0;
		if (io->lru_prev) {
			io->lru_prev->lru_next = io;
		} else {
			b->io_lru_first = io;
		}
		b->io_lru_last = io;
	}

	b->map_lru_first = 0;
	b->map_lru_last = 0;
	for (i = 0; i < sizeof(b->map_hash_first) / sizeof(b->map_hash_first[0]); i++) {
		b->map_hash_first[i] = 0;
		b->map_hash_last[i] = 0;
	}
	for (i = 0; i < sizeof(b->map) / sizeof(b->map[0]); i++) {
		m = &b->map[i];
		m->addr = -1;
		m->mr = NULL;
		m->mw = NULL;

		/* Don't add empty entry to HASH list. */

		/* Add to LRU list. */
		m->lru_prev = b->map_lru_last;
		m->lru_next = 0;
		if (m->lru_prev) {
			m->lru_prev->lru_next = m;
		} else {
			b->map_lru_first = m;
		}
		b->map_lru_last = m;
	}

	return b;
}

void
bus_(main_destroy)(struct bus_(main) *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_PCI_BUS_MAIN);

	free(sig);
}

/* ----------------------------------------------------------------- */

int
bus_(idsel_c0r)(
	struct bus_(idsel) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	unsigned int nr;
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t *);

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->c0r;
		if (func
		 && func(b->member[nr].s, addr, bs, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(idsel_c0w)(
	struct bus_(idsel) *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	unsigned int nr;
	int (*func)(void *, bus_addr_t, unsigned int, bus_data_t);

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->c0w;
		if (func
		 && func(b->member[nr].s, addr, bs, val) == 0) {
			return 0;
		}
	}
}

void
bus_(idsel_connect)(
	struct bus_(idsel) *b,
	void *s,
	const struct bus_(idsel_funcs) *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_PCI_BUS_IDSEL);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

static int
bus_(idsel_s0_c0r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(idsel_merge) *f
			= (struct bus_(idsel_merge) *) _f;

	return bus_(idsel_c0r)(f->s1, f, addr, bs, valp);
}

static int
bus_(idsel_s0_c0w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(idsel_merge) *f
			= (struct bus_(idsel_merge) *) _f;

	return bus_(idsel_c0w)(f->s1, f, addr, bs, val);
}

static int
bus_(idsel_s1_c0r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(idsel_merge) *f
			= (struct bus_(idsel_merge) *) _f;

	return bus_(idsel_c0r)(f->s0, f, addr, bs, valp);
}

static int
bus_(idsel_s1_c0w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(idsel_merge) *f
			= (struct bus_(idsel_merge) *) _f;

	return bus_(idsel_c0w)(f->s0, f, addr, bs, val);
}

struct bus_(idsel_merge) *
bus_(idsel_merge)(
	struct bus_(idsel) *s0,
	struct bus_(idsel) *s1
)
{
	static const struct bus_(idsel_funcs) s0_funcs = {
		.c0r = bus_(idsel_s0_c0r),
		.c0w = bus_(idsel_s0_c0w),
	};
	static const struct bus_(idsel_funcs) s1_funcs = {
		.c0r = bus_(idsel_s1_c0r),
		.c0w = bus_(idsel_s1_c0w),
	};
	struct bus_(idsel_merge) *m;

	m = malloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	bus_(idsel_connect)(s0, m, &s0_funcs);
	m->s1 = s1;
	bus_(idsel_connect)(s1, m, &s1_funcs);

	return m;
}

void
bus_(idsel_split)(struct bus_(idsel_merge) *m)
{
	fixme();
}

struct bus_(idsel) *
bus_(idsel_create)(const char *name)
{
	struct bus_(idsel) *b;

	b = malloc(sizeof(*b));
	assert(b);

	b->type = SIG_GEN_PCI_BUS_IDSEL;
	b->member_count = 0;

	return b;
}

void
bus_(idsel_destroy)(struct bus_(idsel) *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_PCI_BUS_IDSEL);

	free(sig);
}

/* ----------------------------------------------------------------- */

struct bus *
bus_(create)(const char *name)
{
	struct bus *b;
	char n[1000];

	b = malloc(sizeof(*b));
	assert(b);

	b->type = SIG_GEN_PCI_BUS;

	sprintf(n, "%s-+5V", name);
	b->p5V = sig_boolean_create(n);
	sprintf(n, "%s-+12V", name);
	b->p12V = sig_boolean_create(n);
	sprintf(n, "%s--12V", name);
	b->m12V = sig_boolean_create(n);

	sprintf(n, "%s-n_reset", name);
	b->n_reset = sig_boolean_create(n);

	sprintf(n, "%s-idsel", name);
	b->idsel = bus_(idsel_create)(n);
	sprintf(n, "%s-main", name);
	b->main = bus_(main_create)(n);

	sprintf(n, "%s-intA", name);
	b->intA = sig_boolean_or_create(n);
	sprintf(n, "%s-intB", name);
	b->intB = sig_boolean_or_create(n);
	sprintf(n, "%s-intC", name);
	b->intC = sig_boolean_or_create(n);
	sprintf(n, "%s-intD", name);
	b->intD = sig_boolean_or_create(n);

	return b;
}

void
bus_(destroy)(struct bus *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_PCI_BUS);

	sig_boolean_destroy(sig->p5V);
	sig_boolean_destroy(sig->p12V);
	sig_boolean_destroy(sig->m12V);

	sig_boolean_destroy(sig->n_reset);

	bus_(idsel_destroy)(sig->idsel);
	bus_(main_destroy)(sig->main);

	sig_boolean_or_destroy(sig->intA);
	sig_boolean_or_destroy(sig->intB);
	sig_boolean_or_destroy(sig->intC);
	sig_boolean_or_destroy(sig->intD);

	free(sig);
}
