#############################################################
##                                                         ##
## Copyright (c) 2003-2011 by The University of Queensland ##
## Earth Systems Science Computational Centre (ESSCC)      ##
## http://www.uq.edu.au/esscc                              ##
##                                                         ##
## Primary Business: Brisbane, Queensland, Australia       ##
## Licensed under the Open Software License version 3.0    ##
## http://www.opensource.org/licenses/osl-3.0.php          ##
##                                                         ##
#############################################################

"""
Defines the L{ImageFormat} class and functions for mapping a
file name extension to an associated C{ImageFormat} object.
"""
import string
import os
import os.path

class ImageFormat(object):
    """
    Class representing an image format.
    """
    def __init__(self, name):
        """
        Constructor.
        @type name: str
        @param name: Name assigned to this image format.
        """
        self.name = name

    def getName(self):
        """
        Returns the name associated with this image format.
        @rtype: str
        """
        return self.name

    def __str__(self):
        return self.getName()


PNG = ImageFormat("PNG")
PNM = ImageFormat("PNM")

_nameFormatDict = dict()
_nameFormatDict[string.upper(str(PNG))] = PNG
_nameFormatDict[string.upper(str(PNM))] = PNM

def _getDelimitedFormatNameString():
    return ", ".join(map(str,_nameFormatDict.keys()))

def getFormatFromName(formatName, ext=None):
    """
    Returns the C{ImageFormat} object which corresponds
    to a specified image-format name (string).
    @type formatName: str
    @param formatName: The name of an image format, one of: %s
    @type ext: str
    @param ext: File name extension for error message string.
    """ % _getDelimitedFormatNameString()
    if _nameFormatDict.has_key(string.upper(formatName)):
        return _nameFormatDict[string.upper(formatName)]
    raise \
      ValueError(
        (
            "No image format found which matched extension '%s'," +
            " valid image file formats are: %s"
        )
        %
        (ext, _getDelimitedFormatNameString())
      )

def getFormatFromExtension(fileName):
    """
    Returns the C{ImageFormat} object which corresponds
    to a specified file name. Uses the C{fileName} extension
    to try and deduce the corresponding C{ImageFormat} object.
    @type fileName: str
    @param fileName: A file name.
    @rtype: C{ImageFormat}
    @return: An C{ImageFormat} object corresponding to the
    specified file name (and corresponding file name extension).
    """
    (base, ext) = os.path.splitext(fileName)
    if (len(ext) > 0):
        formatName = string.lstrip(ext, ".")
    else:
        raise ValueError(
            "Could not determine image format from file "
            +
            "name " + fileName + ", no extension."
        )
    return getFormatFromName(formatName, ext)
