/*
***************************************************************************
*
* Author: Teunis van Beelen
*
* Copyright (C) 2007, 2008, 2009, 2010 Teunis van Beelen
*
* teuniz@gmail.com
*
***************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2 of the License.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
***************************************************************************
*
* This version of GPL is at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
*
***************************************************************************
*/



#include "mainwindow.h"


#if defined(__APPLE__) || defined(__MACH__) || defined(__APPLE_CC__)

#define fseeko fseek
#define ftello ftell
#define fopeno fopen

#else

#define fseeko fseeko64
#define ftello ftello64
#define fopeno fopen64

#endif




UI_Mainwindow::UI_Mainwindow()
{
  int i, j, k;

  pixmap = new QPixmap(":/images/splash.png");
  splash = new QSplashScreen(this, *pixmap, Qt::WindowStaysOnTopHint);
  splash->show();

  t1 = new QTimer;
  t1->setSingleShot(TRUE);
  t1->start(3000);
  QObject::connect(t1, SIGNAL(timeout()), splash, SLOT(close()));

  setMinimumSize(QSize(640, 480));
  setWindowTitle(PROGRAM_NAME);
  setWindowIcon(QIcon(":/images/edf.png"));

  myfont = new QFont;

  monofont = new QFont;

#ifdef Q_WS_X11
  QApplication::setFont(*myfont);

  myfont->setFamily("Arial");
  myfont->setPixelSize(12);

  monofont->setFamily("andale mono");
  monofont->setPixelSize(12);
#endif

#ifdef Q_WS_MAC
  QApplication::setFont(*myfont);

  myfont->setFamily("Arial");
  myfont->setPixelSize(12);

  monofont->setFamily("andale mono");
  monofont->setPixelSize(12);
#endif

#ifdef Q_WS_WIN
  myfont->setFamily("Tahoma");
  myfont->setPixelSize(11);

  monofont->setFamily("courier");
  monofont->setPixelSize(12);
#endif

  QLocale::setDefault(QLocale::c());

  setlocale(LC_ALL, "C");

  pixelsizefactor = 0.0294382;

  x_pixelsizefactor = 0.0294382;

  auto_dpi = 1;

  show_annot_markers = 1;

  show_baselines = 1;

  clip_to_pane = 0;

  auto_reload_mtg = 1;

  read_biosemi_status_signal = 1;

  exit_in_progress = 0;

  recent_montagedir[0] = 0;
  recent_savedir[0] = 0;
  recent_opendir[0] = 0;

  for(i=0; i<MAXSPECTRUMDIALOGS; i++)
  {
    spectrumdialog[i] = NULL;
  }

  spectrum_colorbar = (struct spectrum_markersblock *)calloc(1, sizeof(struct spectrum_markersblock));
  for(i=0; i < MAXSPECTRUMMARKERS; i++)
  {
    spectrum_colorbar->freq[i] = 1.0;
    spectrum_colorbar->color[i] = Qt::white;
  }
  spectrum_colorbar->items = 5;
  spectrum_colorbar->freq[0] = 4.0;
  spectrum_colorbar->freq[1] = 8.0;
  spectrum_colorbar->freq[2] = 12.0;
  spectrum_colorbar->freq[3] = 30.0;
  spectrum_colorbar->freq[4] = 100.0;
  spectrum_colorbar->color[0] = Qt::darkRed;
  spectrum_colorbar->color[1] = Qt::darkGreen;
  spectrum_colorbar->color[2] = Qt::darkBlue;
  spectrum_colorbar->color[3] = Qt::darkCyan;
  spectrum_colorbar->color[4] = Qt::darkMagenta;
  spectrum_colorbar->method = 1;

  maxdftblocksize = 1000;

  import_annotations_var = (import_annotations_var_block *)calloc(1, sizeof(struct import_annotations_var_block));
  strcpy(import_annotations_var->separator, "tab");
  import_annotations_var->format = 1;
  import_annotations_var->onsettimeformat = 0;
  import_annotations_var->onsetcolumn = 1;
  import_annotations_var->descriptioncolumn = 2;
  import_annotations_var->datastartline = 1;

  export_annotations_var = (export_annotations_var_block *)calloc(1, sizeof(struct export_annotations_var_block));
  export_annotations_var->separator = 0;
  export_annotations_var->format = 1;
  export_annotations_var->duration = 0;

  read_general_settings();

  maincurve = new ViewCurve(this);

  dpix = maincurve->logicalDpiX();

  dpiy = maincurve->logicalDpiY();

  if(auto_dpi)
  {
    pixelsizefactor = 1.0 / ((double)dpiy / 2.54);

    x_pixelsizefactor = 1.0 / ((double)dpix / 2.54);
  }

  read_color_settings();

  setCentralWidget(maincurve);

  menubar = menuBar();

  recent_filesmenu = new QMenu(this);
  recent_filesmenu->setTitle("Recent files");
  connect(recent_filesmenu, SIGNAL(triggered(QAction *)), this, SLOT(recent_file_action_func(QAction *)));

  print_img_menu = new QMenu(this);
  print_img_menu->setTitle("to Image");
  print_img_menu->addAction("640 x 480",   this, SLOT(print_to_img_640x480()));
  print_img_menu->addAction("800 x 600",   this, SLOT(print_to_img_800x600()));
  print_img_menu->addAction("1024 x 768",  this, SLOT(print_to_img_1024x768()));
  print_img_menu->addAction("1280 x 1024", this, SLOT(print_to_img_1280x1024()));
  print_img_menu->addAction("1600 x 1200", this, SLOT(print_to_img_1600x1200()));

  printmenu = new QMenu(this);
  printmenu->setTitle("Print");
  printmenu->addAction("to Printer",    maincurve, SLOT(print_to_printer()), QKeySequence::Print);
  printmenu->addAction("to PostScript", maincurve, SLOT(print_to_postscript()));
  printmenu->addAction("to PDF",        maincurve, SLOT(print_to_pdf()));
  printmenu->addMenu(print_img_menu);
  printmenu->addAction("to EDF",        this,      SLOT(print_to_edf()));
  printmenu->addAction("to BDF",        this,      SLOT(print_to_bdf()));


#ifdef BK_MRS_project
// temporary code for private use in a certain project
// do not use this code, it will be removed in future
  bk_mrs_project_file = NULL;

  keyboard_bk_mrs_project_act = new QAction(this);
  keyboard_bk_mrs_project_act->setShortcut(Qt::Key_P);
  this->addAction(keyboard_bk_mrs_project_act);
  connect(keyboard_bk_mrs_project_act, SIGNAL(triggered()), this, SLOT(keyboard_bk_mrs_project_func()));

#endif

  save_act = new QAction("Save", this);
  save_act->setShortcut(QKeySequence::Save);
  save_act->setEnabled(FALSE);
  connect(save_act, SIGNAL(triggered()), this, SLOT(save_file()));

  filemenu = new QMenu(this);
  filemenu->setTitle("&File");
  filemenu->addAction("Open",         this, SLOT(open_new_file()), QKeySequence::Open);
  filemenu->addAction(save_act);
  filemenu->addMenu(recent_filesmenu);
  filemenu->addMenu(printmenu);
  filemenu->addAction("Info",         this, SLOT(show_file_info()));
  filemenu->addAction("Close all",    this, SLOT(close_all_files()), QKeySequence::Close);
  filemenu->addAction("Exit",         this, SLOT(exit_program()), QKeySequence::Quit);
  menubar->addMenu(filemenu);

  signalmenu = new QMenu(this);
  signalmenu->setTitle("&Signals");
  signalmenu->addAction("Properties", this, SLOT(signalproperties_dialog()));
  signalmenu->addAction("Add",        this, SLOT(add_signals_dialog()));
  signalmenu->addAction("Remove all", this, SLOT(remove_all_signals()));
  menubar->addMenu(signalmenu);

  displaymenu = new QMenu(this);
  displaymenu->setTitle("&Timescale");

  displaymenu->addAction("3 cm/sec", this, SLOT(page_3cmsec()));

  displaymenu->addSeparator();

  page_10m = new QAction("10 mSec/page", this);
  displaymenu->addAction(page_10m);

  page_20m = new QAction("20 mSec/page", this);
  displaymenu->addAction(page_20m);

  page_50m = new QAction("50 mSec/page", this);
  displaymenu->addAction(page_50m);

  page_100m = new QAction("100 mSec/page", this);
  displaymenu->addAction(page_100m);

  page_200m = new QAction("200 mSec/page", this);
  displaymenu->addAction(page_200m);

  page_500m = new QAction("500 mSec/page", this);
  displaymenu->addAction(page_500m);

  page_1 = new QAction("1 Sec/page", this);
  displaymenu->addAction(page_1);

  page_2 = new QAction("2 Sec/page", this);
  displaymenu->addAction(page_2);

  page_5 = new QAction("5 Sec/page", this);
  displaymenu->addAction(page_5);

  page_10 = new QAction("10 Sec/page", this);
  displaymenu->addAction(page_10);

  page_15 = new QAction("15 Sec/page", this);
  displaymenu->addAction(page_15);

  page_20 = new QAction("20 Sec/page", this);
  displaymenu->addAction(page_20);

  page_30 = new QAction("30 Sec/page", this);
  page_30->setChecked(TRUE);
  displaymenu->addAction(page_30);

  page_60 = new QAction("60 Sec/page", this);
  displaymenu->addAction(page_60);

  page_300 = new QAction("5 min/page", this);
  displaymenu->addAction(page_300);

  page_1200 = new QAction("20 min/page", this);
  displaymenu->addAction(page_1200);

  page_3600 = new QAction("1 hour/page", this);
  displaymenu->addAction(page_3600);

  displaymenu->addSeparator();

  page_div2 = new QAction("Timescale / 2", this);
  page_div2->setShortcut(QKeySequence::ZoomIn);
  connect(page_div2, SIGNAL(triggered()), this, SLOT(set_page_div2()));
  displaymenu->addAction(page_div2);

  page_mult2 = new QAction("Timescale x 2", this);
  page_mult2->setShortcut(QKeySequence::ZoomOut);
  connect(page_mult2, SIGNAL(triggered()), this, SLOT(set_page_mult2()));
  displaymenu->addAction(page_mult2);

  displaymenu->addSeparator();

  page_user_defined = new QAction("user defined", this);
  connect(page_user_defined, SIGNAL(triggered()), this, SLOT(set_user_defined_display_time()));
  displaymenu->addAction(page_user_defined);

  displaymenu->addSeparator();

  page_whole_rec = new QAction("whole recording", this);
  connect(page_whole_rec, SIGNAL(triggered()), this, SLOT(set_display_time_whole_rec()));
  displaymenu->addAction(page_whole_rec);

  menubar->addMenu(displaymenu);

  DisplayGroup = new QActionGroup(this);
  DisplayGroup->addAction(page_10m);
  DisplayGroup->addAction(page_20m);
  DisplayGroup->addAction(page_50m);
  DisplayGroup->addAction(page_100m);
  DisplayGroup->addAction(page_200m);
  DisplayGroup->addAction(page_500m);
  DisplayGroup->addAction(page_1);
  DisplayGroup->addAction(page_2);
  DisplayGroup->addAction(page_5);
  DisplayGroup->addAction(page_10);
  DisplayGroup->addAction(page_15);
  DisplayGroup->addAction(page_20);
  DisplayGroup->addAction(page_30);
  DisplayGroup->addAction(page_60);
  DisplayGroup->addAction(page_300);
  DisplayGroup->addAction(page_1200);
  DisplayGroup->addAction(page_3600);
  connect(DisplayGroup, SIGNAL(triggered(QAction *)), this, SLOT(set_display_time(QAction *)));

  amplitudemenu = new QMenu(this);
  amplitudemenu->setTitle("&Amplitude");

  fit_to_pane = new QAction("Fit to pane", this);
  connect(fit_to_pane, SIGNAL(triggered()), this, SLOT(fit_signals_to_pane()));
  amplitudemenu->addAction(fit_to_pane);

  amplitudemenu->addSeparator();

  amplitudemenu->addAction("offset -> 0", this, SLOT(set_dc_offset_to_zero()));

  amplitudemenu->addSeparator();

  amp_50000 = new QAction("50000", this);
  amplitudemenu->addAction(amp_50000);

  amp_20000 = new QAction("20000", this);
  amplitudemenu->addAction(amp_20000);

  amp_10000 = new QAction("10000", this);
  amplitudemenu->addAction(amp_10000);

  amp_5000 = new QAction("5000", this);
  amplitudemenu->addAction(amp_5000);

  amp_2000 = new QAction("2000", this);
  amplitudemenu->addAction(amp_2000);

  amp_1000 = new QAction("1000", this);
  amplitudemenu->addAction(amp_1000);

  amp_500 = new QAction("500", this);
  amplitudemenu->addAction(amp_500);

  amp_200 = new QAction("200", this);
  amplitudemenu->addAction(amp_200);

  amp_100 = new QAction("100", this);
  amplitudemenu->addAction(amp_100);

  amp_50 = new QAction("50", this);
  amplitudemenu->addAction(amp_50);

  amp_20 = new QAction("20", this);
  amplitudemenu->addAction(amp_20);

  amp_10 = new QAction("10", this);
  amplitudemenu->addAction(amp_10);

  amp_5 = new QAction("5", this);
  amplitudemenu->addAction(amp_5);

  amp_2 = new QAction("2", this);
  amplitudemenu->addAction(amp_2);

  amp_1 = new QAction("1", this);
  amplitudemenu->addAction(amp_1);

  amp_05 = new QAction("0.5", this);
  amplitudemenu->addAction(amp_05);

  amp_02 = new QAction("0.2", this);
  amplitudemenu->addAction(amp_02);

  amp_01 = new QAction("0.1", this);
  amplitudemenu->addAction(amp_01);

  amp_005 = new QAction("0.05", this);
  amplitudemenu->addAction(amp_005);

  amp_002 = new QAction("0.02", this);
  amplitudemenu->addAction(amp_002);

  amp_001 = new QAction("0.01", this);
  amplitudemenu->addAction(amp_001);

  amp_0005 = new QAction("0.005", this);
  amplitudemenu->addAction(amp_0005);

  amp_0002 = new QAction("0.002", this);
  amplitudemenu->addAction(amp_0002);

  amp_0001 = new QAction("0.001", this);
  amplitudemenu->addAction(amp_0001);

  amp_00005 = new QAction("0.0005", this);
  amplitudemenu->addAction(amp_00005);

  amp_00002 = new QAction("0.0002", this);
  amplitudemenu->addAction(amp_00002);

  amp_00001 = new QAction("0.0001", this);
  amplitudemenu->addAction(amp_00001);

  menubar->addMenu(amplitudemenu);

  AmplitudeGroup = new QActionGroup(this);
  AmplitudeGroup->addAction(amp_00001);
  AmplitudeGroup->addAction(amp_00002);
  AmplitudeGroup->addAction(amp_00005);
  AmplitudeGroup->addAction(amp_0001);
  AmplitudeGroup->addAction(amp_0002);
  AmplitudeGroup->addAction(amp_0005);
  AmplitudeGroup->addAction(amp_001);
  AmplitudeGroup->addAction(amp_002);
  AmplitudeGroup->addAction(amp_005);
  AmplitudeGroup->addAction(amp_01);
  AmplitudeGroup->addAction(amp_02);
  AmplitudeGroup->addAction(amp_05);
  AmplitudeGroup->addAction(amp_1);
  AmplitudeGroup->addAction(amp_2);
  AmplitudeGroup->addAction(amp_5);
  AmplitudeGroup->addAction(amp_10);
  AmplitudeGroup->addAction(amp_20);
  AmplitudeGroup->addAction(amp_50);
  AmplitudeGroup->addAction(amp_100);
  AmplitudeGroup->addAction(amp_200);
  AmplitudeGroup->addAction(amp_500);
  AmplitudeGroup->addAction(amp_1000);
  AmplitudeGroup->addAction(amp_2000);
  AmplitudeGroup->addAction(amp_5000);
  AmplitudeGroup->addAction(amp_10000);
  AmplitudeGroup->addAction(amp_20000);
  AmplitudeGroup->addAction(amp_50000);
  connect(AmplitudeGroup, SIGNAL(triggered(QAction *)), this, SLOT(set_amplitude(QAction *)));

  filtermenu = new QMenu(this);
  filtermenu->setTitle("&Filter");
  filtermenu->addAction("New", this, SLOT(add_new_filter()));
  filtermenu->addAction("Adjust", this, SLOT(filterproperties_dialog()));
  filtermenu->addAction("Remove all", this, SLOT(remove_all_filters()));
  menubar->addMenu(filtermenu);

  load_predefined_mtg_act[0] = new QAction("Empty", this);
  load_predefined_mtg_act[0]->setShortcut(Qt::Key_F1);
  load_predefined_mtg_act[1] = new QAction("Empty", this);
  load_predefined_mtg_act[1]->setShortcut(Qt::Key_F2);
  load_predefined_mtg_act[2] = new QAction("Empty", this);
  load_predefined_mtg_act[2]->setShortcut(Qt::Key_F3);
  load_predefined_mtg_act[3] = new QAction("Empty", this);
  load_predefined_mtg_act[3]->setShortcut(Qt::Key_F4);
  load_predefined_mtg_act[4] = new QAction("Empty", this);
  load_predefined_mtg_act[4]->setShortcut(Qt::Key_F5);
  load_predefined_mtg_act[5] = new QAction("Empty", this);
  load_predefined_mtg_act[5]->setShortcut(Qt::Key_F6);
  load_predefined_mtg_act[6] = new QAction("Empty", this);
  load_predefined_mtg_act[6]->setShortcut(Qt::Key_F7);
  load_predefined_mtg_act[7] = new QAction("Empty", this);
  load_predefined_mtg_act[7]->setShortcut(Qt::Key_F8);
  load_predefined_mtg_group = new QActionGroup(this);
  for(i=0; i < MAXPREDEFINEDMONTAGES; i++)
  {
    load_predefined_mtg_group->addAction(load_predefined_mtg_act[i]);
  }
  connect(load_predefined_mtg_group, SIGNAL(triggered(QAction *)), this, SLOT(load_predefined_mtg(QAction *)));

  montagemenu = new QMenu(this);
  montagemenu->setTitle("&Montage");
  montagemenu->addAction("View this montage", this, SLOT(show_this_montage()));
  montagemenu->addAction("View saved montages", this, SLOT(view_montage()));
  montagemenu->addAction("Save", this, SLOT(save_montage()));
  montagemenu->addAction("Load", this, SLOT(load_montage()));
  montagemenu->addSeparator();
  montagemenu->addAction("Edit key-bindings for montages", this, SLOT(edit_predefined_montages()));
  montagemenu->addSeparator();
  for(i=0; i < MAXPREDEFINEDMONTAGES; i++)
  {
    montagemenu->addAction(load_predefined_mtg_act[i]);
  }
  menubar->addMenu(montagemenu);

  toolsmenu = new QMenu(this);
  toolsmenu->setTitle("T&ools");
  toolsmenu->addAction("Export EDF/BDF to ASCII", this, SLOT(export_to_ascii()));
  toolsmenu->addAction("Check EDF/BDF compatibility", this, SLOT(check_edf_compatibility()));
  toolsmenu->addAction("Convert Nihon Kohden to EDF+", this, SLOT(nk2edf_converter()));
  toolsmenu->addAction("Convert ASCII to EDF/BDF", this, SLOT(convert_ascii_to_edf()));
  toolsmenu->addAction("Convert Finometer to EDF", this, SLOT(convert_fino_to_edf()));
  toolsmenu->addAction("Convert Nexfin to EDF", this, SLOT(convert_nexfin_to_edf()));
  toolsmenu->addAction("Convert Emsa to EDF+", this, SLOT(convert_emsa_to_edf()));
  toolsmenu->addAction("Convert EDF+D to EDF+C", this, SLOT(edfd_converter()));
  toolsmenu->addAction("Convert Biosemi to BDF+", this, SLOT(biosemi2bdfplus_converter()));
  toolsmenu->addAction("Convert BDF to EDF", this, SLOT(bdf2edf_converter()));
  toolsmenu->addAction("Reduce signals and/or duration", this, SLOT(reduce_signals()));
  toolsmenu->addAction("Edit EDF/BDF header", this, SLOT(edit_header()));
  toolsmenu->addAction("Convert BI9800TL+3 to EDF", this, SLOT(BI98002edf_converter()));
  toolsmenu->addAction("Export annotations/events", this, SLOT(export_annotations()));
  toolsmenu->addAction("Import annotations/events", this, SLOT(import_annotations()));
  toolsmenu->addAction("Options", this, SLOT(show_options_dialog()));
  menubar->addMenu(toolsmenu);

  menubar->addAction("S&ettings", this, SLOT(show_options_dialog()));

  former_page_Act = new QAction("<<", this);
  former_page_Act->setShortcut(QKeySequence::MoveToPreviousPage);
  connect(former_page_Act, SIGNAL(triggered()), this, SLOT(former_page()));
  menubar->addAction(former_page_Act);

  shift_page_left_Act = new QAction("<", this);
  shift_page_left_Act->setShortcut(QKeySequence::MoveToPreviousChar);
  connect(shift_page_left_Act, SIGNAL(triggered()), this, SLOT(shift_page_left()));
  menubar->addAction(shift_page_left_Act);

  shift_page_right_Act = new QAction(">", this);
  shift_page_right_Act->setShortcut(QKeySequence::MoveToNextChar);
  connect(shift_page_right_Act, SIGNAL(triggered()), this, SLOT(shift_page_right()));
  menubar->addAction(shift_page_right_Act);

  next_page_Act = new QAction(">>", this);
  next_page_Act->setShortcut(QKeySequence::MoveToNextPage);
  connect(next_page_Act, SIGNAL(triggered()), this, SLOT(next_page()));
  menubar->addAction(next_page_Act);

  shift_page_up_Act = new QAction("^", this);
  shift_page_up_Act->setShortcut(QKeySequence::MoveToPreviousLine);
  connect(shift_page_up_Act, SIGNAL(triggered()), this, SLOT(shift_page_up()));
  menubar->addAction(shift_page_up_Act);

  shift_page_down_Act = new QAction("v", this);
  shift_page_down_Act->setShortcut(QKeySequence::MoveToNextLine);
  connect(shift_page_down_Act, SIGNAL(triggered()), this, SLOT(shift_page_down()));
  menubar->addAction(shift_page_down_Act);

  zoomback_Act = new QAction("zoomback", this);
  zoomback_Act->setShortcut(Qt::Key_Backspace);
  connect(zoomback_Act, SIGNAL(triggered()), this, SLOT(zoomback()));
  menubar->addAction(zoomback_Act);

  zoomforward_Act = new QAction("zoomforward", this);
  zoomforward_Act->setShortcut(Qt::Key_Insert);
  connect(zoomforward_Act, SIGNAL(triggered()), this, SLOT(forward()));
  menubar->addAction(zoomforward_Act);

  no_timesync_act = new QAction("no timelock", this);
  no_timesync_act->setCheckable(TRUE);

  offset_timesync_act = new QAction("synchronize start of files (offset)", this);
  offset_timesync_act->setCheckable(TRUE);

  absolut_timesync_act = new QAction("synchronize absolute time", this);
  absolut_timesync_act->setCheckable(TRUE);

  user_def_sync_act = new QAction("user defined synchronizing", this);
  user_def_sync_act->setCheckable(TRUE);

  timelock_act_group = new QActionGroup(this);
  timelock_act_group->addAction(no_timesync_act);
  timelock_act_group->addAction(offset_timesync_act);
  timelock_act_group->addAction(absolut_timesync_act);
  timelock_act_group->addAction(user_def_sync_act);
  absolut_timesync_act->setChecked(TRUE);
  connect(timelock_act_group, SIGNAL(triggered(QAction *)), this, SLOT(set_timesync(QAction *)));

  sel_viewtime_act_group = new QActionGroup(this);
  connect(sel_viewtime_act_group, SIGNAL(triggered(QAction *)), this, SLOT(set_timesync_reference(QAction *)));

  timemenu = new QMenu(this);
  timemenu->setTitle("T&imesync");
  timemenu->addAction("Go to start of file", this, SLOT(jump_to_start()), QKeySequence::MoveToStartOfDocument);
  timemenu->addAction("Go to end of file", this, SLOT(jump_to_end()), QKeySequence::MoveToEndOfDocument);
  timemenu->addAction("Jump to", this, SLOT(jump_to_dialog()));
  timemenu->addSeparator()->setText("Timelock");
  timemenu->addAction(no_timesync_act);
  timemenu->addAction(offset_timesync_act);
  timemenu->addAction(absolut_timesync_act);
  timemenu->addAction(user_def_sync_act);
  timemenu->addSeparator();
  timemenu->addAction("synchronize by crosshairs", this, SLOT(sync_by_crosshairs()));
  timemenu->addSeparator()->setText("Time reference");
  menubar->addMenu(timemenu);

  windowmenu = new QMenu(this);
  windowmenu->setTitle("&Window");
  windowmenu->addAction("Annotations", this, SLOT(show_annotations()));
  windowmenu->addAction("Annotation editor", this, SLOT(annotation_editor()));
  windowmenu->addAction("Spectrum", this, SLOT(show_spectrum_dock()));
  menubar->addMenu(windowmenu);

  helpmenu = new QMenu(this);
  helpmenu->setTitle("&Help");
#ifdef Q_WS_WIN
  helpmenu->addAction("Manual",  this, SLOT(show_help()));
#endif
  helpmenu->addAction("Keyboard shortcuts", this, SLOT(show_kb_shortcuts()));
  helpmenu->addAction("About EDFbrowser", this, SLOT(show_about_dialog()));
  helpmenu->addAction("About Qt", qApp, SLOT(aboutQt()));
  helpmenu->addAction("Show splashscreen", this, SLOT(show_splashscreen()));
  menubar->addMenu(helpmenu);

  positionslider = new QSlider(Qt::Horizontal);
  positionslider->setRange(0, 1000000);
  positionslider->setSingleStep(10000);
  positionslider->setPageStep(100000);

  slidertoolbar = new QToolBar();
  slidertoolbar->setFloatable(FALSE);
  slidertoolbar->setAllowedAreas(Qt::TopToolBarArea | Qt::BottomToolBarArea);
  slidertoolbar->addWidget(positionslider);
  addToolBar(Qt::BottomToolBarArea, slidertoolbar);
  QObject::connect(positionslider, SIGNAL(valueChanged(int)), this, SLOT(slider_moved(int)));
  slidertoolbar->hide();

  files_open = 0;
  signalcomps = 0;
  sel_viewtime = 0;
  viewtime_sync = VIEWTIME_SYNCED_ABSOLUT;
  pagetime = 10 * TIME_DIMENSION;
  viewtime_string[0] = 0;
  pagetime_string[0] = 0;
  totalviewbufsize = 0;
  print_to_edf_active = 0;
  annot_editor_active = 0;
  annotations_edited = 0;

  viewbuf = NULL;

  for(i=0; i<MAXFILES; i++)
  {
    annotationlist[i] = NULL;
    annotations_dock[i] = NULL;
  }

  annotationlist_backup = NULL;

  zoomhistory = (struct zoomhistoryblock *)calloc(1, sizeof(struct zoomhistoryblock));

  zoomhistory->history_size_tail = 0;
  zoomhistory->history_size_front = 0;
  for(i=0; i<64; i++)
  {
    zoomhistory->pntr = 0;
    zoomhistory->pagetime[i] = 10 * TIME_DIMENSION;
    for(j=0; j<MAXFILES; j++)
    {
      zoomhistory->viewtime[i][j] = 0;
    }
    for(j=0; j<MAXSIGNALS; j++)
    {
      zoomhistory->voltpercm[i][j] = 70.0;
      zoomhistory->screen_offset[i][j] = 0.0;
      for(k=0; k<MAXSIGNALS; k++)
      {
        zoomhistory->sensitivity[i][j][k] = 0.0475;
      }
    }
  }

  path[0] = 0;
  recent_montagedir[0] = 0;
  recent_savedir[0] = 0;
  recent_opendir[0] = 0;
  montagepath[0] = 0;

  for(i=0; i<MAX_RECENTFILES; i++)
  {
    recent_file_path[i][0] = 0;
    recent_file_mtg_path[i][0] = 0;
  }

  for(i=0; i<MAXPREDEFINEDMONTAGES; i++)
  {
    predefined_mtg_path[i][0] = 0;
  }

  read_recent_file_settings();

  for(i=0; i<MAXPREDEFINEDMONTAGES; i++)
  {
    if(predefined_mtg_path[i][0] != 0)
    {
      load_predefined_mtg_act[i]->setText(predefined_mtg_path[i]);
    }
  }

  annotationEditDock = new UI_AnnotationEditwindow(files_open, this);

  addDockWidget(Qt::BottomDockWidgetArea, annotationEditDock->dockedit, Qt::Horizontal);

  annotationEditDock->dockedit->hide();

  spectrumdock = new UI_SpectrumDockWindow(this);

  addDockWidget(Qt::TopDockWidgetArea, spectrumdock->dock, Qt::Horizontal);

  spectrumdock->dock->hide();

  setCorner(Qt::TopLeftCorner, Qt::LeftDockWidgetArea);
  setCorner(Qt::TopRightCorner, Qt::RightDockWidgetArea);
  setCorner(Qt::BottomLeftCorner, Qt::BottomDockWidgetArea);
  setCorner(Qt::BottomRightCorner, Qt::BottomDockWidgetArea);

  if(QCoreApplication::arguments().size()>1)
  {
    strcpy(path, QCoreApplication::arguments().at(1).toLatin1().data());
    cmdlineargument = 1;

    if(QCoreApplication::arguments().size()>2)
    {
      strcpy(montagepath, QCoreApplication::arguments().at(2).toLatin1().data());
      cmdlineargument = 2;
    }
  }
  else
  {
    cmdlineargument = 0;
  }

  showMaximized();

  oldwindowheight = height();

  if(cmdlineargument)
  {
    open_new_file();
  }

  if(QT_VERSION < MINIMUM_QT_VERSION)
  {
    UI_Messagewindow popupmessage("Warning", "Qt version is too old!");
  }
  else
  {
    int v_nr;

    char v_str[32];

    strncpy(v_str, qVersion(), 32);
    v_str[31] = 0;

    v_nr = 0x10000 * atoi(v_str);
    v_nr += 0x100 * atoi(v_str + 2);
    v_nr += atoi(v_str + 4);

    if(v_nr < QT_VERSION)
    {
      UI_Messagewindow popupmessage("Warning", "Qt version is too old!");
    }
  }
}



UI_Mainwindow::~UI_Mainwindow()
{
  delete pixmap;
  delete splash;
  delete t1;
  delete myfont;
  delete monofont;
  delete maincurve;
  delete annotationEditDock;
  delete spectrumdock;
}



void UI_Mainwindow::exit_program()
{
  close();
}



void UI_Mainwindow::closeEvent(QCloseEvent *event)
{
  int i,
      button_nr=0;


  if(annotations_edited)
  {
    UI_CancelSavewindow ask_to_save_window("Annotations",
                                           "There are unsaved annotations,\n are you sure you want to quit?",
                                           "Quit",
                                           "Cancel",
                                           &button_nr);
  }

  if(button_nr==1)
  {
    event->ignore();
  }
  else
  {
    exit_in_progress = 1;

    for(i=0; i<MAXSPECTRUMDIALOGS; i++)
    {
      if(spectrumdialog[i] != NULL)
      {
        delete spectrumdialog[i];
      }
    }

    annotations_edited = 0;

    close_all_files();

    write_settings();

    free(spectrum_colorbar);
    free(zoomhistory);
    free(import_annotations_var);
    free(export_annotations_var);

    event->accept();
  }
}



void UI_Mainwindow::resizeEvent(QResizeEvent *event)
{
  QWidget::resizeEvent(event);
}


#ifdef BK_MRS_project
// temporary code for private use in a certain project
// do not use this code, it will be removed in future

void UI_Mainwindow::keyboard_bk_mrs_project_func()
{
  int i, j,
      len,
      type,
      model,
      order,
      file_present=0;

  char path[1024],
       str_signal[1024],
       str_dimension[32];

  double result_mrs,
         result_rms,
         frequency,
         frequency2,
         ripple;

  struct edfhdrblock *hdr;


  hdr = edfheaderlist[0];

  if((!files_open) || (!signalcomps))
  {
    return;
  }

  if(bk_mrs_project_file == NULL)
  {
    get_filename_from_path(path, hdr->filename, 1024);
    remove_extension_from_filename(path);
    strcat(path, "_mrs.txt");

    strcpy(path, QFileDialog::getSaveFileName(0, "MRS output file", path, "Text files (*.txt *.TXT)").toLatin1().data());

    if(!strcmp(path, ""))
    {
      return;
    }

    bk_mrs_project_file = fopeno(path, "rb");
    if(bk_mrs_project_file != NULL)
    {
      fclose(bk_mrs_project_file);
      bk_mrs_project_file = NULL;
      file_present = 1;
    }

    if(file_present)
    {
      bk_mrs_project_file = fopeno(path, "ab");
    }
    else
    {
      bk_mrs_project_file = fopeno(path, "wb");
    }

    if(bk_mrs_project_file == NULL)
    {
      UI_Messagewindow popuperror("Error", "Error, can not open outputfile for writing.");
      return;
    }

    if(!file_present)
    {
      fprintf(bk_mrs_project_file, "starttime,pagetime,signal,number_of_samples,mrs,rms,physical_dimension,file,<filters>\n");
    }
  }

  for(i=0; i<signalcomps; i++)
  {
    strcpy(str_signal, signalcomp[i]->signallabel);
    len = strlen(str_signal);
    for(j=0; j<len; j++)
    {
      if(str_signal[j] == ',')
      {
        str_signal[j] = '.';
      }
    }

    if(signalcomp[i]->stat_cnt)
    {
      result_mrs = (signalcomp[i]->stat_sum_rectified / signalcomp[i]->stat_cnt)  * signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].bitvalue;
      result_rms = sqrt(signalcomp[i]->stat_sum_sqr / signalcomp[i]->stat_cnt)  * signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].bitvalue;
    }
    else
    {
      result_mrs = 0.0;
      result_rms = 0.0;
    }

    strcpy(str_dimension, signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].physdimension);
    len = strlen(str_dimension);
    for(j=0; j<len; j++)
    {
      if(str_dimension[j] == ',')
      {
        str_dimension[j] = '.';
      }
    }

#ifdef Q_WS_WIN
    fprintf(bk_mrs_project_file, "%I64d.%07I64d,%I64d.%07I64d,%s,%i,%f,%f,%s",
            hdr->viewtime / TIME_DIMENSION,
            hdr->viewtime % TIME_DIMENSION,
            pagetime / TIME_DIMENSION,
            pagetime % TIME_DIMENSION,
            str_signal,
            signalcomp[i]->stat_cnt,
            result_mrs,
            result_rms,
            str_dimension);
#else
    fprintf(bk_mrs_project_file, "%lli.%07lli,%lli.%07lli,%s,%i,%f,%f,%s",
            hdr->viewtime / TIME_DIMENSION,
            hdr->viewtime % TIME_DIMENSION,
            pagetime / TIME_DIMENSION,
            pagetime % TIME_DIMENSION,
            str_signal,
            signalcomp[i]->stat_cnt,
            result_mrs,
            result_rms,
            str_dimension);
#endif

    fprintf(bk_mrs_project_file, ",%s", hdr->filename);

    for(j=0; j<signalcomp[i]->fidfilter_cnt; j++)
    {
      type = signalcomp[i]->fidfilter_type[j];

      model = signalcomp[i]->fidfilter_model[j];

      frequency = signalcomp[i]->fidfilter_freq[j];

      frequency2 = signalcomp[i]->fidfilter_freq2[j];

      order = signalcomp[i]->fidfilter_order[j];

      ripple = signalcomp[i]->fidfilter_ripple[j];

      if(type == 0)
      {
        fprintf(bk_mrs_project_file, ",HP");
      }

      if(type == 1)
      {
        fprintf(bk_mrs_project_file, ",LP");
      }

      if(type == 2)
      {
        fprintf(bk_mrs_project_file, ",N Res %fHz Q-factor %i", frequency, order);
      }

      if(type == 3)
      {
        fprintf(bk_mrs_project_file, ",BP");
      }

      if(type == 4)
      {
        fprintf(bk_mrs_project_file, ",BS");
      }

      if(type != 2)
      {
        if(model == 0)
        {
          fprintf(bk_mrs_project_file, " Bu");
        }

        if(model == 1)
        {
          fprintf(bk_mrs_project_file, " Ch");
        }

        if(model == 2)
        {
          fprintf(bk_mrs_project_file, " Be");
        }

        if((type == 0) || (type == 1))
        {
          fprintf(bk_mrs_project_file, " %fHz %ith order", frequency, order);
        }

        if((type == 3) || (type == 4))
        {
          fprintf(bk_mrs_project_file, " %f-%fHz %ith order", frequency, frequency2, order);
        }

        if(model == 1)
        {
          fprintf(bk_mrs_project_file, " ripple %fdB", ripple);
        }
      }
    }

    fputc('\n', bk_mrs_project_file);
  }

  fflush(bk_mrs_project_file);

  UI_Messagewindow popupmessage("Message", "Values are written.", NULL, 1000);
}

#endif


void UI_Mainwindow::save_file()
{
  int len;

  char path[1024];

  struct edfhdrblock *hdr;

  struct annotationblock *annot;

  FILE *outputfile;


  if((!annotations_edited)||(!files_open))
  {
    save_act->setEnabled(FALSE);

    return;
  }

  hdr = edfheaderlist[0];

  strcpy(path, recent_savedir);
  strcat(path, "/");
  len = strlen(path);
  get_filename_from_path(path + len, hdr->filename, 1024 - len);
  remove_extension_from_filename(path);
  if(hdr->edf)
  {
    strcat(path, "_edited.edf");
  }
  else
  {
    strcat(path, "_edited.bdf");
  }

  strcpy(path, QFileDialog::getSaveFileName(this, "Save file", path, "EDF/BDF files (*.edf *.EDF *.bdf *.BDF *.rec *.REC)").toLatin1().data());

  if(!strcmp(path, ""))
  {
    return;
  }

  get_directory_from_path(recent_savedir, path, 1024);

  if(file_is_opened(path))
  {
    UI_Messagewindow popuperror("Save file", "Error, selected file is in use.");
    return;
  }

  outputfile = fopeno(path, "wb");
  if(outputfile==NULL)
  {
    UI_Messagewindow popuperror("Save file", "Error, can not create a file for writing.");
    return;
  }

  if(save_annotations(this, outputfile, hdr, annotationlist[0]))
  {
    UI_Messagewindow popuperror("Save file", "An error occurred during saving.");
    fclose(outputfile);
    return;
  }

  fclose(outputfile);

  annot = annotationlist[0];

  annotationlist[0] = annotationlist_backup;

  annotationlist_backup = NULL;

  edfplus_annotation_delete_list(&annot);

  annotations_dock[0]->updateList(0);

  annotations_edited = 0;

  save_act->setEnabled(FALSE);

  annotationEditDock->dockedit->hide();

  maincurve->update();
}



void UI_Mainwindow::slider_moved(int value)
{
  int i;

  long long new_viewtime,
            tmp;


  if(!signalcomps)  return;

  new_viewtime = edfheaderlist[sel_viewtime]->long_data_record_duration * edfheaderlist[sel_viewtime]->datarecords;

  new_viewtime -= pagetime;

  if(new_viewtime<0)
  {
    new_viewtime = 0;
  }
  else
  {
    new_viewtime /= 1000000;

    new_viewtime *= value;
  }

  if(pagetime >= (300LL * TIME_DIMENSION))
  {
    tmp = new_viewtime % (30LL * TIME_DIMENSION);

    new_viewtime -= tmp;
  }
  else
    if(pagetime >= (60LL * TIME_DIMENSION))
    {
      tmp = new_viewtime % (6LL * TIME_DIMENSION);

      new_viewtime -= tmp;
    }
    else
      if(pagetime >= (30LL * TIME_DIMENSION))
      {
        tmp = new_viewtime % (3LL * TIME_DIMENSION);

        new_viewtime -= tmp;
      }
      else
        if(pagetime >= (20LL * TIME_DIMENSION))
        {
          tmp = new_viewtime % (2LL * TIME_DIMENSION);

          new_viewtime -= tmp;
        }
        else
          if(pagetime >= (10LL * TIME_DIMENSION))
          {
            tmp = new_viewtime % TIME_DIMENSION;

            new_viewtime -= tmp;
          }
          else
            if(pagetime >= TIME_DIMENSION)
            {
              tmp = new_viewtime % (TIME_DIMENSION / 10LL);

              new_viewtime -= tmp;
            }

  if(viewtime_sync==VIEWTIME_SYNCED_OFFSET)
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime = new_viewtime;
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime = new_viewtime;
  }

  if(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)
  {
    edfheaderlist[sel_viewtime]->viewtime = new_viewtime;

    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime = edfheaderlist[sel_viewtime]->viewtime - ((edfheaderlist[i]->utc_starttime - edfheaderlist[sel_viewtime]->utc_starttime) * TIME_DIMENSION);
      }
    }
  }

  if(viewtime_sync==VIEWTIME_USER_DEF_SYNCED)
  {
    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime -= (edfheaderlist[sel_viewtime]->viewtime - new_viewtime);
      }
    }

    edfheaderlist[sel_viewtime]->viewtime = new_viewtime;
  }

  setup_viewbuf();
}



void UI_Mainwindow::set_timesync_reference(QAction *action)
{
  int i;

  for(i=0; i<files_open; i++)
  {
    if(!strcmp(edfheaderlist[i]->filename, action->text().toLatin1().data()))
    {
      break;
    }
  }

  sel_viewtime = i;

  setMainwindowTitle(edfheaderlist[sel_viewtime]);

  setup_viewbuf();
}



void UI_Mainwindow::set_timesync(QAction *)
{
  int i;


  if(no_timesync_act->isChecked())
  {
    viewtime_sync = VIEWTIME_UNSYNCED;
  }

  if(offset_timesync_act->isChecked())
  {
    viewtime_sync = VIEWTIME_SYNCED_OFFSET;

    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime = edfheaderlist[sel_viewtime]->viewtime;
    }
  }

  if(absolut_timesync_act->isChecked())
  {
    viewtime_sync = VIEWTIME_SYNCED_ABSOLUT;

    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime = edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset - ((edfheaderlist[i]->utc_starttime - edfheaderlist[sel_viewtime]->utc_starttime) * TIME_DIMENSION) - edfheaderlist[i]->starttime_offset;
      }
    }
  }

  if(user_def_sync_act->isChecked())
  {
    viewtime_sync = VIEWTIME_USER_DEF_SYNCED;
  }

  setup_viewbuf();
}



void UI_Mainwindow::sync_by_crosshairs()
{
  if(files_open<2)  return;
  if(signalcomps<2)  return;

  if(maincurve->crosshair_1_active&&maincurve->crosshair_2_active)
  {
    if(maincurve->crosshair_1_file_num!=maincurve->crosshair_2_file_num)
    {
      viewtime_sync = VIEWTIME_USER_DEF_SYNCED;

      edfheaderlist[maincurve->crosshair_2_file_num]->viewtime
      -= ((maincurve->crosshair_1_time - edfheaderlist[maincurve->crosshair_1_file_num]->l_starttime - edfheaderlist[maincurve->crosshair_1_file_num]->viewtime - edfheaderlist[maincurve->crosshair_1_file_num]->starttime_offset)
      - (maincurve->crosshair_2_time - edfheaderlist[maincurve->crosshair_2_file_num]->l_starttime - edfheaderlist[maincurve->crosshair_2_file_num]->viewtime - edfheaderlist[maincurve->crosshair_2_file_num]->starttime_offset));

      maincurve->crosshair_2_x_position = maincurve->crosshair_1_x_position;

      user_def_sync_act->setChecked(TRUE);

      viewtime_sync = VIEWTIME_USER_DEF_SYNCED;

      setup_viewbuf();
    }
  }
}


void UI_Mainwindow::show_options_dialog()
{
  UI_OptionsDialog OptionsDialog(this);
}


void UI_Mainwindow::nk2edf_converter()
{
  UI_NK2EDFwindow nk2edf(recent_opendir);
}


void UI_Mainwindow::convert_ascii_to_edf()
{
  UI_ASCII2EDFapp ascii2edf(recent_opendir, recent_savedir);
}


void UI_Mainwindow::convert_fino_to_edf()
{
  UI_FINO2EDFwindow fino2edf(recent_opendir, recent_savedir);
}


void UI_Mainwindow::convert_nexfin_to_edf()
{
  UI_NEXFIN2EDFwindow nexfin2edf(recent_opendir, recent_savedir);
}


void UI_Mainwindow::convert_emsa_to_edf()
{
  UI_EMSA2EDFwindow emsa2edf(recent_opendir, recent_savedir);
}


void UI_Mainwindow::edfd_converter()
{
  UI_EDFDwindow edfplusd2edfplusc(recent_opendir, recent_savedir);
}


void UI_Mainwindow::bdf2edf_converter()
{
  UI_BDF2EDFwindow bdf2edfconv(this);
}


void UI_Mainwindow::biosemi2bdfplus_converter()
{
  UI_BIOSEMI2BDFPLUSwindow biosemi2bdfplusconv(this);
}


void UI_Mainwindow::reduce_signals()
{
  UI_ReduceSignalsWindow reduceSignals(this);
}


void UI_Mainwindow::edit_header()
{
  UI_headerEditorWindow header_edit(this);
}


void UI_Mainwindow::BI98002edf_converter()
{
  UI_BI98002EDFwindow BI98002edfconv(recent_opendir, recent_savedir);
}


void UI_Mainwindow::edit_predefined_montages()
{
  UI_edit_predefined_mtg_window edit_predef_mtgs_app(this);
}


void UI_Mainwindow::jump_to_dialog()
{
  UI_JumpMenuDialog jumpmenu(this);
}



void UI_Mainwindow::jump_to_start()
{
  int i;

  if(viewtime_sync==VIEWTIME_SYNCED_OFFSET)
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime = 0;
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime = 0;
  }

  if(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)
  {
    edfheaderlist[sel_viewtime]->viewtime = 0;

    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime = ((edfheaderlist[sel_viewtime]->utc_starttime - edfheaderlist[i]->utc_starttime) * TIME_DIMENSION) + edfheaderlist[sel_viewtime]->starttime_offset - edfheaderlist[i]->starttime_offset;
      }
    }
  }

  if(viewtime_sync==VIEWTIME_USER_DEF_SYNCED)
  {
    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime -= edfheaderlist[sel_viewtime]->viewtime;
      }
    }

    edfheaderlist[sel_viewtime]->viewtime = 0;
  }

  setup_viewbuf();
}



void UI_Mainwindow::jump_to_end()
{
  int i;

  long long new_viewtime;


  new_viewtime = edfheaderlist[sel_viewtime]->datarecords * edfheaderlist[sel_viewtime]->long_data_record_duration - pagetime;

  if(viewtime_sync==VIEWTIME_SYNCED_OFFSET)
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime = new_viewtime;
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime = new_viewtime;
  }

  if(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)
  {
    edfheaderlist[sel_viewtime]->viewtime = new_viewtime;

    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime = edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset - ((edfheaderlist[i]->utc_starttime - edfheaderlist[sel_viewtime]->utc_starttime) * TIME_DIMENSION) - edfheaderlist[i]->starttime_offset;
      }
    }
  }

  if(viewtime_sync==VIEWTIME_USER_DEF_SYNCED)
  {
    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime -= (edfheaderlist[sel_viewtime]->viewtime - new_viewtime);
      }
    }

    edfheaderlist[sel_viewtime]->viewtime = new_viewtime;
  }

  setup_viewbuf();
}



void UI_Mainwindow::add_new_filter()
{
  UI_FilterDialog filterdialog(this);
}



void UI_Mainwindow::zoomback()
{
  int i, j;

  if(!zoomhistory->history_size_tail)
  {
    return;
  }
  zoomhistory->history_size_front++;
  zoomhistory->history_size_tail--;

  for(i=0; i<files_open; i++)
  {
    zoomhistory->viewtime[zoomhistory->pntr][i] = edfheaderlist[i]->viewtime;
  }
  zoomhistory->pagetime[zoomhistory->pntr] = pagetime;
  for(i=0; i<signalcomps; i++)
  {
    zoomhistory->voltpercm[zoomhistory->pntr][i] = signalcomp[i]->voltpercm;
    zoomhistory->screen_offset[zoomhistory->pntr][i] = signalcomp[i]->screen_offset;

    for(j=0; j<signalcomp[i]->num_of_signals; j++)
    {
      zoomhistory->sensitivity[zoomhistory->pntr][i][j] = signalcomp[i]->sensitivity[j];
    }
  }

  zoomhistory->pntr--;
  if(zoomhistory->pntr<0)  zoomhistory->pntr = 63;

  for(i=0; i<files_open; i++)
  {
    edfheaderlist[i]->viewtime = zoomhistory->viewtime[zoomhistory->pntr][i];
  }
  pagetime = zoomhistory->pagetime[zoomhistory->pntr];

  for(i=0; i<signalcomps; i++)
  {
    signalcomp[i]->voltpercm = zoomhistory->voltpercm[zoomhistory->pntr][i];
    signalcomp[i]->screen_offset = zoomhistory->screen_offset[zoomhistory->pntr][i];

    for(j=0; j<signalcomp[i]->num_of_signals; j++)
    {
      signalcomp[i]->sensitivity[j] = zoomhistory->sensitivity[zoomhistory->pntr][i][j];
    }
  }

  setup_viewbuf();
}



void UI_Mainwindow::forward()
{
  int i, j;

  if(!zoomhistory->history_size_front)
  {
    return;
  }
  zoomhistory->history_size_front--;
  zoomhistory->history_size_tail++;

  zoomhistory->pntr++;
  if(zoomhistory->pntr>63)  zoomhistory->pntr = 0;

  for(i=0; i<files_open; i++)
  {
    edfheaderlist[i]->viewtime = zoomhistory->viewtime[zoomhistory->pntr][i];
  }
  pagetime = zoomhistory->pagetime[zoomhistory->pntr];

  for(i=0; i<signalcomps; i++)
  {
    signalcomp[i]->voltpercm = zoomhistory->voltpercm[zoomhistory->pntr][i];
    signalcomp[i]->screen_offset = zoomhistory->screen_offset[zoomhistory->pntr][i];

    for(j=0; j<signalcomp[i]->num_of_signals; j++)
    {
      signalcomp[i]->sensitivity[j] = zoomhistory->sensitivity[zoomhistory->pntr][i][j];
    }
  }

  setup_viewbuf();
}



void UI_Mainwindow::former_page()
{
  int i;

  if((viewtime_sync==VIEWTIME_SYNCED_OFFSET)||(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)||(viewtime_sync==VIEWTIME_USER_DEF_SYNCED))
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime -= pagetime;
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime -= pagetime;
  }

  setup_viewbuf();
}



void UI_Mainwindow::shift_page_left()
{
  int i;

  if((viewtime_sync==VIEWTIME_SYNCED_OFFSET)||(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)||(viewtime_sync==VIEWTIME_USER_DEF_SYNCED))
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime -= (pagetime / 10);
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime -= (pagetime / 10);
  }

  setup_viewbuf();
}



void UI_Mainwindow::shift_page_right()
{
  int i;

  if((viewtime_sync==VIEWTIME_SYNCED_OFFSET)||(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)||(viewtime_sync==VIEWTIME_USER_DEF_SYNCED))
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime += (pagetime / 10);
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime += (pagetime / 10);
  }

  setup_viewbuf();
}



void UI_Mainwindow::next_page()
{
  int i;

  if((viewtime_sync==VIEWTIME_SYNCED_OFFSET)||(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)||(viewtime_sync==VIEWTIME_USER_DEF_SYNCED))
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime += pagetime;
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime += pagetime;
  }

  setup_viewbuf();
}



void UI_Mainwindow::shift_page_up()
{
  int i;

  for(i=0; i<signalcomps; i++)
  {
    signalcomp[i]->screen_offset += (height() / 20.0);

  }

  maincurve->drawCurve_stage_1();
}



void UI_Mainwindow::shift_page_down()
{
  int i;

  for(i=0; i<signalcomps; i++)
  {
    signalcomp[i]->screen_offset -= (height() / 20.0);

  }

  maincurve->drawCurve_stage_1();
}



void UI_Mainwindow::show_annotations()
{
  int i;

  for(i=0; i<files_open; i++)
  {
    if(annotationlist[i])
    {
      annotations_dock[i]->docklist->show();
    }
  }
}



void UI_Mainwindow::annotation_editor()
{
  if(files_open==1)
  {
    if(annotations_dock[0]==NULL)
    {
      annotations_dock[0] = new UI_Annotationswindow(0, this);

      addDockWidget(Qt::RightDockWidgetArea, annotations_dock[0]->docklist, Qt::Vertical);
    }

    annotations_dock[0]->docklist->show();

    annotationEditDock->dockedit->show();
  }
  else
  {
    UI_Messagewindow popupmesg("", "Editing annotations is possible when you have opened one file only.");
  }
}



void UI_Mainwindow::show_spectrum_dock()
{
  UI_SignalChooser signalchooserdialog(this, 1);
}



void UI_Mainwindow::open_new_file()
{
  FILE *newfile;

  int i, len, present, position, button_nr=0;

  char str[2048];

  str[0] = 0;

  struct edfhdrblock *edfhdr=NULL;

  if(annot_editor_active&&files_open)
  {
    UI_Messagewindow popupmesg("", "You can not open multiple files when editing annotations.\n"
                                   "Close the annotation edit window first.");

    cmdlineargument = 0;

    return;
  }

  if(files_open>=MAXFILES)
  {
    UI_Messagewindow popuperror("Error", "Error, there are too many files opened.");
    cmdlineargument = 0;
    return;
  }

  if(cmdlineargument == 0)
  {
    strcpy(path, QFileDialog::getOpenFileName(this, "Open file", recent_opendir, "EDF/BDF files (*.edf *.EDF *.bdf *.BDF *.rec *.REC)").toLatin1().data());

    if(!strcmp(path, ""))
    {
      return;
    }

    get_directory_from_path(recent_opendir, path, 1024);
  }

  if((cmdlineargument == 0) || (cmdlineargument == 1))
  {
    montagepath[0] = 0;
  }

  present = 0;

  for(i=0; i<MAX_RECENTFILES; i++)
  {
    if(!strcmp(&recent_file_path[i][0], path))
    {
      present = 1;

      position = i;

      if(cmdlineargument!=2)
      {
        strcpy(montagepath, &recent_file_mtg_path[i][0]);
      }

      break;
    }
  }

  if(present)
  {
    for(i=position; i>0; i--)
    {
      strcpy(&recent_file_path[i][0], &recent_file_path[i-1][0]);
      strcpy(&recent_file_mtg_path[i][0], &recent_file_mtg_path[i-1][0]);
    }
  }
  else
  {
    for(i=MAX_RECENTFILES-1; i>0; i--)
    {
      strcpy(&recent_file_path[i][0], &recent_file_path[i-1][0]);
      strcpy(&recent_file_mtg_path[i][0], &recent_file_mtg_path[i-1][0]);
    }
  }

  strcpy(&recent_file_path[0][0], path);
  strcpy(&recent_file_mtg_path[0][0], montagepath);

  recent_filesmenu->clear();

  for(i=0; i<MAX_RECENTFILES; i++)
  {
    if(recent_file_path[i][0]==0)
    {
      break;
    }
    recent_filesmenu->addAction(&recent_file_path[i][0]);
  }

  present = 0;

  for(i=0; i<files_open; i++)
  {
    if(!strcmp(edfheaderlist[i]->filename, path))
    {
      present = 1;

      break;
    }
  }

  if(!present)
  {
    len = strlen(path);

    if((strcmp(path + (len - 4), ".edf"))
       &&(strcmp(path + (len - 4), ".EDF"))
       &&(strcmp(path + (len - 4), ".rec"))
       &&(strcmp(path + (len - 4), ".REC"))
       &&(strcmp(path + (len - 4), ".bdf"))
       &&(strcmp(path + (len - 4), ".BDF")))
    {
      snprintf(str, 2048, "File has an unknown extension:  \"%s\"", path + (len - 4));

      UI_Messagewindow popuperror("Error", str);

      cmdlineargument = 0;

      return;
    }

    newfile = fopeno(path, "rb");
    if(newfile==NULL)
    {
      UI_Messagewindow popuperror("Error", "Can not open file for reading");
      cmdlineargument = 0;
      return;
    }

    strcpy(recent_opendir, path);

    if(len)
    {
      for(i=len-1; i>=0; i--)
      {
        if((path[i] == '/')||(path[i] == '\\'))
        {
          break;
        }
      }
    }

    recent_opendir[i] = 0;

    EDFfileCheck EDFfilechecker;

    edfhdr = EDFfilechecker.check_edf_file(newfile, str);
    if(edfhdr==NULL)
    {
      fclose(newfile);

      strcat(str, "\n File is not a valid EDF or BDF file.");

      UI_Messagewindow popuperror("Error", str);

      cmdlineargument = 0;

      return;
    }

    if(edfhdr->discontinuous)
    {
      if(edfhdr->edf)
      {
        UI_Messagewindow popuperror("Error", "EDFbrowser can not show EDF+D (discontinuous) files.\n"
                                    "Convert this file to EDF+C first. You can find this converter\n"
                                    "in the Tools menu (EDF+D to EDF+C converter).");
      }

      if(edfhdr->bdf)
      {
        UI_Messagewindow popuperror("Error", "EDFbrowser can not show BDF+D (discontinuous) files.\n"
                                    "Convert this file to BDF+C first. You can find this converter\n"
                                    "in the Tools menu (EDF+D to EDF+C converter).");
      }

      free(edfhdr->edfparam);
      free(edfhdr);
      fclose(newfile);
      cmdlineargument = 0;

      return;
    }

    strcpy(edfhdr->filename, path);

    edfhdr->file_hdl = newfile;

    edfheaderlist[files_open] = edfhdr;

    annotationlist[files_open] = NULL;

    annotations_dock[files_open] = NULL;

    if(edfhdr->edfplus || edfhdr->bdfplus)
    {
      QLabel waitlabel("Looking for annotations...", this);
      waitlabel.setGeometry(width() / 2, height() / 2, 325, 25);
      waitlabel.show();

      EDF_annotations annotations;

      annotations.get_annotations(files_open, edfhdr, &annotationlist[files_open]);

      waitlabel.hide();

      annotations_dock[files_open] = new UI_Annotationswindow(files_open, this);

      addDockWidget(Qt::RightDockWidgetArea, annotations_dock[files_open]->docklist, Qt::Vertical);

      if(!annotationlist[files_open])
      {
        annotations_dock[files_open]->docklist->hide();
      }
    }

    if(edfhdr->bdf && (!edfhdr->bdfplus) && read_biosemi_status_signal)
    {
      QLabel waitlabel("Looking for trigger inputs...", this);
      waitlabel.setGeometry(width() / 2, height() / 2, 325, 25);
      waitlabel.show();

      BDF_triggers bdf_triggers_obj;

      bdf_triggers_obj.get_triggers(edfhdr, &annotationlist[files_open]);

      waitlabel.hide();

      annotations_dock[files_open] = new UI_Annotationswindow(files_open, this);

      addDockWidget(Qt::RightDockWidgetArea, annotations_dock[files_open]->docklist, Qt::Vertical);

      if(!annotationlist[files_open])
      {
        annotations_dock[files_open]->docklist->hide();
      }
    }

    if(!files_open)
    {
      edfheaderlist[0]->viewtime = 0;

      setMainwindowTitle(edfhdr);
    }
    else
    {
      if(viewtime_sync==VIEWTIME_SYNCED_OFFSET)
      {
        edfheaderlist[files_open]->viewtime = edfheaderlist[sel_viewtime]->viewtime;
      }

      if(viewtime_sync==VIEWTIME_UNSYNCED)
      {
        edfheaderlist[files_open]->viewtime = 0;
      }

      if(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)
      {
        edfheaderlist[files_open]->viewtime = edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset - edfheaderlist[files_open]->starttime_offset - ((edfheaderlist[files_open]->utc_starttime - edfheaderlist[sel_viewtime]->utc_starttime) * TIME_DIMENSION);
      }

      if(viewtime_sync==VIEWTIME_USER_DEF_SYNCED)
      {
          edfheaderlist[files_open]->viewtime = 0;
      }
    }

    sel_viewtime_act[files_open] = new QAction(edfhdr->filename, this);
    sel_viewtime_act[files_open]->setCheckable(TRUE);
    if(!files_open)
    {
      sel_viewtime_act[files_open]->setChecked(TRUE);
    }
    sel_viewtime_act_group->addAction(sel_viewtime_act[files_open]);
    timemenu->addAction(sel_viewtime_act[files_open]);

    files_open++;
  }

  if((montagepath[0]!=0)&&(cmdlineargument==2))
  {
    UI_LoadMontagewindow load_montage(this, montagepath);
    strcpy(&recent_file_mtg_path[0][0], montagepath);
  }
  else
  {
    if((recent_file_mtg_path[0][0] != 0) && (files_open == 1) && auto_reload_mtg)
    {
      sprintf(str, "Load last used montage?\n\n(%s)", &recent_file_mtg_path[0][0]);

      UI_CancelSavewindow popupmsg("Load Montage", str, "Yes", "No", &button_nr);

      if(button_nr == 1)
      {
        recent_file_mtg_path[0][0] = 0;

        UI_Signalswindow signalwindow(this);
      }
      else
      {
        UI_LoadMontagewindow load_montage(this, &recent_file_mtg_path[0][0]);

        if(recent_file_mtg_path[0][0] == 0)
        {
          UI_Signalswindow signalwindow(this);
        }
      }
    }
    else
    {
      UI_Signalswindow signalwindow(this);
    }
  }

  cmdlineargument = 0;
}



void UI_Mainwindow::remove_recent_file_mtg_path(const char *mtg_path)
{
  int i;


  for(i=0; i<MAX_RECENTFILES; i++)
  {
    if(!strcmp(&recent_file_mtg_path[i][0], mtg_path))
    {
      recent_file_mtg_path[i][0] = 0;
    }
  }
}



void UI_Mainwindow::setMainwindowTitle(struct edfhdrblock *edfhdr)
{
  int i, len;

  char str[256];

  struct date_time_struct date_time;


  if(edfhdr==NULL)
  {
    setWindowTitle(PROGRAM_NAME);

    return;
  }

  if(edfhdr->edfplus || edfhdr->bdfplus)
  {
    snprintf(str, 256, PROGRAM_NAME "  patient %s  birthdate %s  startdate %s",
                  edfhdr->plus_patient_name,
                  edfhdr->plus_birthdate,
                  edfhdr->plus_startdate);
  }
  else
  {
    utc_to_date_time(edfhdr->utc_starttime, &date_time);

    date_time.month_str[0] += 32;
    date_time.month_str[1] += 32;
    date_time.month_str[2] += 32;

    snprintf(str, 256, PROGRAM_NAME "  %s  startdate %i %s %i",
                  edfhdr->patient,
                  date_time.day,
                  date_time.month_str,
                  date_time.year);
    len = strlen(str);
    for(i=0; i<len; i++)
    {
      if(str[i]=='_')
      {
        str[i] = ' ';
      }
    }
  }

  setWindowTitle(str);
}



void UI_Mainwindow::signalproperties_dialog()
{
  UI_SignalChooser signalchooserdialog(this, 0);
}



void UI_Mainwindow::filterproperties_dialog()
{
  UI_SignalChooser signalchooserdialog(this, 2);
}



void UI_Mainwindow::add_signals_dialog()
{
  UI_Signalswindow signalwindow(this);
}



void UI_Mainwindow::show_splashscreen()
{
  splash = new QSplashScreen(this, *pixmap, Qt::WindowStaysOnTopHint);

  splash->show();
}



void UI_Mainwindow::show_file_info()
{
  UI_EDFhdrwindow showhdr(this);
}



void UI_Mainwindow::remove_all_filters()
{
  int i, j,
      update=0;

  for(i=0; i<signalcomps; i++)
  {
    for(j=0; j<signalcomp[i]->filter_cnt; j++)
    {
      free(signalcomp[i]->filter[j]);

      update = 1;
    }

    signalcomp[i]->filter_cnt = 0;

    for(j=0; j<signalcomp[i]->fidfilter_cnt; j++)
    {
      free(signalcomp[i]->fidfilter[j]);
      fid_run_free(signalcomp[i]->fid_run[j]);
      fid_run_freebuf(signalcomp[i]->fidbuf[j]);
      fid_run_freebuf(signalcomp[i]->fidbuf2[j]);

      update = 1;
    }

    signalcomp[i]->fidfilter_cnt = 0;
  }

  if(update)
  {
    maincurve->drawCurve_stage_1();
  }
}



void UI_Mainwindow::remove_all_signals()
{
  int i;

  spectrumdock->clear();
  spectrumdock->dock->hide();

  for(i=0; i<MAXSPECTRUMDIALOGS; i++)
  {
    if(spectrumdialog[i] != NULL)
    {
      delete spectrumdialog[i];

      spectrumdialog[i] = NULL;
    }
  }

  maincurve->crosshair_1_active = 0;
  maincurve->crosshair_2_active = 0;
  maincurve->crosshair_1_moving = 0;
  maincurve->crosshair_2_moving = 0;

  remove_all_filters();

  for(i=0; i<signalcomps; i++)
  {
    free(signalcomp[i]);
  }

  signalcomps = 0;

  if(viewbuf!=NULL)
  {
    free(viewbuf);
    viewbuf = NULL;
  }

  slidertoolbar->hide();
}


void UI_Mainwindow::close_all_files()
{
  int i, j, k,
      button_nr=0;


  if(annotations_edited)
  {
    UI_CancelSavewindow ask_to_save_window("Annotations",
                                           "There are unsaved annotations,\n are you sure you want to close this file?",
                                           "Close file",
                                           "Cancel",
                                           &button_nr);
  }

  if(button_nr==1)
  {
    return;
  }

  annotations_edited = 0;

#ifdef BK_MRS_project
// temporary code for private use in a certain project
// do not use this code, it will be removed in future

  if(bk_mrs_project_file != NULL)
  {
    fclose(bk_mrs_project_file);
    bk_mrs_project_file = NULL;
  }

#endif

  remove_all_signals();

  while(files_open)
  {
    files_open--;
    fclose(edfheaderlist[files_open]->file_hdl);
    free(edfheaderlist[files_open]->edfparam);
    free(edfheaderlist[files_open]);
    edfplus_annotation_delete_list(&annotationlist[files_open]);

    if(annotations_dock[files_open])
    {
      annotations_dock[files_open]->docklist->close();
      delete annotations_dock[files_open];
      annotations_dock[files_open] = NULL;
    }

    delete sel_viewtime_act[files_open];
  }

  edfplus_annotation_delete_list(&annotationlist_backup);

  sel_viewtime = 0;

  pagetime = 10 * TIME_DIMENSION;

  for(i=0; i<64; i++)
  {
    zoomhistory->pntr = 0;
    zoomhistory->pagetime[i] = 10 * TIME_DIMENSION;
    for(j=0; j<MAXFILES; j++)
    {
      zoomhistory->viewtime[i][j] = 0;
    }
    for(j=0; j<MAXSIGNALS; j++)
    {
      zoomhistory->voltpercm[i][j] = 70;
      zoomhistory->screen_offset[i][j] = 0.0;
      for(k=0; k<MAXSIGNALS; k++)
      {
        zoomhistory->sensitivity[i][j][k] = 0.0475;
      }
    }
  }

  annotationEditDock->dockedit->hide();

  save_act->setEnabled(FALSE);

  annotations_edited = 0;

  setWindowTitle(PROGRAM_NAME);

  if(!exit_in_progress)
  {
    maincurve->update();
  }
}


void UI_Mainwindow::show_about_dialog()
{
  UI_Aboutwindow aboutwindow;
}


void UI_Mainwindow::page_3cmsec()
{
  double mm;

  if(auto_dpi)
  {
    mm = maincurve->widthMM();

    pagetime = mm * 333333.3333;
  }
  else
  {
    pagetime = (long long)((((double)maincurve->width()) / (1.0 / x_pixelsizefactor) / 3.0) * TIME_DIMENSION);
  }

  setup_viewbuf();
}


void UI_Mainwindow::set_page_div2()
{
  int i;

  if((viewtime_sync==VIEWTIME_SYNCED_OFFSET)||(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)||(viewtime_sync==VIEWTIME_USER_DEF_SYNCED))
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime += (pagetime / 4);
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime += (pagetime / 4);
  }

  pagetime /= 2;

  setup_viewbuf();
}


void UI_Mainwindow::set_page_mult2()
{
  int i;

  if((viewtime_sync==VIEWTIME_SYNCED_OFFSET)||(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)||(viewtime_sync==VIEWTIME_USER_DEF_SYNCED))
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime -= (pagetime / 2);
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime -= (pagetime / 2);
  }

  pagetime *= 2;

  setup_viewbuf();
}


void UI_Mainwindow::set_display_time(QAction *action)
{
  if(action==page_10m)   pagetime = TIME_DIMENSION / 100;
  if(action==page_20m)   pagetime = TIME_DIMENSION / 50;
  if(action==page_50m)   pagetime = TIME_DIMENSION / 20;
  if(action==page_100m)  pagetime = TIME_DIMENSION / 10;
  if(action==page_200m)  pagetime = TIME_DIMENSION / 5;
  if(action==page_500m)  pagetime = TIME_DIMENSION / 2;
  if(action==page_1)     pagetime = TIME_DIMENSION;
  if(action==page_2)     pagetime = TIME_DIMENSION * 2;
  if(action==page_5)     pagetime = TIME_DIMENSION * 5;
  if(action==page_10)    pagetime = TIME_DIMENSION * 10;
  if(action==page_15)    pagetime = TIME_DIMENSION * 15;
  if(action==page_20)    pagetime = TIME_DIMENSION * 20;
  if(action==page_30)    pagetime = TIME_DIMENSION * 30;
  if(action==page_60)    pagetime = TIME_DIMENSION * 60;
  if(action==page_300)   pagetime = TIME_DIMENSION * 300;
  if(action==page_1200)  pagetime = TIME_DIMENSION * 1200;
  if(action==page_3600)  pagetime = TIME_DIMENSION * 3600;

  setup_viewbuf();
}


void UI_Mainwindow::set_user_defined_display_time()
{
  UI_Userdefined_timepage_Dialog set_displaytime_dialog(this);
}


void UI_Mainwindow::set_display_time_whole_rec()
{
  int i;

  if(!files_open)  return;

  if(viewtime_sync==VIEWTIME_SYNCED_OFFSET)
  {
    for(i=0; i<files_open; i++)
    {
      edfheaderlist[i]->viewtime = 0;
    }
  }

  if(viewtime_sync==VIEWTIME_UNSYNCED)
  {
    edfheaderlist[sel_viewtime]->viewtime = 0;
  }

  if(viewtime_sync==VIEWTIME_SYNCED_ABSOLUT)
  {
    edfheaderlist[sel_viewtime]->viewtime = 0;

    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime = ((edfheaderlist[sel_viewtime]->utc_starttime - edfheaderlist[i]->utc_starttime) * TIME_DIMENSION) + edfheaderlist[sel_viewtime]->starttime_offset - - edfheaderlist[i]->starttime_offset;
      }
    }
  }

  if(viewtime_sync==VIEWTIME_USER_DEF_SYNCED)
  {
    for(i=0; i<files_open; i++)
    {
      if(i!=sel_viewtime)
      {
        edfheaderlist[i]->viewtime -= edfheaderlist[sel_viewtime]->viewtime;
      }
    }

    edfheaderlist[sel_viewtime]->viewtime = 0;
  }

  pagetime = edfheaderlist[sel_viewtime]->datarecords * edfheaderlist[sel_viewtime]->long_data_record_duration;

  setup_viewbuf();
}


void UI_Mainwindow::fit_signals_to_pane()
{
  int i, j,
      pane_size;

  if(!signalcomps)  return;

  pane_size = maincurve->height() / (signalcomps + 1);

  for(i=0; i<signalcomps; i++)
  {
    for(j=0; j<signalcomp[i]->num_of_signals; j++)
    {
      if(signalcomp[i]->max_dig_value!=signalcomp[i]->min_dig_value)
      {
        signalcomp[i]->sensitivity[j] = (double)pane_size / (double)(signalcomp[i]->max_dig_value - signalcomp[i]->min_dig_value);
      }
      else
      {
        signalcomp[i]->sensitivity[j] = pane_size;
      }
    }

    signalcomp[i]->voltpercm =
     signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].bitvalue
     / (signalcomp[i]->sensitivity[0] * pixelsizefactor);

    signalcomp[i]->screen_offset = ((signalcomp[i]->max_dig_value + signalcomp[i]->min_dig_value) / 2.0) * signalcomp[i]->sensitivity[0];
  }

  maincurve->drawCurve_stage_1();
}


void UI_Mainwindow::set_amplitude(QAction *action)
{
  int i, j;

  double value=50.0, original_value;


  if(action==amp_00001) value = 0.0001;
  if(action==amp_00002) value = 0.0002;
  if(action==amp_00005) value = 0.0005;
  if(action==amp_0001)  value = 0.001;
  if(action==amp_0002)  value = 0.002;
  if(action==amp_0005)  value = 0.005;
  if(action==amp_001)   value = 0.01;
  if(action==amp_002)   value = 0.02;
  if(action==amp_005)   value = 0.05;
  if(action==amp_01)    value = 0.1;
  if(action==amp_02)    value = 0.2;
  if(action==amp_05)    value = 0.5;
  if(action==amp_1)     value = 1.0;
  if(action==amp_2)     value = 2.0;
  if(action==amp_5)     value = 5.0;
  if(action==amp_10)    value = 10.0;
  if(action==amp_20)    value = 20.0;
  if(action==amp_50)    value = 50.0;
  if(action==amp_100)   value = 100.0;
  if(action==amp_200)   value = 200.0;
  if(action==amp_500)   value = 500.0;
  if(action==amp_1000)  value = 1000.0;
  if(action==amp_2000)  value = 2000.0;
  if(action==amp_5000)  value = 5000.0;
  if(action==amp_10000) value = 10000.0;
  if(action==amp_20000) value = 20000.0;
  if(action==amp_50000) value = 50000.0;

  for(i=0; i<signalcomps; i++)
  {
    for(j=0; j<signalcomp[i]->num_of_signals; j++)
    {
      signalcomp[i]->sensitivity[j] = (signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[j]].bitvalue / value) / pixelsizefactor;
    }

    original_value = signalcomp[i]->voltpercm;

    signalcomp[i]->voltpercm = value;

    signalcomp[i]->screen_offset = signalcomp[i]->screen_offset * (original_value / value);
  }

  maincurve->drawCurve_stage_1();
}


void UI_Mainwindow::load_predefined_mtg(QAction *action)
{
  int i;

  if(!files_open)
  {
    return;
  }

  for(i=0; i < MAXPREDEFINEDMONTAGES; i++)
  {
    if(action==load_predefined_mtg_act[i])
    {
      if(predefined_mtg_path[i][0] != 0)
      {
        strcpy(montagepath, &predefined_mtg_path[i][0]);

        UI_LoadMontagewindow load_montage(this, montagepath);

        return;
      }
    }
  }
}


void UI_Mainwindow::setup_viewbuf()
{
  int i, j, k, s,
      temp=0,
      skip,
      totalsize,
      hasprefilter=0,
      readsize=0,
      dif,
      dig_value;

  double time=0.0,
         d_temp;

  long long l_temp,
            datarecords;

  union {
          unsigned int one;
          signed int one_signed;
          unsigned short two[2];
          signed short two_signed[2];
          unsigned char four[4];
        } var;



  for(i=0; i<files_open; i++) edfheaderlist[i]->prefiltertime = 0;

  for(i=0; i<signalcomps; i++)
  {
    if(signalcomp[i]->filter_cnt)
    {
      hasprefilter = 1;

      for(k=0; k<signalcomp[i]->filter_cnt; k++)
      {
        if(time < (1.0 / signalcomp[i]->filter[k]->cutoff_frequency))
        {
          time = (1.0 / signalcomp[i]->filter[k]->cutoff_frequency);
        }
      }
    }

    if(signalcomp[i]->fidfilter_cnt)
    {
      hasprefilter = 1;

      for(k=0; k<signalcomp[i]->fidfilter_cnt; k++)
      {
        if(time < ((2.0 * signalcomp[i]->fidfilter_order[k]) / signalcomp[i]->fidfilter_freq[k]))
        {
          time = (2.0 * signalcomp[i]->fidfilter_order[k]) / signalcomp[i]->fidfilter_freq[k];
        }
      }
    }
  }

  if(hasprefilter)
  {
    for(i=0; i<signalcomps; i++)
    {
      if((signalcomp[i]->filter_cnt) || (signalcomp[i]->fidfilter_cnt))
      {
        edfheaderlist[signalcomp[i]->filenum]->prefiltertime = (long long)(time * ((double)TIME_DIMENSION));
        if(edfheaderlist[signalcomp[i]->filenum]->prefiltertime>edfheaderlist[signalcomp[i]->filenum]->viewtime)
        {
          edfheaderlist[signalcomp[i]->filenum]->prefiltertime = edfheaderlist[signalcomp[i]->filenum]->viewtime;
          if(edfheaderlist[signalcomp[i]->filenum]->prefiltertime<0) edfheaderlist[signalcomp[i]->filenum]->prefiltertime = 0;
        }
      }
    }

    totalsize = 0;

    for(i=0; i<signalcomps; i++)
    {
      if(edfheaderlist[signalcomp[i]->filenum]->prefiltertime)  signalcomp[i]->records_in_viewbuf = (edfheaderlist[signalcomp[i]->filenum]->viewtime / signalcomp[i]->edfhdr->long_data_record_duration) - ((edfheaderlist[signalcomp[i]->filenum]->viewtime - edfheaderlist[signalcomp[i]->filenum]->prefiltertime) / signalcomp[i]->edfhdr->long_data_record_duration) + 1;
      else signalcomp[i]->records_in_viewbuf = 0;

      signalcomp[i]->viewbufsize = signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize;

      for(j=0; j<signalcomp[i]->num_of_signals; j++)
      {
        if(edfheaderlist[signalcomp[i]->filenum]->prefiltertime)
        {
          signalcomp[i]->samples_in_prefilterbuf = (signalcomp[i]->records_in_viewbuf - 1) * signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[j]].smp_per_record;

          signalcomp[i]->samples_in_prefilterbuf
          += (int)(((double)(edfheaderlist[signalcomp[i]->filenum]->viewtime % signalcomp[i]->edfhdr->long_data_record_duration)
          / (double)signalcomp[i]->edfhdr->long_data_record_duration)
          * (double)signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[j]].smp_per_record);
        }
        else
        {
          signalcomp[i]->samples_in_prefilterbuf = 0;
        }
      }

      if(!i)
      {
        signalcomp[i]->viewbufoffset = 0;
        totalsize = signalcomp[i]->viewbufsize;
      }
      else
      {
        skip = 0;

        for(j=0; j<i; j++)
        {
          if(signalcomp[i]->edfhdr->file_hdl==signalcomp[j]->edfhdr->file_hdl)
          {
            skip = 1;
            signalcomp[i]->viewbufoffset = signalcomp[j]->viewbufoffset;
            signalcomp[i]->records_in_viewbuf = signalcomp[j]->records_in_viewbuf;
            signalcomp[i]->viewbufsize = signalcomp[j]->viewbufsize;
            break;
          }
        }

        if(!skip)
        {
          signalcomp[i]->viewbufoffset = signalcomp[i-1]->viewbufoffset + signalcomp[i-1]->viewbufsize;
          totalsize += signalcomp[i]->viewbufsize;
        }
      }
    }

    if(viewbuf!=NULL)
    {
      free(viewbuf);
      viewbuf = NULL;
    }

    viewbuf = (char *)malloc(totalsize);
    if(viewbuf==NULL)
    {
      UI_Messagewindow popuperror("Error", "Internal error: Memory allocation error:\n\"prefilterbuf\"");
      remove_all_signals();
      return;
    }

    for(i=0; i<signalcomps; i++)
    {
      if(!i)
      {
        datarecords = (edfheaderlist[signalcomp[i]->filenum]->viewtime - edfheaderlist[signalcomp[i]->filenum]->prefiltertime) / signalcomp[i]->edfhdr->long_data_record_duration;

        if((signalcomp[i]->viewbufsize>0)&&(datarecords<signalcomp[i]->edfhdr->datarecords))
        {
          fseeko(signalcomp[i]->edfhdr->file_hdl, (long long)(signalcomp[i]->edfhdr->hdrsize + (datarecords * signalcomp[i]->edfhdr->recordsize)), SEEK_SET);

          if(signalcomp[i]->viewbufsize>((signalcomp[i]->edfhdr->datarecords - datarecords) * signalcomp[i]->edfhdr->recordsize))
          {
            signalcomp[i]->viewbufsize = (signalcomp[i]->edfhdr->datarecords - datarecords) * signalcomp[i]->edfhdr->recordsize;
          }

          if(fread(viewbuf + signalcomp[i]->viewbufoffset, signalcomp[i]->viewbufsize, 1, signalcomp[i]->edfhdr->file_hdl)!=1)
          {
            UI_Messagewindow popuperror("Error", "A read error occurred. 2");
            remove_all_signals();
            return;
          }
        }
      }
      else
      {
        skip = 0;

        for(j=0; j<i; j++)
        {
          if(signalcomp[i]->edfhdr->file_hdl==signalcomp[j]->edfhdr->file_hdl)
          {
            skip = 1;
            break;
          }
        }

        if(!skip)
        {
          datarecords = (edfheaderlist[signalcomp[i]->filenum]->viewtime - edfheaderlist[signalcomp[i]->filenum]->prefiltertime) / signalcomp[i]->edfhdr->long_data_record_duration;

          if((signalcomp[i]->viewbufsize>0)&&(datarecords<signalcomp[i]->edfhdr->datarecords))
          {
            fseeko(signalcomp[i]->edfhdr->file_hdl, (long long)(signalcomp[i]->edfhdr->hdrsize + (datarecords * signalcomp[i]->edfhdr->recordsize)), SEEK_SET);

            if(signalcomp[i]->viewbufsize>((signalcomp[i]->edfhdr->datarecords - datarecords) * signalcomp[i]->edfhdr->recordsize))
            {
              signalcomp[i]->viewbufsize = (signalcomp[i]->edfhdr->datarecords - datarecords) * signalcomp[i]->edfhdr->recordsize;
            }

            if(fread(viewbuf + signalcomp[i]->viewbufoffset, signalcomp[i]->viewbufsize, 1, signalcomp[i]->edfhdr->file_hdl)!=1)
            {
              UI_Messagewindow popuperror("Error", "A read error occurred. 3");
              remove_all_signals();
              return;
            }
          }
        }
      }
    }

    for(i=0; i<signalcomps; i++)
    {
      if((!signalcomp[i]->filter_cnt) && (!signalcomp[i]->fidfilter_cnt)) continue;

      for(s=0; s<signalcomp[i]->samples_in_prefilterbuf; s++)
      {
        dig_value = 0;

        for(k=0; k<signalcomp[i]->num_of_signals; k++)
        {
          if(signalcomp[i]->edfhdr->bdf)
          {
            var.two[0] = *((unsigned short *)(
              viewbuf
              + signalcomp[i]->viewbufoffset
              + (signalcomp[i]->edfhdr->recordsize * (s / signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].smp_per_record))
              + signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].buf_offset
              + ((s % signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].smp_per_record) * 3)));

            var.four[2] = *((unsigned char *)(
              viewbuf
              + signalcomp[i]->viewbufoffset
              + (signalcomp[i]->edfhdr->recordsize * (s / signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].smp_per_record))
              + signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].buf_offset
              + ((s % signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].smp_per_record) * 3)
              + 2));

            if(var.four[2]&0x80)
            {
              var.four[3] = 0xff;
            }
            else
            {
              var.four[3] = 0x00;
            }

            temp = var.one_signed;
          }

          if(signalcomp[i]->edfhdr->edf)
          {
            temp = *(((short *)(
            viewbuf
            + signalcomp[i]->viewbufoffset
            + (signalcomp[i]->edfhdr->recordsize * (s / signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].smp_per_record))
            + signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].buf_offset))
            + (s % signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].smp_per_record));
          }

          temp += signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[k]].offset;
          temp *= signalcomp[i]->factor[k];

          dig_value += temp;
        }

        for(j=0; j<signalcomp[i]->filter_cnt; j++)
        {
          dig_value = first_order_filter(dig_value, signalcomp[i]->filter[j]);
        }

        for(j=0; j<signalcomp[i]->fidfilter_cnt; j++)
        {
          dig_value = signalcomp[i]->fidfuncp[j](signalcomp[i]->fidbuf[j], dig_value);
        }
      }
    }

    for(i=0; i<signalcomps; i++)
    {
      for(k=0; k<signalcomp[i]->num_of_signals; k++)
      {
        for(j=0; j<signalcomp[i]->filter_cnt; j++)
        {
          signalcomp[i]->filterpreset_a[j] = signalcomp[i]->filter[j]->old_input;
          signalcomp[i]->filterpreset_b[j] = signalcomp[i]->filter[j]->old_output;
        }

        for(j=0; j<signalcomp[i]->fidfilter_cnt; j++)
        {
          memcpy(signalcomp[i]->fidbuf2[j], signalcomp[i]->fidbuf[j], fid_run_bufsize(signalcomp[i]->fid_run[j]));
        }
      }
    }
  }

  totalsize = 0;

  for(i=0; i<signalcomps; i++)
  {
    if(edfheaderlist[signalcomp[i]->filenum]->viewtime>=0)  signalcomp[i]->records_in_viewbuf = ((pagetime + (edfheaderlist[signalcomp[i]->filenum]->viewtime % signalcomp[i]->edfhdr->long_data_record_duration)) / signalcomp[i]->edfhdr->long_data_record_duration) + 1;
    else  signalcomp[i]->records_in_viewbuf = ((pagetime + ((-(edfheaderlist[signalcomp[i]->filenum]->viewtime)) % signalcomp[i]->edfhdr->long_data_record_duration)) / signalcomp[i]->edfhdr->long_data_record_duration) + 1;

    signalcomp[i]->viewbufsize = signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize;

    signalcomp[i]->samples_on_screen = (int)(((double)pagetime / (double)signalcomp[i]->edfhdr->long_data_record_duration) * (double)signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].smp_per_record);

    if(edfheaderlist[signalcomp[i]->filenum]->viewtime<0)
    {
      d_temp =
        (((double)(-(edfheaderlist[signalcomp[i]->filenum]->viewtime)))
        / (double)signalcomp[i]->edfhdr->long_data_record_duration)
        * (double)signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].smp_per_record;

      if(d_temp>2147483648.0)
      {
        signalcomp[i]->sample_start = 2147483647LL;
      }
      else
      {
        signalcomp[i]->sample_start = (int)d_temp;
      }
    }
    else
    {
      signalcomp[i]->sample_start = 0;
    }

    if(edfheaderlist[signalcomp[i]->filenum]->viewtime>=0)
    {
      signalcomp[i]->sample_timeoffset_part = ((double)(edfheaderlist[signalcomp[i]->filenum]->viewtime % signalcomp[i]->edfhdr->long_data_record_duration) / (double)signalcomp[i]->edfhdr->long_data_record_duration) * (double)signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].smp_per_record;
      signalcomp[i]->sample_timeoffset = (int)(signalcomp[i]->sample_timeoffset_part);
      signalcomp[i]->sample_timeoffset_part -= signalcomp[i]->sample_timeoffset;
    }
    else
    {
      signalcomp[i]->sample_timeoffset_part = 0.0;
      signalcomp[i]->sample_timeoffset = 0;
    }

    if(!i)
    {
      signalcomp[i]->viewbufoffset = 0;
      totalsize = signalcomp[i]->viewbufsize;
    }
    else
    {
      skip = 0;

      for(j=0; j<i; j++)
      {
        if(signalcomp[i]->edfhdr->file_hdl==signalcomp[j]->edfhdr->file_hdl)
        {
          skip = 1;
          signalcomp[i]->viewbufoffset = signalcomp[j]->viewbufoffset;
          signalcomp[i]->records_in_viewbuf = signalcomp[j]->records_in_viewbuf;
          signalcomp[i]->viewbufsize = signalcomp[j]->viewbufsize;
          break;
        }
      }

      if(!skip)
      {
        signalcomp[i]->viewbufoffset = signalcomp[i-1]->viewbufoffset + signalcomp[i-1]->viewbufsize;
        totalsize += signalcomp[i]->viewbufsize;
      }
    }
  }

  if(viewbuf!=NULL)
  {
    free(viewbuf);
    viewbuf = NULL;
  }

  if(totalsize)
  {
    viewbuf = (char *)malloc(totalsize);
    if(viewbuf==NULL)
    {
      UI_Messagewindow popuperror("Error", "The system was not able to provide enough resources (memory) to perform the requested action.\n"
                                  "Decrease the displaytime and/or the amount of signals on the screen and retry.");

      remove_all_signals();
      return;
    }
  }

  for(i=0; i<signalcomps; i++)
  {
    if(!i)
    {
      if(edfheaderlist[signalcomp[i]->filenum]->viewtime>=0)
      {
        datarecords = edfheaderlist[signalcomp[i]->filenum]->viewtime / signalcomp[i]->edfhdr->long_data_record_duration;
      }
      else
      {
        datarecords = 0;
      }

      dif = signalcomp[i]->edfhdr->datarecords - datarecords;

      if(dif<=0)
      {
        memset(viewbuf + signalcomp[i]->viewbufoffset, 0, signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize);

        signalcomp[i]->sample_stop = 0;
      }
      else
      {
        if(dif<signalcomp[i]->records_in_viewbuf)
        {
          readsize = dif * signalcomp[i]->edfhdr->recordsize;

          memset(viewbuf + signalcomp[i]->viewbufoffset + readsize, 0, (signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize) - readsize);

          signalcomp[i]->sample_stop = (dif * signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].smp_per_record) - signalcomp[i]->sample_timeoffset;
        }
        else
        {
          readsize = signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize;

          signalcomp[i]->sample_stop = signalcomp[i]->samples_on_screen;
        }

        l_temp = signalcomp[i]->edfhdr->hdrsize;
        l_temp += (datarecords * signalcomp[i]->edfhdr->recordsize);

        fseeko(signalcomp[i]->edfhdr->file_hdl, l_temp, SEEK_SET);

        if(fread(viewbuf + signalcomp[i]->viewbufoffset, readsize, 1, signalcomp[i]->edfhdr->file_hdl)!=1)
        {
          UI_Messagewindow popuperror("Error", "A read error occurred. 5");
          remove_all_signals();
          return;
        }
      }
    }
    else
    {
      skip = 0;

      for(j=0; j<i; j++)
      {
        if(signalcomp[i]->edfhdr->file_hdl==signalcomp[j]->edfhdr->file_hdl)
        {
          skip = 1;
          break;
        }
      }

      if(edfheaderlist[signalcomp[i]->filenum]->viewtime>=0)
      {
        datarecords = edfheaderlist[signalcomp[i]->filenum]->viewtime / signalcomp[i]->edfhdr->long_data_record_duration;
      }
      else
      {
        datarecords = 0;
      }

      dif = signalcomp[i]->edfhdr->datarecords - datarecords;

      if(dif<=0)
      {
        if(!skip)
        {
          memset(viewbuf + signalcomp[i]->viewbufoffset, 0, signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize);
        }

        signalcomp[i]->sample_stop = 0;
      }
      else
      {
        if(dif<signalcomp[i]->records_in_viewbuf)
        {
          if(!skip)
          {
            readsize = dif * signalcomp[i]->edfhdr->recordsize;

            memset(viewbuf + signalcomp[i]->viewbufoffset + readsize, 0, (signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize) - readsize);
          }

          signalcomp[i]->sample_stop = (dif * signalcomp[i]->edfhdr->edfparam[signalcomp[i]->edfsignal[0]].smp_per_record) - signalcomp[i]->sample_timeoffset;
        }
        else
        {
          if(!skip)
          {
            readsize = signalcomp[i]->records_in_viewbuf * signalcomp[i]->edfhdr->recordsize;
          }

          signalcomp[i]->sample_stop = signalcomp[i]->samples_on_screen;
        }

        if(!skip)
        {
          l_temp = signalcomp[i]->edfhdr->hdrsize;
          l_temp += (datarecords * signalcomp[i]->edfhdr->recordsize);

          fseeko(signalcomp[i]->edfhdr->file_hdl, l_temp, SEEK_SET);

          if(fread(viewbuf + signalcomp[i]->viewbufoffset, readsize, 1, signalcomp[i]->edfhdr->file_hdl)!=1)
          {
            UI_Messagewindow popuperror("Error", "A read error occurred. 6");
            remove_all_signals();
            return;
          }
        }
      }
    }

    signalcomp[i]->sample_stop += signalcomp[i]->sample_start;
  }

  if(signalcomps)
  {
    if((edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset)>=0LL)
    {
      snprintf(viewtime_string, 32, "%2i:%02i:%02i.%04i",
              (int)((((edfheaderlist[sel_viewtime]->l_starttime + edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset) / TIME_DIMENSION)/ 3600LL) % 24LL),
              (int)((((edfheaderlist[sel_viewtime]->l_starttime + edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset) / TIME_DIMENSION) % 3600LL) / 60LL),
              (int)(((edfheaderlist[sel_viewtime]->l_starttime + edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset) / TIME_DIMENSION) % 60LL),
              (int)(((edfheaderlist[sel_viewtime]->l_starttime + edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset) % TIME_DIMENSION) / 1000LL));

      snprintf(viewtime_string + strlen(viewtime_string), 32, " (%i:%02i:%02i.%04i)",
              (int)((edfheaderlist[sel_viewtime]->viewtime / TIME_DIMENSION)/ 3600LL),
              (int)(((edfheaderlist[sel_viewtime]->viewtime / TIME_DIMENSION) % 3600LL) / 60LL),
              (int)((edfheaderlist[sel_viewtime]->viewtime / TIME_DIMENSION) % 60LL),
              (int)((edfheaderlist[sel_viewtime]->viewtime % TIME_DIMENSION) / 1000LL));
    }
    else
    {
      l_temp = edfheaderlist[sel_viewtime]->l_starttime + ((edfheaderlist[sel_viewtime]->viewtime + edfheaderlist[sel_viewtime]->starttime_offset) % (86400LL * TIME_DIMENSION));
      if(l_temp<=0)
      {
        l_temp = (86400LL * TIME_DIMENSION) + l_temp;
      }
      snprintf(viewtime_string, 32, "%2i:%02i:%02i.%04i",
              (int)((((l_temp) / TIME_DIMENSION)/ 3600LL) % 24LL),
              (int)((((l_temp) / TIME_DIMENSION) % 3600LL) / 60LL),
              (int)(((l_temp) / TIME_DIMENSION) % 60LL),
              (int)(((l_temp) % TIME_DIMENSION) / 1000LL));

      l_temp = -edfheaderlist[sel_viewtime]->viewtime;

      snprintf(viewtime_string + strlen(viewtime_string), 32, " (-%i:%02i:%02i.%04i)",
              (int)((l_temp / TIME_DIMENSION)/ 3600LL),
              (int)(((l_temp / TIME_DIMENSION) % 3600LL) / 60LL),
              (int)((l_temp / TIME_DIMENSION) % 60LL),
              (int)((l_temp % TIME_DIMENSION) / 1000LL));
    }

    snprintf(pagetime_string, 32, "%i.%04i",
            (int)(pagetime / TIME_DIMENSION),
            (int)((pagetime % TIME_DIMENSION) / 1000LL));

    remove_trailing_zeros(viewtime_string);
    remove_trailing_zeros(pagetime_string);
  }

  if(print_to_edf_active)
  {
    print_to_edf_active = 0;
  }
  else
  {
    if(signalcomps && slidertoolbar->isHidden())
    {
      slidertoolbar->show();
    }

    maincurve->drawCurve_stage_1();

    if(signalcomps)
    {
      QObject::disconnect(positionslider, SIGNAL(valueChanged(int)), this, SLOT(slider_moved(int)));

      long long record_duration = edfheaderlist[sel_viewtime]->long_data_record_duration * edfheaderlist[sel_viewtime]->datarecords;

      record_duration -= pagetime;

      if(edfheaderlist[sel_viewtime]->viewtime<=0)
      {
        positionslider->setValue(0);
      }
      else
      {
        if(edfheaderlist[sel_viewtime]->viewtime>=record_duration)
        {
          positionslider->setValue(1000000);
        }
        else
        {
          if(record_duration<pagetime)
          {
            positionslider->setValue(1000000);
          }
          else
          {
            positionslider->setValue(edfheaderlist[sel_viewtime]->viewtime * 1000000LL / record_duration);
          }
        }
      }

      QObject::connect(positionslider, SIGNAL(valueChanged(int)), this, SLOT(slider_moved(int)));
    }
    else
    {
      slidertoolbar->hide();
    }
  }

  if(spectrumdock->dock->isVisible())
  {
    spectrumdock->rescan();
  }
}


void UI_Mainwindow::export_to_ascii()
{
  UI_AsciiExportwindow exportdialog(this);
}


void UI_Mainwindow::export_annotations()
{
  UI_ExportAnnotationswindow exportAnnotsDialog(this);
}


void UI_Mainwindow::import_annotations()
{
  UI_ImportAnnotationswindow importAnnotsDialog(this);
}


void UI_Mainwindow::check_edf_compatibility()
{
  UI_EDFCompatwindow edfcompatdialog(this);
}


void UI_Mainwindow::print_to_img_640x480()
{
  maincurve->print_to_image(640, 480);
}


void UI_Mainwindow::print_to_img_800x600()
{
  maincurve->print_to_image(800, 600);
}


void UI_Mainwindow::print_to_img_1024x768()
{
  maincurve->print_to_image(1024, 768);
}


void UI_Mainwindow::print_to_img_1280x1024()
{
  maincurve->print_to_image(1280, 1024);
}


void UI_Mainwindow::print_to_img_1600x1200()
{
  maincurve->print_to_image(1600, 1200);
}


long long UI_Mainwindow::get_long_time(char *str)
{
  int i, len, hasdot=0, dotposition=0;

  long long value=0, radix;

  str = str + 1;

  len = strlen(str);

  for(i=0; i<len; i++)
  {
    if(str[i]=='.')
    {
      hasdot = 1;
      dotposition = i;
      break;
    }
  }

  if(hasdot)
  {
    radix = TIME_DIMENSION;

    for(i=dotposition-1; i>=0; i--)
    {
        value += ((long long)(str[i] - 48)) * radix;
        radix *= 10;
    }

    radix = TIME_DIMENSION / 10;

    for(i=dotposition+1; i<len; i++)
    {
        value += ((long long)(str[i] - 48)) * radix;
        radix /= 10;
    }
  }
  else
  {
    radix = TIME_DIMENSION;

    for(i=len-1; i>=0; i--)
    {
        value += ((long long)(str[i] - 48)) * radix;
        radix *= 10;
    }
  }

  if(str[-1]=='-')  value = -value;

  return(value);
}


void UI_Mainwindow::get_rgbcolor_settings(struct xml_handle *xml_hdl, const char *id, int cnt, QColor *rgb_color)
{
  char *result;

  QColor tmp_color;


  if(xml_goto_nth_element_inside(xml_hdl, id, cnt))
  {
    return;
  }

  if(xml_goto_nth_element_inside(xml_hdl, "red", 0))
  {
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    return;
  }
  tmp_color.setRed(atoi(result));
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "green", 0))
  {
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    return;
  }
  tmp_color.setGreen(atoi(result));
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "blue", 0))
  {
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    return;
  }
  tmp_color.setBlue(atoi(result));
  free(result);

  *rgb_color = tmp_color;

  xml_go_up(xml_hdl);
  xml_go_up(xml_hdl);
}


void UI_Mainwindow::read_color_settings()
{
  char cfg_path[1024],
       *result;

  struct xml_handle *xml_hdl;

  cfg_path[0] = 0;

#ifdef Q_WS_X11
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_MAC
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_WIN
  strcpy(cfg_path, specialFolder(CSIDL_APPDATA).toLatin1().data());
  strcat(cfg_path, "\\");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "\\settings.xml");
#endif

  xml_hdl = xml_get_handle(cfg_path);
  if(!xml_hdl)
  {
    return;
  }

  if(strcmp(xml_hdl->elementname, "config"))
  {
    xml_close(xml_hdl);
    return;
  }

  if(xml_goto_nth_element_inside(xml_hdl, "UI", 0))
  {
    xml_close(xml_hdl);
    return;
  }

  if(xml_goto_nth_element_inside(xml_hdl, "colors", 0))
  {
    xml_close(xml_hdl);
    return;
  }

  get_rgbcolor_settings(xml_hdl, "backgroundcolor", 0, &maincurve->backgroundcolor);

  get_rgbcolor_settings(xml_hdl, "small_ruler_color", 0, &maincurve->small_ruler_color);

  get_rgbcolor_settings(xml_hdl, "big_ruler_color", 0, &maincurve->big_ruler_color);

  get_rgbcolor_settings(xml_hdl, "mouse_rect_color", 0, &maincurve->mouse_rect_color);

  get_rgbcolor_settings(xml_hdl, "text_color", 0, &maincurve->text_color);

  get_rgbcolor_settings(xml_hdl, "baseline_color", 0, &maincurve->baseline_color);

  get_rgbcolor_settings(xml_hdl, "annot_marker_color", 0, &maincurve->annot_marker_color);

  if(xml_goto_nth_element_inside(xml_hdl, "signal_color", 0))
  {
    xml_close(xml_hdl);
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }
  maincurve->signal_color = atoi(result);
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "floating_ruler_color", 0))
  {
    xml_close(xml_hdl);
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }
  maincurve->floating_ruler_color = atoi(result);
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "blackwhite_printing", 0))
  {
    xml_close(xml_hdl);
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }
  maincurve->blackwhite_printing = atoi(result);
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "show_annot_markers", 0))
  {
    xml_close(xml_hdl);
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }
  show_annot_markers = atoi(result);
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "show_baselines", 0))
  {
    xml_close(xml_hdl);
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }
  show_baselines = atoi(result);
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "crosshair_1_color", 0))
  {
    xml_close(xml_hdl);
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }
  maincurve->crosshair_1_color = atoi(result);
  free(result);

  xml_go_up(xml_hdl);

  if(xml_goto_nth_element_inside(xml_hdl, "crosshair_2_color", 0))
  {
    xml_close(xml_hdl);
    return;
  }
  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }
  maincurve->crosshair_2_color = atoi(result);
  free(result);

  xml_close(xml_hdl);
}



void UI_Mainwindow::read_recent_file_settings()
{
  int i;

  char cfg_path[1024],
       *result;

  struct xml_handle *xml_hdl;

  cfg_path[0] = 0;

#ifdef Q_WS_X11
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_MAC
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_WIN
  strcpy(cfg_path, specialFolder(CSIDL_APPDATA).toLatin1().data());
  strcat(cfg_path, "\\");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "\\settings.xml");
#endif

  xml_hdl = xml_get_handle(cfg_path);
  if(xml_hdl==NULL)
  {
    return;
  }

  if(strcmp(xml_hdl->elementname, "config"))
  {
    xml_close(xml_hdl);
    return;
  }

  if(xml_goto_nth_element_inside(xml_hdl, "UI", 0))
  {
    xml_close(xml_hdl);
    return;
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "recent_montagedir", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    strcpy(recent_montagedir, result);
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "recent_savedir", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    strcpy(recent_savedir, result);
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "recent_opendir", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    strcpy(recent_opendir, result);
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "recent_file", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }
    if(result[0] != 0)
    {
      strcpy(&recent_file_path[0][0], result);
      recent_filesmenu->addAction(&recent_file_path[0][0]);
      free(result);

      for(i=1; i<MAX_RECENTFILES; i++)
      {
        if(xml_goto_next_element_with_same_name(xml_hdl))
        {
          break;
        }
        result = xml_get_content_of_element(xml_hdl);
        if(result==NULL)
        {
          xml_close(xml_hdl);
          return;
        }
        if(result[0] == 0)
        {
          free(result);
          break;
        }
        strcpy(&recent_file_path[i][0], result);
        recent_filesmenu->addAction(&recent_file_path[i][0]);
        free(result);
      }
    }
    else
    {
      free(result);
    }

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "recent_file_mtg", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }
    strcpy(&recent_file_mtg_path[0][0], result);
    free(result);
    for(i=1; i<MAX_RECENTFILES; i++)
    {
      if(xml_goto_next_element_with_same_name(xml_hdl))
      {
        break;
      }
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }
      strcpy(&recent_file_mtg_path[i][0], result);
      free(result);
    }

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "predefined_mtg_path", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }
    strncpy(&predefined_mtg_path[0][0], result, 1024);
    predefined_mtg_path[0][1023] = 0;
    free(result);

    for(i=1; i < MAXPREDEFINEDMONTAGES; i++)
    {
      if(xml_goto_next_element_with_same_name(xml_hdl))
      {
        break;
      }
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }
      strncpy(&predefined_mtg_path[i][0], result, 1024);
      predefined_mtg_path[i][1023] = 0;
      free(result);
    }

    xml_go_up(xml_hdl);
  }

  xml_close(xml_hdl);
}



void UI_Mainwindow::read_general_settings()
{
  int i;

  char cfg_path[1024],
       *result;

  struct xml_handle *xml_hdl;

  cfg_path[0] = 0;

#ifdef Q_WS_X11
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_MAC
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_WIN
  strcpy(cfg_path, specialFolder(CSIDL_APPDATA).toLatin1().data());
  strcat(cfg_path, "\\");
  strcat(cfg_path, PROGRAM_NAME);
  strcat(cfg_path, "\\settings.xml");
#endif

  xml_hdl = xml_get_handle(cfg_path);
  if(xml_hdl==NULL)
  {
    return;
  }

  if(strcmp(xml_hdl->elementname, "config"))
  {
    xml_close(xml_hdl);
    return;
  }

  if(xml_goto_nth_element_inside(xml_hdl, "UI", 0))
  {
    xml_close(xml_hdl);
    return;
  }

  if(xml_goto_nth_element_inside(xml_hdl, "pixelsizefactor", 0))
  {
    xml_close(xml_hdl);
    return;
  }

  result = xml_get_content_of_element(xml_hdl);
  if(result==NULL)
  {
    xml_close(xml_hdl);
    return;
  }

  pixelsizefactor = atof(result);
  if(pixelsizefactor < 0.00001)
  {
    pixelsizefactor = 0.0294382;
  }
  free(result);

  xml_go_up(xml_hdl);

  if(!(xml_goto_nth_element_inside(xml_hdl, "auto_dpi", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    auto_dpi = atoi(result);
    if((auto_dpi < 0) || (auto_dpi > 1))
    {
      auto_dpi = 1;
    }
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "x_pixelsizefactor", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    x_pixelsizefactor = atof(result);
    if(x_pixelsizefactor < 0.00001)
    {
      x_pixelsizefactor = 0.0294382;
    }
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "clip_to_pane", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    clip_to_pane = atoi(result);
    if((clip_to_pane < 0) || (clip_to_pane > 1))
    {
      clip_to_pane = 0;
    }
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "auto_reload_mtg", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    auto_reload_mtg = atoi(result);
    if((auto_reload_mtg < 0) || (auto_reload_mtg > 1))
    {
      auto_reload_mtg = 1;
    }
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "read_biosemi_status_signal", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    read_biosemi_status_signal = atoi(result);
    if((read_biosemi_status_signal < 0) || (read_biosemi_status_signal > 1))
    {
      read_biosemi_status_signal = 1;
    }
    free(result);

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "spectrummarkerblock", 0)))
  {
    if(xml_goto_nth_element_inside(xml_hdl, "items", 0))
    {
      xml_close(xml_hdl);
      return;
    }

    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    if((atoi(result) > MAXSPECTRUMMARKERS) || (atoi(result) < 0))
    {
      xml_close(xml_hdl);
      free(result);
      return;
    }

    spectrum_colorbar->items = atoi(result);
    free(result);

    xml_go_up(xml_hdl);

    if(xml_goto_nth_element_inside(xml_hdl, "method", 0))
    {
      xml_close(xml_hdl);
      return;
    }

    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    if((atoi(result) > 2) || (atoi(result) < 0))
    {
      xml_close(xml_hdl);
      free(result);
      return;
    }

    spectrum_colorbar->method = atoi(result);
    free(result);

    xml_go_up(xml_hdl);
    if(xml_goto_nth_element_inside(xml_hdl, "frequency", 0))
    {
      xml_close(xml_hdl);
      return;
    }

    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    if((atof(result) > 1000000.0) || (atof(result) < 0.00001))
    {
      xml_close(xml_hdl);
      free(result);
      return;
    }

    spectrum_colorbar->freq[0] = atof(result);
    free(result);

    for(i=1; i < spectrum_colorbar->items; i++)
    {
      if(xml_goto_next_element_with_same_name(xml_hdl))
      {
        xml_close(xml_hdl);
        return;
      }

      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      if(i < spectrum_colorbar->items)
      {
        if((atof(result) > 1000000.0) || (atof(result) <= spectrum_colorbar->freq[i-1]))
        {
          xml_close(xml_hdl);
          free(result);
          return;
        }
      }

      spectrum_colorbar->freq[i] = atof(result);
      free(result);
    }

    xml_go_up(xml_hdl);

    if(xml_goto_nth_element_inside(xml_hdl, "color", 0))
    {
      xml_close(xml_hdl);
      return;
    }

    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    if((atoi(result) > 18) || (atoi(result) < 2))
    {
      xml_close(xml_hdl);
      free(result);
      return;
    }

    spectrum_colorbar->color[0] = atoi(result);
    free(result);

    for(i=1; i < spectrum_colorbar->items; i++)
    {
      if(xml_goto_next_element_with_same_name(xml_hdl))
      {
        xml_close(xml_hdl);
        return;
      }

      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      if((atoi(result) > 18) || (atoi(result) < 2))
      {
        xml_close(xml_hdl);
        free(result);
        return;
      }

      spectrum_colorbar->color[i] = atoi(result);
      free(result);
    }

    xml_go_up(xml_hdl);

    if(!(xml_goto_nth_element_inside(xml_hdl, "label", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      strncpy(spectrum_colorbar->label[0], result, 16);
      spectrum_colorbar->label[0][16] = 0;
      free(result);

      for(i=1; i < spectrum_colorbar->items; i++)
      {
        if(xml_goto_next_element_with_same_name(xml_hdl))
        {
          xml_close(xml_hdl);
          return;
        }

        result = xml_get_content_of_element(xml_hdl);
        if(result==NULL)
        {
          xml_close(xml_hdl);
          return;
        }

        strncpy(spectrum_colorbar->label[i], result, 16);
        spectrum_colorbar->label[i][16] = 0;
        free(result);
      }

      xml_go_up(xml_hdl);
    }
  }

  xml_go_up(xml_hdl);

  if(!(xml_goto_nth_element_inside(xml_hdl, "maxdftblocksize", 0)))
  {
    result = xml_get_content_of_element(xml_hdl);
    if(result==NULL)
    {
      xml_close(xml_hdl);
      return;
    }

    if(atoi(result) < 10)
    {
      xml_close(xml_hdl);
      free(result);
      return;
    }

    maxdftblocksize = atoi(result);

    free(result);
  }

  xml_go_up(xml_hdl);

  if(!(xml_goto_nth_element_inside(xml_hdl, "annotations_import_var", 0)))
  {
    if(!(xml_goto_nth_element_inside(xml_hdl, "format", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      import_annotations_var->format = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    if(!(xml_goto_nth_element_inside(xml_hdl, "onsettimeformat", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      import_annotations_var->onsettimeformat = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    if(!(xml_goto_nth_element_inside(xml_hdl, "onsetcolumn", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      import_annotations_var->onsetcolumn = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    if(!(xml_goto_nth_element_inside(xml_hdl, "descriptioncolumn", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      import_annotations_var->descriptioncolumn = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    if(!(xml_goto_nth_element_inside(xml_hdl, "datastartline", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      import_annotations_var->datastartline = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    if(!(xml_goto_nth_element_inside(xml_hdl, "separator", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      strncpy(import_annotations_var->separator, result, 3);

      import_annotations_var->separator[3] = 0;

      free(result);

      xml_go_up(xml_hdl);
    }

    xml_go_up(xml_hdl);
  }

  if(!(xml_goto_nth_element_inside(xml_hdl, "annotations_export_var", 0)))
  {
    if(!(xml_goto_nth_element_inside(xml_hdl, "separator", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      export_annotations_var->separator = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    if(!(xml_goto_nth_element_inside(xml_hdl, "format", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      export_annotations_var->format = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    if(!(xml_goto_nth_element_inside(xml_hdl, "duration", 0)))
    {
      result = xml_get_content_of_element(xml_hdl);
      if(result==NULL)
      {
        xml_close(xml_hdl);
        return;
      }

      export_annotations_var->duration = atoi(result);

      free(result);

      xml_go_up(xml_hdl);
    }

    xml_go_up(xml_hdl);
  }

  xml_close(xml_hdl);
}



void UI_Mainwindow::recent_file_action_func(QAction *action)
{
  strcpy(path, action->text().toLatin1().data());

  cmdlineargument = 1;

  open_new_file();
}



void UI_Mainwindow::write_settings()
{
  int i, len;

  char cfg_path[1024];

  FILE *cfgfile;

  cfg_path[0] = 0;

#ifdef Q_WS_X11
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  mkdir(cfg_path,  S_IRWXU);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_MAC
  strcpy(cfg_path, getenv("HOME"));
  strcat(cfg_path, "/.");
  strcat(cfg_path, PROGRAM_NAME);
  mkdir(cfg_path,  S_IRWXU);
  strcat(cfg_path, "/settings.xml");
#endif
#ifdef Q_WS_WIN
  strcpy(cfg_path, specialFolder(CSIDL_APPDATA).toLatin1().data());
  strcat(cfg_path, "\\");
  strcat(cfg_path, PROGRAM_NAME);
  _mkdir(cfg_path);
  strcat(cfg_path, "\\settings.xml");
#endif

  cfgfile = fopeno(cfg_path, "wb");
  if(cfgfile)
  {
    fprintf(cfgfile, "<?xml version=\"1.0\"?>\n<config>\n  <UI>\n    <colors>\n");

    fprintf(cfgfile, "      <backgroundcolor>\n"
                    "        <red>%i</red>\n"
                    "        <green>%i</green>\n"
                    "        <blue>%i</blue>\n"
                    "      </backgroundcolor>\n",
                    maincurve->backgroundcolor.red(),
                    maincurve->backgroundcolor.green(),
                    maincurve->backgroundcolor.blue());

    fprintf(cfgfile, "      <small_ruler_color>\n"
                    "        <red>%i</red>\n"
                    "        <green>%i</green>\n"
                    "        <blue>%i</blue>\n"
                    "      </small_ruler_color>\n",
                    maincurve->small_ruler_color.red(),
                    maincurve->small_ruler_color.green(),
                    maincurve->small_ruler_color.blue());

    fprintf(cfgfile, "      <big_ruler_color>\n"
                    "        <red>%i</red>\n"
                    "        <green>%i</green>\n"
                    "        <blue>%i</blue>\n"
                    "      </big_ruler_color>\n",
                    maincurve->big_ruler_color.red(),
                    maincurve->big_ruler_color.green(),
                    maincurve->big_ruler_color.blue());

    fprintf(cfgfile, "      <mouse_rect_color>\n"
                    "        <red>%i</red>\n"
                    "        <green>%i</green>\n"
                    "        <blue>%i</blue>\n"
                    "      </mouse_rect_color>\n",
                    maincurve->mouse_rect_color.red(),
                    maincurve->mouse_rect_color.green(),
                    maincurve->mouse_rect_color.blue());

    fprintf(cfgfile, "      <text_color>\n"
                    "        <red>%i</red>\n"
                    "        <green>%i</green>\n"
                    "        <blue>%i</blue>\n"
                    "      </text_color>\n",
                    maincurve->text_color.red(),
                    maincurve->text_color.green(),
                    maincurve->text_color.blue());

    fprintf(cfgfile, "      <baseline_color>\n"
                    "        <red>%i</red>\n"
                    "        <green>%i</green>\n"
                    "        <blue>%i</blue>\n"
                    "      </baseline_color>\n",
                    maincurve->baseline_color.red(),
                    maincurve->baseline_color.green(),
                    maincurve->baseline_color.blue());

    fprintf(cfgfile, "      <annot_marker_color>\n"
                    "        <red>%i</red>\n"
                    "        <green>%i</green>\n"
                    "        <blue>%i</blue>\n"
                    "      </annot_marker_color>\n",
                    maincurve->annot_marker_color.red(),
                    maincurve->annot_marker_color.green(),
                    maincurve->annot_marker_color.blue());

    fprintf(cfgfile, "      <signal_color>%i</signal_color>\n",
                    maincurve->signal_color);

    fprintf(cfgfile, "      <crosshair_1_color>%i</crosshair_1_color>\n",
                    maincurve->crosshair_1_color);

    fprintf(cfgfile, "      <crosshair_2_color>%i</crosshair_2_color>\n",
                    maincurve->crosshair_2_color);

    fprintf(cfgfile, "      <floating_ruler_color>%i</floating_ruler_color>\n",
                    maincurve->floating_ruler_color);

    fprintf(cfgfile, "      <blackwhite_printing>%i</blackwhite_printing>\n",
                    maincurve->blackwhite_printing);

    fprintf(cfgfile, "      <show_annot_markers>%i</show_annot_markers>\n",
                    show_annot_markers);

    fprintf(cfgfile, "      <show_baselines>%i</show_baselines>\n",
                    show_baselines);

    fprintf(cfgfile, "      <clip_to_pane>%i</clip_to_pane>\n",
                    clip_to_pane);

    fprintf(cfgfile, "      <auto_reload_mtg>%i</auto_reload_mtg>\n",
                    auto_reload_mtg);

    fprintf(cfgfile, "      <read_biosemi_status_signal>%i</read_biosemi_status_signal>\n",
                    read_biosemi_status_signal);

    len = strlen(path);
    for(i=len-1; i>=0; i--)
    {
      if((path[i] == '/')||(path[i] == '\\'))
      {
        break;
      }
    }
    path[i+1] = 0;

    fprintf(cfgfile, "    </colors>\n    <pixelsizefactor>%.10f</pixelsizefactor>\n    <auto_dpi>%i</auto_dpi>\n    <x_pixelsizefactor>%.10f</x_pixelsizefactor>\n    <recent_dir>",
                     pixelsizefactor, auto_dpi, x_pixelsizefactor);

    xml_fwrite_encode_entity(cfgfile, path);

    fprintf(cfgfile, "</recent_dir>\n");

    for(i=0; i<MAX_RECENTFILES; i++)
    {
      fprintf(cfgfile, "    <recent_file>");

      xml_fwrite_encode_entity(cfgfile, &recent_file_path[i][0]);

      fprintf(cfgfile, "</recent_file>\n");
    }

    for(i=0; i<MAX_RECENTFILES; i++)
    {
      fprintf(cfgfile, "    <recent_file_mtg>");

      xml_fwrite_encode_entity(cfgfile, &recent_file_mtg_path[i][0]);

      fprintf(cfgfile, "</recent_file_mtg>\n");
    }

    fprintf(cfgfile, "    <recent_montagedir>");

    xml_fwrite_encode_entity(cfgfile, recent_montagedir);

    fprintf(cfgfile, "</recent_montagedir>\n");

    fprintf(cfgfile, "    <recent_savedir>");

    xml_fwrite_encode_entity(cfgfile, recent_savedir);

    fprintf(cfgfile, "</recent_savedir>\n");

    fprintf(cfgfile, "    <recent_opendir>");

    xml_fwrite_encode_entity(cfgfile, recent_opendir);

    fprintf(cfgfile, "</recent_opendir>\n");

    for(i=0; i < MAXPREDEFINEDMONTAGES; i++)
    {
      fprintf(cfgfile, "    <predefined_mtg_path>");

      xml_fwrite_encode_entity(cfgfile, &predefined_mtg_path[i][0]);

      fprintf(cfgfile, "</predefined_mtg_path>\n");
    }

    fprintf(cfgfile, "    <spectrummarkerblock>\n");

    fprintf(cfgfile, "      <items>%i</items>\n", spectrum_colorbar->items);

    for(i=0; i < MAXSPECTRUMMARKERS; i++)
    {
      fprintf(cfgfile, "      <frequency>%f</frequency>\n", spectrum_colorbar->freq[i]);
    }

    for(i=0; i < MAXSPECTRUMMARKERS; i++)
    {
      fprintf(cfgfile, "      <color>%i</color>\n", spectrum_colorbar->color[i]);
    }

    for(i=0; i < MAXSPECTRUMMARKERS; i++)
    {
      fprintf(cfgfile, "      <label>");

      xml_fwrite_encode_entity(cfgfile, spectrum_colorbar->label[i]);

      fprintf(cfgfile, "</label>\n");
    }

    fprintf(cfgfile, "      <method>%i</method>\n", spectrum_colorbar->method);

    fprintf(cfgfile, "    </spectrummarkerblock>\n");

    fprintf(cfgfile, "    <maxdftblocksize>%i</maxdftblocksize>\n", maxdftblocksize);

    fprintf(cfgfile, "    <annotations_import_var>\n");

    fprintf(cfgfile, "      <format>%i</format>\n", import_annotations_var->format);

    fprintf(cfgfile, "      <onsettimeformat>%i</onsettimeformat>\n", import_annotations_var->onsettimeformat);

    fprintf(cfgfile, "      <onsetcolumn>%i</onsetcolumn>\n", import_annotations_var->onsetcolumn);

    fprintf(cfgfile, "      <descriptioncolumn>%i</descriptioncolumn>\n", import_annotations_var->descriptioncolumn);

    fprintf(cfgfile, "      <datastartline>%i</datastartline>\n", import_annotations_var->datastartline);

    fprintf(cfgfile, "      <separator>%s</separator>\n", import_annotations_var->separator);

    fprintf(cfgfile, "    </annotations_import_var>\n");

    fprintf(cfgfile, "    <annotations_export_var>\n");

    fprintf(cfgfile, "      <separator>%i</separator>\n", export_annotations_var->separator);

    fprintf(cfgfile, "      <format>%i</format>\n", export_annotations_var->format);

    fprintf(cfgfile, "      <duration>%i</duration>\n", export_annotations_var->duration);

    fprintf(cfgfile, "    </annotations_export_var>\n");

    fprintf(cfgfile, "  </UI>\n</config>\n");

    fclose(cfgfile);
  }
}




void UI_Mainwindow::save_montage()
{
  if(!files_open)  return;

  if(!signalcomps)  return;

  UI_SaveMontagewindow save_montage(this);
}




void UI_Mainwindow::load_montage()
{
  if(!files_open)  return;

  UI_LoadMontagewindow load_montage(this);
}



void UI_Mainwindow::view_montage()
{
  UI_ViewMontagewindow view_montage(this);
}



void UI_Mainwindow::show_this_montage()
{
  UI_ShowActualMontagewindow show_montage(this);
}


void UI_Mainwindow::print_to_edf()
{
  print_screen_to_edf(this);
}


void UI_Mainwindow::print_to_bdf()
{
  print_screen_to_bdf(this);
}


void UI_Mainwindow::set_dc_offset_to_zero()
{
  int i;

  if(!signalcomps)
  {
    return;
  }

  for(i=0; i<signalcomps; i++)
  {
    signalcomp[i]->screen_offset = 0.0;
  }

  maincurve->drawCurve_stage_1();
}


void UI_Mainwindow::show_help()
{
#ifdef Q_WS_WIN
  char path[1024];

  strcpy(path, "file:///");
  strcat(path, specialFolder(CSIDL_PROGRAM_FILES).toLatin1().data());
  strcat(path, "\\");
  strcat(path, PROGRAM_NAME);
  strcat(path, "\\");
  strcat(path, PROGRAM_NAME);
  strcat(path, "\%20manual.html");
  QDesktopServices::openUrl(QUrl(path));
#endif
}



void UI_Mainwindow::show_kb_shortcuts()
{
  UI_Messagewindow popuperror("Keyboard shortcuts",
   "PgDn\tnext page\n"
   "PgUp\tformer page\n"
   "Right Arrow\tshift right one tenth of pagetime\n"
   "Left Arrow\tshift left one tenth of pagetime\n"
   "Up Arrow\tshift up\n"
   "Down Arrow\tshift down\n"
   "Ctrl-Home\tgo to start of file\n"
   "Ctrl-End\tgo to end of file\n"
   "Ctrl++\tzoom in\n"
   "Ctrl+-\tzoom out\n"
   "F1 - F8\tload predefined montage\n"
   "\nafter zooming in by dragging a rectangle:\n"
   "Backspace\tzoom back\n"
   "Insert\tzoom in\n");
}





#ifdef Q_WS_WIN

QString UI_Mainwindow::specialFolder(int type)
{
    QString result;

    QLibrary library(QLatin1String("shell32"));
    QT_WA( {
        typedef BOOL (WINAPI*GetSpecialFolderPath)(HWND, LPTSTR, int, BOOL);
        GetSpecialFolderPath SHGetSpecialFolderPath = (GetSpecialFolderPath)library.resolve("SHGetSpecialFolderPathW");
        if (SHGetSpecialFolderPath) {
            TCHAR path[MAX_PATH];
            SHGetSpecialFolderPath(0, path, type, FALSE);
            result = QString::fromUtf16((ushort*)path);
        }
    } , {
        typedef BOOL (WINAPI*GetSpecialFolderPath)(HWND, char*, int, BOOL);
        GetSpecialFolderPath SHGetSpecialFolderPath = (GetSpecialFolderPath)library.resolve("SHGetSpecialFolderPathA");
        if (SHGetSpecialFolderPath) {
            char path[MAX_PATH];
            SHGetSpecialFolderPath(0, path, type, FALSE);
            result = QString::fromLocal8Bit(path);
        }
    } );

    if (result.isEmpty()) {
         result = QLatin1String("C:\\temp");
    }

    return result;
}

#endif



int UI_Mainwindow::file_is_opened(const char *path)
{
  int i;


  for(i=0; i<files_open; i++)
  {
    if(!strcmp(edfheaderlist[i]->filename, path))
    {
      return(1);
    }
  }

  return(0);
}











