__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2010 Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# Last changed: 2010-06-03
from ufl.algorithms import preprocess, replace
from dolfin import info, plot, interactive
from dolfin import FunctionSpace, MixedFunctionSpace, DirichletBC, FunctionSpaceFromCpp
from dolfin import interpolate

import dolfin.cpp as cpp

from dolfin.function.function import Argument

__all__ = ["update_form", "update_bcs"]

def update_form(F, mesh, extract=None, exchange=None):
    """
    Update a form (including arguments and coefficients to a (new)
    mesh 'mesh'.
    """

    # FIXME: Copy form here first

    # Preprocess F to obtain arguments and coefficents
    G = preprocess(F)

    # Update Arguments in F
    arguments = G.form_data().original_arguments

    # FIXME: counting HACK to deal with JIT-signature problem (as usual)
    new_arguments = [Argument(update_function_space(v.function_space(), mesh),
                              v._count - 2) for v in arguments]
    F = replace(F, dict(zip(arguments, new_arguments)))

    # Update Coefficients in F. There is some extra logic here because
    # of how to deal with non-linear problems
    coefficients = G.form_data().original_coefficients
    for f in coefficients:

        if exchange and exchange[0] == f:
            g = exchange[1]
        else:
            g = update_coefficient(f, mesh)
        F = replace(F, {f: g})

        if extract and extract == f:
            extract = g

    return (F, extract)

def update_coefficient(f, mesh):
    """
    Update a Coefficient f to (new) mesh 'mesh'. Coefficients that are
    not defined over a function space are ignored.
    """

    if not hasattr(f, "function_space"):
        info("Ignoring coefficient (%s)" % str(f.__class__))
        return f

    info("Updating coefficient by interpolation")
    V = update_function_space(f.function_space(), mesh)
    g = interpolate(f, V)

    return g


def update_bcs(bcs, mesh, V_h):
    """
    Update Dirichlet boundary conditions defined over a sub-domain to
    'mesh'.
    """

    if not isinstance(bcs, (list, tuple)):
        return update_bcs([bcs])

    new_bcs = []

    for bc in bcs:

        # FIXME: Must figure out how to refine mesh functions
        if len(bc.domain_args) > 1:
            cpp.error("Unable to increase bc order for DirichletBC defined with MeshFunction")

        V_i = bc.function_space()
        component = V_i.component()
        if component.size() > 0:
            # FIXME: Component/Array issues to be resolved
            W = update_function_space(V_h, mesh).sub(component[0])
        else:
            W = update_function_space(V_i, mesh)

        new_bcs += [DirichletBC(W, update_coefficient(bc.value(), mesh),
                                bc.domain_args[0])]
    return new_bcs

def update_function_space(V, mesh):
    "Update a function space V to a (new) mesh 'mesh'."

    n = V.num_sub_spaces()
    if n > 0:
        return MixedFunctionSpace([update_function_space(V.sub(i), mesh)
                                   for i in range(n)])
    element = V.ufl_element()
    return FunctionSpace(mesh, element.family(), element.degree())
