"""This module handles the Function class in Python.
"""
# Copyright (C) 2009 Johan Hake
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN.  If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2009-10-06
# Last changed: 2011-04-18

__all__ = ["Function", "TestFunction", "TrialFunction", "Argument",
           "TestFunctions", "TrialFunctions"]

import types

# Import UFL and SWIG-generated extension module (DOLFIN C++)
import ufl
import dolfin.cpp as cpp

from dolfin.functions.functionspace import FunctionSpaceBase
from dolfin.functions.expression import expression__call__

class MetaNoEvalOverloading(type):
    def __init__(mcs, name, bases, dictionary):
        if "eval" in dictionary:
            raise TypeError, "cannot overload 'eval'"

class Function(ufl.Coefficient, cpp.Function):
    """This class represents a function u_h in a finite element

    function space V_h, given by

      u_h = sum_i U_i phi_i

    where {phi_i}_i is a basis for V_h, and U is a vector of
    expansion coefficients for u_h.

    *Initialize Function*

    Create a Function:

    - from a FunctionSpace

      >>> f = Function(V)

    - from another Function

      >>> g = Function(f)

    - from a FunctionSpace and a GenericVector

      >>> g = Function(V, v)

    - from a FunctionSpace and a filename containing a GenericVector

      >>> g = Function(V, 'MyVectorValues.xml')"""
    __metaclass__ = MetaNoEvalOverloading

    def __init__(self, *args):
        """Initialize Function"""
        # Check arguments
        if len(args) == 0:
            raise TypeError, "expected 1 or more arguments"

        if not isinstance(args[0], (FunctionSpaceBase, Function)):
            raise TypeError, "expected a FunctionSpace or a Function as argument 1"

        # If using the copy constuctor
        if isinstance(args[0], Function):
            other = args[0]
            # If using the copy constuctor
            if len(args) == 1:
                # Instantiate base classes
                cpp.Function.__init__(self, other)
                ufl.Coefficient.__init__(self, other._element)
                return

            # If using sub-function constructor
            elif len(args) == 2 and isinstance(args[1], int):
                i = args[1]
                num_sub_spaces = other.function_space().num_sub_spaces()

                if num_sub_spaces == 1:
                    raise RuntimeError, "No subfunctions to extract"
                if not i < num_sub_spaces:
                    raise RuntimeError, "Can only extract subfunctions with i = 0..%d"% num_sub_spaces
                cpp.Function.__init__(self, other, i)
                ufl.Coefficient.__init__(self, self.function_space().ufl_element())
                return
            else:
                raise TypeError, "expected one or two arguments when instantiating from another Function"

        V = args[0]

        # Instantiate ufl base class
        ufl.Coefficient.__init__(self, V.ufl_element())

        # Passing only the FunctionSpace
        if len(args) == 1:
            # Instantiate cpp base classes
            cpp.Function.__init__(self, V)
        elif len(args) == 2:
            # If passing FunctionSpace together with cpp.Function
            # Attached passed FunctionSpace and initialize the cpp.Function
            # using the passed Function
            if isinstance(args[1], cpp.Function):
                if args[1].function_space().dim() != V.dim():
                    raise ValueError, "non matching dimensions on passed FunctionSpaces"

                cpp.Function.__init__(self, args[1])
            else:
                cpp.Function.__init__(self, *args)
        else:
            raise TypeError, "too many arguments"

    def _sub(self, i, deepcopy = False):
        """Return a sub function

        The sub functions are numbered from i = 0..N-1, where N is the total number
        of sub spaces.

        *Arguments*
            i
                The number of the sub function
        """
        if not isinstance(i, int):
            raise TypeError, "expects an 'int' as first argument"
        num_sub_spaces = self.function_space().num_sub_spaces()
        if num_sub_spaces == 1:
            raise RuntimeError, "No subfunctions to extract"
        if not i < num_sub_spaces:
            raise RuntimeError, "Can only extract subfunctions with i = 0..%d"% num_sub_spaces

        # Create and instantiate the Function
        if deepcopy:
            return Function(self.function_space().sub(i), cpp.Function._sub(self, i))
        else:
            return Function(self, i)

    def split(self, deepcopy = False):
        """Extract any sub functions

        A sub function can be extracted from a discrete function that is in a
        a MixedFunctionSpace or in a VectorFunctionSpace. The sub function is a
        function that resides in a sub space of the mixed space.

        *Arguments*
            deepcopy
                Copy sub function vector instead of sharing

        """
        num_sub_spaces = self.function_space().num_sub_spaces()
        if num_sub_spaces == 1:
            raise RuntimeError, "No subfunctions to extract"
        return tuple(self._sub(i, deepcopy) for i in xrange(num_sub_spaces))

    def ufl_element(self):
        "Return ufl element"
        return self._element

    def __str__(self):
        # FIXME: We might change this using rank and dimension instead
        return "<Function in %s>" % str(self.function_space())

    def __repr__(self):
        return ufl.Coefficient.__repr__(self)

#expression__call__.__doc__ = """ Evaluate the Function by interpolation

#    Example of use:
#    1) Using an iterable as x:

#    >>> Vs = FunctionSpace(mesh,"CG",1)
#    >>> fs = Function(Vs)
#    >>> x0 = (1.,0.5,0.5)
#    >>> x1 = [1.,0.5,0.5]
#    >>> x2 = numpy.array([1.,0.5,0.5])
#    >>> v0 = fs(x0)
#    >>> v1 = fs(x1)
#    >>> v2 = fs(x2)

#    2) Using multiple scalar args for x, interpreted as a point coordinate
#    >>> v0 = f(1.,0.5,0.5)

#    3) Passing return array
#    >>> Vv = VectorFunctionSpace(mesh,"CG",1)
#    >>> fv = Function(Vv)
#    >>> x0 = numpy.array([1.,0.5,0.5])
#    >>> v0 = numpy.zeros(3)
#    >>> fv(x0, values = v0)

#    Note: A longer values array may be passed. In this way one can fast fill up
#          an array with different evaluations.
#    >>> values = numpy.zeros(9)
#    >>> for i in xrange(0,10,3):
#            fv(x[i:i+3], values = values[i:i+3])

#    """

Function.__call__ = types.MethodType(expression__call__, None, Function)

#--- Subclassing of ufl.{Basis, Trial, Test}Function ---

class Argument(ufl.Argument):
    """An Argument represents a possibly differentiated component
    of an argument on the reference cell.
    """
    def __init__(self, V, index=None):
        if not isinstance(V, FunctionSpaceBase):
            raise TypeError, "Illegal argument for creation of Argument, not a FunctionSpace: " + str(V)
        ufl.Argument.__init__(self, V.ufl_element(), index)
        self._V = V

    def function_space(self):
        "Return the FunctionSpace"
        return self._V

def TrialFunction(V):
    """A TrialFunction is the Argument with the next lowest primary
    index. We simply pick an index lower than almost all others (-1).
    """
    return Argument(V, -1)

def TestFunction(V):
    """A TestFunction is the Argument with the lowest primary
    index. We simply pick an index lower than all others (-2).
    """
    return Argument(V, -2)

#--- TestFunctions and TrialFunctions ---

def TestFunctions(V):
    "Create test functions from mixed function space."
    return ufl.split(TestFunction(V))

def TrialFunctions(V):
    "Create trial functions from mixed function space."
    return ufl.split(TrialFunction(V))

