"""This module provides a simple way to compute the projection of
a Function (discrete or user-defined) onto a finite element space."""

# Copyright (C) 2008-2009 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN.  If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2008-07-13
# Last changed: 2009-12-11

__all__ = ['project']

# Import UFL and SWIG-generated extension module (DOLFIN C++)
import ufl
import dolfin.cpp as cpp

# Local imports
from dolfin.functions.function import *
from dolfin.functions.expression import *
from dolfin.functions.functionspace import *
from dolfin.fem.assemble import *

def project(v, V=None, mesh=None, solver_type="cg", preconditioner_type="default"):
    """Return projection of given expression v onto the finite element space V.
    Example usage:

        V = FunctionSpace(mesh, "Lagrange", 1)

        Pv = project(v, V)

    This is useful for post-processing functions or expressions which
    are not readily handled by visualization tools (such as for
    example discontinuous functions)."""

    # If trying to project an Expression
    if V is None and isinstance(v, Expression):
        if mesh is not None and isinstance(mesh, cpp.Mesh):
            V = FunctionSpaceBase(mesh, v.ufl_element())
        else:
            raise TypeError, "expected a mesh when projecting an Expression"

    # Try extracting function space if not specified
    if V is None:
        V = _extract_function_space(v)

    # Check arguments
    if not isinstance(V, FunctionSpaceBase):
        cpp.error("Illegal function space for projection, not a FunctionSpace: " + str(v))

    # Define variational problem for projection
    w = TestFunction(V)
    Pv = TrialFunction(V)
    a = ufl.inner(w, Pv)*ufl.dx
    L = ufl.inner(w, v)*ufl.dx

    # Assemble linear system
    A = assemble(a)
    b = assemble(L)

    # Solve linear system for projection
    Pv = Function(V)
    cpp.solve(A, Pv.vector(), b, solver_type, preconditioner_type)

    return Pv

def _extract_function_space(expression):
    """Try to extract a suitable function space for projection of
    given expression."""

    # Extract functions
    functions = ufl.algorithms.extract_coefficients(expression)

    # Extract mesh from functions
    mesh = None
    for f in functions:
        try:
            mesh = f.function_space().mesh()
            break
        except:
            pass
    if mesh is None:
        raise RuntimeError, "Unable to project expression, can't find a suitable mesh."

    # Create function space
    if expression.rank() == 0:
        V = FunctionSpace(mesh, "CG", 1)
    elif expression.rank() == 1:
        V = VectorFunctionSpace(mesh, "CG", 1)
    else:
        raise RuntimeError, "Unable to project expression, unhandled rank."

    return V
