# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.

"""Test network IO."""


import pygtk
pygtk.require('2.0')

from twisted.internet import reactor, task

from desktopcouch.application.pair.couchdb_pairing.network_io import (
    start_send_invitation, ListenForInvitations)
from desktopcouch.application.pair.couchdb_pairing.dbus_io import (
    get_local_hostname)

import unittest

LOCAL_HOSTNAME = ".".join(get_local_hostname())

A_HOSTID = "hostid a"
A_OAUTH_DATA = dict(
    token="atoken", token_secret="atokensecret",
    consumer="aconsumer", consumer_secret="aconsumersecret")

B_HOSTID = "hostid b"
B_OAUTH_DATA = dict(
    token="btoken", token_secret="btokensecret",
    consumer="bconsumer", consumer_secret="bconsumersecret")


class TestNetworkIO(unittest.TestCase):
    """Test class for  network IO."""

    def setUp(self):
        self._listener_socket_state = "open"
        self._inviter_socket_state = "open"
        self._listener_auth_completed = False
        self._inviter_auth_completed = False

    def test_successful_lifespan(self):
        """Test a successful lifespan."""
        secret = "sekrit"

        # pylint: disable=W0613
        def listener_get_secret_from_user(sender, function, send_secret,
                                          hostid, oauth_info):
            """Get secret from user.  Try several first."""

            self.assertEquals("hostid b", hostid)
            self.assertTrue("token" in oauth_info)

            self.assertFalse(function("sek"))
            self.assertFalse(function(""))
            self.assertFalse(function("                                  fs"))
            ret = function(secret)
            self.assertTrue(ret)
            if ret:
                send_secret(secret)
                listener_complete_auth()
        # pylint: enable=W0613

        def listener_close_socket():
            """Close the listener socket."""
            self._listener_socket_state = "closed"

        def inviter_close_socket():
            """Close the inviter socket."""
            self._inviter_socket_state = "closed"

        # pylint: disable=W0613
        def inviter_complete_auth(hostname, hostid, oauth_info):
            """Complete the inviter OAuth."""
            self.assertEquals("hostid a", hostid)
            self.assertTrue("token" in oauth_info)
            self._inviter_auth_completed = True
        # pylint: enable=W0613

        def listener_complete_auth():
            """Complete the listener OAuth."""
            self._listener_auth_completed = True
        # pylint: disable=W0201
        self.listener = ListenForInvitations(listener_get_secret_from_user,
                listener_close_socket, A_HOSTID, A_OAUTH_DATA)
        listener_port = self.listener.get_local_port()

        self.inviter = start_send_invitation(
            LOCAL_HOSTNAME, listener_port, inviter_complete_auth, secret,
            "seed", inviter_close_socket, B_HOSTID, B_OAUTH_DATA)
        # pylint: enable=W0201

        def exit_on_success():
            """Exit on success."""
            if self._listener_auth_completed and self._inviter_auth_completed:
                reactor.stop()          # pylint: disable=E1101
        task.LoopingCall(exit_on_success).start(1.0)

        def exit_on_timeout():
            """Exit on timeout."""
            reactor.stop()              # pylint: disable=E1101
        reactor.callLater(30, exit_on_timeout)  # pylint: disable=E1101

        reactor.run()                   # pylint: disable=E1101

        self.assertTrue(self._listener_auth_completed)
        self.assertTrue(self._inviter_auth_completed)

        # still open at end
        self.assertEqual(self._inviter_socket_state, "open")
        # still open at end
        self.assertEqual(self._listener_socket_state, "open")


if __name__ == "__main__":
    unittest.main()
