/* transpositions.rectangular-controls.c - controls for manipulating a rectangular transposition
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"
#include "common.transpositions.rectangular.h"

/* Global data */
int rows, cols;
gridpath readpath, writepath;
static GList *path_name_list = NULL;

/* Function declares */
static void cb_spinbox_changed (GtkEditable *editable, gpointer user_data);
static void cb_combo_changed(GtkWidget *widget, gpointer gdata);
static void cb_view_path(GtkWidget *widget, gpointer gdata);

/* Plugin Interface */
/* ---------------- */

const char name[] = "transpositions.grid-controls";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "Various ways to read and write out text from rectangular grids.";
const int interface_version = 1;
const int plugin_type = PLUGIN_MODE;
const char menu_string[] = "/Transposition/Grid Controls";

int boot(void) {
    int i;
    cols = rows = 5;
    writepath = HORIZONTALS_STRAIGHT_TL;
    readpath = VERTICALS_DESCENDING_TL;
    for (i = 1; i <= MAX_GRIDPATH; i++)
	path_name_list = g_list_append(path_name_list, (gpointer) grid_path_names[i]);
    return PLUGIN_BOOT_OK;
}

void init_plugin(void) {;}

void deactivate_plugin(void) {;}

GtkWidget *make_widget(char *text) {
    GtkWidget *hbox;
    GtkObject *adjustment_rows, *adjustment_cols;
    GtkWidget *spin_rows, *spin_cols;
    GtkWidget *label_rows, *label_cols;
    GtkWidget *combo_writepath, *combo_readpath;
    GtkWidget *view_writepath, *view_readpath;
    
    hbox = gtk_hbox_new(FALSE, 0);
    adjustment_rows = gtk_adjustment_new((float) rows, 1.0, 40.0, 1.0, 1.0, 0.0);
    adjustment_cols = gtk_adjustment_new((float) rows, 1.0, 40.0, 1.0, 1.0, 0.0);

    label_rows = gtk_label_new("Number of rows: ");
    gtk_box_pack_start(GTK_BOX(hbox), label_rows, FALSE, FALSE, 0);
    gtk_widget_show(label_rows);
    
    spin_rows = gtk_spin_button_new(GTK_ADJUSTMENT(adjustment_rows), 0.0, 0); 
    gtk_signal_connect(GTK_OBJECT(spin_rows), "changed", GTK_SIGNAL_FUNC(cb_spinbox_changed), (gpointer) &rows);
    gtk_box_pack_start(GTK_BOX(hbox), spin_rows, FALSE, FALSE, 0);
    gtk_widget_show(spin_rows);

    label_cols = gtk_label_new("Number of columns: ");
    gtk_box_pack_start(GTK_BOX(hbox), label_cols, FALSE, FALSE, 0);
    gtk_widget_show(label_cols);
    
    spin_cols = gtk_spin_button_new(GTK_ADJUSTMENT(adjustment_cols), 0.0, 0);  
    gtk_signal_connect(GTK_OBJECT(spin_cols), "changed", GTK_SIGNAL_FUNC(cb_spinbox_changed), (gpointer) &cols);
    gtk_box_pack_start(GTK_BOX(hbox), spin_cols, FALSE, FALSE, 0);
    gtk_widget_show(spin_cols);

    view_writepath = gtk_button_new_with_label("Write Path");
    gtk_box_pack_start(GTK_BOX(hbox), view_writepath, FALSE, FALSE, 0);
    gtk_signal_connect(GTK_OBJECT(view_writepath), "clicked",
                       GTK_SIGNAL_FUNC(cb_view_path), (gpointer) &writepath);
    gtk_widget_show(view_writepath);

    combo_writepath = gtk_combo_new();
    gtk_combo_set_popdown_strings (GTK_COMBO (combo_writepath), path_name_list);
    gtk_box_pack_start(GTK_BOX(hbox), combo_writepath, FALSE, FALSE, 0);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo_writepath)->entry), FALSE);
    gtk_combo_set_value_in_list(GTK_COMBO(combo_writepath), TRUE, FALSE);
    gtk_list_select_item(GTK_LIST(GTK_COMBO(combo_writepath)->list), writepath - 1);
    gtk_widget_show(combo_writepath);
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo_writepath)->entry), "changed",
                       GTK_SIGNAL_FUNC (cb_combo_changed), (gpointer) &writepath);
    
    view_readpath = gtk_button_new_with_label("Read Path");
    gtk_box_pack_start(GTK_BOX(hbox), view_readpath, FALSE, FALSE, 0);
    gtk_signal_connect(GTK_OBJECT(view_readpath), "clicked",
                       GTK_SIGNAL_FUNC(cb_view_path), (gpointer) &readpath);
    gtk_widget_show(view_readpath);

    combo_readpath = gtk_combo_new();
    gtk_combo_set_popdown_strings (GTK_COMBO (combo_readpath), path_name_list);
    gtk_box_pack_start(GTK_BOX(hbox), combo_readpath, FALSE, FALSE, 0);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo_readpath)->entry), FALSE);
    gtk_combo_set_value_in_list(GTK_COMBO(combo_readpath), TRUE, FALSE);
    gtk_list_select_item(GTK_LIST(GTK_COMBO(combo_readpath)->list), readpath - 1);
    gtk_widget_show(combo_readpath);
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo_readpath)->entry), "changed",
                       GTK_SIGNAL_FUNC (cb_combo_changed), (gpointer) &readpath);

    return hbox;
}

char *transform(char *text) {
    return transform_with_grid(rows, cols, text, writepath, readpath);
}

void reset(void) {;}

/* Plugin implementation */
/* --------------------- */

static gridpath name_to_path(char *name) {
    int i, found = FALSE;
    for (i = 1; i <= MAX_GRIDPATH; i++) 
	if (!strcmp(name, grid_path_names[i]))
	    found = i;
    return found;
}

GtkWidget *make_path_view_window(gridpath path, int rows, int cols) {
    GtkWidget *dialog;
    GtkWidget *button;
    GtkWidget *table;
    GtkWidget *label, *label_temp;
    Grid grid;
    char *text, character, *pos, *written_string, buf[BUFFER_SIZE];
    const char LOW_CHAR = '\0' + 1;
    int r, c, length = rows * cols;

    /* Write a sequence of characters into a text string */
    text = malloc(sizeof(char) * (length + 1));
    pos = text;
    for (character = LOW_CHAR; character < LOW_CHAR + (char) length; character++)
	*(pos++) = character;
    *pos = '\0';
    
    /* Make a grid according to the pattern */
    grid = make_new_grid(rows, cols, text, path);
    written_string = extract_from_grid(grid, rows, cols, HORIZONTALS_STRAIGHT_TL);
    
    dialog  = gtk_dialog_new();
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_widget_set_usize(dialog, -1, -1);
    gtk_window_set_title(GTK_WINDOW(dialog), "Route View");
    button = gtk_button_new_with_label("Dismiss");
    gtk_signal_connect_object(GTK_OBJECT(button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, FALSE, FALSE, 0);
    gtk_widget_show(button);

    label = gtk_label_new(grid_path_names[path]);
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 5);

    table = gtk_table_new(rows, cols, TRUE);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    for (r = 0; r < rows; r++) {
	for (c = 0; c < cols; c++) {
	    sprintf(buf, "%d", (int) (written_string[r * cols + c] - LOW_CHAR + 1));
	    label_temp = gtk_label_new(buf);
	    gtk_widget_show(label_temp);
	    gtk_table_attach_defaults(GTK_TABLE(table), label_temp, c, c + 1, r, r + 1);
	}
    }
    free(written_string);
    free(grid);
    free(text);					     
    return dialog;
}

/* Callbacks */
/* --------- */

static void cb_spinbox_changed(GtkEditable *editable, gpointer user_data) {
    * (int *) user_data = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(editable));
    display_text();
}

static void cb_combo_changed(GtkWidget *widget, gpointer gdata) {
    char *txt;
    txt = gtk_editable_get_chars(GTK_EDITABLE(widget), 0, -1);
    * (int *) gdata = name_to_path(txt);
    g_free(txt);
    display_text();
}

static void cb_view_path(GtkWidget *widget, gpointer gdata) {
    gtk_widget_show(make_path_view_window(* (int *) gdata, rows, cols));
}
