/* transpositions.grid-brute-force.c - automatic brute force checker for route ciphers
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"
#include "common.statistics.n-grams.h"
#include "common.transpositions.rectangular.h"

/* Global data */

int rows, cols;
gridpath readpath, writepath;
static GtkWidget *progress_bar;
static GtkWidget *stop_button;
static int abort_computation = FALSE;     /* Whether the user has signalled to abort a computation */ 
static float *slft_std;
static float *bift_std;
static float *trift_std;

/* fn decls */
static void do_break(void);
static void cb_break(GtkWidget *widget, gpointer gdata);
static gint cb_abort(GtkWidget *widget, gpointer gdata);

/* Plugin Interface */
/* ---------------- */

const char name[] = "transpositions.grid-brute-force";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "A brute force automatic cracker for rectangular grid route ciphers.";
const int interface_version = 1;
const int plugin_type = PLUGIN_MODE;
const char menu_string[] = "/Transposition/Brute Force Cracker";

int boot(void) {
    /* Load the standard frequency tables */
    slft_std = load_slft_std(DEFAULT_SLFT);
    bift_std = load_bift_std(DEFAULT_BIFT);
    trift_std = load_trift_std(DEFAULT_TRIFT);

    /* Set up default transposition */
    cols = rows = 5;
    writepath = HORIZONTALS_STRAIGHT_TL;
    readpath = VERTICALS_DESCENDING_TL;

    return PLUGIN_BOOT_OK;
}

void init_plugin(void) {;}

void deactivate_plugin(void) {;}

GtkWidget *make_widget(char *text) {
    GtkWidget *plugin_controls;
    GtkWidget *break_button;
    plugin_controls = gtk_hbox_new(FALSE, 0); 
    
    break_button = gtk_button_new_with_label("Start");
    gtk_signal_connect(GTK_OBJECT(break_button), "clicked", GTK_SIGNAL_FUNC(cb_break), NULL);
    gtk_box_pack_start(GTK_BOX(plugin_controls), break_button, FALSE, TRUE, 0);
    gtk_widget_show(break_button);

    progress_bar = gtk_progress_bar_new();
    gtk_box_pack_start(GTK_BOX(plugin_controls), progress_bar, FALSE, TRUE, 0);
    gtk_widget_show(progress_bar);

    stop_button = gtk_button_new_with_label("Stop");
    gtk_signal_connect(GTK_OBJECT(stop_button), "clicked", GTK_SIGNAL_FUNC(cb_abort), NULL);
    gtk_box_pack_start(GTK_BOX(plugin_controls), stop_button, FALSE, TRUE, 0);
    gtk_widget_show(stop_button);
    
    return plugin_controls;
}

char *transform(char *text) {
    return transform_with_grid(rows, cols, text, writepath, readpath);
}

void reset(void) {
  ;
}

/* Plugin implementation */
/* --------------------- */

static void cb_break(GtkWidget *widget, gpointer gdata) {
    do_break();
} 

static gint cb_abort(GtkWidget *widget, gpointer gdata) {
    abort_computation = TRUE;
    return (FALSE);
}

/* Exhaustive Search */
static void do_break(void) {
    gridpath current_writepath, current_readpath;
    stats *text_stats;
    float best_error, current_error;
    char *temp;
    
    abort_computation = FALSE;

    /* Calculate the stats */
    text_stats = make_stats(get_text(), slft_std, bift_std, trift_std);
    
    /* Check for zero length text */
    if (!text_stats->letter_count) {
	free(text_stats);
	return;
    }

    /* Lock off the rest of the UI while processing */
    gtk_grab_add(stop_button);

    best_error = 1000000.0;

    for (current_writepath = 1; current_writepath <= MAX_GRIDPATH; current_writepath++) {
	for (current_readpath = 1; current_readpath <= MAX_GRIDPATH; current_readpath++) {
	    if (abort_computation) 
		goto end_main_loop;
	    //	    free_stats(text_stats);
	    temp = transform_with_grid(rows, cols, get_text(), current_writepath, current_readpath);
	    //	    text_stats = make_stats(temp, slft_std, bift_std, trift_std);
	    current_error = (*global_fitness)(temp);
	    free(temp);
	    if (current_error < best_error) {
		best_error = current_error;
		readpath = current_readpath;
		writepath = current_writepath;
		display_text();
	    }
	    gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), (float) ((current_writepath - 1) * MAX_GRIDPATH + current_readpath) / (float) (MAX_GRIDPATH * MAX_GRIDPATH));
	    yield_to_ui();
        }
    }
end_main_loop:

    /* Clean up */
    gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), 0.0);
    abort_computation = TRUE;
    while (gtk_grab_get_current())
	gtk_grab_remove(gtk_grab_get_current());

    if (text_stats) 
	free_stats(text_stats);

    display_text();
}
