# cpushare client
# Copyright (C) 2005-2006  Andrea Arcangeli <andrea@cpushare.com>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation;
# only version 2.1 of the License.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import re, struct, os, md5

# local
from cpushare.proto_const import HEADER_FMT

PROTO_SECCOMP_MAX_LENGTH = 4*1024*1024

class seccomp_gen_class(object):
	def __init__(self, bytecode, name = None):
		self.bytecode = bytecode
		self.name = name
		self.load()

	def size(self):
		return self.text_size + self.data_size
	def static_size(self):
		return self.size() + self.bss_size

	def load(self):
		text = file(self.bytecode + '.text.bin').read()
		data = file(self.bytecode + '.data.bin').read()
		self.text_data = text + data
		self.digest = md5.new(self.text_data).digest()
		self._size = len(self.text_data)
		if self._size > PROTO_SECCOMP_MAX_LENGTH:
			raise Exception('seccomp bytecode too big')

		headers = os.popen('objdump -h ' + self.bytecode).readlines()
		for line in headers:
			m = re.search(r"\s\.text\s+(\w+)", line)
			if m:
				if hasattr(self, 'text_size'):
					raise Exception('x')
				self.text_size = long(m.group(1), 16)
				continue
			m = re.search(r"\s\.data\s+(\w+)\s+(\w+)", line)
			if m:
				if hasattr(self, 'data_size'):
					raise Exception('x')
				self.data_size = long(m.group(1), 16)
				self.data_start = long(m.group(2), 16)
				continue
			m = re.search(r"\s\.bss\s+(\w+)\s+(\w+)", line)
			if m:
				if hasattr(self, 'bss_size'):
					raise Exception('x')
				self.bss_size = long(m.group(1), 16)
				self.bss_start = long(m.group(2), 16)
				continue
		if hasattr(self, 'data_size'):
			assert self.bss_start >= self.data_start+self.data_size
			assert self.data_size <= self.bss_start-self.data_start
			self.bss_size += self.bss_start - \
					 (self.data_start+self.data_size)
		else:
			self.data_size = 0

		nm = os.popen('nm ' + self.bytecode).readlines()
		for line in nm:
			m = re.search(r"(\w+) T bytecode", line)
			if m:
				if hasattr(self, 'call_address'):
					raise Exception('x')
				self.call_address = long(m.group(1), 16)
				continue
			m = re.search(r"(\w+) T sighandler", line)
			if m:
				if hasattr(self, 'signal_address'):
					raise Exception('x')
				self.signal_address = long(m.group(1), 16)
				continue

		assert len(text) == self.text_size
		assert len(data) == self.data_size
		assert self._size == len(self.text_data)
		assert self._size == self.size()

	def __build_header(self, cksum, heap, stack):
		return struct.pack(HEADER_FMT,
				   self.text_size,
				   self.data_size,
				   self.bss_size,
				   self.call_address,
				   self.signal_address,
				   heap, stack, cksum)

	def _build_header(self, heap, stack):
		if heap < 0 or heap > 2**32-1:
			raise Exception('wrong heap')
		if stack < 0 or stack > 2**32-1:
			raise Exception('wrong stack')
		obj = self.__build_header(0, heap, stack)
		obj += self.text_data

		cksum = 0L
		for c in obj:
			cksum += ord(c)
			cksum &= 2**32-1

		return self.__build_header(cksum, heap, stack)

	def build_header(self, heap, stack):
		return chr(len(self.name)) + self.name + self._build_header(heap, stack) + self.digest
