/* $Id: cpl_image_filter-test.c,v 1.47 2010/11/11 09:23:18 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/11/11 09:23:18 $
 * $Revision: 1.47 $
 * $Name: cpl-5_3_0-BRANCH $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <float.h>
#include <assert.h>

#include "cpl_image_filter.h"
#include "cpl_image_gen.h"
#include "cpl_image_io.h"
#include "cpl_tools.h"
#include "cpl_test.h"
#include "cpl_image_bpm.h"
#include "cpl_memory.h"

/*-----------------------------------------------------------------------------
                                Defines
 -----------------------------------------------------------------------------*/

#define CPL_IMAGE_FILTER_LINEAR 0
#define CPL_IMAGE_FILTER_MORPHO 1
#define CPL_IMAGE_FILTER_SIZE   2

#ifndef IMAGE_SIZE_X
#define IMAGE_SIZE_X 64
#endif
#ifndef IMAGE_SIZE_Y
#define IMAGE_SIZE_Y IMAGE_SIZE_X
#endif

/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/

static void cpl_image_filter_test(int, int, int, int);

/*-----------------------------------------------------------------------------
                                  Main
 -----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */
    cpl_image_filter_test(8, 8, 0, 0);
    cpl_image_filter_test(8, 8, 1, 0);
    cpl_image_filter_test(8, 8, 0, 1);
    cpl_image_filter_test(8, 8, 1, 1);
    cpl_image_filter_test(8, 8, 2, 0);
    cpl_image_filter_test(8, 8, 0, 2);
    cpl_image_filter_test(8, 8, 2, 2);

    return cpl_test_end(0);
}


/*----------------------------------------------------------------------------*/
/**
  @brief  Test cpl_image_filter() and cpl_image_filter_mask()
  @param  nx      The image X-size
  @param  ny      The image Y-size
  @param  hm      The kernel X-halfsize
  @param  hn      The kernel Y-halfsize
  @param  type    The filtering type
  @return void

 */
/*----------------------------------------------------------------------------*/
static void cpl_image_filter_test(int nx, int ny, int hm, int hn)
{

    const cpl_filter_mode filter_mode[] = {CPL_FILTER_MEDIAN,
                                           CPL_FILTER_STDEV, 
                                           CPL_FILTER_STDEV_FAST, 
                                           CPL_FILTER_AVERAGE, 
                                           CPL_FILTER_AVERAGE_FAST,
                                           CPL_FILTER_LINEAR};

    const cpl_boolean filter_mask[] = {CPL_TRUE, CPL_TRUE, CPL_TRUE,
                                       CPL_TRUE, CPL_TRUE, CPL_FALSE};
    /* Compare with morpho on a kernel of ones */
    const cpl_boolean filter_morph[]  = {CPL_FALSE, CPL_FALSE, CPL_FALSE,
                     CPL_TRUE, CPL_FALSE, CPL_TRUE};

    const cpl_type pixel_type[] = {CPL_TYPE_INT, CPL_TYPE_FLOAT,
                                   CPL_TYPE_DOUBLE};

    const int m = 1 + 2 * hm;
    const int n = 1 + 2 * hn;
    const double xmin = -100.0;
    const double xmax =  200.0;

    cpl_mask * window1 = cpl_mask_new(m, n);
    cpl_mask * window2 = cpl_mask_new(m+2, n+2);
    cpl_mask * shift   = cpl_mask_new(m, n);
    cpl_matrix * xwindow1 = cpl_matrix_new(n, m);
    cpl_matrix * xwindow2 = cpl_matrix_new(n+2, m+2);
    cpl_matrix * xshift   = cpl_matrix_new(n, m);
    unsigned itype1;
    int i, j;

    cpl_test_eq(sizeof(filter_mode)/sizeof(filter_mode[0]),
                sizeof(filter_mask)/sizeof(filter_mask[0]));

    cpl_test_zero(cpl_mask_not(window1));
    cpl_test_zero(cpl_matrix_fill(xwindow1, 1.0));

    for (j = 2; j <= n+1; j++) {
        for (i = 2; i <= m+1; i++) {
            cpl_mask_set(window2, i, j, CPL_BINARY_1);
            cpl_matrix_set(xwindow2, j-1, i-1, 1.0);
        }
    }
    cpl_test_eq(cpl_mask_count(window1), m*n);
    cpl_test_eq(cpl_mask_count(window2), m*n);

    for (itype1 = 0; 
         itype1 < sizeof(pixel_type)/sizeof(pixel_type[0]); 
         itype1++) {
        unsigned itype2;

        for (itype2 = 0; 
             itype2 < sizeof(pixel_type)/sizeof(pixel_type[0]); 
             itype2++) {
            unsigned ifilt;

            for (ifilt = 0; 
                 ifilt < sizeof(filter_mode)/sizeof(filter_mode[0]); 
                 ifilt++) {
                const cpl_error_code exp_err = m == 1 && n == 1
                    && (filter_mode[ifilt] == CPL_FILTER_STDEV ||
                        filter_mode[ifilt] == CPL_FILTER_STDEV_FAST)
                    ? CPL_ERROR_DATA_NOT_FOUND
                    : CPL_ERROR_NONE;

                if (filter_mode[ifilt] != CPL_FILTER_MEDIAN ||
                    itype1 == itype2) {

                    /* FIXME: Need to test with bad pixels as well */
                    cpl_image * testimg
                        = cpl_image_new(nx, ny, pixel_type[itype1]);
                    cpl_image * filtim1
                        = cpl_image_new(nx, ny, pixel_type[itype2]);
                    cpl_image * filtim2
                        = cpl_image_new(nx, ny, pixel_type[itype2]);

                    cpl_error_code error;
                    double tol = 0.0;


                    cpl_msg_info(cpl_func, "%u-filtering %u X %u %s to %s with "
                                 "%u X %u", filter_mode[ifilt], nx, ny,
                                 cpl_type_get_name(pixel_type[itype1]),
                                 cpl_type_get_name(pixel_type[itype2]), m, n);

                    cpl_test_zero(cpl_image_fill_noise_uniform(testimg,
                                                               xmin, xmax));

                    error = filter_mask[ifilt]
                        ? cpl_image_filter_mask(filtim1, testimg, window1,
                                                filter_mode[ifilt],
                                                CPL_BORDER_FILTER)
                        : cpl_image_filter(filtim1, testimg, xwindow1,
                                           filter_mode[ifilt],
                                           CPL_BORDER_FILTER);
                    cpl_test_eq_error(error, exp_err);

                    error = filter_mask[ifilt]
                        ? cpl_image_filter_mask(filtim2, testimg, window2,
                                                filter_mode[ifilt],
                                                CPL_BORDER_FILTER)
                        : cpl_image_filter(filtim2, testimg, xwindow2,
                                           filter_mode[ifilt],
                                           CPL_BORDER_FILTER);
                    cpl_test_eq_error(error, exp_err);

                    if (exp_err == CPL_ERROR_NONE) {

                        if (filter_mode[ifilt] == CPL_FILTER_MEDIAN) {
                            /* FIXME: Cannot trust border (with even numbers) */
                            for (j = 1; j <= ny; j++) {
                                for (i = 1; i <= nx; i++) {
                                    if (j < n || ny - n < j ||
                                        i < m || nx - m < i) {
                                        cpl_image_reject(filtim1, i, j);
                                        cpl_image_reject(filtim2, i, j);
                                    }
                                }
                            }
                        } else {
                            /* filt1 and filt2 must be identical, because the
                               mask of filt2 equals filt1, except that is has a
                               halo of zeros */
                            if (filter_mode[ifilt] == CPL_FILTER_AVERAGE_FAST
                || filter_mode[ifilt] == CPL_FILTER_STDEV_FAST) {
                                /* In this case filtim1 is actually done with 
                                   CPL_FILTER_AVERAGE */
                                tol = 10.0 * (xmax - xmin)
                                    * ((pixel_type[itype1] == CPL_TYPE_FLOAT || 
                                        pixel_type[itype2] == CPL_TYPE_FLOAT)
                                       ? FLT_EPSILON : DBL_EPSILON);
                            } else if (filter_mode[ifilt] == CPL_FILTER_STDEV
                                       && pixel_type[itype2] == CPL_TYPE_DOUBLE
                                       && m*n > 1 && nx > m && ny > n) {
                                /* Verify a non-border value */

                                int is_bad;
                                const double stdev
                                    = cpl_image_get_stdev_window(testimg,
                                                                 nx-m+1,
                                                                 ny-n+1, nx,
                                                                 ny);

                                const double filtered = cpl_image_get(filtim1,
                                                                      nx-hm,
                                                                      ny-hn,
                                                                      &is_bad);
                                if (!is_bad) {
                                    cpl_test_rel(stdev, filtered,
                                                 DBL_EPSILON);
                                }
                            }

                /* The precision loss is higher in this case */
                if (filter_mode[ifilt] == CPL_FILTER_STDEV_FAST)
                tol = 1.0;
                        }
                        cpl_test_image_abs(filtim1, filtim2, tol);

                        if (filter_morph[ifilt]) {
                            /* Result should equal that of a morpho with
                               all ones */
                            /* The additions are reordered (sorted),
                               i.e. different round-off */
                            tol = (xmax - xmin)
                                * ((pixel_type[itype1] == CPL_TYPE_FLOAT || 
                                    pixel_type[itype2] == CPL_TYPE_FLOAT)
                                   ? FLT_EPSILON : DBL_EPSILON);

                            error = cpl_image_filter(filtim2, testimg,
                                                     xwindow1,
                                                     CPL_FILTER_MORPHO,
                                                     CPL_BORDER_FILTER);
                            cpl_test_eq_error(error, exp_err);
                            cpl_test_image_abs(filtim1, filtim2, tol);
                        }

                        if (itype1 == itype2) {

                            /* If the input and output pixel types are identical
                               a mask with 1 element corresponds to a shift */

                            /* Try all possible 1-element masks */
                            for (j = 1; j <= n; j++) {
                                for (i = 1; i <= m; i++) {

                                    /* Empty mask */
                                    cpl_test_zero(cpl_mask_xor(shift, shift));
                                    cpl_test_zero(cpl_matrix_fill(xshift, 0.0));

                                    /* Set one element */
                                    cpl_test_zero(cpl_mask_set(shift, i, j,
                                                               CPL_BINARY_1));
                                    cpl_test_zero(cpl_matrix_set(xshift, n-j,
                                                                 i-1, 1.0));


                                    cpl_test_eq(cpl_mask_count(shift), 1);

                                    /* This filter corresponds to a shift of
                                       ((hm+1) - i, (hn+1) - j) */
                                    error = filter_mask[ifilt]
                                        ? cpl_image_filter_mask(filtim1,
                                                                testimg,
                                                                shift,
                                                                filter_mode[ifilt],
                                                                CPL_BORDER_FILTER)

                                        : cpl_image_filter(filtim1,
                                                           testimg,
                                                           xshift,
                                                           filter_mode[ifilt],
                                                           CPL_BORDER_FILTER);


                                    cpl_test_error(error);
                                    if (filter_mode[ifilt] == CPL_FILTER_STDEV
                    || filter_mode[ifilt] == CPL_FILTER_STDEV_FAST) {
                                        cpl_test_eq(error,
                                                    CPL_ERROR_DATA_NOT_FOUND);
                                    } else {
                                        cpl_test_zero(error);

                                        cpl_test_zero(cpl_image_copy(filtim2,
                                                                     testimg,
                                                                     1, 1));
                                        cpl_test_zero(cpl_image_shift(filtim2,
                                                                      (hm+1) - i,
                                                                      (hn+1) - j));

                                        cpl_test_image_abs(filtim1, filtim2, 0.0);
                                    }
                                }
                            }
                        }
                    }

                    cpl_image_delete(testimg);
                    cpl_image_delete(filtim1);
                    cpl_image_delete(filtim2);

                }
            }
        }
    }

    cpl_mask_delete(shift);
    cpl_mask_delete(window1);
    cpl_mask_delete(window2);
    cpl_matrix_delete(xshift);
    cpl_matrix_delete(xwindow1);
    cpl_matrix_delete(xwindow2);

    return;
}
