!-----------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations         !
!   Copyright (C) 2000 - 2010  CP2K developers group                          !
!-----------------------------------------------------------------------------!

! *****************************************************************************
!> \brief Perform an abnormal program termination.
!> \par Index
!>      SUBROUTINE set_error_unit(lunit)
!>      SUBROUTINE stop_allocate(routine,array,memory)
!>      SUBROUTINE stop_deallocate(routine,array)
!>      SUBROUTINE stop_program(routine,message)
!> \par History
!>      none
!> \author Matthias Krack (12.02.2001)
! *****************************************************************************
MODULE termination

  USE cp_files,                        ONLY: close_file,&
                                             open_file
  USE cp_para_types,                   ONLY: cp_para_env_type
  USE f77_blas
  USE global_types,                    ONLY: global_environment_type
  USE kinds,                           ONLY: default_string_length,&
                                             dp
  USE machine,                         ONLY: default_output_unit,&
                                             m_flush_internal,&
                                             m_walltime
  USE message_passing,                 ONLY: mp_abort,&
                                             mp_bcast
  USE string_utilities,                ONLY: compress
  USE timings,                         ONLY: print_stack,&
                                             timeset,&
                                             timestop
#include "cp_common_uses.h"

  IMPLICIT NONE

  PRIVATE

! *** Global parameters ***

  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'termination'

  INTEGER :: error_unit

  PUBLIC :: external_control,&
            set_error_unit,&
            stop_memory,&
            stop_program

  INTERFACE stop_memory
     MODULE PROCEDURE stop_allocate,stop_allocate_new,&
                      stop_deallocate,stop_deallocate_new
  END INTERFACE

  INTERFACE stop_program
     MODULE PROCEDURE stop_program_new,stop_program_old
  END INTERFACE

CONTAINS

! *****************************************************************************
!> \brief Initialise the output unit number for error messages.
!> \par History
!>      none
!> \author JGH
! *****************************************************************************
  SUBROUTINE set_error_unit(lunit)
    INTEGER, INTENT(IN)                      :: lunit

    CHARACTER(len=*), PARAMETER :: routineN = 'set_error_unit', &
      routineP = moduleN//':'//routineN

    error_unit = lunit

  END SUBROUTINE set_error_unit

! *****************************************************************************
!> \brief The memory allocation for an array failed. Print an error message and
!>      stop the program execution.
!> \note
!>      array  : Name of the array.
!>      memory : Size of array in bytes.
!>      routine: Name of the calling routine.
!> \par History
!>      CP2K by JGH 21.08.2000
!> \author Matthias Krack (12.10.1999)
! *****************************************************************************
  SUBROUTINE stop_allocate(routine,array,memory)

    CHARACTER(LEN=*), INTENT(IN)             :: routine, array
    INTEGER, INTENT(IN)                      :: memory

    CHARACTER(len=*), PARAMETER :: routineN = 'stop_allocate', &
      routineP = moduleN//':'//routineN

    CHARACTER(LEN=240)                       :: message

! just in case there is not enough memory do even trim... and fail with
! with a mystery core dump

    WRITE(default_output_unit,*) "Out of memory ..."
    CALL m_flush_internal(default_output_unit)
    IF (memory == 0) THEN
      WRITE (message,"(A)")&
        "The memory allocation for the data object <"//TRIM(array)//"> failed"
    ELSE
      WRITE (message,"(A,I12,A)")&
        "The memory allocation for the data object <"//TRIM(array)//&
        "> failed. The requested memory size is ",memory/1024," Kbytes"
    END IF

    CALL compress(message)

    CALL stop_program(routine,message)

  END SUBROUTINE stop_allocate

! *****************************************************************************
  SUBROUTINE stop_allocate_new(routineN,moduleN,line_number,object,memory)

    CHARACTER(LEN=*), INTENT(IN)             :: routineN, moduleN
    INTEGER, INTENT(IN)                      :: line_number
    CHARACTER(LEN=*), INTENT(IN)             :: object
    INTEGER, INTENT(IN)                      :: memory

    CHARACTER(LEN=240)                       :: message

    IF (memory == 0) THEN
      message = "The memory allocation for the data object <"//TRIM(object)//&
                "> failed"
    ELSE
      WRITE (message,"(A,I12,A)")&
        "The memory allocation for the data object <"//TRIM(object)//&
        "> failed. The requested memory size is ",memory/1024," Kbytes"
      CALL compress(message)
    END IF

    CALL stop_program(routineN,moduleN,line_number,message)

  END SUBROUTINE stop_allocate_new

! *****************************************************************************
!> \brief The memory deallocation for an array failed. Print an error message and
!>      stop the program execution.
!> \note
!>      array  : Name of the array.
!>      routine: Name of the calling routine.
!> \par History
!>      CP2K by JGH 21.08.2000
!> \author Matthias Krack (20.10.1999)
! *****************************************************************************
  SUBROUTINE stop_deallocate(routine,array)

    CHARACTER(LEN=*), INTENT(IN)             :: routine, array

    CHARACTER(len=*), PARAMETER :: routineN = 'stop_deallocate', &
      routineP = moduleN//':'//routineN

    CHARACTER(LEN=240)                       :: message

    WRITE (message,"(A)")&
      "The memory deallocation for the data object <"//TRIM(array)//"> failed"

    CALL stop_program(routine,message)

  END SUBROUTINE stop_deallocate

! *****************************************************************************
  SUBROUTINE stop_deallocate_new(routineN,moduleN,line_number,object)

    CHARACTER(LEN=*), INTENT(IN)             :: routineN, moduleN
    INTEGER, INTENT(IN)                      :: line_number
    CHARACTER(LEN=*), INTENT(IN)             :: object

    CHARACTER(LEN=240)                       :: message

    message = "The memory deallocation for the data object <"//TRIM(object)//&
              "> failed"

    CALL stop_program(routineN,moduleN,line_number,message)

  END SUBROUTINE stop_deallocate_new

! *****************************************************************************
!> \brief Stop the program run and write an error message.
!> \note
!>      routine: Name of the calling routine
!> \par History
!>      Translated to Fortran 90 (07.10.99, MK)
!> \author Matthias Krack (28.08.1996)
! *****************************************************************************
  SUBROUTINE stop_program_old(routine,error_message,para_env)

    CHARACTER(LEN=*), INTENT(IN)             :: routine, error_message
    TYPE(cp_para_env_type), OPTIONAL, &
      POINTER                                :: para_env

    CHARACTER(len=*), PARAMETER :: routineN = 'stop_program_old', &
      routineP = moduleN//':'//routineN

    LOGICAL                                  :: ionode

    IF (PRESENT(para_env)) THEN
      ionode = para_env%ionode
    ELSE
      ionode = .TRUE.
    END IF

!   *** Print the error message ***

    IF (ionode) THEN
      CALL print_message("ERROR in "//TRIM(routine),error_unit,2,2,0)
      CALL print_message(error_message,error_unit,1,1,1)
      CALL print_stack(error_unit)
      CALL m_flush_internal(error_unit)
      CALL mp_abort()
    END IF

  END SUBROUTINE stop_program_old

! *****************************************************************************
  SUBROUTINE stop_program_new(routineN,moduleN,line_number,error_message,para_env)

    CHARACTER(LEN=*), INTENT(IN)             :: routineN, moduleN
    INTEGER, INTENT(IN)                      :: line_number
    CHARACTER(LEN=*), INTENT(IN)             :: error_message
    TYPE(cp_para_env_type), OPTIONAL, &
      POINTER                                :: para_env

    CHARACTER(LEN=80)                        :: message
    LOGICAL                                  :: ionode

    IF (PRESENT(para_env)) THEN
      ionode = para_env%ionode
    ELSE
      ionode = .TRUE.
    END IF

!   *** Print the error message ***

    IF (ionode) THEN
      message = "ERROR in "//TRIM(routineN)//&
                " (MODULE "//TRIM(moduleN)//")"
      CALL print_message(message,error_unit,2,2,0)
      CALL print_message(error_message,error_unit,1,1,0)
      WRITE (UNIT=message,FMT="(A,I6,A)")&
        "Program stopped at line number ",line_number,&
        " of MODULE "//TRIM(moduleN)
      CALL compress(message)
      CALL print_message(message,error_unit,1,1,0)
      CALL print_stack(error_unit)
      CALL m_flush_internal(error_unit)
      CALL mp_abort()
    END IF

  END SUBROUTINE stop_program_new

! *****************************************************************************
!> \brief External manipulations during a run : when the <PROJECT_NAME>.EXIT_$runtype
!>      command is sent the program stops at the level of $runtype
!>      when a general <PROJECT_NAME>.EXIT command is sent the program is stopped
!>      at all levels (at least those that call this function)
!>      if the file WAIT exists, the program waits here till it disappears
!> \author MI (10.03.2005)
! *****************************************************************************
  SUBROUTINE external_control(should_stop,flag,globenv,target_time,start_time,error)

    LOGICAL, INTENT(OUT)                     :: should_stop
    CHARACTER(LEN=*), INTENT(IN)             :: flag
    TYPE(global_environment_type), &
      OPTIONAL, POINTER                      :: globenv
    REAL(dp), OPTIONAL                       :: target_time, start_time
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'external_control', &
      routineP = moduleN//':'//routineN

    CHARACTER(LEN=default_string_length)     :: exit_fname, exit_fname_level, &
                                                exit_gname, exit_gname_level
    INTEGER                                  :: handle, i, unit_number
    LOGICAL                                  :: failure, should_wait
    LOGICAL, SAVE                            :: check_always = .FALSE.
    REAL(KIND=dp)                            :: my_start_time, &
                                                my_target_time, t1, t2, &
                                                time_check
    REAL(KIND=dp), SAVE                      :: t_last_file_check = 0.0_dp
    TYPE(cp_logger_type), POINTER            :: logger

    CALL timeset(routineN,handle)

    failure=.FALSE.
    logger => cp_error_get_logger(error)
    should_stop = .FALSE.

    exit_gname       = "EXIT"
    exit_gname_level = TRIM(exit_gname)//"_"//TRIM(flag)
    exit_fname       = TRIM(logger%iter_info%project_name)//"."//TRIM(exit_gname)
    exit_fname_level = TRIM(logger%iter_info%project_name)//"."//TRIM(exit_gname_level)

    IF (logger%para_env%source==logger%para_env%mepos) THEN
       ! files will only be checked every 20 seconds, or if the clock wraps/does not exist,
       ! otherwise 64 waters on 64 cores can spend up to 10% of time here, on lustre
       ! however, if should_stop has been true, we should always check
       ! (at each level scf, md, ... the file must be there to guarantee termination)
       t1=m_walltime()
       IF (t1>t_last_file_check+20.0_dp .OR. t1 <= t_last_file_check .OR. check_always) THEN
          t_last_file_check=t1
          ! allows for halting execution for a while
          ! this is useful to copy a consistent snapshot of the output
          ! while a simulation is running
          INQUIRE (FILE="WAIT",EXIST=should_wait)
          IF (should_wait) THEN
             CALL open_file(file_name="WAITING",file_status="UNKNOWN",&
                  file_form="FORMATTED",file_action="WRITE",&
                  unit_number=unit_number)
             WRITE (UNIT=cp_logger_get_default_unit_nr(logger),FMT="(/,T2,A,/)")&
                  "*** waiting till the file WAIT has been removed ***"
             DO
                ! sleep a bit (to save the file system)
                t1=m_walltime()
                DO I=1,100000000
                   t2=m_walltime()
                   IF (t2-t1>1.0_dp) EXIT
                ENDDO
                ! and ask again
                INQUIRE (FILE="WAIT",EXIST=should_wait)
                IF (.NOT.should_wait) EXIT
             ENDDO
             CALL close_file(unit_number=unit_number,file_status="DELETE")
          ENDIF
          ! EXIT control sequence
          ! Check for <PROJECT_NAME>.EXIT_<FLAG>
          IF (.NOT.should_stop) THEN
             INQUIRE (FILE=exit_fname_level,EXIST=should_stop)
             IF (should_stop) THEN
                CALL open_file(file_name=exit_fname_level,unit_number=unit_number)
                CALL close_file(unit_number=unit_number,file_status="DELETE")
                WRITE (UNIT=cp_logger_get_default_unit_nr(logger),FMT="(/,T2,A,/)")&
                     "*** "//flag//" run terminated by external request ***"
             END IF
          END IF
          ! Check for <PROJECT_NAME>.EXIT
          IF (.NOT.should_stop) THEN
             INQUIRE (FILE=exit_fname,EXIST=should_stop)
             IF(should_stop) THEN
                WRITE (UNIT=cp_logger_get_default_unit_nr(logger),FMT="(/,T2,A,/)")&
                     "*** "//TRIM(flag)//" run terminated by external request ***"
             ENDIF
          END IF
          ! Check for EXIT_<FLAG>
          IF (.NOT.should_stop) THEN
             INQUIRE (FILE=exit_gname_level,EXIST=should_stop)
             IF (should_stop) THEN
                CALL open_file(file_name=exit_gname_level,unit_number=unit_number)
                CALL close_file(unit_number=unit_number,file_status="DELETE")
                WRITE (UNIT=cp_logger_get_default_unit_nr(logger),FMT="(/,T2,A,/)")&
                     "*** "//flag//" run terminated by external request ***"
             END IF
          END IF
          ! Check for EXIT
          IF (.NOT.should_stop) THEN
             INQUIRE (FILE=exit_gname,EXIST=should_stop)
             IF(should_stop) THEN
                WRITE (UNIT=cp_logger_get_default_unit_nr(logger),FMT="(/,T2,A,/)")&
                     "*** "//TRIM(flag)//" run terminated by external request ***"
             ENDIF
          END IF
       ENDIF

       IF(PRESENT(target_time)) THEN
          my_target_time = target_time
          my_start_time = start_time
       ELSEIF(PRESENT(globenv)) THEN
          my_target_time=globenv%cp2k_target_time
          my_start_time=globenv%cp2k_start_time
       ELSE
          ! If none of the two arguments is present abort.. This routine should always check about time.
          CPPrecondition(.FALSE.,cp_fatal_level,routineP,error,failure)
       END IF

       IF ( (.NOT.should_stop).AND.(my_target_time > 0.0_dp)) THEN
          ! Check for execution time
          time_check  = m_walltime() - my_start_time
          IF (time_check .GT. my_target_time) THEN
             should_stop = .TRUE.
             WRITE (UNIT=cp_logger_get_default_unit_nr(logger),FMT="(/,T2,A,f12.3,A)")&
                  "*** "//TRIM(flag)//" run terminated - exceeded requested execution time:",&
                  my_target_time," seconds.",&
                  "*** Execution time now: ",time_check," seconds."
          END IF
       END IF
    END IF
    CALL mp_bcast(should_stop,logger%para_env%source,logger%para_env%group)

    check_always=should_stop

    CALL timestop(handle)

  END SUBROUTINE external_control

END MODULE termination

