!-----------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations         !
!   Copyright (C) 2000 - 2010  CP2K developers group                          !
!-----------------------------------------------------------------------------!

! *****************************************************************************
!> \brief Generate the atomic neighbor lists.
!> \par History
!>      - List rebuild for sab_orb neighbor list (10.09.2002,MK)
!>      - List rebuild for all lists (25.09.2002,MK)
!>      - Row-wise parallelized version (16.06.2003,MK)
!>      - Row- and column-wise parallelized version (19.07.2003,MK)
!>      - bug fix for non-periodic case (23.02.06,MK)
!> \author Matthias Krack (08.10.1999,26.03.2002,16.06.2003)
! *****************************************************************************
MODULE qs_neighbor_lists

  USE atomic_kind_types,               ONLY: atomic_kind_type,&
                                             get_atomic_kind,&
                                             get_atomic_kind_set
  USE basis_set_types,                 ONLY: get_gto_basis_set,&
                                             gto_basis_set_type
  USE cell_types,                      ONLY: cell_type,&
                                             get_cell,&
                                             pbc,&
                                             plane_distance,&
                                             real_to_scaled,&
                                             scaled_to_real
  USE cp_output_handling,              ONLY: cp_p_file,&
                                             cp_print_key_finished_output,&
                                             cp_print_key_should_output,&
                                             cp_print_key_unit_nr
  USE cp_para_types,                   ONLY: cp_para_env_type
  USE cp_units,                        ONLY: cp_unit_from_cp2k
  USE distribution_1d_types,           ONLY: distribution_1d_type
  USE distribution_2d_types,           ONLY: distribution_2d_type
  USE ewald_environment_types,         ONLY: ewald_env_get,&
                                             ewald_environment_type
  USE external_potential_types,        ONLY: all_potential_type,&
                                             get_potential,&
                                             gth_potential_type
  USE f77_blas
  USE input_constants,                 ONLY: do_se_IS_slater,&
                                             rel_none,&
                                             xc_vdw_fun_pairpot
  USE input_section_types,             ONLY: section_vals_get_subs_vals,&
                                             section_vals_type,&
                                             section_vals_val_get
  USE kinds,                           ONLY: default_string_length,&
                                             dp
  USE message_passing,                 ONLY: mp_max,&
                                             mp_sum
  USE molecule_types_new,              ONLY: molecule_type
  USE particle_types,                  ONLY: particle_type
  USE paw_proj_set_types,              ONLY: get_paw_proj_set,&
                                             paw_proj_set_type
  USE periodic_table,                  ONLY: ptable
  USE physcon,                         ONLY: bohr
  USE qs_dftb_types,                   ONLY: qs_dftb_atom_type,&
                                             qs_dftb_pairpot_type
  USE qs_dftb_utils,                   ONLY: get_dftb_atom_param
  USE qs_dispersion_types,             ONLY: qs_dispersion_type
  USE qs_environment_types,            ONLY: get_qs_env,&
                                             qs_environment_type,&
                                             set_qs_env
  USE qs_local_rho_types,              ONLY: rhoz_type
  USE qs_neighbor_list_types,          ONLY: &
       add_neighbor_list, add_neighbor_node, allocate_neighbor_list_set, &
       deallocate_neighbor_list_set, first_list, first_node, &
       get_neighbor_list, get_neighbor_list_set, get_neighbor_node, &
       neighbor_list_p_type, neighbor_list_set_p_type, &
       neighbor_list_set_type, neighbor_list_type, neighbor_node_type, next
  USE qs_rho0_types,                   ONLY: get_rho0_mpole,&
                                             rho0_mpole_type
  USE string_utilities,                ONLY: compress
  USE termination,                     ONLY: stop_program
  USE timings,                         ONLY: timeset,&
                                             timestop
#include "cp_common_uses.h"

  IMPLICIT NONE

  PRIVATE

  ! Type definitions

! *****************************************************************************
  TYPE local_atoms_type
    INTEGER, DIMENSION(:), POINTER                   :: list,&
                                                        list_local_a_index,&
                                                        list_local_b_index,&
                                                        list_a_mol,&
                                                        list_b_mol
    REAL(dp), DIMENSION(:,:), POINTER                :: r_pbc,&
                                                        s_pbc
    REAL ( dp )                                      :: qeff
  END TYPE local_atoms_type

! *****************************************************************************
  TYPE subcell_type
    INTEGER                        :: natom
    REAL(KIND=dp), DIMENSION(3)    :: s_max,s_min
    INTEGER, DIMENSION(:), POINTER :: atom_list
    REAL(KIND=dp), DIMENSION(3,8)  :: corners
  END TYPE subcell_type

  ! Global parameters
  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'qs_neighbor_lists'

  ! private counter, used to version qs neighbor lists
  INTEGER, SAVE, PRIVATE :: last_qs_neighbor_list_id_nr=0

  ! Public data types
  PUBLIC :: local_atoms_type,&
            subcell_type

  ! Public subroutines
  PUBLIC :: allocate_subcell,&
            build_gth_ppl_neighbor_lists,&
            build_qs_neighbor_lists,&
            deallocate_subcell,&
            write_neighbor_lists

CONTAINS

! *****************************************************************************
!> \brief    Build all the required neighbor lists for Quickstep.
!> \author  MK
!> \date    28.08.2000
!> \par History
!>          - Parallized version (16.06.2003,MK) 
!> \version 1.0
! *****************************************************************************
  SUBROUTINE build_qs_neighbor_lists(qs_env,para_env,molecular,force_env_section,error)
    TYPE(qs_environment_type), POINTER       :: qs_env
    TYPE(cp_para_env_type), POINTER          :: para_env
    LOGICAL, OPTIONAL                        :: molecular
    TYPE(section_vals_type), POINTER         :: force_env_section
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'build_qs_neighbor_lists', &
      routineP = moduleN//':'//routineN

    CHARACTER(LEN=default_string_length)     :: print_key_path, unit_str
    INTEGER :: atom_a, handle, iatom, iatom_local, ikind, imol, iw, jkind, &
      maxatom, maxatom_local, natom_a, natom_local_a, natom_local_b, nkind, &
      output_unit, stat, zat
    INTEGER, DIMENSION(:), POINTER           :: local_cols_array, &
                                                local_rows_array
    LOGICAL :: all_pairs_list, all_potential_present, cls, epr, failure, &
      gth_potential_present, mic, molecule_only, nmr, paw_atom, &
      paw_atom_present, rho0_present
    LOGICAL, ALLOCATABLE, DIMENSION(:) :: all_present, aux_fit_present, &
      aux_present, default_present, oce_present, orb_present, ppl_present, &
      ppnl_present, Qlm_present, semi_present
    REAL(dp)                                 :: alpha, r_skin, subcells, &
                                                unit_conv
    REAL(dp), ALLOCATABLE, DIMENSION(:) :: all_pot_rad, aux_fit_radius, &
      c_radius, oce_radius, orb_rad_3c, orb_radius, ppl_radius, ppnl_radius, &
      radius_3c, short_aux_fit_radius, short_orb_radius
    REAL(dp), ALLOCATABLE, DIMENSION(:, :)   :: ab_radius
    REAL(dp), DIMENSION(3)                   :: r_pbc
    TYPE(all_potential_type), POINTER        :: all_potential
    TYPE(atomic_kind_type), DIMENSION(:), &
      POINTER                                :: atomic_kind_set
    TYPE(atomic_kind_type), POINTER          :: atomic_kind
    TYPE(cell_type), POINTER                 :: cell
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(distribution_1d_type), POINTER      :: distribution_1d
    TYPE(distribution_2d_type), POINTER      :: distribution_2d
    TYPE(ewald_environment_type), POINTER    :: ewald_env
    TYPE(gth_potential_type), POINTER        :: gth_potential
    TYPE(gto_basis_set_type), POINTER        :: aux_basis_set, &
                                                aux_fit_basis_set, &
                                                orb_basis_set
    TYPE(local_atoms_type), ALLOCATABLE, &
      DIMENSION(:)                           :: atom
    TYPE(molecule_type), DIMENSION(:), &
      POINTER                                :: molecule_set
    TYPE(neighbor_list_set_p_type), DIMENSION(:), POINTER :: sab_2c, sab_all, &
      sab_aux_fit, sab_aux_fit_asymm, sab_aux_fit_vs_orb, sab_cn, sab_lrc, &
      sab_orb, sab_scp, sab_vdw, sac_3c, sac_ae, sac_ppl, sap_oce, sap_ppnl, &
      sbc_3c
    TYPE(particle_type), DIMENSION(:), &
      POINTER                                :: particle_set
    TYPE(paw_proj_set_type), POINTER         :: paw_proj
    TYPE(qs_dftb_atom_type), POINTER         :: dftb_atom
    TYPE(qs_dftb_pairpot_type), &
      DIMENSION(:, :), POINTER               :: dftb_potential
    TYPE(qs_dispersion_type), POINTER        :: dispersion_env
    TYPE(rho0_mpole_type), POINTER           :: rho0_mpole
    TYPE(rhoz_type), DIMENSION(:), POINTER   :: rhoz_set
    TYPE(section_vals_type), POINTER         :: neighbor_list_section

    CALL timeset(routineN,handle)
    failure = .FALSE.
    NULLIFY(logger)
    logger => cp_error_get_logger(error)

    NULLIFY (atomic_kind, neighbor_list_section)
    NULLIFY (atomic_kind_set)
    NULLIFY (cell)
    NULLIFY (distribution_2d)
    NULLIFY (distribution_1d)
    NULLIFY (gth_potential)
    NULLIFY (orb_basis_set)
    NULLIFY (particle_set)
    NULLIFY (molecule_set)
    NULLIFY (rho0_mpole)
    NULLIFY (rhoz_set)
    NULLIFY (sab_orb)
    NULLIFY (sac_ae)
    NULLIFY (sac_ppl)
    NULLIFY (sap_ppnl)
    NULLIFY (sap_oce)
    NULLIFY (sab_2c)
    NULLIFY (sab_lrc)
    NULLIFY (sac_3c)
    NULLIFY (sbc_3c)
    NULLIFY (sab_all)
    NULLIFY (sab_vdw)
    NULLIFY (sab_cn)
    NULLIFY (sab_aux_fit)
    NULLIFY (sab_aux_fit_vs_orb)
    NULLIFY (sab_scp)

    output_unit = cp_print_key_unit_nr(logger=logger,&
                                       basis_section=force_env_section,&
                                       print_key_path="DFT%PRINT%SUBCELL",&
                                       extension=".Log",&
                                       error=error)
    CALL section_vals_val_get(force_env_section,"DFT%PRINT%SUBCELL%UNIT",c_val=unit_str,error=error)
    unit_conv = cp_unit_from_cp2k(1.0_dp,TRIM(unit_str),error=error)
    neighbor_list_section => section_vals_get_subs_vals(force_env_section,"DFT%PRINT%NEIGHBOR_LISTS",error=error)

    molecule_only = .FALSE.
    IF (PRESENT(molecular)) molecule_only = molecular

    cls = qs_env%dft_control%do_xas_calculation
    CALL section_vals_val_get(qs_env%input,&
         "PROPERTIES%LINRES%NMR%_SECTION_PARAMETERS_",l_val=nmr,error=error)
    CALL section_vals_val_get(qs_env%input,&
         "PROPERTIES%LINRES%EPR%_SECTION_PARAMETERS_",l_val=epr,error=error)
    CALL section_vals_val_get(qs_env%input,"DFT%SUBCELLS",r_val=subcells,error=error)
    IF(cls .OR. nmr .OR. epr) THEN
       all_pairs_list = .TRUE.
    ELSE
       all_pairs_list = .FALSE.
    END IF

    ! This sets the id number of the qs neighbor lists, new lists, means new version
    ! new version implies new sparsity of the matrices

    last_qs_neighbor_list_id_nr=last_qs_neighbor_list_id_nr+1
    CALL set_qs_env(qs_env=qs_env,neighbor_list_id=last_qs_neighbor_list_id_nr,error=error)

    ! Set Verlet skin

    r_skin = 0.0_dp

    CALL get_qs_env(qs_env=qs_env,&
                    atomic_kind_set=atomic_kind_set,&
                    cell=cell,&
                    distribution_2d=distribution_2d,&
                    local_particles=distribution_1d,&
                    particle_set=particle_set,&
                    molecule_set=molecule_set,&
                    rho0_mpole=rho0_mpole,&
                    rhoz_set=rhoz_set,&
                    sab_orb=sab_orb,&
                    sab_aux_fit=sab_aux_fit,&
                    sab_aux_fit_vs_orb=sab_aux_fit_vs_orb,&
                    sab_aux_fit_asymm=sab_aux_fit_asymm,&
                    sac_ae=sac_ae,&
                    sac_ppl=sac_ppl,&
                    sab_vdw=sab_vdw,&
                    sab_cn=sab_cn,&
                    sap_ppnl=sap_ppnl,&
                    sap_oce=sap_oce,&
                    sab_2c=sab_2c,&
                    sab_lrc=sab_lrc,&
                    sac_3c=sac_3c,&
                    sbc_3c=sbc_3c,&
                    sab_all=sab_all, &
                    sab_scp=sab_scp, & ! *** SCP
                    error=error)

    rho0_present=ASSOCIATED(rho0_mpole)

    CALL get_atomic_kind_set(atomic_kind_set=atomic_kind_set,&
                             gth_potential_present=gth_potential_present,&
                             paw_atom_present=paw_atom_present,&
                             all_potential_present=all_potential_present,&
                             maxatom=maxatom)

    ! Allocate work storage

    nkind = SIZE(atomic_kind_set)

    ALLOCATE (atom(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE (orb_present(nkind),default_present(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE (aux_fit_present(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE (aux_present(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    ALLOCATE (orb_radius(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    orb_radius(:) = 0.0_dp

    ALLOCATE (short_orb_radius(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    short_orb_radius(:) = 0.0_dp

    ALLOCATE (aux_fit_radius(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    aux_fit_radius(:) = 0.0_dp

    ALLOCATE (short_aux_fit_radius(nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    short_aux_fit_radius(:) = 0.0_dp

    IF (gth_potential_present) THEN
       ALLOCATE (ppl_present(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

       ALLOCATE (ppl_radius(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       ppl_radius = 0.0_dp

       ALLOCATE (ppnl_present(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

       ALLOCATE (ppnl_radius(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       ppnl_radius = 0.0_dp
    END IF

    IF (paw_atom_present) THEN
       ALLOCATE (oce_present(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

       ALLOCATE (oce_radius(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       oce_radius = 0.0_dp
    END IF

    IF (rho0_present) THEN
       CPPrecondition(ASSOCIATED(rhoz_set),cp_failure_level,routineP,error,failure)

       ALLOCATE (Qlm_present(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

!MI get rid of full gapw
       IF ( qs_env%rel_control%rel_method/=rel_none) THEN
         ALLOCATE (radius_3c(nkind),STAT=stat)
         CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
         radius_3c = 0.0_dp
         ALLOCATE (orb_rad_3c(nkind),STAT=stat)
         CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
         orb_rad_3c = 0.0_dp
       END IF
!MI get rid of full gapw
    END IF

    IF (all_potential_present) THEN
       ALLOCATE (all_present(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

       ALLOCATE (all_pot_rad(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       all_pot_rad = 0.0_dp
    END IF

    ! Initialize the local data structures

    maxatom_local = 0
    DO ikind=1,nkind
       atomic_kind => atomic_kind_set(ikind)
       NULLIFY (atom(ikind)%list)
       NULLIFY (atom(ikind)%list_local_a_index)
       NULLIFY (atom(ikind)%list_local_b_index)
       NULLIFY (atom(ikind)%list_a_mol)
       NULLIFY (atom(ikind)%list_b_mol)
       NULLIFY (atom(ikind)%r_pbc)
       NULLIFY (atom(ikind)%s_pbc)

       CALL get_atomic_kind(atomic_kind=atomic_kind,&
                            atom_list=atom(ikind)%list,&
                            all_potential=all_potential,&
                            gth_potential=gth_potential,&
                            orb_basis_set=orb_basis_set,&
                            aux_basis_set=aux_basis_set,&
                            aux_fit_basis_set=aux_fit_basis_set,&
                            paw_proj_set=paw_proj,&
                            paw_atom=paw_atom)
       
       natom_a = SIZE(atom(ikind)%list)

       IF ( molecule_only ) THEN
          natom_local_a = distribution_1d%n_el(ikind)
          natom_local_b = distribution_1d%n_el(ikind)
          local_rows_array => distribution_1d%list(ikind)%array
          local_cols_array => distribution_1d%list(ikind)%array
       ELSE
          natom_local_a = distribution_2d%n_local_rows(ikind)
          natom_local_b = distribution_2d%n_local_cols(ikind)
          local_rows_array => distribution_2d%local_rows(ikind)%array
          local_cols_array => distribution_2d%local_cols(ikind)%array
       END IF

       maxatom_local = MAX(maxatom_local,natom_local_a,natom_local_b)
       ! SCP
       IF (ASSOCIATED(aux_basis_set)) THEN
          aux_present(ikind) = .TRUE.
       ELSE
          aux_present(ikind) = .FALSE.
       ENDIF
       ! SCP

       IF (ASSOCIATED(orb_basis_set)) THEN
          orb_present(ikind) = .TRUE.

          CALL get_gto_basis_set(gto_basis_set=orb_basis_set,&
                                 kind_radius=orb_radius(ikind),&
                                 short_kind_radius=short_orb_radius(ikind))

       ELSE IF (qs_env%dft_control%qs_control%dftb) THEN
          ! Set the interaction radius for the neighbor lists (DFTB case)
          CALL get_atomic_kind(atomic_kind=atomic_kind,dftb_parameter=dftb_atom)
          CALL get_dftb_atom_param(dftb_parameter=dftb_atom,&
                                   cutoff=orb_radius(ikind),&
                                   defined=orb_present(ikind),error=error)
       ELSE
          orb_present(ikind) = .FALSE.
       END IF

       IF (ASSOCIATED(aux_fit_basis_set)) THEN
          aux_fit_present(ikind) = .TRUE.

          CALL get_gto_basis_set(gto_basis_set=aux_fit_basis_set,&
                                 kind_radius=aux_fit_radius(ikind),&
                                 short_kind_radius=short_aux_fit_radius(ikind))
       ELSE IF (qs_env%dft_control%qs_control%dftb) THEN
          ! Set the interaction radius for the neighbor lists (DFTB case)
          CALL get_atomic_kind(atomic_kind=atomic_kind,dftb_parameter=dftb_atom)
          CALL get_dftb_atom_param(dftb_parameter=dftb_atom,&
                                   cutoff=orb_radius(ikind),&
                                   defined=orb_present(ikind),error=error)
       ELSE
          aux_fit_present(ikind) = .FALSE.
       END IF

       IF ( orb_present(ikind) ) THEN
          ! Block rows
          IF (natom_local_a > 0) THEN
             ALLOCATE (atom(ikind)%list_local_a_index(natom_local_a),STAT=stat)
             CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
             ALLOCATE (atom(ikind)%list_a_mol(natom_local_a),STAT=stat)
             CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
             atom(ikind)%list_a_mol(:) = 0

             ! Build index vector for mapping
             DO iatom_local=1,natom_local_a
                atom_a = local_rows_array(iatom_local)
                DO iatom=1,natom_a
                   IF (atom_a == atom(ikind)%list(iatom)) THEN
                      atom(ikind)%list_local_a_index(iatom_local) = iatom
                      EXIT
                   END IF
                END DO
                IF ( molecule_only ) THEN
                   DO imol=1,SIZE(molecule_set)
                      IF ( atom_a >= molecule_set(imol)%first_atom .AND. &
                           atom_a <= molecule_set(imol)%last_atom ) THEN
                         atom(ikind)%list_a_mol(iatom_local) = imol
                         EXIT
                      END IF
                   END DO
                END IF
             END DO

          END IF

          ! Block columns
          IF (natom_local_b > 0) THEN

             ALLOCATE (atom(ikind)%list_local_b_index(natom_local_b),STAT=stat)
             CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
             ALLOCATE (atom(ikind)%list_b_mol(natom_local_b),STAT=stat)
             CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
             atom(ikind)%list_b_mol(:) = 0

             ! Build index vector for mapping
             DO iatom_local=1,natom_local_b
                atom_a = local_cols_array(iatom_local)
                DO iatom=1,natom_a
                   IF (atom_a == atom(ikind)%list(iatom)) THEN
                      atom(ikind)%list_local_b_index(iatom_local) = iatom
                      EXIT
                   END IF
                END DO
                IF (molecule_only) THEN
                   DO imol=1,SIZE(molecule_set)
                      IF ((atom_a >= molecule_set(imol)%first_atom).AND.&
                           (atom_a <= molecule_set(imol)%last_atom)) THEN
                         atom(ikind)%list_b_mol(iatom_local) = imol
                         EXIT
                      END IF
                   END DO
                END IF
             END DO

          END IF

          ALLOCATE (atom(ikind)%r_pbc(3,natom_a),STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
          ALLOCATE (atom(ikind)%s_pbc(3,natom_a),STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

          ! Calculate PBC coordinates
          DO iatom=1,natom_a
             atom_a = atom(ikind)%list(iatom)
             r_pbc(:) = pbc(particle_set(atom_a)%r(:),cell)
             atom(ikind)%r_pbc(:,iatom) = r_pbc(:)
             CALL real_to_scaled(atom(ikind)%s_pbc(:,iatom),r_pbc(:),cell)
          END DO

       ELSE

          orb_present(ikind) = .FALSE.

       END IF

       IF (gth_potential_present) THEN
          IF (ASSOCIATED(gth_potential)) THEN
             CALL get_potential(potential=gth_potential,&
                                ppl_present=ppl_present(ikind),&
                                ppl_radius=ppl_radius(ikind),&
                                ppnl_present=ppnl_present(ikind),&
                                ppnl_radius=ppnl_radius(ikind))
          ELSE
             ppl_present(ikind) = .FALSE.
             ppnl_present(ikind) = .FALSE.
          END IF
       END IF

       ! Check the presence of rho1_hard and rho1_soft

       IF (paw_atom_present) THEN
          IF (paw_atom) THEN
             oce_present(ikind) = .TRUE.
             CALL get_paw_proj_set(paw_proj_set=paw_proj,&
                                   rcprj=oce_radius(ikind))
          ELSE
             oce_present(ikind) = .FALSE.
          END IF
       END IF

       ! Check the presence of rho0_hard and rho0_soft

!MI get rid of full gapw
       IF ( qs_env%rel_control%rel_method/=rel_none) THEN
         IF(rho0_present) THEN
            IF(ABS(rhoz_set(ikind)%one_atom) > 1.0E-8_dp) THEN
               Qlm_present(ikind) = .TRUE.
               CALL get_rho0_mpole(rho0_mpole=rho0_mpole, ikind=ikind,&
                                     rpgf0_s=radius_3c(ikind))
              END IF
              ! These radii determine the extension of the 3c lists used for both
              ! the all electron core term 
              orb_rad_3c(ikind) = rho0_mpole%orb_radius_3c(ikind)
           END IF
       END IF
!MI get rid of full gapw

       ! Check the presence of an all electron potential

       IF (all_potential_present) THEN
          IF (ASSOCIATED(all_potential)) THEN
             all_present(ikind) = .TRUE.
             CALL get_potential(potential=all_potential,&
                                core_charge_radius=all_pot_rad(ikind))
          ELSE
             all_present(ikind) = .FALSE.
             all_pot_rad(ikind) = 0.0_dp
          END IF

       END IF

    END DO

    ! Build the orbital-orbital overlap neighbor lists
    IF (qs_env%dft_control%qs_control%semi_empirical) THEN
       ! we need strict minimum image convention for the interaction lists
       mic = .TRUE.
    ELSE IF (qs_env%dft_control%qs_control%dftb) THEN
       ! we need strict minimum image convention for the interaction lists
       mic = .TRUE.
    ELSE
       mic = .FALSE.
    END IF

    IF (molecule_only) THEN
       CALL build_restricted_neighbor_lists(sab_orb,atom,cell,r_skin,&
                                            maxatom_local,unit_conv,unit_str,&
                                            output_unit,orb_present,orb_present,&
                                            orb_radius,orb_radius,&
                                            name="ORBITAL ORBITAL",subcells=subcells,&
                                            error=error)
    ELSE
! MI get rid of full gapw
       IF(rho0_present .AND. qs_env%rel_control%rel_method/=rel_none) THEN
         CALL build_neighbor_lists(sab_orb,atom,cell,r_skin,maxatom_local,&
                                 unit_conv,unit_str,output_unit,&
                                 orb_present,orb_present,orb_radius,orb_radius,&
                                 name="ORBITAL ORBITAL",&
                                 l3c_oce=paw_atom_present,&
                                 l3c_rho0=rho0_present,&
                                 mic=mic,subcells=subcells,&
                                 error=error)
       ELSE
         CALL build_neighbor_lists(sab_orb,atom,cell,r_skin,maxatom_local,&
                                 unit_conv,unit_str,output_unit,&
                                 orb_present,orb_present,orb_radius,orb_radius,&
                                 name="ORBITAL ORBITAL",&
                                 mic=mic,subcells=subcells,&
                                 error=error)

       END IF
! MI get rid of full gapw

    END IF

    ! Put the generated neighbor lists into the QS environment
    CALL set_qs_env(qs_env=qs_env,sab_orb=sab_orb,error=error)

    ! Dump neighbor list if requested
    CALL write_neighbor_lists(sab_orb,particle_set,cell,para_env,neighbor_list_section,&
         "/SAB_ORB","sab_orb","ORBITAL ORBITAL",error)

    IF( qs_env%dft_control%do_admm ) THEN
       IF (molecule_only) THEN
          ! *** this needs to be checked
          CALL build_restricted_neighbor_lists(sab_aux_fit,atom,cell,r_skin,&
                                               maxatom_local,unit_conv,unit_str,&
                                               output_unit,aux_fit_present,aux_fit_present,&
                                               aux_fit_radius,aux_fit_radius,&
                                               name="AUX_FIT_ORBITAL AUX_FIT_ORBITAL",&
                                               subcells=subcells,&
                                               error=error)
       ELSE
          CALL build_neighbor_lists(sab_aux_fit,atom,cell,r_skin,maxatom_local,&
                                    unit_conv,unit_str,output_unit,&
                                    aux_fit_present,aux_fit_present,aux_fit_radius,aux_fit_radius,&
                                    name="AUX_FIT_ORBITAL AUX_FIT_ORBITAL",&
                                    mic=mic,subcells=subcells,&
                                    error=error)
          CALL build_neighbor_lists(sab_aux_fit_asymm,atom,cell,r_skin,maxatom_local,&
                                    unit_conv,unit_str,output_unit,&
                                    aux_fit_present,aux_fit_present,aux_fit_radius,aux_fit_radius,&
                                    name="AUX_FIT_ORBITAL AUX_FIT_ORBITAL_ASYMM",&
                                    mic=mic,&
                                    symmetric=.FALSE.,subcells=subcells,&
                                    error=error)
       END IF

       IF (molecule_only) THEN
          ! *** This needs to be checked
          CALL build_restricted_neighbor_lists(sab_aux_fit_vs_orb,atom,cell,r_skin,&
                                               maxatom_local,unit_conv,unit_str,&
                                               output_unit,aux_fit_present,orb_present,&
                                               aux_fit_radius,orb_radius,&
                                               name="AUX_FIT_ORBITAL AUX_FIT_ORBITAL",&
                                               subcells=subcells,&
                                               error=error)
       ELSE
          CALL build_neighbor_lists(sab_aux_fit_vs_orb,atom,cell,r_skin,maxatom_local,&
                                    unit_conv,unit_str,output_unit,&
                                    aux_fit_present,orb_present,aux_fit_radius,orb_radius,&
                                    name="ORBITAL AUX_FIT_ORBITAL",&
                                    mic=mic,&
                                    symmetric=.FALSE.,subcells=subcells,&
                                    error=error)

       END IF

       ! Put the generated neighbor lists into the QS environment
       CALL set_qs_env(qs_env=qs_env,sab_aux_fit=sab_aux_fit,error=error)
       CALL set_qs_env(qs_env=qs_env,sab_aux_fit_vs_orb=sab_aux_fit_vs_orb,error=error)
       CALL set_qs_env(qs_env=qs_env,sab_aux_fit_asymm=sab_aux_fit_asymm,error=error)

       ! Dump neighbor list if requested
       CALL write_neighbor_lists(sab_aux_fit,particle_set,cell,para_env,neighbor_list_section,&
            "/SAB_AUX_FIT","sab_aux_fit","AUX_FIT_ORBITAL AUX_FIT_ORBITAL",error)
       CALL write_neighbor_lists(sab_aux_fit_vs_orb,particle_set,cell,para_env,neighbor_list_section,&
            "/SAB_AUX_FIT_VS_ORB","sab_aux_fit_vs_orb","ORBITAL AUX_FIT_ORBITAL",error)
    END IF

    ! Build orbital-orbital list containing all the pairs, to be used with
    ! non-symmetric operators. Beware: the cutoff of the orbital-orbital overlap
    ! might be not optimal. It should be verified for each operator.
    IF(all_pairs_list) THEN
       CALL build_neighbor_lists(sab_all,atom,cell,r_skin,maxatom_local,&
                                 unit_conv,unit_str,output_unit,&
                                 orb_present,orb_present,orb_radius,orb_radius,&
                                 name="ALL_ORBITAL ORBITAL",&
                                 symmetric=.FALSE.,subcells=subcells,&
                                 error=error)
       ! Put the generated neighbor lists into the QS environment
       CALL set_qs_env(qs_env=qs_env,sab_all=sab_all,error=error)
    END IF

    ! Build orbital GTH-PPL operator overlap list

    IF (gth_potential_present) THEN
       IF (ANY(ppl_present)) THEN
          CALL build_gth_ppl_neighbor_lists(sac_ppl,atom,cell,r_skin,&
                                            maxatom_local,unit_conv,unit_str,&
                                            output_unit,orb_present,ppl_present,&
                                            orb_radius,ppl_radius,&
                                            name="ORBITAL GTH-PPL",&
                                            subcells=subcells,&
                                            error=error)

          ! Put the generated neighbor lists into the QS environment
          CALL set_qs_env(qs_env=qs_env,sac_ppl=sac_ppl,error=error)

          ! Dump neighbor list if requested
          CALL write_neighbor_lists(sac_ppl,particle_set,cell,para_env,neighbor_list_section,&
               "/SAC_PPL","sac_ppl","ORBITAL GTH-PPL",error)
       END IF

       IF (ANY(ppnl_present)) THEN
          CALL build_projector_neighbor_lists(sap_ppnl,atom,cell,r_skin,&
                                             maxatom,2*maxatom_local,unit_conv,unit_str,&
                                             output_unit,orb_present,ppnl_present,&
                                             orb_radius,ppnl_radius,&
                                             name="ORBITAL GTH-PPNL",&
                                             subcells=subcells,&
                                             error=error)

          ! Put the generated neighbor lists into the QS environment
          CALL set_qs_env(qs_env=qs_env,sap_ppnl=sap_ppnl,error=error)

          ! Dump neighbor list if requested
          CALL write_neighbor_lists(sap_ppnl,particle_set,cell,para_env,neighbor_list_section,&
               "/SAP_PPNL","sap_ppnl","ORBITAL GTH-PPNL",error)
       END IF
    END IF


    IF (paw_atom_present) THEN
       ! Build orbital-GAPW projector overlap list
       IF (ANY(oce_present)) THEN
          CALL build_projector_neighbor_lists(sap_oce,atom,cell,r_skin,&
                                             maxatom,2*maxatom_local,unit_conv,unit_str,&
                                             output_unit,orb_present,oce_present,&
                                             orb_radius,oce_radius,&
                                             name="ORBITAL PAW-PRJ",&
                                             subcells=subcells,&
                                             error=error)

          ! Put the generated neighbor lists into the QS environment
          CALL set_qs_env(qs_env=qs_env,sap_oce=sap_oce,error=error)

          ! Dump neighbor list if requested
          CALL write_neighbor_lists(sap_oce,particle_set,cell,para_env,neighbor_list_section,&
               "/SAP_OCE","sap_oce","ORBITAL(A) PAW-PRJ",error)

       END IF
    END IF

    ! Build orbital-ERFC potential list
    IF (all_potential_present) THEN
       CALL build_gth_ppl_neighbor_lists(sac_ae,atom,cell,r_skin,&
                                         maxatom_local,unit_conv,unit_str,&
                                         output_unit,orb_present,all_present,&
                                         orb_radius,all_pot_rad,&
                                         name="ORBITAL ERFC POTENTIAL",&
                                         subcells=subcells,&
                                         error=error)

       ! Put the generated neighbor lists into the QS environment
       CALL set_qs_env(qs_env=qs_env,sac_ae=sac_ae,error=error)

       ! Dump neighbor list if requested
       CALL write_neighbor_lists(sac_ae,particle_set,cell,para_env,neighbor_list_section,&
               "/SAC_AE","sac_ae","ORBITAL ERFC POTENTIAL",error)
     END IF

    IF (qs_env%dft_control%qs_control%semi_empirical) THEN
       ! Semi-empirical neighbor lists
       ALLOCATE (semi_present(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       ALLOCATE (c_radius(nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       semi_present = .TRUE.
       c_radius = qs_env%dft_control%qs_control%se_control%cutoff_cou

       ! Build the neighbor lists for the Hartree terms
       IF (qs_env%dft_control%qs_control%se_control%do_ewald_gks)  THEN
! Use MIC for the periodic code of GKS
         CALL build_neighbor_lists(sab_2c,atom,cell,r_skin,maxatom_local,&
                                   unit_conv,unit_str,output_unit,semi_present,&
                                   semi_present,c_radius,c_radius, mic=mic, &
                                   name="HARTREE INTERACTIONS",&
                                   subcells=subcells,&
                                   error=error)
       ELSE
         CALL build_neighbor_lists(sab_2c,atom,cell,r_skin,maxatom_local,&
                                   unit_conv,unit_str,output_unit,semi_present,&
                                   semi_present,c_radius,c_radius,&
                                   name="HARTREE INTERACTIONS",&
                                   subcells=subcells,&
                                   error=error)
       END IF
       CALL set_qs_env(qs_env=qs_env,sab_2c=sab_2c,error=error)

       ! Dump neighbor list if requested
       CALL write_neighbor_lists(sab_2c,particle_set,cell,para_env,neighbor_list_section,&
            "/SAB_2C","sab_2c","HARTREE INTERACTIONS",error)

       ! If requested build the SE long-range correction neighbor list 
       IF ((qs_env%dft_control%qs_control%se_control%do_ewald).AND.&
           (qs_env%dft_control%qs_control%se_control%integral_screening/=do_se_IS_slater)) THEN
          c_radius = qs_env%dft_control%qs_control%se_control%cutoff_lrc
          CALL build_neighbor_lists(sab_lrc,atom,cell,r_skin,maxatom_local,&
                                    unit_conv,unit_str,output_unit,semi_present,&
                                    semi_present,c_radius,c_radius,&
                                    name="SE LONG-RANGE CORRECTION",&
                                    subcells=subcells,error=error)
          CALL set_qs_env(qs_env=qs_env,sab_lrc=sab_lrc,error=error)
          
          ! Dump neighbor list if requested
          CALL write_neighbor_lists(sab_lrc,particle_set,cell,para_env,neighbor_list_section,&
               "/SAB_LRC","sab_lrc","SE LONG-RANGE CORRECTION",error)       
       END IF

       DEALLOCATE (semi_present,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       DEALLOCATE (c_radius,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ELSEIF (qs_env%dft_control%scp) THEN
       IF ( qs_env%dft_control%scp_control%dispersion ) THEN
          ALLOCATE (c_radius(nkind),STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
          DO ikind = 1, nkind
             c_radius(ikind) = 10.45_dp ! hard wired for now
          END DO

          CALL build_neighbor_lists(sab_scp,atom,cell,r_skin,maxatom_local,&
               unit_conv,unit_str,output_unit,aux_present,aux_present,&
               c_radius,c_radius,name="SCP_DISPERSION",subcells=subcells,&
               error=error)

          ! Put the generated neighbor lists into the QS environment
          CALL set_qs_env(qs_env=qs_env,sab_scp=sab_scp,error=error)

          ! Dump neighbor list if requested
          CALL write_neighbor_lists(sab_scp,particle_set,cell,para_env,neighbor_list_section,&
               "/SAB_SCP","sab_scp","SCP DISPERSION PAIR POTENTIAL",error)

          DEALLOCATE (c_radius,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF

    ELSEIF (qs_env%dft_control%qs_control%dftb) THEN

       ! Build the neighbor lists for the DFTB repulsive pair-potential
       CALL get_qs_env(qs_env=qs_env,&
            dftb_potential=dftb_potential,error=error)
       ALLOCATE(ab_radius(nkind,nkind),STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       DO ikind=1,nkind
          DO jkind=1,ikind
             IF ( dftb_potential(ikind,jkind)%n_urpoly > 0 .OR. &
                  dftb_potential(ikind,jkind)%spdim > 0 ) THEN
                ab_radius(ikind,jkind) = dftb_potential(ikind,jkind)%urep_cut
                ab_radius(jkind,ikind) = dftb_potential(ikind,jkind)%urep_cut
             ELSE
                ab_radius(ikind,jkind) = 0._dp
                ab_radius(jkind,ikind) = 0._dp
             END IF
          END DO
       END DO
       mic=.TRUE.
       CALL build_neighbor_lists(sab_2c,atom,cell,r_skin,maxatom_local,&
                                 unit_conv,unit_str,output_unit,orb_present,orb_present,&
                                 orb_radius,orb_radius,pair_radius=ab_radius,&
                                 mic=mic,name="DFTB REPULSIVE PAIR POTENTIAL",&
                                 subcells=subcells,error=error)
       ! Put the generated neighbor lists into the QS environment
       CALL set_qs_env(qs_env=qs_env,sab_2c=sab_2c,error=error)

       ! Dump neighbor list if requested
       CALL write_neighbor_lists(sab_2c,particle_set,cell,para_env,neighbor_list_section,&
            "/SAB_2C","sab_2c","DFTB REPULSIVE PAIR POTENTIAL",error)

       DEALLOCATE(ab_radius,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

       ! Build the neighbor lists for the DFTB Ewald methods
       IF ( qs_env%dft_control%qs_control%dftb_control%do_ewald ) THEN
          ALLOCATE (c_radius(nkind),STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

          CALL get_qs_env(qs_env=qs_env,ewald_env=ewald_env,error=error)
          CALL ewald_env_get ( ewald_env, alpha=alpha )
          c_radius = 0.5_dp*SQRT(-LOG(3.5_dp*alpha**3*1.e-12_dp))/alpha
          CALL build_neighbor_lists(sac_3c,atom,cell,r_skin,maxatom_local,&
               unit_conv,unit_str,output_unit,orb_present,orb_present,&
               c_radius,c_radius,mic=mic,name="EWALD OVERLAP",&
               subcells=subcells,error=error)
          ! Put the generated neighbor lists into the QS environment
          CALL set_qs_env(qs_env=qs_env,sac_3c=sac_3c,error=error)
          DEALLOCATE (c_radius,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF

       ! Build the neighbor lists for the DFTB vdW pair potential
       IF ( qs_env%dft_control%qs_control%dftb_control%dispersion ) THEN
          ALLOCATE (c_radius(nkind),STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
          DO ikind = 1, nkind
             atomic_kind => atomic_kind_set(ikind)
             CALL get_atomic_kind(atomic_kind=atomic_kind,dftb_parameter=dftb_atom)
             CALL get_dftb_atom_param(dftb_parameter=dftb_atom,&
                                      rcdisp=c_radius(ikind),error=error)
          END DO

          default_present=.TRUE.
          CALL build_neighbor_lists(sab_vdw,atom,cell,r_skin,maxatom_local,&
               unit_conv,unit_str,output_unit,default_present,default_present,&
               c_radius,c_radius,name="DISPERSION",subcells=subcells,&
               error=error)
          ! Put the generated neighbor lists into the QS environment
          CALL set_qs_env(qs_env=qs_env,sab_vdw=sab_vdw,error=error)

          DEALLOCATE (c_radius,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF

    ELSE

       ! Build the neighbor lists for the vdW pair potential
       CALL get_qs_env(qs_env=qs_env,dispersion_env=dispersion_env,error=error)
       IF ( dispersion_env%type == xc_vdw_fun_pairpot ) THEN
          ALLOCATE (c_radius(nkind),STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
          c_radius(:) = dispersion_env%rc_disp
          default_present=.TRUE. !include all atoms in vdW (even without basis)

          CALL build_neighbor_lists(sab_vdw,atom,cell,r_skin,maxatom_local,&
               unit_conv,unit_str,output_unit,default_present,default_present,&
               c_radius,c_radius,name="DISPERSION",subcells=subcells,&
               error=error)
          ! Put the generated neighbor lists into the QS environment
          CALL set_qs_env(qs_env=qs_env,sab_vdw=sab_vdw,error=error)

          IF ( TRIM(dispersion_env%pp_type) == "DFTD3" ) THEN
             ! Build the neighbor lists for coordination numbers as needed by the DFT-D3 method
             DO ikind = 1, nkind
                atomic_kind => atomic_kind_set(ikind)
                CALL get_atomic_kind(atomic_kind=atomic_kind,z=zat)
                c_radius(ikind) = 4._dp*ptable(zat)%covalent_radius*bohr
             END DO
             CALL build_neighbor_lists(sab_cn,atom,cell,0._dp,maxatom_local,&
                unit_conv,unit_str,output_unit,default_present,default_present,&
                c_radius,c_radius,name="COORDINATION",subcells=subcells,&
                error=error)
             ! Put the generated neighbor lists into the QS environment
             CALL set_qs_env(qs_env=qs_env,sab_cn=sab_cn,error=error)
          END IF

          DEALLOCATE (c_radius,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF

    END IF

! MI get rid of full GAPW
    IF (rho0_present) THEN
       IF ( qs_env%rel_control%rel_method/=rel_none) THEN
         IF (all_potential_present) THEN
            DO ikind=1,nkind
               radius_3c(ikind) = MAX(radius_3c(ikind),all_pot_rad(ikind))
            END DO
         END IF
!       ! orb_rad_3c is the max distance from the A and B centers to have
!       ! a not negligible interaction of the density with the potential
!       ! Zc erfc(zetc|r-Rc|)/|r-Rc|
         CALL build_gth_ppnl_neighbor_lists(sac_3c,sbc_3c,atom,cell,r_skin,&
                                          maxatom,maxatom_local,unit_conv,unit_str,&
                                          output_unit,orb_present,Qlm_present,&
                                          orb_rad_3c,radius_3c,&
                                          name="THREE-CENTER ORBITAL V_HARTREE_LOCAL",&
                                          subcells=subcells,error=error)

         ! Put the generated neighbor lists into the QS environment
         CALL set_qs_env(qs_env=qs_env,sac_3c=sac_3c,sbc_3c=sbc_3c,error=error)
         ! Dump neighbor list if requested
         CALL write_neighbor_lists(sac_3c,particle_set,cell,para_env,neighbor_list_section,&
              "/SAC_3C","sac_3c","THREE-CENTER ORBITAL(A) V_HARTREE_LOCAL",error)
         CALL write_neighbor_lists(sbc_3c,particle_set,cell,para_env,neighbor_list_section,&
              "/SBC_3C","sbc_3c","THREE-CENTER ORBITAL(B) V_HARTREE_LOCAL",error)
       END IF  ! rel_method
    END IF
! MI get rid of full GAPW

    ! Print particle distribution
    print_key_path = "PRINT%DISTRIBUTION"
    IF (BTEST(cp_print_key_should_output(logger%iter_info,force_env_section,&
         print_key_path,error=error),&
         cp_p_file)) THEN
       iw = cp_print_key_unit_nr(logger=logger,&
                                 basis_section=force_env_section,&
                                 print_key_path=print_key_path,&
                                 extension=".out",&
                                 error=error)
       CALL write_neighbor_distribution(sab_orb,atomic_kind_set,iw,para_env,error)
       CALL cp_print_key_finished_output(unit_nr=iw,&
                                         logger=logger,&
                                         basis_section=force_env_section,&
                                         print_key_path=print_key_path,&
                                         error=error)
    END IF

    ! Release work storage
    DO ikind=1,nkind
       NULLIFY (atom(ikind)%list)
       IF (ASSOCIATED(atom(ikind)%list_local_a_index)) THEN
          DEALLOCATE (atom(ikind)%list_local_a_index,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
       IF (ASSOCIATED(atom(ikind)%list_local_b_index)) THEN
          DEALLOCATE (atom(ikind)%list_local_b_index,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
       IF (ASSOCIATED(atom(ikind)%list_a_mol)) THEN
          DEALLOCATE (atom(ikind)%list_a_mol,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
       IF (ASSOCIATED(atom(ikind)%list_b_mol)) THEN
          DEALLOCATE (atom(ikind)%list_b_mol,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
       IF (ASSOCIATED(atom(ikind)%r_pbc)) THEN
          DEALLOCATE (atom(ikind)%r_pbc,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
       IF (ASSOCIATED(atom(ikind)%s_pbc)) THEN
          DEALLOCATE (atom(ikind)%s_pbc,STAT=stat)
          CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
    END DO

    DEALLOCATE (atom,STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    DEALLOCATE (orb_present,default_present,STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    DEALLOCATE (orb_radius,STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    DEALLOCATE (short_orb_radius,STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    IF (gth_potential_present) THEN
       DEALLOCATE (ppl_present,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       DEALLOCATE (ppl_radius,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       DEALLOCATE (ppnl_present,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       DEALLOCATE (ppnl_radius,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

    IF (paw_atom_present) THEN
       DEALLOCATE (oce_present,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       DEALLOCATE (oce_radius,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ENDIF

    IF (rho0_present) THEN
       IF(ALLOCATED(radius_3c)) THEN
         DEALLOCATE (radius_3c,STAT=stat)
         CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
       IF(ALLOCATED(orb_rad_3c)) THEN
         DEALLOCATE(orb_rad_3c,STAT=stat)
         CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       END IF
    END IF

    IF (all_potential_present) THEN
       DEALLOCATE (all_present,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
       DEALLOCATE (all_pot_rad,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

    CALL cp_print_key_finished_output(unit_nr=output_unit,&
                                      logger=logger,&
                                      basis_section=force_env_section,&
                                      print_key_path="DFT%PRINT%SUBCELL",&
                                      error=error)
    CALL timestop(handle)

  END SUBROUTINE build_qs_neighbor_lists

! *****************************************************************************
!> \brief   Build simple pair neighbor lists. 
!> \author  MK
!> \date    20.03.2002
!> \version 1.0
! *****************************************************************************
  SUBROUTINE build_neighbor_lists(ab,atom,cell,r_skin,maxatom_local, unit_conv,&
       unit_str, output_unit, present_a,present_b,radius_a,radius_b, pair_radius,&
       name,l3c_oce,l3c_rho0,mic,symmetric,subcells,error)

    TYPE(neighbor_list_set_p_type), &
      DIMENSION(:), POINTER                  :: ab
    TYPE(local_atoms_type), DIMENSION(:), &
      INTENT(IN)                             :: atom
    TYPE(cell_type), POINTER                 :: cell
    REAL(dp), INTENT(IN)                     :: r_skin
    INTEGER, INTENT(IN)                      :: maxatom_local
    REAL(dp), INTENT(IN)                     :: unit_conv
    CHARACTER(LEN=*), INTENT(IN)             :: unit_str
    INTEGER, INTENT(IN)                      :: output_unit
    LOGICAL, DIMENSION(:), INTENT(IN)        :: present_a, present_b
    REAL(dp), DIMENSION(:), INTENT(IN)       :: radius_a, radius_b
    REAL(dp), DIMENSION(:, :), INTENT(IN), &
      OPTIONAL                               :: pair_radius
    CHARACTER(LEN=*), INTENT(IN)             :: name
    LOGICAL, INTENT(IN), OPTIONAL            :: l3c_oce, l3c_rho0, mic, &
                                                symmetric
    REAL(dp), INTENT(IN)                     :: subcells
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'build_neighbor_lists', &
      routineP = moduleN//':'//routineN

    INTEGER :: atom_a, atom_b, handle, i, iab, iatom, iatom_local, &
      iatom_subcell, icell, ikind, j, jatom, jatom_local, jcell, jkind, k, &
      kcell, natom_local_a, natom_local_b, nkind, stat
    INTEGER, DIMENSION(3)                    :: cell_a, cell_b, ncell, &
                                                nsubcell, periodic
    LOGICAL                                  :: failure, include_ab, &
                                                my_l3c_oce, my_l3c_rho0, &
                                                my_mic, my_symmetric
    REAL(dp)                                 :: rab2, rab2_max, rab_max
    REAL(dp), DIMENSION(3)                   :: r, rab, rb, s, sab_max, sb, &
                                                sb_max, sb_min, sb_pbc
    TYPE(neighbor_list_p_type), &
      DIMENSION(maxatom_local)               :: kind_a
    TYPE(neighbor_list_set_type), POINTER    :: neighbor_list_set
    TYPE(subcell_type), DIMENSION(:, :, :), &
      POINTER                                :: subcell

    CALL timeset(routineN,handle)
    failure = .FALSE.
    ! Deallocate the old neighbor list structure
    IF (ASSOCIATED(ab)) THEN
       DO iab=1,SIZE(ab)
          CALL deallocate_neighbor_list_set(ab(iab)%neighbor_list_set)
       END DO
       DEALLOCATE (ab,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

    ! Allocate and initialize the new neighbor list structure

    nkind = SIZE(atom)
    my_l3c_oce = .FALSE.
    IF (PRESENT(l3c_oce)) my_l3c_oce = l3c_oce
    my_l3c_rho0 = .FALSE.
    IF (PRESENT(l3c_rho0)) my_l3c_rho0 = l3c_rho0
    my_mic = .FALSE.
    IF (PRESENT(mic)) my_mic = mic
    my_symmetric = .TRUE.
    IF (PRESENT(symmetric)) my_symmetric = symmetric

    ALLOCATE (ab(nkind*nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    DO iab=1,SIZE(ab)
      NULLIFY (ab(iab)%neighbor_list_set)
    END DO

    ! Load informations about the simulation cell
    CALL get_cell(cell=cell,&
                  periodic=periodic)

    ! Print headline
    IF (output_unit>0) THEN
      WRITE (UNIT=output_unit,FMT="(/,/,T2,A,/,/,T3,A,T29,A,T54,A)")&
        "SUBCELL GRID FOR THE "//TRIM(name)//" NEIGHBOR LISTS",&
        "Atomic kind pair","Grid size",&
        "Subcell size in "//TRIM(unit_str)
    END IF

    ! Loop over all atomic kind pairs
    DO ikind=1,nkind

      IF (.NOT.present_a(ikind)) CYCLE

      IF (.NOT.ASSOCIATED(atom(ikind)%list_local_a_index)) CYCLE

      natom_local_a = SIZE(atom(ikind)%list_local_a_index)

      DO jkind=1,nkind

        IF (.NOT.present_b(jkind)) CYCLE

        iab = ikind + nkind*(jkind - 1)

        IF (.NOT.ASSOCIATED(atom(jkind)%list_local_b_index)) CYCLE

        natom_local_b = SIZE(atom(jkind)%list_local_b_index)

        ! Calculate the square of the maximum interaction distance
        IF ( PRESENT(pair_radius) ) THEN
          IF ( pair_radius(ikind,jkind) <= 0._dp ) CYCLE
          rab_max = pair_radius(ikind,jkind) + r_skin
        ELSE
          rab_max = radius_a(ikind) + radius_b(jkind) + r_skin
        END IF
        rab2_max = rab_max*rab_max

        ! some kinds, like basis none kinds have no radius and no associated pair lists
        IF (rab2_max == 0.0_dp) CYCLE 

        sab_max(1) = rab_max/plane_distance(1,0,0,cell)
        sab_max(2) = rab_max/plane_distance(0,1,0,cell)
        sab_max(3) = rab_max/plane_distance(0,0,1,cell)

        IF ( my_mic ) THEN
          ncell(:) = 0
          periodic = 0
        ELSE
          ncell(:) = (INT(sab_max(:)) + 1)*periodic(:)
        END IF
        nsubcell(:) = MIN(MAX(1,NINT(0.5_dp*subcells/sab_max(:))),20)

        ! Print subcell information for the current atomic kind pair
        IF (output_unit>0) THEN
          s(:) = 1.0_dp/REAL(nsubcell(:),KIND=dp)
          CALL scaled_to_real(r,s,cell)
          WRITE (UNIT=output_unit,FMT="(T3,2I8,4X,3I5,6X,3F12.6)")&
            ikind,jkind,nsubcell(1:3),r(1:3)*unit_conv
        END IF

        CALL allocate_neighbor_list_set(neighbor_list_set=ab(iab)%neighbor_list_set,&
                                        r_max=rab_max)
        neighbor_list_set => ab(iab)%neighbor_list_set

        cell_a = (/0,0,0/)

        ! Check, if we have to consider a subcell grid
        IF (SUM(nsubcell) == 3) THEN

          ! Case 1: No subcell grid is needed
          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            CALL add_neighbor_list(neighbor_list_set=neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO jatom_local=1,natom_local_b

            jatom = atom(jkind)%list_local_b_index(jatom_local)
            atom_b = atom(jkind)%list(jatom)
            sb_pbc(:) = atom(jkind)%s_pbc(:,jatom)

            loop1_kcell: DO kcell=-ncell(3),ncell(3)

              sb(3) = sb_pbc(3) + REAL(kcell,dp)
              sb_min(3) = sb(3) - sab_max(3)
              sb_max(3) = sb(3) + sab_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sb_min(3) >= 0.5_dp) EXIT loop1_kcell
                IF (sb_max(3) < -0.5_dp) CYCLE loop1_kcell
              END IF
              cell_b(3) = kcell

              loop1_jcell: DO jcell=-ncell(2),ncell(2)

                sb(2) = sb_pbc(2) + REAL(jcell,dp)
                sb_min(2) = sb(2) - sab_max(2)
                sb_max(2) = sb(2) + sab_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sb_min(2) >= 0.5_dp) EXIT loop1_jcell
                  IF (sb_max(2) < -0.5_dp) CYCLE loop1_jcell
                END IF
                cell_b(2) = jcell

                loop1_icell: DO icell=-ncell(1),ncell(1)

                  sb(1) = sb_pbc(1) + REAL(icell,dp)
                  sb_min(1) = sb(1) - sab_max(1)
                  sb_max(1) = sb(1) + sab_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sb_min(1) >= 0.5_dp) EXIT loop1_icell
                    IF (sb_max(1) < -0.5_dp) CYCLE loop1_icell
                  END IF
                  cell_b(1) = icell

                  CALL scaled_to_real(rb,sb(:),cell)

                  DO iatom_local=1,natom_local_a
                    iatom = atom(ikind)%list_local_a_index(iatom_local)
                    atom_a = atom(ikind)%list(iatom)
                    IF (my_symmetric) THEN
                      IF (atom_a > atom_b) THEN
                        include_ab = (MODULO(atom_a + atom_b,2) /= 0)
                      ELSE
                        include_ab = (MODULO(atom_a + atom_b,2) == 0)
                      END IF
                    ELSE
                      include_ab = .TRUE.
                    END IF
                    IF (include_ab) THEN
                      rab(:) = rb(:) - atom(ikind)%r_pbc(:,iatom)
                      IF ( my_mic ) rab(:) = pbc(rab(:),cell)
                      rab2 = rab(1)*rab(1) + rab(2)*rab(2) + rab(3)*rab(3)
                      IF (rab2 < rab2_max) THEN
                        CALL add_neighbor_node(&
                          neighbor_list=kind_a(iatom_local)%neighbor_list,&
                          neighbor=atom_b,&
                          cell=cell_b,&
                          r=rab,&
                          l3c_oce=my_l3c_oce,&
                          l3c_rho0=my_l3c_rho0,&
                          nkind=nkind)
                      END IF
                    END IF
                  END DO

                END DO loop1_icell
              END DO loop1_jcell
            END DO loop1_kcell

          END DO

        ELSE

          ! Case 2: A subcell grid is needed
          CALL allocate_subcell(subcell,nsubcell,maxatom_local,error=error)

          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            s(:) = atom(ikind)%s_pbc(:,iatom) + 0.5_dp
            i = INT(s(1)*REAL(nsubcell(1),dp)) + 1
            j = INT(s(2)*REAL(nsubcell(2),dp)) + 1
            k = INT(s(3)*REAL(nsubcell(3),dp)) + 1
            i = MIN(MAX(i,1),nsubcell(1))
            j = MIN(MAX(j,1),nsubcell(2))
            k = MIN(MAX(k,1),nsubcell(3))
            subcell(i,j,k)%natom =  subcell(i,j,k)%natom + 1
            subcell(i,j,k)%atom_list(subcell(i,j,k)%natom) = iatom_local
            CALL add_neighbor_list(neighbor_list_set=neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO jatom_local=1,natom_local_b

            jatom = atom(jkind)%list_local_b_index(jatom_local)
            atom_b = atom(jkind)%list(jatom)
            sb_pbc(:) = atom(jkind)%s_pbc(:,jatom)

            loop2_kcell: DO kcell=-ncell(3),ncell(3)

              sb(3) = sb_pbc(3) + REAL(kcell,dp)
              sb_min(3) = sb(3) - sab_max(3)
              sb_max(3) = sb(3) + sab_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sb_min(3) >= 0.5_dp) EXIT loop2_kcell
                IF (sb_max(3) < -0.5_dp) CYCLE loop2_kcell
              END IF
              cell_b(3) = kcell

              loop2_jcell: DO jcell=-ncell(2),ncell(2)

                sb(2) = sb_pbc(2) + REAL(jcell,dp)
                sb_min(2) = sb(2) - sab_max(2)
                sb_max(2) = sb(2) + sab_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sb_min(2) >= 0.5_dp) EXIT loop2_jcell
                  IF (sb_max(2) < -0.5_dp) CYCLE loop2_jcell
                END IF
                cell_b(2) = jcell

                loop2_icell: DO icell=-ncell(1),ncell(1)

                  sb(1) = sb_pbc(1) + REAL(icell,dp)
                  sb_min(1) = sb(1) - sab_max(1)
                  sb_max(1) = sb(1) + sab_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sb_min(1) >= 0.5_dp) EXIT loop2_icell
                    IF (sb_max(1) < -0.5_dp) CYCLE loop2_icell
                  END IF
                  cell_b(1) = icell

                  CALL scaled_to_real(rb,sb,cell)

                  loop_k: DO k=1,nsubcell(3)
                    loop_j: DO j=1,nsubcell(2)
                      loop_i: DO i=1,nsubcell(1)

                        IF (periodic(3) /= 0) THEN
                          IF (sb_max(3) < subcell(i,j,k)%s_min(3)) EXIT loop_k
                          IF (sb_min(3) >= subcell(i,j,k)%s_max(3)) CYCLE loop_k
                        END IF

                        IF (periodic(2) /= 0) THEN
                          IF (sb_max(2) < subcell(i,j,k)%s_min(2)) EXIT loop_j
                          IF (sb_min(2) >= subcell(i,j,k)%s_max(2)) CYCLE loop_j
                        END IF

                        IF (periodic(1) /= 0) THEN
                          IF (sb_max(1) < subcell(i,j,k)%s_min(1)) EXIT loop_i
                          IF (sb_min(1) >= subcell(i,j,k)%s_max(1)) CYCLE loop_i
                        END IF

                        IF (subcell(i,j,k)%natom == 0) CYCLE

                        DO iatom_subcell=1,subcell(i,j,k)%natom
                          iatom_local = subcell(i,j,k)%atom_list(iatom_subcell)
                          iatom = atom(ikind)%list_local_a_index(iatom_local)
                          atom_a = atom(ikind)%list(iatom)
                           IF (my_symmetric) THEN
                             IF (atom_a > atom_b) THEN
                               include_ab = (MODULO(atom_a + atom_b,2) /= 0)
                             ELSE
                               include_ab = (MODULO(atom_a + atom_b,2) == 0)
                             END IF
                           ELSE
                             include_ab = .TRUE.
                           END IF
                          IF (include_ab) THEN
                            rab(:) = rb(:) - atom(ikind)%r_pbc(:,iatom)
                            IF ( my_mic ) rab(:) = pbc(rab(:),cell)
                            rab2 = rab(1)*rab(1) + rab(2)*rab(2) + rab(3)*rab(3)
                            IF (rab2 < rab2_max) THEN
                              CALL add_neighbor_node(&
                                neighbor_list=kind_a(iatom_local)%neighbor_list,&
                                neighbor=atom_b,&
                                cell=cell_b,&
                                r=rab,&
                                l3c_oce=my_l3c_oce,&
                                l3c_rho0=my_l3c_rho0,&
                                nkind=nkind)
                            END IF
                          END IF
                        END DO

                      END DO loop_i
                    END DO loop_j
                  END DO loop_k

                END DO loop2_icell
              END DO loop2_jcell
            END DO loop2_kcell

          END DO

          CALL deallocate_subcell(subcell,error=error)

        END IF

      END DO
    END DO

    CALL timestop(handle)

  END SUBROUTINE build_neighbor_lists

! *****************************************************************************
!> \brief   Build simple pair neighbor lists restricted to molecules.
!> \version 1.0
! *****************************************************************************
  SUBROUTINE build_restricted_neighbor_lists(ab,atom,cell,r_skin,&
    maxatom_local,unit_conv,unit_str, output_unit,present_a,present_b,&
    radius_a,radius_b,name,subcells,error)
    TYPE(neighbor_list_set_p_type), &
      DIMENSION(:), POINTER                  :: ab
    TYPE(local_atoms_type), DIMENSION(:), &
      INTENT(IN)                             :: atom
    TYPE(cell_type), POINTER                 :: cell
    REAL(dp), INTENT(IN)                     :: r_skin
    INTEGER, INTENT(IN)                      :: maxatom_local
    REAL(dp), INTENT(IN)                     :: unit_conv
    CHARACTER(LEN=*), INTENT(IN)             :: unit_str
    INTEGER, INTENT(IN)                      :: output_unit
    LOGICAL, DIMENSION(:), INTENT(IN)        :: present_a, present_b
    REAL(dp), DIMENSION(:), INTENT(IN)       :: radius_a, radius_b
    CHARACTER(LEN=*), INTENT(IN)             :: name
    REAL(dp), INTENT(IN)                     :: subcells
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: &
      routineN = 'build_restricted_neighbor_lists', &
      routineP = moduleN//':'//routineN

    INTEGER :: atom_a, atom_b, handle, i, iab, iatom, iatom_local, &
      iatom_subcell, ikind, j, jatom, jatom_local, jkind, k, mol_a, mol_b, &
      natom_local_a, natom_local_b, nkind, stat
    INTEGER, DIMENSION(3)                    :: cell_a, cell_b, ncell, &
                                                nsubcell, periodic
    LOGICAL                                  :: failure, include_ab
    REAL(dp)                                 :: rab2, rab2_max, rab_max
    REAL(dp), DIMENSION(3)                   :: r, rab, rb, s, sab_max, sb, &
                                                sb_max, sb_min
    TYPE(neighbor_list_p_type), &
      DIMENSION(maxatom_local)               :: kind_a
    TYPE(neighbor_list_set_type), POINTER    :: neighbor_list_set
    TYPE(subcell_type), DIMENSION(:, :, :), &
      POINTER                                :: subcell

    CALL timeset(routineN,handle)
    failure = .FALSE.
    ! Deallocate the old neighbor list structure
    IF (ASSOCIATED(ab)) THEN
      DO iab=1,SIZE(ab)
        CALL deallocate_neighbor_list_set(ab(iab)%neighbor_list_set)
      END DO
      DEALLOCATE (ab,STAT=stat)
      CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

   ! Allocate and initialize the new neighbor list structure
    nkind = SIZE(atom)

    ALLOCATE (ab(nkind*nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    DO iab=1,SIZE(ab)
      NULLIFY (ab(iab)%neighbor_list_set)
    END DO

    ! Load informations about the simulation cell
    CALL get_cell(cell=cell,periodic=periodic)

    ! Print headline
    IF (output_unit>0) THEN
      WRITE (UNIT=output_unit,FMT="(/,/,T2,A,/,/,T3,A,T29,A,T54,A)")&
        "SUBCELL GRID FOR THE "//TRIM(name)//" NEIGHBOR LISTS",&
        "Atomic kind pair","Grid size",&
        "Subcell size in "//TRIM(unit_str)
    END IF

    ! Loop over all atomic kind pairs
    DO ikind=1,nkind

      IF (.NOT.present_a(ikind)) CYCLE

      IF (.NOT.ASSOCIATED(atom(ikind)%list_local_a_index)) CYCLE

      natom_local_a = SIZE(atom(ikind)%list_local_a_index)

      DO jkind=1,nkind

        IF (.NOT.present_b(jkind)) CYCLE

        iab = ikind + nkind*(jkind - 1)

        IF (.NOT.ASSOCIATED(atom(jkind)%list_local_b_index)) CYCLE

        natom_local_b = SIZE(atom(jkind)%list_local_b_index)

        ! Calculate the square of the maximum interaction distance

        rab_max = radius_a(ikind) + radius_b(jkind) + r_skin
        rab2_max = rab_max*rab_max

        sab_max(1) = rab_max/plane_distance(1,0,0,cell)
        sab_max(2) = rab_max/plane_distance(0,1,0,cell)
        sab_max(3) = rab_max/plane_distance(0,0,1,cell)

        ncell(:) = 0
        nsubcell(:) = MIN(MAX(1,NINT(0.5_dp*subcells/sab_max(:))),20)

        ! Print subcell information for the current atomic kind pair

        IF (output_unit>0) THEN
          s(:) = 1.0_dp/REAL(nsubcell(:),KIND=dp)
          CALL scaled_to_real(r,s,cell)
          WRITE (UNIT=output_unit,FMT="(T3,2I8,4X,3I5,6X,3F12.6)")&
            ikind,jkind,nsubcell(1:3),r(1:3)*unit_conv
        END IF

        CALL allocate_neighbor_list_set(neighbor_list_set=ab(iab)%neighbor_list_set,&
                                        r_max=rab_max)
        neighbor_list_set => ab(iab)%neighbor_list_set

        cell_a = (/0,0,0/)

        ! Check, if we have to consider a subcell grid

        IF (SUM(nsubcell) == 3) THEN

          ! Case 1: No subcell grid is needed

          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            CALL add_neighbor_list(neighbor_list_set=neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO jatom_local=1,natom_local_b

            jatom = atom(jkind)%list_local_b_index(jatom_local)
            atom_b = atom(jkind)%list(jatom)
            mol_b = atom(jkind)%list_b_mol(jatom_local)
            sb(:) = atom(jkind)%s_pbc(:,jatom)
            CALL scaled_to_real(rb,sb,cell)

            DO iatom_local=1,natom_local_a
              iatom = atom(ikind)%list_local_a_index(iatom_local)
              atom_a = atom(ikind)%list(iatom)
              mol_a = atom(ikind)%list_a_mol(iatom_local)
              IF (mol_a /= mol_b) CYCLE
              IF (atom_a > atom_b) THEN
                include_ab = (MODULO(atom_a + atom_b,2) /= 0)
              ELSE
                include_ab = (MODULO(atom_a + atom_b,2) == 0)
              END IF
              IF (include_ab) THEN
                rab(:) = rb(:) - atom(ikind)%r_pbc(:,iatom)
                r(:) = pbc(rab(:),cell)
                rab(:) = r(:)
                rab2 = r(1)*r(1) + r(2)*r(2) + r(3)*r(3)
                IF (rab2 < rab2_max) THEN
                  r(:) = atom(ikind)%r_pbc(:,iatom)+rab(:)
                  CALL real_to_scaled(sb,r,cell)
                  cell_b = NINT(sb)
                  CALL add_neighbor_node(&
                    neighbor_list=kind_a(iatom_local)%neighbor_list,&
                    neighbor=atom_b,cell=cell_b,r=rab)
                END IF
              END IF
            END DO

          END DO

        ELSE

          ! Case 2: A subcell grid is needed

          CALL allocate_subcell(subcell,nsubcell,maxatom_local,error=error)

          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            s(:) = atom(ikind)%s_pbc(:,iatom) + 0.5_dp
            i = INT(s(1)*REAL(nsubcell(1),KIND=dp)) + 1
            j = INT(s(2)*REAL(nsubcell(2),KIND=dp)) + 1
            k = INT(s(3)*REAL(nsubcell(3),KIND=dp)) + 1
            i = MIN(MAX(i,1),nsubcell(1))
            j = MIN(MAX(j,1),nsubcell(2))
            k = MIN(MAX(k,1),nsubcell(3))
            subcell(i,j,k)%natom =  subcell(i,j,k)%natom + 1
            subcell(i,j,k)%atom_list(subcell(i,j,k)%natom) = iatom_local
            CALL add_neighbor_list(neighbor_list_set=neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO jatom_local=1,natom_local_b

            jatom = atom(jkind)%list_local_b_index(jatom_local)
            atom_b = atom(jkind)%list(jatom)
            mol_b = atom(jkind)%list_b_mol(jatom_local)
            sb(:) = atom(jkind)%s_pbc(:,jatom)
            CALL scaled_to_real(rb,sb,cell)
            sb_min(:) = sb(:) - sab_max(:)
            sb_max(:) = sb(:) + sab_max(:)

            loop_k: DO k=1,nsubcell(3)
              loop_j: DO j=1,nsubcell(2)
                loop_i: DO i=1,nsubcell(1)

                  IF (periodic(3) /= 0) THEN
                    IF (sb_max(3) < subcell(i,j,k)%s_min(3)) EXIT loop_k
                    IF (sb_min(3) >= subcell(i,j,k)%s_max(3)) CYCLE loop_k
                  END IF

                  IF (periodic(2) /= 0) THEN
                    IF (sb_max(2) < subcell(i,j,k)%s_min(2)) EXIT loop_j
                    IF (sb_min(2) >= subcell(i,j,k)%s_max(2)) CYCLE loop_j
                  END IF

                  IF (periodic(1) /= 0) THEN
                    IF (sb_max(1) < subcell(i,j,k)%s_min(1)) EXIT loop_i
                    IF (sb_min(1) >= subcell(i,j,k)%s_max(1)) CYCLE loop_i
                  END IF

                  IF (subcell(i,j,k)%natom == 0) CYCLE

                  DO iatom_subcell=1,subcell(i,j,k)%natom
                    iatom_local = subcell(i,j,k)%atom_list(iatom_subcell)
                    iatom = atom(ikind)%list_local_a_index(iatom_local)
                    atom_a = atom(ikind)%list(iatom)
                    mol_a = atom(ikind)%list_a_mol(iatom_local)
                    IF (mol_a /= mol_b) CYCLE
                    IF (atom_a > atom_b) THEN
                      include_ab = (MODULO(atom_a + atom_b,2) /= 0)
                    ELSE
                      include_ab = (MODULO(atom_a + atom_b,2) == 0)
                    END IF
                    IF (include_ab) THEN
                      rab(:) = rb(:) - atom(ikind)%r_pbc(:,iatom)
                      r(:) = pbc(rab(:),cell)
                      rab(:) = r(:)
                      rab2 = r(1)*r(1) + r(2)*r(2) + r(3)*r(3)
                      IF (rab2 < rab2_max) THEN
                        r(:) = atom(ikind)%r_pbc(:,iatom)+rab(:)
                        CALL real_to_scaled(sb,r,cell)
                        cell_b = NINT(sb)
                        CALL add_neighbor_node(&
                          neighbor_list=kind_a(iatom_local)%neighbor_list,&
                          neighbor=atom_b,&
                          cell=cell_b,&
                          r=rab)
                      END IF
                    END IF
                  END DO

                END DO loop_i
              END DO loop_j
            END DO loop_k

          END DO

          CALL deallocate_subcell(subcell,error=error)

        END IF

      END DO
    END DO

    CALL timestop(handle)

  END SUBROUTINE build_restricted_neighbor_lists

! *****************************************************************************
!> \brief   Build the neighbor lists for the calculation of the local part of
!>          the GTH pseudo potential (PPL). Atom c is a PPL operator atom,
!>          atom a is an atom in the cell(0,0,0). 
!> \author  MK
!> \date    20.03.2002
!> \version 1.0
! *****************************************************************************
  SUBROUTINE build_gth_ppl_neighbor_lists(ac,atom,cell,r_skin,maxatom_local,&
    unit_conv,unit_str,output_unit, present_a,present_c, radius_a,radius_c,name,&
    particle_set,subcells,error)
    TYPE(neighbor_list_set_p_type), &
      DIMENSION(:), POINTER                  :: ac
    TYPE(local_atoms_type), DIMENSION(:), &
      INTENT(IN)                             :: atom
    TYPE(cell_type), POINTER                 :: cell
    REAL(dp), INTENT(IN)                     :: r_skin
    INTEGER, INTENT(IN)                      :: maxatom_local
    REAL(dp), INTENT(IN)                     :: unit_conv
    CHARACTER(LEN=*), INTENT(IN)             :: unit_str
    INTEGER, INTENT(IN)                      :: output_unit
    LOGICAL, DIMENSION(:), INTENT(IN)        :: present_a, present_c
    REAL(dp), DIMENSION(:), INTENT(IN)       :: radius_a, radius_c
    CHARACTER(LEN=*), INTENT(IN)             :: name
    TYPE(particle_type), DIMENSION(:), &
      OPTIONAL, POINTER                      :: particle_set
    REAL(dp), INTENT(IN)                     :: subcells
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'build_gth_ppl_neighbor_lists', &
      routineP = moduleN//':'//routineN

    INTEGER :: atom_a, atom_c, handle, i, iac, iatom, iatom_local, &
      iatom_subcell, icell, ikind, j, jcell, k, katom, kcell, kkind, natom_c, &
      natom_local_a, nkind, stat
    INTEGER, DIMENSION(3)                    :: cell_a, cell_c, ncell, &
                                                nsubcell, periodic
    LOGICAL                                  :: cell000, exclude, failure
    REAL(dp)                                 :: rac2, rac2_max, rac_max
    REAL(dp), DIMENSION(3)                   :: r, rac, rc, s, sac_max, sc, &
                                                sc_max, sc_min, sc_pbc
    TYPE(neighbor_list_p_type), &
      DIMENSION(maxatom_local)               :: kind_a
    TYPE(neighbor_list_set_type), POINTER    :: neighbor_list_set
    TYPE(subcell_type), DIMENSION(:, :, :), &
      POINTER                                :: subcell

    CALL timeset(routineN,handle)
    failure = .FALSE.
    ! Deallocate the old neighbor list structure

    IF (ASSOCIATED(ac)) THEN
      DO iac=1,SIZE(ac)
        CALL deallocate_neighbor_list_set(ac(iac)%neighbor_list_set)
      END DO
      DEALLOCATE (ac,STAT=stat)
      CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

    ! Allocate and initialize the new neighbor list structure

    nkind = SIZE(atom)

    ALLOCATE (ac(nkind*nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    DO iac=1,SIZE(ac)
      NULLIFY (ac(iac)%neighbor_list_set)
    END DO

    exclude = .FALSE.

    ! Load informations about the simulation cell

    CALL get_cell(cell=cell,periodic=periodic)

    ! Print headline

    IF (output_unit>0) THEN
      WRITE (UNIT=output_unit,FMT="(/,/,T2,A,/,/,T3,A,T29,A,T54,A)")&
        "SUBCELL GRID FOR THE "//TRIM(name)//" NEIGHBOR LISTS",&
        "Atomic kind pair","Grid size",&
        "Subcell size in "//TRIM(unit_str)
    END IF

    ! Loop over all atomic kind pairs

    DO ikind=1,nkind

      IF (.NOT.present_a(ikind)) CYCLE

      IF (.NOT.ASSOCIATED(atom(ikind)%list_local_a_index)) CYCLE

      natom_local_a = SIZE(atom(ikind)%list_local_a_index)

      DO kkind=1,nkind

        IF (.NOT.present_c(kkind)) CYCLE

        iac = ikind + nkind*(kkind - 1)

        natom_c = SIZE(atom(kkind)%list)

        ! Calculate the square of the maximum interaction distance

        rac_max = radius_a(ikind) + radius_c(kkind) + r_skin
        rac2_max = rac_max*rac_max

        sac_max(1) = rac_max/plane_distance(1,0,0,cell)
        sac_max(2) = rac_max/plane_distance(0,1,0,cell)
        sac_max(3) = rac_max/plane_distance(0,0,1,cell)

        ncell(:) = (INT(sac_max(:)) + 1)*periodic(:)
        nsubcell(:) = MIN(MAX(1,NINT(0.5_dp*subcells/sac_max(:))),20)

        ! Print subcell information for the current atomic kind pair

        IF (output_unit>0) THEN
          s(:) = 1.0_dp/REAL(nsubcell(:),KIND=dp)
          CALL scaled_to_real(r,s,cell)
          WRITE (UNIT=output_unit,FMT="(T3,2I8,4X,3I5,6X,3F12.6)")&
            ikind,kkind,nsubcell(1:3),r(1:3)*unit_conv
        END IF

        CALL allocate_neighbor_list_set(neighbor_list_set=ac(iac)%neighbor_list_set,&
                                        r_max=rac_max)
        neighbor_list_set => ac(iac)%neighbor_list_set

        cell_a = (/0,0,0/)

        ! Check, if we have to consider a subcell grid

        IF (SUM(nsubcell) == 3) THEN

          ! Case 1: No subcell grid is needed

          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            CALL add_neighbor_list(neighbor_list_set=neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO katom=1,natom_c

            atom_c = atom(kkind)%list(katom)
            sc_pbc(:) = atom(kkind)%s_pbc(:,katom)

            loop1_kcell: DO kcell=-ncell(3),ncell(3)

              sc(3) = sc_pbc(3) + REAL(kcell,dp)
              sc_min(3) = sc(3) - sac_max(3)
              sc_max(3) = sc(3) + sac_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sc_min(3) >= 0.5_dp) EXIT loop1_kcell
                IF (sc_max(3) < -0.5_dp) CYCLE loop1_kcell
              END IF
              cell_c(3) = kcell

              loop1_jcell: DO jcell=-ncell(2),ncell(2)

                sc(2) = sc_pbc(2) + REAL(jcell,dp)
                sc_min(2) = sc(2) - sac_max(2)
                sc_max(2) = sc(2) + sac_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sc_min(2) >= 0.5_dp) EXIT loop1_jcell
                  IF (sc_max(2) < -0.5_dp) CYCLE loop1_jcell
                END IF
                cell_c(2) = jcell

                loop1_icell: DO icell=-ncell(1),ncell(1)

                  sc(1) = sc_pbc(1) + REAL(icell,dp)
                  sc_min(1) = sc(1) - sac_max(1)
                  sc_max(1) = sc(1) + sac_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sc_min(1) >= 0.5_dp) EXIT loop1_icell
                    IF (sc_max(1) < -0.5_dp) CYCLE loop1_icell
                  END IF
                  cell_c(1) = icell

                  CALL scaled_to_real(rc,sc,cell)

                  cell000 = ((icell == 0).AND.&
                             (jcell == 0).AND.&
                             (kcell == 0))

                  DO iatom_local=1,natom_local_a

                    iatom = atom(ikind)%list_local_a_index(iatom_local)
                    atom_a = atom(ikind)%list(iatom)

                    ! Check for exclusions
                    !MK: That is not needed for QS, but most likely for KG

                    IF (PRESENT(particle_set)) THEN
                       IF (ANY(particle_set(atom_a)%list_exclude_ei==atom_c)&
                            .AND.cell000) THEN
                          exclude = .TRUE.
                       ELSE
                          exclude = .FALSE.
                       END IF
                       IF(exclude) CYCLE
                    END IF

                    rac(:) = rc(:) - atom(ikind)%r_pbc(:,iatom)
                    rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                    IF (rac2 < rac2_max) THEN
                      CALL add_neighbor_node(&
                        neighbor_list=kind_a(iatom_local)%neighbor_list,&
                        neighbor=atom_c,&
                        cell=cell_c,&
                        r=rac)
                    END IF
                  END DO

                END DO loop1_icell
              END DO loop1_jcell
            END DO loop1_kcell

          END DO

        ELSE

          ! Case 2: A subcell grid is needed

          CALL allocate_subcell(subcell,nsubcell,maxatom_local,error=error)

          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            s(:) = atom(ikind)%s_pbc(:,iatom) + 0.5_dp
            i = INT(s(1)*REAL(nsubcell(1),KIND=dp)) + 1
            j = INT(s(2)*REAL(nsubcell(2),KIND=dp)) + 1
            k = INT(s(3)*REAL(nsubcell(3),KIND=dp)) + 1
            i = MIN(MAX(i,1),nsubcell(1))
            j = MIN(MAX(j,1),nsubcell(2))
            k = MIN(MAX(k,1),nsubcell(3))
            subcell(i,j,k)%natom =  subcell(i,j,k)%natom + 1
            subcell(i,j,k)%atom_list(subcell(i,j,k)%natom) = iatom_local
            CALL add_neighbor_list(neighbor_list_set=neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO katom=1,natom_c

            atom_c = atom(kkind)%list(katom)
            sc_pbc(:) = atom(kkind)%s_pbc(:,katom)

            loop2_kcell: DO kcell=-ncell(3),ncell(3)

              sc(3) = sc_pbc(3) + REAL(kcell,dp)
              sc_min(3) = sc(3) - sac_max(3)
              sc_max(3) = sc(3) + sac_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sc_min(3) >= 0.5_dp) EXIT loop2_kcell
                IF (sc_max(3) < -0.5_dp) CYCLE loop2_kcell
              END IF
              cell_c(3) = kcell

              loop2_jcell: DO jcell=-ncell(2),ncell(2)

                sc(2) = sc_pbc(2) + REAL(jcell,dp)
                sc_min(2) = sc(2) - sac_max(2)
                sc_max(2) = sc(2) + sac_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sc_min(2) >= 0.5_dp) EXIT loop2_jcell
                  IF (sc_max(2) < -0.5_dp) CYCLE loop2_jcell
                END IF
                cell_c(2) = jcell

                loop2_icell: DO icell=-ncell(1),ncell(1)

                  sc(1) = sc_pbc(1) + REAL(icell,dp)
                  sc_min(1) = sc(1) - sac_max(1)
                  sc_max(1) = sc(1) + sac_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sc_min(1) >= 0.5_dp) EXIT loop2_icell
                    IF (sc_max(1) < -0.5_dp) CYCLE loop2_icell
                  END IF
                  cell_c(1) = icell

                  CALL scaled_to_real(rc,sc,cell)

                  cell000 = ((icell == 0).AND.&
                             (jcell == 0).AND.&
                             (kcell == 0))

                  loop_k: DO k=1,nsubcell(3)
                    loop_j: DO j=1,nsubcell(2)
                      loop_i: DO i=1,nsubcell(1)

                        IF (periodic(3) /= 0) THEN
                          IF (sc_max(3) < subcell(i,j,k)%s_min(3)) EXIT loop_k
                          IF (sc_min(3) >= subcell(i,j,k)%s_max(3)) CYCLE loop_k
                        END IF

                        IF (periodic(2) /= 0) THEN
                          IF (sc_max(2) < subcell(i,j,k)%s_min(2)) EXIT loop_j
                          IF (sc_min(2) >= subcell(i,j,k)%s_max(2)) CYCLE loop_j
                        END IF

                        IF (periodic(1) /= 0) THEN
                          IF (sc_max(1) < subcell(i,j,k)%s_min(1)) EXIT loop_i
                          IF (sc_min(1) >= subcell(i,j,k)%s_max(1)) CYCLE loop_i
                        END IF

                        IF (subcell(i,j,k)%natom == 0) CYCLE

                        DO iatom_subcell=1,subcell(i,j,k)%natom
                          iatom_local = subcell(i,j,k)%atom_list(iatom_subcell)
                          iatom = atom(ikind)%list_local_a_index(iatom_local)
                          atom_a = atom(ikind)%list(iatom)

                          ! Check for exclusions
                          !MK: That is not needed for QS, but most likely for KG

                          IF (PRESENT(particle_set)) THEN
                             IF (ANY(particle_set(atom_a)%list_exclude_ei==atom_c)&
                                  .AND.cell000) THEN
                                exclude = .TRUE.
                             ELSE
                                exclude = .FALSE.
                             END IF
                             IF (exclude) CYCLE
                          END IF

                          rac(:) = rc(:) - atom(ikind)%r_pbc(:,iatom)
                          rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                          IF (rac2 < rac2_max) THEN
                            CALL add_neighbor_node(&
                              neighbor_list=kind_a(iatom_local)%neighbor_list,&
                              neighbor=atom_c,&
                              cell=cell_c,&
                              r=rac)
                          END IF
                        END DO

                      END DO loop_i
                    END DO loop_j
                  END DO loop_k

                END DO loop2_icell
              END DO loop2_jcell
            END DO loop2_kcell

          END DO

          CALL deallocate_subcell(subcell,error=error)

        END IF

      END DO
    END DO

    CALL timestop(handle)

  END SUBROUTINE build_gth_ppl_neighbor_lists

! *****************************************************************************
!> \brief   Build the neighbor lists for the calculation of the non-local
!>          part of the GTH pseudo potential (PPNL). Atom c is a PPNL
!>          operator atom, atom a is an atom in the cell(0,0,0), and atom b
!>          is an atom in any cell(i,j,k). 
!> \author  MK
!> \date    30.05.2003
!> \version 1.0
! *****************************************************************************
  SUBROUTINE build_gth_ppnl_neighbor_lists(ac,bc,atom,cell,r_skin,&
    maxatom,maxatom_local, unit_conv, unit_str, output_unit, present_a,present_c,&
    radius_a,radius_c,name,subcells,error)
    TYPE(neighbor_list_set_p_type), &
      DIMENSION(:), POINTER                  :: ac, bc
    TYPE(local_atoms_type), DIMENSION(:), &
      INTENT(IN)                             :: atom
    TYPE(cell_type), POINTER                 :: cell
    REAL(dp), INTENT(IN)                     :: r_skin
    INTEGER, INTENT(IN)                      :: maxatom, maxatom_local
    REAL(dp), INTENT(IN)                     :: unit_conv
    CHARACTER(LEN=*), INTENT(IN)             :: unit_str
    INTEGER, INTENT(IN)                      :: output_unit
    LOGICAL, DIMENSION(:), INTENT(IN)        :: present_a, present_c
    REAL(dp), DIMENSION(:), INTENT(IN)       :: radius_a, radius_c
    CHARACTER(LEN=*), INTENT(IN)             :: name
    REAL(dp), INTENT(IN)                     :: subcells
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: &
      routineN = 'build_gth_ppnl_neighbor_lists', &
      routineP = moduleN//':'//routineN

    INTEGER :: atom_a, atom_c, handle, i, iac, iatom, iatom_local, &
      iatom_subcell, icell, icell_b, ikind, j, jcell, jcell_b, k, katom, &
      kcell, kcell_b, kkind, natom_c, natom_local_a, natom_local_b, nkind, &
      stat
    INTEGER, DIMENSION(3)                    :: cell_a, cell_c, ncell, &
                                                nsubcell, periodic
    LOGICAL                                  :: failure
    REAL(dp)                                 :: rac2, rac2_max, rac_max
    REAL(dp), DIMENSION(3)                   :: r, rac, rc, s, s2r, s_max, &
                                                s_min, sac_max, sc, sc_max, &
                                                sc_min, sc_pbc
    TYPE(neighbor_list_p_type), &
      DIMENSION(maxatom_local)               :: kind_a, kind_b
    TYPE(neighbor_list_set_type), POINTER    :: ac_neighbor_list_set, &
                                                bc_neighbor_list_set
    TYPE(subcell_type), DIMENSION(:, :, :), &
      POINTER                                :: subcell_a, subcell_b

    CALL timeset(routineN,handle)
    failure = .FALSE.
    ! Deallocate the old neighbor list structures

    IF (ASSOCIATED(ac)) THEN
      DO iac=1,SIZE(ac)
        CALL deallocate_neighbor_list_set(ac(iac)%neighbor_list_set)
      END DO
      DEALLOCATE (ac,STAT=stat)
      CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

    IF (ASSOCIATED(bc)) THEN
      DO iac=1,SIZE(bc)
        CALL deallocate_neighbor_list_set(bc(iac)%neighbor_list_set)
      END DO
      DEALLOCATE (bc,STAT=stat)
      CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

    ! Allocate and initialize the new neighbor list structures

    nkind = SIZE(atom)

    ALLOCATE (ac(nkind*nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE (bc(nkind*nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    DO iac=1,SIZE(ac)
      NULLIFY (ac(iac)%neighbor_list_set)
      NULLIFY (bc(iac)%neighbor_list_set)
    END DO

    ! Load informations about the simulation cell

    CALL get_cell(cell=cell,periodic=periodic)

    ! Print headline

    IF (output_unit>0) THEN
      WRITE (UNIT=output_unit,FMT="(/,/,T2,A,/,/,T3,A,T29,A,T54,A)")&
        "SUBCELL GRID FOR THE"//TRIM(name)//" NEIGHBOR LISTS",&
        "Atomic kind pair","Grid size",&
        "Subcell size in "//TRIM(unit_str)
    END IF

   ! Loop over all atomic kind pairs

    DO ikind=1,nkind

      IF (.NOT.present_a(ikind)) CYCLE

      IF (ASSOCIATED(atom(ikind)%list_local_a_index)) THEN
        natom_local_a = SIZE(atom(ikind)%list_local_a_index)
      ELSE
        natom_local_a = 0
      END IF

      IF (ASSOCIATED(atom(ikind)%list_local_b_index)) THEN
        natom_local_b = SIZE(atom(ikind)%list_local_b_index)
      ELSE
        natom_local_b = 0
      END IF

      DO kkind=1,nkind

        IF (.NOT.present_c(kkind)) CYCLE

        iac = ikind + nkind*(kkind - 1)

        natom_c = SIZE(atom(kkind)%list)

        ! Calculate the square of the maximum interaction distance
        ! for sac_max / ncell this must be the maximum over all kinds
        ! to be correct for three center terms involving different kinds
        rac_max = MAXVAL(radius_a(:)) + radius_c(kkind) + r_skin
        sac_max(1) = rac_max/plane_distance(1,0,0,cell)
        sac_max(2) = rac_max/plane_distance(0,1,0,cell)
        sac_max(3) = rac_max/plane_distance(0,0,1,cell)
        ncell(:) = (INT(sac_max(:)) + 1)*periodic(:)
        nsubcell(:) = MIN(MAX(1,NINT(0.5_dp*subcells/sac_max(:))),20)

        ! actual inclusion still follows the normal rule
        rac_max = radius_a(ikind) + radius_c(kkind) + r_skin
        rac2_max = rac_max*rac_max

        ! Print subcell information for the current atomic kind pair

        IF (output_unit>0) THEN
          s(:) = 1.0_dp/REAL(nsubcell(:),KIND=dp)
          CALL scaled_to_real(r,s,cell)
          WRITE (UNIT=output_unit,FMT="(T3,2I8,4X,3I5,6X,3F12.6)")&
            ikind,kkind,nsubcell(1:3),r(1:3)*unit_conv
        END IF

        CALL allocate_neighbor_list_set(neighbor_list_set=ac(iac)%neighbor_list_set,&
                                        r_max=rac_max)
        ac_neighbor_list_set => ac(iac)%neighbor_list_set

        CALL allocate_neighbor_list_set(neighbor_list_set=bc(iac)%neighbor_list_set,&
                                        r_max=rac_max)
        bc_neighbor_list_set => bc(iac)%neighbor_list_set

        cell_a = (/0,0,0/)

        ! Check, if we have to consider a subcell grid

        IF (SUM(nsubcell) == 3) THEN

          ! Case 1: No subcell grid is needed

          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            CALL add_neighbor_list(neighbor_list_set=ac_neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO iatom_local=1,natom_local_b
            iatom = atom(ikind)%list_local_b_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            CALL add_neighbor_list(neighbor_list_set=bc_neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_b(iatom_local)%neighbor_list)
          END DO

          DO katom=1,natom_c

            atom_c = atom(kkind)%list(katom)
            sc_pbc(:) = atom(kkind)%s_pbc(:,katom)

            loop1_kcell: DO kcell=-ncell(3),ncell(3)

              sc(3) = sc_pbc(3) + REAL(kcell,dp)
              sc_min(3) = sc(3) - sac_max(3)
              sc_max(3) = sc(3) + sac_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sc_min(3) >= 0.5_dp) EXIT loop1_kcell
                IF (sc_max(3) < -0.5_dp) CYCLE loop1_kcell
              END IF
              cell_c(3) = kcell

              loop1_jcell: DO jcell=-ncell(2),ncell(2)

                sc(2) = sc_pbc(2) + REAL(jcell,dp)
                sc_min(2) = sc(2) - sac_max(2)
                sc_max(2) = sc(2) + sac_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sc_min(2) >= 0.5_dp) EXIT loop1_jcell
                  IF (sc_max(2) < -0.5_dp) CYCLE loop1_jcell
                END IF
                cell_c(2) = jcell

                loop1_icell: DO icell=-ncell(1),ncell(1)

                  sc(1) = sc_pbc(1) + REAL(icell,dp)
                  sc_min(1) = sc(1) - sac_max(1)
                  sc_max(1) = sc(1) + sac_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sc_min(1) >= 0.5_dp) EXIT loop1_icell
                    IF (sc_max(1) < -0.5_dp) CYCLE loop1_icell
                  END IF
                  cell_c(1) = icell

                  CALL scaled_to_real(rc,sc,cell)

                  DO iatom_local=1,natom_local_a
                    iatom = atom(ikind)%list_local_a_index(iatom_local)
                    rac(:) = rc(:) - atom(ikind)%r_pbc(:,iatom)
                    rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                    IF (rac2 < rac2_max) THEN
                      CALL add_neighbor_node(&
                        neighbor_list=kind_a(iatom_local)%neighbor_list,&
                        neighbor=atom_c,&
                        cell=cell_c,&
                        r=rac)
                    END IF
                  END DO

                  DO icell_b=cell_c(1)-ncell(1),cell_c(1)+ncell(1)

                    s(1) = REAL(icell_b,dp)

                    DO jcell_b=cell_c(2)-ncell(2),cell_c(2)+ncell(2)

                      s(2) = REAL(jcell_b,dp)

                      DO kcell_b=cell_c(3)-ncell(3),cell_c(3)+ncell(3)

                        s(3) = REAL(kcell_b,dp)

                        CALL scaled_to_real(s2r,s,cell)
                        r(:) = rc(:) - s2r

                        DO iatom_local=1,natom_local_b
                          iatom = atom(ikind)%list_local_b_index(iatom_local)
                          rac(:) = r(:) - atom(ikind)%r_pbc(:,iatom)
                          rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                          IF (rac2 < rac2_max) THEN
                            CALL add_neighbor_node(&
                              neighbor_list=kind_b(iatom_local)%neighbor_list,&
                              neighbor=atom_c,&
                              cell=cell_c,&
                              r=rac)
                          END IF
                        END DO

                      END DO
                    END DO
                  END DO

                END DO loop1_icell
              END DO loop1_jcell
            END DO loop1_kcell

          END DO

        ELSE

          CALL allocate_subcell(subcell_a,nsubcell,maxatom_local,error=error)

          DO iatom_local=1,natom_local_a
            iatom = atom(ikind)%list_local_a_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            s(:) = atom(ikind)%s_pbc(:,iatom) + 0.5_dp
            i = INT(s(1)*REAL(nsubcell(1),KIND=dp)) + 1
            j = INT(s(2)*REAL(nsubcell(2),KIND=dp)) + 1
            k = INT(s(3)*REAL(nsubcell(3),KIND=dp)) + 1
            i = MIN(MAX(i,1),nsubcell(1))
            j = MIN(MAX(j,1),nsubcell(2))
            k = MIN(MAX(k,1),nsubcell(3))
            subcell_a(i,j,k)%natom =  subcell_a(i,j,k)%natom + 1
            subcell_a(i,j,k)%atom_list(subcell_a(i,j,k)%natom) = iatom_local
            CALL add_neighbor_list(neighbor_list_set=ac_neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          CALL allocate_subcell(subcell_b,nsubcell,maxatom_local,error=error)

          DO iatom_local=1,natom_local_b
            iatom = atom(ikind)%list_local_b_index(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            s(:) = atom(ikind)%s_pbc(:,iatom) + 0.5_dp
            i = INT(s(1)*REAL(nsubcell(1),KIND=dp)) + 1
            j = INT(s(2)*REAL(nsubcell(2),KIND=dp)) + 1
            k = INT(s(3)*REAL(nsubcell(3),KIND=dp)) + 1
            i = MIN(MAX(i,1),nsubcell(1))
            j = MIN(MAX(j,1),nsubcell(2))
            k = MIN(MAX(k,1),nsubcell(3))
            subcell_b(i,j,k)%natom =  subcell_b(i,j,k)%natom + 1
            subcell_b(i,j,k)%atom_list(subcell_b(i,j,k)%natom) = iatom_local
            CALL add_neighbor_list(neighbor_list_set=bc_neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_b(iatom_local)%neighbor_list)
          END DO

          DO katom=1,natom_c

            atom_c = atom(kkind)%list(katom)
            sc_pbc(:) = atom(kkind)%s_pbc(:,katom)

            loop2_kcell: DO kcell=-ncell(3),ncell(3)

              sc(3) = sc_pbc(3) + REAL(kcell,dp)
              sc_min(3) = sc(3) - sac_max(3)
              sc_max(3) = sc(3) + sac_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sc_min(3) >= 0.5_dp) EXIT loop2_kcell
                IF (sc_max(3) < -0.5_dp) CYCLE loop2_kcell
              END IF
              cell_c(3) = kcell

              loop2_jcell: DO jcell=-ncell(2),ncell(2)

                sc(2) = sc_pbc(2) + REAL(jcell,dp)
                sc_min(2) = sc(2) - sac_max(2)
                sc_max(2) = sc(2) + sac_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sc_min(2) >= 0.5_dp) EXIT loop2_jcell
                  IF (sc_max(2) < -0.5_dp) CYCLE loop2_jcell
                END IF
                cell_c(2) = jcell

                loop2_icell: DO icell=-ncell(1),ncell(1)

                  sc(1) = sc_pbc(1) + REAL(icell,dp)
                  sc_min(1) = sc(1) - sac_max(1)
                  sc_max(1) = sc(1) + sac_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sc_min(1) >= 0.5_dp) EXIT loop2_icell
                    IF (sc_max(1) < -0.5_dp) CYCLE loop2_icell
                  END IF
                  cell_c(1) = icell

                  CALL scaled_to_real(rc,sc,cell)

                  loop1_k: DO k=1,nsubcell(3)
                    loop1_j: DO j=1,nsubcell(2)
                      loop1_i: DO i=1,nsubcell(1)

                        IF (periodic(3) /= 0) THEN
                          IF (sc_max(3) < subcell_a(i,j,k)%s_min(3)) EXIT loop1_k
                          IF (sc_min(3) >= subcell_a(i,j,k)%s_max(3)) CYCLE loop1_k
                        END IF

                        IF (periodic(2) /= 0) THEN
                          IF (sc_max(2) < subcell_a(i,j,k)%s_min(2)) EXIT loop1_j
                          IF (sc_min(2) >= subcell_a(i,j,k)%s_max(2)) CYCLE loop1_j
                        END IF

                        IF (periodic(1) /= 0) THEN
                          IF (sc_max(1) < subcell_a(i,j,k)%s_min(1)) EXIT loop1_i
                          IF (sc_min(1) >= subcell_a(i,j,k)%s_max(1)) CYCLE loop1_i
                        END IF

                        IF (subcell_a(i,j,k)%natom == 0) CYCLE

                        DO iatom_subcell=1,subcell_a(i,j,k)%natom
                          iatom_local = subcell_a(i,j,k)%atom_list(iatom_subcell)
                          iatom = atom(ikind)%list_local_a_index(iatom_local)
                          rac(:) = rc(:) - atom(ikind)%r_pbc(:,iatom)
                          rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                          IF (rac2 < rac2_max) THEN
                            CALL add_neighbor_node(&
                              neighbor_list=kind_a(iatom_local)%neighbor_list,&
                              neighbor=atom_c,&
                              cell=cell_c,&
                              r=rac)
                          END IF
                        END DO

                      END DO loop1_i
                    END DO loop1_j
                  END DO loop1_k

                  loop_kcell_b: DO kcell_b=cell_c(3)-ncell(3),cell_c(3)+ncell(3)

                    s(3) = REAL(kcell_b,dp)
                    s_min(3) = sc_min(3) - s(3)
                    s_max(3) = sc_max(3) - s(3)
                    IF (periodic(3) /= 0) THEN
                      IF (s_min(3) >= 0.5_dp) CYCLE loop_kcell_b
                      IF (s_max(3) < -0.5_dp) EXIT loop_kcell_b
                    END IF

                    loop_jcell_b: DO jcell_b=cell_c(2)-ncell(2),cell_c(2)+ncell(2)

                      s(2) = REAL(jcell_b,dp)
                      s_min(2) = sc_min(2) - s(2)
                      s_max(2) = sc_max(2) - s(2)
                      IF (periodic(2) /= 0) THEN
                        IF (s_min(2) >= 0.5_dp) CYCLE loop_jcell_b
                        IF (s_max(2) < -0.5_dp) EXIT loop_jcell_b
                      END IF

                      loop_icell_b: DO icell_b=cell_c(1)-ncell(1),cell_c(1)+ncell(1)

                        s(1) = REAL(icell_b,dp)
                        s_min(1) = sc_min(1) - s(1)
                        s_max(1) = sc_max(1) - s(1)
                        IF (periodic(1) /= 0) THEN
                          IF (s_min(1) >= 0.5_dp) CYCLE loop_icell_b
                          IF (s_max(1) < -0.5_dp) EXIT loop_icell_b
                        END IF

                        CALL scaled_to_real(s2r,s,cell)
                        r(:) = rc(:) - s2r

                        loop2_k: DO k=1,nsubcell(3)
                          loop2_j: DO j=1,nsubcell(2)
                            loop2_i: DO i=1,nsubcell(1)

                              IF (periodic(3) /= 0) THEN
                                IF (s_max(3) < subcell_b(i,j,k)%s_min(3)) EXIT loop2_k
                                IF (s_min(3) >= subcell_b(i,j,k)%s_max(3)) CYCLE loop2_k
                              END IF

                              IF (periodic(2) /= 0) THEN
                                IF (s_max(2) < subcell_b(i,j,k)%s_min(2)) EXIT loop2_j
                                IF (s_min(2) >= subcell_b(i,j,k)%s_max(2)) CYCLE loop2_j
                              END IF

                              IF (periodic(1) /= 0) THEN
                                IF (s_max(1) < subcell_b(i,j,k)%s_min(1)) EXIT loop2_i
                                IF (s_min(1) >= subcell_b(i,j,k)%s_max(1)) CYCLE loop2_i
                              END IF

                              IF (subcell_b(i,j,k)%natom == 0) CYCLE

                              DO iatom_subcell=1,subcell_b(i,j,k)%natom
                                iatom_local = subcell_b(i,j,k)%atom_list(iatom_subcell)
                                iatom = atom(ikind)%list_local_b_index(iatom_local)
                                rac(:) = r(:) - atom(ikind)%r_pbc(:,iatom)
                                rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                                IF (rac2 < rac2_max) THEN
                                  CALL add_neighbor_node(&
                                    neighbor_list=kind_b(iatom_local)%neighbor_list,&
                                    neighbor=atom_c,&
                                    cell=cell_c,&
                                    r=rac)
                                END IF
                              END DO

                            END DO loop2_i
                          END DO loop2_j
                        END DO loop2_k

                      END DO loop_icell_b
                    END DO loop_jcell_b
                  END DO loop_kcell_b

                END DO loop2_icell
              END DO loop2_jcell
            END DO loop2_kcell

          END DO

          CALL deallocate_subcell(subcell_a,error=error)
          CALL deallocate_subcell(subcell_b,error=error)

        END IF

      END DO

    END DO

    CALL timestop(handle)

  END SUBROUTINE build_gth_ppnl_neighbor_lists

! *****************************************************************************
!> \brief   Build the neighbor lists for the calculation of the overlap
!>          between a basis function and an atomic projector function.
!>          Atom a is an atom in the cell(0,0,0) and atom c is a projector 
!>          atom in cell(i,j,k). 
!> \author  jhu
!> \date    15.01.2009
!> \version 1.0
! *****************************************************************************
  SUBROUTINE build_projector_neighbor_lists(ap_lists,atom,cell,r_skin,&
    maxatom,maxatom_local,unit_conv,unit_str,output_unit,present_a,present_c,&
    radius_a,radius_c,name,subcells,error)
    TYPE(neighbor_list_set_p_type), &
      DIMENSION(:), POINTER                  :: ap_lists
    TYPE(local_atoms_type), DIMENSION(:), &
      INTENT(IN)                             :: atom
    TYPE(cell_type), POINTER                 :: cell
    REAL(dp), INTENT(IN)                     :: r_skin
    INTEGER, INTENT(IN)                      :: maxatom, maxatom_local
    REAL(dp), INTENT(IN)                     :: unit_conv
    CHARACTER(LEN=*), INTENT(IN)             :: unit_str
    INTEGER, INTENT(IN)                      :: output_unit
    LOGICAL, DIMENSION(:), INTENT(IN)        :: present_a, present_c
    REAL(dp), DIMENSION(:), INTENT(IN)       :: radius_a, radius_c
    CHARACTER(LEN=*), INTENT(IN)             :: name
    REAL(dp), INTENT(IN)                     :: subcells
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: &
      routineN = 'build_projector_neighbor_lists', &
      routineP = moduleN//':'//routineN

    INTEGER :: atom_a, atom_c, handle, i, iac, iatom, iatom_local, &
      iatom_subcell, icell, ikind, j, jcell, k, katom, kcell, kkind, natom_c, &
      nkind, nlist, stat
    INTEGER, DIMENSION(3)                    :: cell_a, cell_c, ncell, &
                                                nsubcell, periodic
    INTEGER, DIMENSION(:), POINTER           :: local_list
    LOGICAL                                  :: failure
    REAL(dp)                                 :: rac2, rac2_max, rac_max
    REAL(dp), DIMENSION(3)                   :: r, rac, rc, s, sac_max, sc, &
                                                sc_max, sc_min, sc_pbc
    TYPE(neighbor_list_p_type), &
      DIMENSION(maxatom_local)               :: kind_a
    TYPE(neighbor_list_set_type), POINTER    :: ap_neighbor_list_set
    TYPE(subcell_type), DIMENSION(:, :, :), &
      POINTER                                :: subcell_a

    CALL timeset(routineN,handle)
    failure = .FALSE.

    ! Deallocate the old neighbor list structures
    IF (ASSOCIATED(ap_lists)) THEN
      DO iac=1,SIZE(ap_lists)
        CALL deallocate_neighbor_list_set(ap_lists(iac)%neighbor_list_set)
      END DO
      DEALLOCATE (ap_lists,STAT=stat)
      CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    END IF

    ! Allocate and initialize the new neighbor list structures

    nkind = SIZE(atom)

    ALLOCATE (ap_lists(nkind*nkind),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    DO iac=1,SIZE(ap_lists)
      NULLIFY (ap_lists(iac)%neighbor_list_set)
    END DO

    ! Load informations about the simulation cell
    CALL get_cell(cell=cell,periodic=periodic)

    ! Print headline
    IF (output_unit>0) THEN
      WRITE (UNIT=output_unit,FMT="(/,/,T2,A,/,/,T3,A,T29,A,T54,A)")&
        "SUBCELL GRID FOR THE"//TRIM(name)//" NEIGHBOR LISTS",&
        "Atomic kind pair","Grid size",&
        "Subcell size in "//TRIM(unit_str)
    END IF

    ! Loop over all atomic kind pairs
    DO ikind=1,nkind
      IF (.NOT.present_a(ikind)) CYCLE

      NULLIFY(local_list)
      CALL combine_lists(local_list,nlist,ikind,atom,error)

      DO kkind=1,nkind
        IF (.NOT.present_c(kkind)) CYCLE

        iac = ikind + nkind*(kkind - 1)

        natom_c = SIZE(atom(kkind)%list)

        ! Calculate the square of the maximum interaction distance
        ! for sac_max / ncell this must be the maximum over all kinds
        ! to be correct for three center terms involving different kinds
        rac_max = MAXVAL(radius_a(:)) + radius_c(kkind) + r_skin
        sac_max(1) = rac_max/plane_distance(1,0,0,cell)
        sac_max(2) = rac_max/plane_distance(0,1,0,cell)
        sac_max(3) = rac_max/plane_distance(0,0,1,cell)
        ncell(:) = (INT(sac_max(:)) + 1)*periodic(:)
        nsubcell(:) = MIN(MAX(1,NINT(0.5_dp*subcells/sac_max(:))),20)

        ! actual inclusion still follows the normal rule
        rac_max = radius_a(ikind) + radius_c(kkind) + r_skin
        rac2_max = rac_max*rac_max

        ! Print subcell information for the current atomic kind pair

        IF (output_unit>0) THEN
          s(:) = 1.0_dp/REAL(nsubcell(:),KIND=dp)
          CALL scaled_to_real(r,s,cell)
          WRITE (UNIT=output_unit,FMT="(T3,2I8,4X,3I5,6X,3F12.6)")&
            ikind,kkind,nsubcell(1:3),r(1:3)*unit_conv
        END IF

        CALL allocate_neighbor_list_set(neighbor_list_set=ap_lists(iac)%neighbor_list_set,&
                                        r_max=rac_max)
        ap_neighbor_list_set => ap_lists(iac)%neighbor_list_set

        cell_a = (/0,0,0/)

        ! Check, if we have to consider a subcell grid

        IF (SUM(nsubcell) == 3) THEN

          ! Case 1: No subcell grid is needed

          DO iatom_local=1,nlist
            iatom = local_list(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            CALL add_neighbor_list(neighbor_list_set=ap_neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO katom=1,natom_c

            atom_c = atom(kkind)%list(katom)
            sc_pbc(:) = atom(kkind)%s_pbc(:,katom)

            loop1_kcell: DO kcell=-ncell(3),ncell(3)

              sc(3) = sc_pbc(3) + REAL(kcell,dp)
              sc_min(3) = sc(3) - sac_max(3)
              sc_max(3) = sc(3) + sac_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sc_min(3) >= 0.5_dp) EXIT loop1_kcell
                IF (sc_max(3) < -0.5_dp) CYCLE loop1_kcell
              END IF
              cell_c(3) = kcell

              loop1_jcell: DO jcell=-ncell(2),ncell(2)

                sc(2) = sc_pbc(2) + REAL(jcell,dp)
                sc_min(2) = sc(2) - sac_max(2)
                sc_max(2) = sc(2) + sac_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sc_min(2) >= 0.5_dp) EXIT loop1_jcell
                  IF (sc_max(2) < -0.5_dp) CYCLE loop1_jcell
                END IF
                cell_c(2) = jcell

                loop1_icell: DO icell=-ncell(1),ncell(1)

                  sc(1) = sc_pbc(1) + REAL(icell,dp)
                  sc_min(1) = sc(1) - sac_max(1)
                  sc_max(1) = sc(1) + sac_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sc_min(1) >= 0.5_dp) EXIT loop1_icell
                    IF (sc_max(1) < -0.5_dp) CYCLE loop1_icell
                  END IF
                  cell_c(1) = icell

                  CALL scaled_to_real(rc,sc,cell)

                  DO iatom_local=1,nlist
                    iatom = local_list(iatom_local)
                    rac(:) = rc(:) - atom(ikind)%r_pbc(:,iatom)
                    rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                    IF (rac2 < rac2_max) THEN
                      CALL add_neighbor_node(&
                        neighbor_list=kind_a(iatom_local)%neighbor_list,&
                        neighbor=atom_c,&
                        cell=cell_c,&
                        r=rac)
                    END IF
                  END DO

                END DO loop1_icell
              END DO loop1_jcell
            END DO loop1_kcell

          END DO

        ELSE

          CALL allocate_subcell(subcell_a,nsubcell,maxatom_local,error=error)

          DO iatom_local=1,nlist
            iatom = local_list(iatom_local)
            atom_a = atom(ikind)%list(iatom)
            s(:) = atom(ikind)%s_pbc(:,iatom) + 0.5_dp
            i = INT(s(1)*REAL(nsubcell(1),KIND=dp)) + 1
            j = INT(s(2)*REAL(nsubcell(2),KIND=dp)) + 1
            k = INT(s(3)*REAL(nsubcell(3),KIND=dp)) + 1
            i = MIN(MAX(i,1),nsubcell(1))
            j = MIN(MAX(j,1),nsubcell(2))
            k = MIN(MAX(k,1),nsubcell(3))
            subcell_a(i,j,k)%natom =  subcell_a(i,j,k)%natom + 1
            subcell_a(i,j,k)%atom_list(subcell_a(i,j,k)%natom) = iatom_local
            CALL add_neighbor_list(neighbor_list_set=ap_neighbor_list_set,&
                                   atom=atom_a,&
                                   cell=cell_a,&
                                   neighbor_list=kind_a(iatom_local)%neighbor_list)
          END DO

          DO katom=1,natom_c

            atom_c = atom(kkind)%list(katom)
            sc_pbc(:) = atom(kkind)%s_pbc(:,katom)

            loop2_kcell: DO kcell=-ncell(3),ncell(3)

              sc(3) = sc_pbc(3) + REAL(kcell,dp)
              sc_min(3) = sc(3) - sac_max(3)
              sc_max(3) = sc(3) + sac_max(3)
              IF (periodic(3) /= 0) THEN
                IF (sc_min(3) >= 0.5_dp) EXIT loop2_kcell
                IF (sc_max(3) < -0.5_dp) CYCLE loop2_kcell
              END IF
              cell_c(3) = kcell

              loop2_jcell: DO jcell=-ncell(2),ncell(2)

                sc(2) = sc_pbc(2) + REAL(jcell,dp)
                sc_min(2) = sc(2) - sac_max(2)
                sc_max(2) = sc(2) + sac_max(2)
                IF (periodic(2) /= 0) THEN
                  IF (sc_min(2) >= 0.5_dp) EXIT loop2_jcell
                  IF (sc_max(2) < -0.5_dp) CYCLE loop2_jcell
                END IF
                cell_c(2) = jcell

                loop2_icell: DO icell=-ncell(1),ncell(1)

                  sc(1) = sc_pbc(1) + REAL(icell,dp)
                  sc_min(1) = sc(1) - sac_max(1)
                  sc_max(1) = sc(1) + sac_max(1)
                  IF (periodic(1) /= 0) THEN
                    IF (sc_min(1) >= 0.5_dp) EXIT loop2_icell
                    IF (sc_max(1) < -0.5_dp) CYCLE loop2_icell
                  END IF
                  cell_c(1) = icell

                  CALL scaled_to_real(rc,sc,cell)

                  loop1_k: DO k=1,nsubcell(3)
                    loop1_j: DO j=1,nsubcell(2)
                      loop1_i: DO i=1,nsubcell(1)

                        IF (periodic(3) /= 0) THEN
                          IF (sc_max(3) < subcell_a(i,j,k)%s_min(3)) EXIT loop1_k
                          IF (sc_min(3) >= subcell_a(i,j,k)%s_max(3)) CYCLE loop1_k
                        END IF

                        IF (periodic(2) /= 0) THEN
                          IF (sc_max(2) < subcell_a(i,j,k)%s_min(2)) EXIT loop1_j
                          IF (sc_min(2) >= subcell_a(i,j,k)%s_max(2)) CYCLE loop1_j
                        END IF

                        IF (periodic(1) /= 0) THEN
                          IF (sc_max(1) < subcell_a(i,j,k)%s_min(1)) EXIT loop1_i
                          IF (sc_min(1) >= subcell_a(i,j,k)%s_max(1)) CYCLE loop1_i
                        END IF

                        IF (subcell_a(i,j,k)%natom == 0) CYCLE

                        DO iatom_subcell=1,subcell_a(i,j,k)%natom
                          iatom_local = subcell_a(i,j,k)%atom_list(iatom_subcell)
                          iatom = local_list(iatom_local)
                          rac(:) = rc(:) - atom(ikind)%r_pbc(:,iatom)
                          rac2 = rac(1)*rac(1) + rac(2)*rac(2) + rac(3)*rac(3)
                          IF (rac2 < rac2_max) THEN
                            CALL add_neighbor_node(&
                              neighbor_list=kind_a(iatom_local)%neighbor_list,&
                              neighbor=atom_c,&
                              cell=cell_c,&
                              r=rac)
                          END IF
                        END DO

                      END DO loop1_i
                    END DO loop1_j
                  END DO loop1_k

                END DO loop2_icell
              END DO loop2_jcell
            END DO loop2_kcell

          END DO

          CALL deallocate_subcell(subcell_a,error=error)

        END IF

      END DO

      DEALLOCATE(local_list,STAT=stat)
      CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    END DO

    CALL timestop(handle)

  END SUBROUTINE build_projector_neighbor_lists

  SUBROUTINE combine_lists(list,n,ikind,atom,error)
    INTEGER, DIMENSION(:), POINTER           :: list
    INTEGER, INTENT(OUT)                     :: n
    INTEGER, INTENT(IN)                      :: ikind
    TYPE(local_atoms_type), DIMENSION(:), &
      INTENT(IN)                             :: atom
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'combine_lists', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: i, ib, ierr, na, nb
    INTEGER, DIMENSION(:), POINTER           :: lista, listb
    LOGICAL                                  :: failure = .FALSE.

    CPPrecondition(.NOT.ASSOCIATED(list),cp_failure_level,routineP,error,failure)


    lista => atom(ikind)%list_local_a_index
    listb => atom(ikind)%list_local_b_index

    IF (ASSOCIATED(lista)) THEN
       na = SIZE(lista)
    ELSE
       na = 0
    END IF

    IF (ASSOCIATED(listb)) THEN
       nb = SIZE(listb)
    ELSE
       nb = 0
    END IF

    ALLOCATE(list(na+nb),STAT=ierr)
    CPPostcondition(ierr==0,cp_failure_level,routineP,error,failure)

    n = na
    IF(na.GT.0) list(1:na) = lista(1:na)
    IF(nb.GT.0) THEN
       loopb: DO ib = 1, nb
          DO i = 1, na
             IF(listb(ib) == list(i)) CYCLE loopb
          END DO
          n = n+1
          list(n) = listb(ib)
       END DO loopb
    ENDIF
  END SUBROUTINE combine_lists

! *****************************************************************************
!> \brief Allocate and initialize a subcell grid structure for the atomic neighbor search.  
!> \author MK  
!> \date    12.06.2003
!> \version 1.0
! *****************************************************************************
  SUBROUTINE allocate_subcell(subcell,nsubcell,maxatom,cell,error)

    TYPE(subcell_type), DIMENSION(:, :, :), &
      POINTER                                :: subcell
    INTEGER, DIMENSION(3), INTENT(IN)        :: nsubcell
    INTEGER, INTENT(IN), OPTIONAL            :: maxatom
    TYPE(cell_type), OPTIONAL, POINTER       :: cell
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'allocate_subcell', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: i, j, k, na, nb, nc, stat
    LOGICAL                                  :: failure
    REAL(dp)                                 :: a_max, a_min, b_max, b_min, &
                                                c_max, c_min, delta_a, &
                                                delta_b, delta_c

    failure = .FALSE.
    na = nsubcell(1)
    nb = nsubcell(2)
    nc = nsubcell(3)

    ALLOCATE (subcell(na,nb,nc),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    delta_a = 1.0_dp/REAL(na,dp)
    delta_b = 1.0_dp/REAL(nb,dp)
    delta_c = 1.0_dp/REAL(nc,dp)

    c_min = -0.5_dp

    DO k=1,nc
      c_max = c_min + delta_c
      b_min = -0.5_dp
      DO j=1,nb
        b_max = b_min + delta_b
        a_min = -0.5_dp
        DO i=1,na
          a_max = a_min + delta_a
          subcell(i,j,k)%s_min(1) = a_min
          subcell(i,j,k)%s_min(2) = b_min
          subcell(i,j,k)%s_min(3) = c_min
          subcell(i,j,k)%s_max(1) = a_max
          subcell(i,j,k)%s_max(2) = b_max
          subcell(i,j,k)%s_max(3) = c_max
          subcell(i,j,k)%natom = 0
          IF (PRESENT(cell)) THEN
             CALL scaled_to_real(subcell(i,j,k)%corners(:,1),(/a_min,b_min,c_min/),cell)
             CALL scaled_to_real(subcell(i,j,k)%corners(:,2),(/a_max,b_min,c_min/),cell)
             CALL scaled_to_real(subcell(i,j,k)%corners(:,3),(/a_min,b_max,c_min/),cell)
             CALL scaled_to_real(subcell(i,j,k)%corners(:,4),(/a_max,b_max,c_min/),cell)
             CALL scaled_to_real(subcell(i,j,k)%corners(:,5),(/a_min,b_min,c_max/),cell)
             CALL scaled_to_real(subcell(i,j,k)%corners(:,6),(/a_max,b_min,c_max/),cell)
             CALL scaled_to_real(subcell(i,j,k)%corners(:,7),(/a_min,b_max,c_max/),cell)
             CALL scaled_to_real(subcell(i,j,k)%corners(:,8),(/a_max,b_max,c_max/),cell)
          END IF
          IF (PRESENT(maxatom)) THEN
             ALLOCATE (subcell(i,j,k)%atom_list(maxatom),STAT=stat)
             CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
          END IF
          a_min = a_max
        END DO
        b_min = b_max
      END DO
      c_min = c_max
    END DO

  END SUBROUTINE allocate_subcell

! *****************************************************************************
!> \brief   Deallocate a subcell grid structure.
!> \author  MK
!> \date    16.06.2003
!> \version 1.0
! *****************************************************************************
  SUBROUTINE deallocate_subcell(subcell,error)

    TYPE(subcell_type), DIMENSION(:, :, :), &
      POINTER                                :: subcell
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'deallocate_subcell', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: i, j, k, stat
    LOGICAL                                  :: failure

    failure = .FALSE.
    IF (ASSOCIATED(subcell)) THEN

       DO k=1,SIZE(subcell,3)
          DO j=1,SIZE(subcell,2)
             DO i=1,SIZE(subcell,1)
                DEALLOCATE (subcell(i,j,k)%atom_list,STAT=stat)
                CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
             END DO
          END DO
       END DO

       DEALLOCATE (subcell,STAT=stat)
       CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ELSE
       CALL stop_program(routineN,moduleN,__LINE__,&
            "The pointer subcell is not associated")
    END IF

  END SUBROUTINE deallocate_subcell

! *****************************************************************************
!> \brief   Print the distribution of the simple pair neighbor list.
!> \author  MK
!> \date    19.06.2003
!> \version 1.0
! *****************************************************************************
  SUBROUTINE write_neighbor_distribution(ab,atomic_kind_set,output_unit,para_env,error)
    TYPE(neighbor_list_set_p_type), &
      DIMENSION(:), POINTER                  :: ab
    TYPE(atomic_kind_type), DIMENSION(:), &
      POINTER                                :: atomic_kind_set
    INTEGER, INTENT(in)                      :: output_unit
    TYPE(cp_para_env_type), POINTER          :: para_env
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'write_neighbor_distribution', &
      routineP = moduleN//':'//routineN
    LOGICAL, PARAMETER                       :: full_output = .FALSE.

    INTEGER :: group, handle, iab, ikind, ilist, ipe, jkind, mype, n, &
      nblock_max, nblock_sum, nelement_max, nelement_sum, nkind, nlist, &
      nnode, npe, nsgf_a, nsgf_b, stat, tmp(2)
    INTEGER, ALLOCATABLE, DIMENSION(:)       :: nblock, nelement
    LOGICAL                                  :: failure
    TYPE(atomic_kind_type), POINTER          :: atomic_kind
    TYPE(gto_basis_set_type), POINTER        :: orb_basis_set
    TYPE(neighbor_list_set_type), POINTER    :: neighbor_list_set
    TYPE(neighbor_list_type), POINTER        :: neighbor_list

    CALL timeset(routineN,handle)
    failure = .FALSE.
    group = para_env%group
    mype = para_env%mepos + 1
    npe = para_env%num_pe

    ! Allocate work storage

    ALLOCATE (nblock(npe),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    nblock(:) = 0

    ALLOCATE (nelement(npe),STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    nelement(:) = 0

    nkind = SIZE(atomic_kind_set)

    DO ikind=1,nkind

      atomic_kind => atomic_kind_set(ikind)

      CALL get_atomic_kind(atomic_kind=atomic_kind,&
                           orb_basis_set=orb_basis_set)

      IF (ASSOCIATED(orb_basis_set)) THEN
        CALL get_gto_basis_set(gto_basis_set=orb_basis_set,nsgf=nsgf_a)
      ELSE
        nsgf_a = 1
      END IF

      DO jkind=1,nkind

        iab = ikind + nkind*(jkind - 1)

        neighbor_list_set => ab(iab)%neighbor_list_set

        IF (.NOT.ASSOCIATED(neighbor_list_set)) CYCLE

        atomic_kind => atomic_kind_set(jkind)

        CALL get_atomic_kind(atomic_kind=atomic_kind,&
                             orb_basis_set=orb_basis_set)

        IF (ASSOCIATED(orb_basis_set)) THEN
          CALL get_gto_basis_set(gto_basis_set=orb_basis_set,nsgf=nsgf_b)
        ELSE
          nsgf_b = 1
        END IF

        n = nsgf_a*nsgf_b

        CALL get_neighbor_list_set(neighbor_list_set=neighbor_list_set,&
                                   nlist=nlist)

        neighbor_list => first_list(neighbor_list_set)

        DO ilist=1,nlist

          CALL get_neighbor_list(neighbor_list=neighbor_list,&
                                 nnode=nnode)

          nblock(mype) = nblock(mype) + nnode
          nelement(mype) = nelement(mype) + n*nnode

          neighbor_list => next(neighbor_list)

        END DO

      END DO

    END DO

    IF (full_output) THEN
       ! XXXXXXXX should gather/scatter this on ionode
       CALL mp_sum(nblock,group)
       CALL mp_sum(nelement,group)

       nblock_sum = SUM(nblock)
       nelement_sum = SUM(nelement)
    ELSE
       nblock_sum = nblock(mype)
       nblock_max = nblock(mype)
       nelement_sum = nelement(mype)
       nelement_max = nelement(mype)
       tmp=(/nblock_sum,nelement_sum/)
       CALL mp_sum(tmp,group)
       nblock_sum=tmp(1) ; nelement_sum=tmp(2)
       tmp=(/nblock_max,nelement_max/)
       CALL mp_max(tmp,group)
       nblock_max=tmp(1) ; nelement_max=tmp(2)
    ENDIF

    IF (output_unit > 0) THEN
      IF (full_output) THEN
         WRITE (UNIT=output_unit,&
                FMT="(/,/,T2,A,/,/,T3,A,/,/,(T4,I6,T27,I10,T55,I10))")&
           "DISTRIBUTION OF THE NEIGHBOR LISTS",&
           "Process   Number of particle pairs   Number of matrix elements",&
           (ipe-1,nblock(ipe),nelement(ipe),ipe=1,npe)
         WRITE (UNIT=output_unit,FMT="(/,T7,A3,T27,I10,T55,I10)")&
           "Sum",SUM(nblock),SUM(nelement)
      ELSE
         WRITE (UNIT=output_unit,FMT="(/,T2,A)") "DISTRIBUTION OF THE NEIGHBOR LISTS"
         WRITE (UNIT=output_unit,FMT="(/,T3,A,T50,I13)") "Total number of particle pairs:",nblock_sum
         WRITE (UNIT=output_unit,FMT="(T3,A,T50,I13)") "Total number of matrix elements:",nelement_sum
         WRITE (UNIT=output_unit,FMT="(T3,A,T50,I13)") "Average number of particle pairs:",(nblock_sum+npe-1)/npe
         WRITE (UNIT=output_unit,FMT="(T3,A,T50,I13)") "Maximum number of particle pairs:",nblock_max
         WRITE (UNIT=output_unit,FMT="(T3,A,T50,I13)") "Average number of matrix element:",(nelement_sum+npe-1)/npe
         WRITE (UNIT=output_unit,FMT="(T3,A,T50,I13)") "Maximum number of matrix elements:",nelement_max
      ENDIF
    END IF

    ! Release work storage

    DEALLOCATE (nblock,STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    DEALLOCATE (nelement,STAT=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    CALL timestop(handle)

  END SUBROUTINE write_neighbor_distribution

! *****************************************************************************
!> \brief   Write a set of neighbor lists to the output unit.
!> \author  MK
!> \date    04.03.2002
!> \par History
!>       - Adapted to the new parallelized neighbor list version
!>         (26.06.2003,MK) 
!> \version 1.0
! *****************************************************************************
  SUBROUTINE write_neighbor_lists(ab,particle_set,cell,para_env,neighbor_list_section,&
    nl_type,middle_name,name,error)

    TYPE(neighbor_list_set_p_type), &
      DIMENSION(:), INTENT(IN)               :: ab
    TYPE(particle_type), DIMENSION(:), &
      POINTER                                :: particle_set
    TYPE(cell_type), POINTER                 :: cell
    TYPE(cp_para_env_type), POINTER          :: para_env
    TYPE(section_vals_type), POINTER         :: neighbor_list_section
    CHARACTER(LEN=*), INTENT(IN)             :: nl_type, middle_name, name
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=default_string_length)     :: string, unit_str
    INTEGER                                  :: atom_a, atom_b, iab, ilist, &
                                                inode, iw, mype, nlist, &
                                                nneighbor, nnode
    INTEGER, DIMENSION(3)                    :: cell_a, cell_b
    LOGICAL                                  :: failure, print_headline
    REAL(dp)                                 :: dab, unit_conv
    REAL(dp), DIMENSION(3)                   :: ra, rab, rb
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(neighbor_list_set_type), POINTER    :: neighbor_list_set
    TYPE(neighbor_list_type), POINTER        :: neighbor_list
    TYPE(neighbor_node_type), POINTER        :: neighbor_node

    failure = .FALSE.
    NULLIFY(logger)
    logger => cp_error_get_logger(error)
    IF (BTEST(cp_print_key_should_output(logger%iter_info,neighbor_list_section,&
                                         TRIM(nl_type),error=error),&
              cp_p_file)) THEN
       iw = cp_print_key_unit_nr(logger=logger,&
                                 basis_section=neighbor_list_section,&
                                 print_key_path=TRIM(nl_type),&
                                 extension=".out",&
                                 middle_name=TRIM(middle_name),&
                                 local=.TRUE.,&
                                 log_filename=.FALSE.,&
                                 file_position="REWIND",&
                                 error=error)
       mype = para_env%mepos
       CALL section_vals_val_get(neighbor_list_section,"UNIT",c_val=unit_str,error=error)
       unit_conv = cp_unit_from_cp2k(1.0_dp,TRIM(unit_str),error=error)

       ! Print headline
       string = ""
       WRITE (UNIT=string,FMT="(A,I5,A)")&
         TRIM(name)//" IN "//TRIM(unit_str)//" (PROCESS",mype,")"
       CALL compress(string)
       IF (iw > 0) WRITE (UNIT=iw,FMT="(/,/,T2,A)") TRIM(string)

       print_headline = .TRUE.

       nneighbor = 0

       DO iab=1,SIZE(ab)

         neighbor_list_set => ab(iab)%neighbor_list_set

         IF (.NOT.ASSOCIATED(neighbor_list_set)) CYCLE

         ! Loop over all atoms and their corresponding neighbor lists
         CALL get_neighbor_list_set(neighbor_list_set=neighbor_list_set,&
                                    nlist=nlist)

         neighbor_list => first_list(neighbor_list_set)

         DO ilist=1,nlist

           CALL get_neighbor_list(neighbor_list=neighbor_list,&
                                  atom=atom_a,&
                                  cell=cell_a,&
                                  nnode=nnode)

           nneighbor = nneighbor + nnode

           IF (iw > 0) THEN

             ! Print second part of the headline
             IF (print_headline) THEN
               WRITE (UNIT=iw,FMT="(/,T3,A,7X,A,2(11X,A),10X,A)")&
                 "Atom  Neighbors  Cell(i,j,k)","X","Y","Z","Distance"
               print_headline = .FALSE.
             END IF

             ra(:) = pbc(particle_set(atom_a)%r,cell,cell_a)

             WRITE (UNIT=iw,FMT="(/,T2,I5,3X,I6,3X,3I4,3F12.6)")&
               atom_a,nnode,cell_a(1:3),ra(1:3)*unit_conv

             ! Direct the work pointer to the start point of the current list

             neighbor_node => first_node(neighbor_list)

             ! Traverse the neighbor list of the current
             ! atom and print the stored information

             DO inode=1,nnode

               CALL get_neighbor_node(neighbor_node=neighbor_node,&
                                      neighbor=atom_b,&
                                      cell=cell_b,&
                                      r=rab)

               rb(:) = ra(:) + rab(:)
               dab = SQRT(rab(1)*rab(1) + rab(2)*rab(2) + rab(3)*rab(3))

               WRITE (UNIT=iw,&
                      FMT="(T10,I6,3X,3I4,3F12.6,2X,F12.6)")&
                 atom_b,cell_b(1:3),rb(1:3)*unit_conv,dab*unit_conv

               neighbor_node => next(neighbor_node)

             END DO ! inode

           END IF

           neighbor_list => next(neighbor_list)

         END DO ! ilist

       END DO ! iab

       string = ""
       WRITE (UNIT=string,FMT="(A,I12,A,I12)")&
         "Total number of neighbor interactions for process",mype,":",&
         nneighbor
       CALL compress(string)
       IF (iw > 0) WRITE (UNIT=iw,FMT="(/,T2,A)") TRIM(string)
       CALL cp_print_key_finished_output(unit_nr=iw,&
                                         logger=logger,&
                                         basis_section=neighbor_list_section,&
                                         print_key_path=TRIM(nl_type),&
                                         local=.TRUE.,&
                                         error=error)
    END IF

  END SUBROUTINE write_neighbor_lists

END MODULE qs_neighbor_lists
