!-----------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations         !
!   Copyright (C) 2000 - 2010  CP2K developers group                          !
!-----------------------------------------------------------------------------!

! *****************************************************************************
!> \brief Some utilities to handle the molecular blocks of the full matrix
!>      of the molecular orbitals
!> \par History
!>      none
!> \author MI (29.11.2004)
! *****************************************************************************
MODULE kg_gpw_fm_mol_utils

  USE cp_dbcsr_interface,              ONLY: cp_dbcsr_get_block_p,&
                                             cp_dbcsr_set
  USE cp_dbcsr_operations,             ONLY: cp_dbcsr_deallocate_matrix,&
                                             cp_dbcsr_from_sm,&
                                             sm_from_dbcsr
  USE cp_dbcsr_types,                  ONLY: cp_dbcsr_type
  USE cp_files,                        ONLY: close_file,&
                                             open_file
  USE cp_output_handling,              ONLY: cp_p_file,&
                                             cp_print_key_finished_output,&
                                             cp_print_key_should_output,&
                                             cp_print_key_unit_nr
  USE cp_para_types,                   ONLY: cp_para_env_type
  USE distribution_2d_types,           ONLY: distribution_2d_type
  USE input_section_types,             ONLY: section_vals_get_subs_vals,&
                                             section_vals_type
  USE kg_gpw_fm_mol_types,             ONLY: fm_mol_blocks_type,&
                                             get_fm_mol_block,&
                                             get_kg_fm_mol_set,&
                                             get_mol_mo_set,&
                                             kg_fm_mol_set_type,&
                                             kg_fm_p_type,&
                                             mol_mo_set_p_type,&
                                             mol_mo_set_type
  USE kinds,                           ONLY: default_path_length,&
                                             dp
  USE memory_utilities,                ONLY: reallocate
  USE message_passing,                 ONLY: mp_bcast,&
                                             mp_sum
  USE qs_mo_types,                     ONLY: wfn_restart_file_name
  USE sparse_matrix_types,             ONLY: real_matrix_type
  USE timings,                         ONLY: timeset,&
                                             timestop
#include "cp_common_uses.h"

  IMPLICIT NONE

  PRIVATE

  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'kg_gpw_fm_mol_utils'

! *** Public subroutines ***

  PUBLIC :: calculate_w_matrix_per_molecule, &
            copy_sparse2mol_block, fm_mol_cholesky_restore, &
            fm_mol_power, fm_mol_syevd, fm_mol_syevx, mol_mo_random,&
            read_mo_mol_restart, write_mo_mol_restart

CONTAINS

! *****************************************************************************
!> \brief Create the sparse matrix W =  fE*MO by MO^T, where fE is the
!>      diagonal matrix of the eigenvalues multiplied by the occupation numbers
!>      Here each molecular block of the MO is treated independently and the
!>      result is copied inthe right part of the sparse W matrix.
!>      Therefore everything is done in a loop over the molecules
!> \param fm_mol_set molecular MO blocks and other info about the molecules
!> \param matrix_w sparse matrix, the structure must be already there
!> \param ispi n
!> \param erro r
!> \author MI
! *****************************************************************************
  SUBROUTINE calculate_w_matrix_per_molecule(fm_mol_set,w_matrix_sm,ispin,distribution_2d,error)
    TYPE(kg_fm_mol_set_type), DIMENSION(:), &
      POINTER                                :: fm_mol_set
    TYPE(real_matrix_type), POINTER          :: w_matrix_sm
    INTEGER, INTENT(IN)                      :: ispin
    TYPE(distribution_2d_type), POINTER      :: distribution_2d
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: &
      routineN = 'calculate_w_matrix_per_molecule', &
      routineP = moduleN//':'//routineN

    INTEGER :: homo, iao, iat_mol, iatom, ifs, ils, imo, imol, &
      imolecule_kind, inao, istat, jao, jat_mol, jatom, jfs, jls, jnao, ldao, &
      ldmo, nao_mol, nat_mol, nmo_mol(2), nmol_local, nmolecule_kind
    INTEGER, DIMENSION(:), POINTER           :: i_atom, ifirst_ao, ilast_ao
    LOGICAL                                  :: failure, found
    REAL(dp)                                 :: eigocc
    REAL(dp), DIMENSION(:), POINTER          :: eigenvalues, occupation
    REAL(dp), DIMENSION(:, :), POINTER       :: mo, w_block, wmo, wmo_x_mot
    TYPE(cp_dbcsr_type), POINTER             :: w_matrix
    TYPE(fm_mol_blocks_type), DIMENSION(:), &
      POINTER                                :: fm_mol_blocks
    TYPE(fm_mol_blocks_type), POINTER        :: mol_block
    TYPE(kg_fm_mol_set_type), POINTER        :: fm_mol
    TYPE(mol_mo_set_p_type), DIMENSION(:), &
      POINTER                                :: mos_mol
    TYPE(mol_mo_set_type), POINTER           :: mo_set

    failure = .FALSE.

    ALLOCATE(w_matrix)!sm->dbcsr
    !CALL cp_dbcsr_init(w_matrix, error)!sm->dbcsr
    CALL cp_dbcsr_from_sm(w_matrix, w_matrix_sm, error, distribution_2d)!sm->dbcsr

    nmolecule_kind = SIZE(fm_mol_set,1)
    NULLIFY(wmo_x_mot,mo,wmo)
    ldmo = 1
    ldao = 1
    ALLOCATE(wmo(ldao,ldmo),STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
    wmo = 0.0_dp
    CALL cp_dbcsr_set(w_matrix,0.0_dp,error=error)

    DO imolecule_kind = 1, nmolecule_kind

      NULLIFY (fm_mol, fm_mol_blocks)
      fm_mol => fm_mol_set(imolecule_kind)
      CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol,&
                             nmolecule_local=nmol_local,&
                             natom=nat_mol, n_ao=nao_mol, n_mo=nmo_mol,&
                             fm_mol_blocks=fm_mol_blocks)

      IF((ldao < nao_mol) .OR. (ldmo < nmo_mol(ispin))) THEN
        ldao = nao_mol
        ldmo = nmo_mol(ispin)
        CALL reallocate(wmo,1,ldao,1,ldmo)
      END IF

      DO imol = 1, nmol_local
        NULLIFY(i_atom,ifirst_ao,ilast_ao,mos_mol, mol_block)
        mol_block => fm_mol_blocks(imol)
        CALL get_fm_mol_block(fm_mol_block = mol_block,&
                              mos=mos_mol,&
                              work = wmo_x_mot, &
                              index_atom = i_atom,&
                              ifirst_ao = ifirst_ao,&
                              ilast_ao = ilast_ao )

        NULLIFY(eigenvalues, mo, mo_set, occupation)
        mo_set => mos_mol(ispin)%mo_set
        CALL get_mol_mo_set(mo_set, homo = homo, mo = mo ,&
                            eigenvalues = eigenvalues,&
                            occupation_numbers = occupation )

        DO imo = 1,homo
          eigocc = eigenvalues(imo)*occupation(imo)
          CALL DCOPY(nao_mol,mo(1,imo),1,wmo(1,imo),1)
          CALL DSCAL(nao_mol,eigocc,wmo(1,imo),1)
        END DO
        CALL DGEMM('N','T',nao_mol,nao_mol,homo,1.0_dp,wmo(1,1),ldao,&
                   mo(1,1),ldao,0.0_dp,wmo_x_mot,SIZE(wmo_x_mot,1))

        DO iat_mol = 1,nat_mol
          iatom = i_atom(iat_mol)
          ifs = ifirst_ao(iat_mol)
          ils = ilast_ao(iat_mol)
          inao = ils - ifs + 1

          DO jat_mol = iat_mol,nat_mol
            jatom = i_atom(jat_mol)
            jfs = ifirst_ao(jat_mol)
            jls = ilast_ao(jat_mol)
            jnao = jls - jfs + 1

            NULLIFY(w_block)
            IF(iatom <= jatom) THEN

              CALL cp_dbcsr_get_block_p(matrix=w_matrix,&
                   row=iatom,col=jatom,block=w_block,found=found)
              CPPrecondition(ASSOCIATED(w_block),cp_failure_level,routineP,error,failure)
              CPPrecondition(inao==SIZE(w_block,1),cp_failure_level,routineP,error,failure)
              CPPrecondition(jnao==SIZE(w_block,2),cp_failure_level,routineP,error,failure)

              DO jao = 1,jnao
                DO iao = 1,inao
                  w_block(iao,jao) = wmo_x_mot(ifs+iao-1,jfs+jao-1)
                END DO  ! iao
              END DO  ! jao

            ELSE

               CALL cp_dbcsr_get_block_p(matrix=w_matrix,&
                    row=jatom,col=iatom,block=w_block,found=found)
              CPPrecondition(ASSOCIATED(w_block),cp_failure_level,routineP,error,failure)
              CPPrecondition(jnao==SIZE(w_block,1),cp_failure_level,routineP,error,failure)
              CPPrecondition(inao==SIZE(w_block,2),cp_failure_level,routineP,error,failure)

              DO iao = 1,inao
                DO jao = 1,jnao
                  w_block(jao,iao) = wmo_x_mot(jfs+jao-1,ifs+iao-1)
                END DO  ! jao
              END DO  ! iao

            END IF

          END DO  ! jat_mol
        END DO  ! iat_mol
      END DO  ! imol

    END DO  ! imolecule_kind

    DEALLOCATE(wmo,STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

    CALL sm_from_dbcsr ( w_matrix_sm, w_matrix, distribution_2d, error=error )!sm->dbcsr
    CALL cp_dbcsr_deallocate_matrix ( w_matrix, error=error )!sm->dbcsr

  END SUBROUTINE calculate_w_matrix_per_molecule

! *****************************************************************************
!> \brief Multiply one block (molecular block for KG_GPW) of the MO full matrix by the
!>      corresponding cholesky decomposition of the overlap matrix, U
!> \param vectors_in set of MO as input
!> \param nmo number of MO we need
!> \param upper_dec cholesky decomposition of the overlap matrix, U
!> \param vectors_out set of MO as output after restore
!> \param op can be "SOLVE" (out = U^-1 * in ) or "MULTIPLY"   (out = U * in )
!> \param pos can be "LEFT" or "RIGHT" (U at the left or at the right)
!> \author MI
! *****************************************************************************
  SUBROUTINE fm_mol_cholesky_restore(vectors_in,nmo,upper_dec,vectors_out,&
                                     op,pos,error)

    REAL(dp), DIMENSION(:, :)                :: vectors_in
    INTEGER, INTENT(IN)                      :: nmo
    REAL(dp), DIMENSION(:, :), POINTER       :: upper_dec, vectors_out
    CHARACTER(LEN=*), INTENT(IN)             :: op
    CHARACTER(LEN=*), INTENT(IN), OPTIONAL   :: pos
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'fm_mol_cholesky_restore', &
      routineP = moduleN//':'//routineN

    CHARACTER                                :: my_pos
    INTEGER                                  :: nao
    LOGICAL                                  :: failure
    REAL(dp)                                 :: alpha

    failure = .FALSE.

    CALL cp_assert(op == "SOLVE" .OR. op == "MULTIPLY",&
         cp_failure_level,cp_assertion_failed,routineP,&
         "wrong argument op"//&
                CPSourceFileRef,&
         only_ionode=.TRUE.)

    IF (PRESENT(pos)) THEN
       SELECT CASE(pos)
       CASE("LEFT")
         my_pos='L'
       CASE("RIGHT")
         my_pos='R'
       CASE DEFAULT
          CALL cp_assert(.FALSE.,&
               cp_failure_level,cp_assertion_failed,routineP,&
               "wrong argument pos"//&
                CPSourceFileRef,&
         only_ionode=.TRUE.)
       END SELECT
    ELSE
       my_pos='L'
    ENDIF

    nao = SIZE(upper_dec,1)
    CPPostcondition(SIZE(vectors_out,1)==nao,cp_failure_level,routineP,error,failure)
    CPPostcondition(SIZE(vectors_out,2)>=nmo,cp_failure_level,routineP,error,failure)
    alpha = 1.0_dp
    CALL dcopy(nmo*nao,vectors_in(1,1),1,vectors_out(1,1),1)
    IF (op.EQ."SOLVE") THEN
      CALL dtrsm(my_pos,'U','N','N',nao,nmo,alpha,upper_dec(1,1),nao,&
                 vectors_out(1,1),nao)
    ELSE
      CALL dtrmm(my_pos,'U','N','N',nao,nmo,alpha,upper_dec(1,1),nao,&
                 vectors_out(1,1),nao)
    ENDIF

  END SUBROUTINE fm_mol_cholesky_restore

! *****************************************************************************
!> \brief Raise the real symmetric n by n matrix to the power given by
!>      exponent. All eigenvectors with a corresponding eigenvalue lower
!>      than threshold are quenched. Here the n by n matrix is the block of
!>      the overlap matrix corresponding to one molecule
!> \param matrix block of the overlap matrix
!> \param eigenvectors :
!> \param ldv leading dimension of the matrix
!> \param ncv number of vectors
!> \param exponen t
!> \param threshold to quench the eigenvalues
!> \param n_dependen t
!> \author MI
! *****************************************************************************
  SUBROUTINE fm_mol_power(matrix,eigenvectors,ldv,ncv,exponent,threshold,n_dependent,error)

    REAL(dp), DIMENSION(:, :), POINTER       :: matrix
    REAL(dp), DIMENSION(:, :)                :: eigenvectors
    INTEGER, INTENT(IN)                      :: ldv, ncv
    REAL(dp), INTENT(IN)                     :: exponent, threshold
    INTEGER, INTENT(OUT)                     :: n_dependent
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'fm_mol_power', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle, icol, istat, ncol, &
                                                nrow
    LOGICAL                                  :: failure
    REAL(dp)                                 :: f, p
    REAL(dp), ALLOCATABLE, DIMENSION(:)      :: eigenvalues

    CALL timeset(routineN,handle)
    failure = .FALSE.

    nrow = SIZE(matrix,1)
    ncol = SIZE(matrix,2)

    n_dependent = 0

    ALLOCATE(eigenvalues(ncol),STAT=istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)
    eigenvalues(:) = 0.0_dp

    CALL fm_mol_syevx(matrix,eigenvectors,ldv,ncv,eigenvalues,ncol,error=error)

    p = 0.5_dp*exponent

    DO icol = 1,ncol
       IF(eigenvalues(icol) < threshold) THEN
         n_dependent = n_dependent + 1
         eigenvectors(1:nrow,icol) = 0.0_dp
       ELSE

         f = eigenvalues(icol)**p
         eigenvectors(1:nrow,icol) =&
             f*eigenvectors(1:nrow,icol)
       END IF
    END DO

    CALL dsyrk("U","N",nrow,ncol,1.0_dp,eigenvectors(1,1),ldv,&
                                 0.0_dp,matrix(1,1),nrow)

    DEALLOCATE(eigenvalues, STAT = istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)

    CALL timestop(handle)

  END SUBROUTINE fm_mol_power

! *****************************************************************************
!> \brief computes all eigenvalues and vectors of a real symmetric matrix
!> \param matrix block of the full matrix
!> \param nrow leading dimension of the matrix
!> \param eigenvalue s
!> \param matrix_ou t
!> \param erro r
!> \author MI
! *****************************************************************************
  SUBROUTINE fm_mol_syevd(matrix,nrow,eigenvalues,matrix_out,error)

    REAL(dp), DIMENSION(:, :), POINTER       :: matrix
    INTEGER, INTENT(IN)                      :: nrow
    REAL(dp), DIMENSION(:)                   :: eigenvalues
    REAL(dp), DIMENSION(:, :)                :: matrix_out
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'fm_mol_syevd', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: info, istat, lda, liwork, &
                                                lwork
    INTEGER, DIMENSION(:), POINTER           :: iwork
    LOGICAL                                  :: failure
    REAL(KIND=dp), DIMENSION(:), POINTER     :: work

    failure = .FALSE.

    IF(ASSOCIATED(matrix)) THEN
      lda = SIZE(matrix,1)
      CPPostcondition(nrow<=lda,cp_failure_level,routineP,error,failure)

      lwork=1+6*nrow+2*nrow**2+5000
      liwork=5*nrow+3
      ALLOCATE(work(lwork),STAT=istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)
      ALLOCATE(iwork(liwork),STAT=istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)
      CALL DSYEVD('V','U', nrow, matrix(1,1), nrow, eigenvalues(1),&
                  work(1), lwork, iwork(1), liwork, info)
    CPPrecondition(info==0,cp_failure_level,routineP,error,failure)

      CALL DCOPY (nrow*nrow,matrix(1,1),1,matrix_out(1,1),1)

    ELSE
      CALL cp_assert(.FALSE.,cp_failure_level,cp_assertion_failed,routineP,&
              " Error in dsyevd"//&
              CPSourceFileRef,&
              only_ionode=.TRUE.)
    END IF

  END SUBROUTINE fm_mol_syevd

! *****************************************************************************
!> \brief Diagonalise the symmetric n by n matrix using the LAPACK library
!>      Compute ncol egenvalues and eigenvectors
!> \param matrix block of the full matrix
!> \param eigenvetor s
!> \param ldv leading dimension of the matrix
!> \param ncv number of columns available
!> \param eigenvalue s
!> \param ncol number of vector to be calculated
!> \param erro r
!> \author MI
! *****************************************************************************
  SUBROUTINE fm_mol_syevx(matrix,eigenvetors,ldv,ncv,eigenvalues,ncol,error)

    REAL(dp), DIMENSION(:, :), POINTER       :: matrix
    REAL(dp), DIMENSION(:, :)                :: eigenvetors
    INTEGER, INTENT(IN)                      :: ldv, ncv
    REAL(dp), DIMENSION(:), INTENT(OUT)      :: eigenvalues
    INTEGER, INTENT(IN)                      :: ncol
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'fm_mol_syevx', &
      routineP = moduleN//':'//routineN
    REAL(KIND=dp), PARAMETER                 :: vl = 0.0_dp, vu = 0.0_dp

    INTEGER                                  :: info, istat, ldb, liwork, &
                                                lwork, m, nb, nrow
    INTEGER, ALLOCATABLE, DIMENSION(:)       :: ifail, iwork
    INTEGER, EXTERNAL                        :: ilaenv
    LOGICAL                                  :: failure
    REAL(dp)                                 :: abstol
    REAL(dp), ALLOCATABLE, DIMENSION(:)      :: w, work
    REAL(KIND=dp), EXTERNAL                  :: dlamch

    failure =  .FALSE.
    abstol = 2.0_dp*dlamch("S")
    nrow = SIZE(matrix,1)
    ldb = ldv  ! SIZE(eigenvetors,1)
!    CPPostcondition(ncol<=SIZE(eigenvetors,2),cp_failure_level,routineP,error,failure)
    CPPostcondition(ncol<=ncv,cp_failure_level,routineP,error,failure)
    CPPostcondition(ncol<=SIZE(eigenvalues,1),cp_failure_level,routineP,error,failure)

    ALLOCATE(w(nrow), STAT =istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)
    w(:) = 0.0_dp

    eigenvalues(:) = 0.0_dp

    nb = MAX(ilaenv(1,"DSYTRD","U",nrow,-1,-1,-1),&
             ilaenv(1,"DORMTR","U",nrow,-1,-1,-1))

    lwork = MAX((nb + 3)*nrow,8*nrow)+nrow ! sun bug fix
    liwork = 5*nrow

    ALLOCATE (ifail(nrow),STAT=istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE (iwork(liwork),STAT=istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE (work(lwork),STAT=istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)

    CALL dsyevx("V","I","U",nrow,matrix(1,1),nrow,vl,vu,1,ncol,abstol,&
                m,w,eigenvetors(1,1),ldb,work(1),lwork,&
                iwork(1),ifail(1),info)
    CPPrecondition(info==0,cp_failure_level,routineP,error,failure)

    eigenvalues(1:ncol) = w(1:ncol)

!   *** Release work storage ***
    DEALLOCATE(w,ifail,iwork,work,STAT =istat)
    CPPrecondition(istat==0,cp_failure_level,routineP,error,failure)

  END SUBROUTINE fm_mol_syevx

! *****************************************************************************
!> \brief Assign a random value to the MO block matrix
!> \param mo block of the full matrix
!> \param mepo s
!> \param num_p e
!> \param ncol number of vector to be calculated
!> \param start_col from which column to start
!> \param erro r
!> \author MI
! *****************************************************************************
  SUBROUTINE mol_mo_random(mo,mepos,num_pe,ncol,start_col,error)

    REAL(dp), DIMENSION(:, :), POINTER       :: mo
    INTEGER, INTENT(IN)                      :: mepos, num_pe
    INTEGER, INTENT(IN), OPTIONAL            :: ncol, start_col
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'mol_mo_random', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: icol, my_ncol, my_start, nrow
    INTEGER, DIMENSION(4), SAVE              :: ISEED
    LOGICAL                                  :: failure
    LOGICAL, SAVE                            :: FIRST = .TRUE.

    failure = .FALSE.

    IF (FIRST) THEN
      ISEED(1)= mepos
      ISEED(2)= mepos + num_pe
      ISEED(3)= mepos + 2*num_pe
      ISEED(4)= mepos + 3*num_pe ! last one has to be odd
      IF(MOD(ISEED(4),2) .EQ. 0) ISEED(4)=ISEED(4)+1
      FIRST=.FALSE.
    ENDIF

    CPPostcondition(ASSOCIATED(mo),cp_failure_level,routineP,error,failure)
    my_start = 1
    IF(PRESENT(start_col)) my_start = start_col
    my_ncol = SIZE(mo,2)
    IF(PRESENT(ncol)) THEN
      CPPostcondition(my_ncol>=ncol,cp_failure_level,routineP,error,failure)
      my_ncol = ncol
    END IF

    nrow = SIZE(mo,1)
    DO icol = my_start ,my_start + my_ncol - 1
      CALL dlarnv(1,iseed,nrow,mo(1,icol))
    END DO

  END SUBROUTINE mol_mo_random

! *****************************************************************************
!> \brief Copy a block of a sparse matrix into the right part of the
!>      molecular block of a full matrix (full for one molecule not the whole system)
!> \param matri x
!> \param mol_bloc k
!> \param matrix_mol_bloc k
!> \param nato m
!> \param nro w
!> \param nco l
!> \param erro r
!> \author MI
! *****************************************************************************
  SUBROUTINE copy_sparse2mol_block(matrix_sm, mol_block, matrix_mol_block, &
                                   natom, nrow, ncol, error)

    TYPE(real_matrix_type), POINTER          :: matrix_sm
    TYPE(fm_mol_blocks_type), POINTER        :: mol_block
    REAL(dp), DIMENSION(:, :)                :: matrix_mol_block
    INTEGER, INTENT(IN)                      :: natom, nrow, ncol
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'copy_sparse2mol_block', &
      routineP = moduleN//':'//routineN

    INTEGER :: col_first, col_last, iat, iatom, icol, icol_mol_s, irow, &
      irow_mol_s, jat, jatom, row_first, row_last
    INTEGER, DIMENSION(:), POINTER           :: i_atom, ifirst_ao, ilast_ao
    LOGICAL                                  :: failure, found
    REAL(dp), DIMENSION(:, :), POINTER       :: s_block
    TYPE(cp_dbcsr_type), POINTER             :: matrix

    failure = .FALSE.

    ALLOCATE(matrix)!sm->dbcsr
    !CALL cp_dbcsr_init(matrix, error)!sm->dbcsr
    CALL cp_dbcsr_from_sm(matrix, matrix_sm, error)!sm->dbcsr

    NULLIFY(i_atom,ifirst_ao,ilast_ao)

    CALL get_fm_mol_block(fm_mol_block = mol_block,&
                          index_atom = i_atom,&
                          ifirst_ao = ifirst_ao,&
                          ilast_ao = ilast_ao )

    CPPrecondition(SIZE(matrix_mol_block,1)>=nrow,cp_failure_level,routineP,error,failure)
    CPPrecondition(SIZE(matrix_mol_block,2)>=ncol,cp_failure_level,routineP,error,failure)

 ! Transfer the sparse matrix in the molecular block
    DO iat = 1,natom
      iatom = i_atom(iat)
      row_first = ifirst_ao(iat)
      row_last  = ilast_ao(iat)
      DO jat = 1 , natom
        jatom = i_atom(jat)
        col_first = ifirst_ao(jat)
        col_last  = ilast_ao(jat)

        NULLIFY(s_block)
        IF (iatom <= jatom) THEN
           CALL cp_dbcsr_get_block_p(matrix=matrix,&
                row=iatom,col=jatom,BLOCK=s_block,found=found)
          icol =  1
          DO icol_mol_s = col_first, col_last
            irow = 1
            DO irow_mol_s = row_first, row_last
              matrix_mol_block(irow_mol_s,icol_mol_s) = s_block(irow,icol)
              irow = irow + 1
            END DO  ! irow_mol_s
            icol = icol + 1
          END DO   ! icol_mol_s
        ELSE
          CALL cp_dbcsr_get_block_p(matrix=matrix,&
               row=jatom,col=iatom,BLOCK=s_block,found=found)
          icol =  1
          DO icol_mol_s = col_first, col_last
            irow = 1
            DO irow_mol_s = row_first, row_last
              matrix_mol_block(irow_mol_s,icol_mol_s) = s_block(icol,irow)
              irow = irow + 1
            END DO  ! irow_mol_s
            icol = icol + 1
          END DO  ! icol_mol_s
        END IF
      END DO  ! jat
    END DO  ! iat

    CALL cp_dbcsr_deallocate_matrix ( matrix, error=error )!sm->dbcsr

  END SUBROUTINE copy_sparse2mol_block

! *****************************************************************************
!> \brief Write a RESTART file for the KG_GPW calculation
!>      The MO are written in a list of blocks, each one corresponding
!>      to one molecule
!>      Since the molecule are distributed over the projectors, some commuication
!>      is required to write the list in the correct order (order of the molecule)
!> \param kg_fm containes all the info abut the molecules and the MOof each molecule
!> \param id_nr give an index to the restart if =/ 1
!> \note
!>      The KG_GPW RESTART file has a special form. It should be read and written only
!>      in a KG_GPW calculation. However the defoult name is still RESTART.
!>      An initial flag identifies this as a KG_GPW RESTART.
!> \author MI
! *****************************************************************************
  SUBROUTINE write_mo_mol_restart(kg_fm,para_env,nspins,id_nr,force_env_section,error)

    TYPE(kg_fm_p_type)                       :: kg_fm
    TYPE(cp_para_env_type), POINTER          :: para_env
    INTEGER, INTENT(IN)                      :: nspins, id_nr
    TYPE(section_vals_type), POINTER         :: force_env_section
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'write_mo_mol_restart', &
      routineP = moduleN//':'//routineN

    CHARACTER(LEN=30), DIMENSION(2) :: keys = (/&
      "DFT%SCF%PRINT%RESTART_HISTORY","DFT%SCF%PRINT%RESTART        "/)
    CHARACTER(LEN=6)                         :: method
    INTEGER :: group, handle, ikey, imo, imol_found, imol_global, imol_local, &
      imol_proc, imolecule_kind, ires, ispin, istat, n_ao, n_mo(2), nao_max, &
      nelectron_global, nelectron_spin(2), nmo_max, nmolecule_global, &
      nmolecule_kind, nmolecule_local, source
    INTEGER, ALLOCATABLE, DIMENSION(:)       :: nao_molkind, natom_molkind, &
                                                nmol_molkind
    INTEGER, ALLOCATABLE, DIMENSION(:, :)    :: nel_molkind
    LOGICAL                                  :: failure
    REAL(dp), ALLOCATABLE, DIMENSION(:, :)   :: eig_buffer, occ_buffer
    REAL(dp), ALLOCATABLE, &
      DIMENSION(:, :, :)                     :: mo_buffer
    REAL(dp), DIMENSION(:), POINTER          :: eigenvalues, &
                                                occupation_numbers
    REAL(dp), DIMENSION(:, :), POINTER       :: mo
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(fm_mol_blocks_type), DIMENSION(:), &
      POINTER                                :: mol_blocks
    TYPE(fm_mol_blocks_type), POINTER        :: mol_block
    TYPE(kg_fm_mol_set_type), POINTER        :: fm_mol_set
    TYPE(mol_mo_set_p_type), DIMENSION(:), &
      POINTER                                :: mos
    TYPE(mol_mo_set_type), POINTER           :: mo_set

    CALL timeset(routineN,handle)
    failure = .FALSE.
    group = para_env%group
    source = para_env%source
    logger=>cp_error_get_logger(error)
    
   IF (BTEST(cp_print_key_should_output(logger%iter_info,&
           force_env_section,keys(1),error=error),cp_p_file) .OR.  &
        BTEST(cp_print_key_should_output(logger%iter_info,&
           force_env_section,keys(2),error=error),cp_p_file) ) THEN

      method ="KG_GPW"
      nmolecule_kind = SIZE(kg_fm%kg_fm_mol_set)
      nmolecule_global = kg_fm%nmolecule_global
      nelectron_global = kg_fm%nelectron_global
      nao_max = kg_fm%nao_max
      nmo_max = kg_fm%nmo_max

      DO ikey=1,SIZE(keys)

         IF (BTEST(cp_print_key_should_output(logger%iter_info,&
              force_env_section,keys(ikey),error=error),cp_p_file)) THEN

            ires = cp_print_key_unit_nr(logger,force_env_section,keys(ikey),&
                 extension=".wfn", file_status="REPLACE", file_action="WRITE",&
                 do_backup=.TRUE., file_form="UNFORMATTED", error=error)
    
            IF (ires>0) THEN

               !     *** create some info about the molecules ***
               ALLOCATE(nmol_molkind(nmolecule_kind),STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
               ALLOCATE(nao_molkind(nmolecule_kind),STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
               ALLOCATE(natom_molkind(nmolecule_kind),STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
               ALLOCATE(nel_molkind(2,nmolecule_kind),STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
               
               DO imolecule_kind = 1,nmolecule_kind
                  NULLIFY(fm_mol_set)
                  fm_mol_set => kg_fm%kg_fm_mol_set(imolecule_kind)
                  CALL  get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_set,&
                                          nmol_kind_global=nmol_molkind(imolecule_kind),&
                                          n_ao = nao_molkind(imolecule_kind),&
                                          natom = natom_molkind(imolecule_kind), &
                                          nelectron_spin=nel_molkind(1:2,imolecule_kind) )
               END DO

               WRITE (ires) method, nmolecule_kind, nmolecule_global, nelectron_global, nspins
               WRITE (ires) nmol_molkind
               WRITE (ires) nao_molkind
               WRITE (ires) natom_molkind
               WRITE (ires) nel_molkind(1,1:nmolecule_kind)
               WRITE (ires) nel_molkind(2,1:nmolecule_kind)
               
               DEALLOCATE(nmol_molkind,STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
               DEALLOCATE(nao_molkind,STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
               DEALLOCATE(natom_molkind,STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
               DEALLOCATE(nel_molkind,STAT=istat)
               CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
            END IF

            ! The following is done by all the processor
            ! The MOs etc are communicated to all the processor, one molecule at a time

            ALLOCATE(mo_buffer(nao_max,nmo_max,nspins),STAT=istat)
            CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
            ALLOCATE(eig_buffer(nmo_max,nspins),STAT=istat)
            CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
            ALLOCATE(occ_buffer(nmo_max,nspins),STAT=istat)
            CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

            DO imol_global = 1,nmolecule_global
               imol_found = 0
               imol_proc = -1
               CALL dcopy(nao_max*nmo_max*nspins,0.0_dp,0,mo_buffer(1,1,1),1)
               CALL dcopy(nmo_max*nspins,0.0_dp,0,eig_buffer(1,1),1)
               CALL dcopy(nmo_max*nspins,0.0_dp,0,occ_buffer(1,1),1)
               DO imolecule_kind = 1,nmolecule_kind
                  NULLIFY(fm_mol_set,mol_blocks,mos, mo_set)
                  NULLIFY(mo,eigenvalues,occupation_numbers)
                  fm_mol_set => kg_fm%kg_fm_mol_set(imolecule_kind)
                  CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_set,  &
                       nmolecule_local = nmolecule_local,&
                       n_ao=n_ao, n_mo=n_mo,nelectron_spin=nelectron_spin,&
                       fm_mol_blocks=mol_blocks)
                  DO imol_local = 1,nmolecule_local
                     mol_block => mol_blocks(imol_local)
                     IF(imol_global == mol_block%imol_global) THEN
                        ! Molecule found, now retrieve the MO etc
                        ! fill mo_buffer, eig_buffer, occ_buffer one spin at a time
                        ! write them down in the restart
                        ! and ciaociao
                        imol_found = 1
                        imol_proc = para_env%mepos
                        CALL get_fm_mol_block(fm_mol_block=mol_block, mos = mos)
                        DO ispin = 1,nspins
                           mo_set => mos(ispin)%mo_set
                           CALL get_mol_mo_set(mo_set=mo_set,mo=mo,eigenvalues=eigenvalues,&
                                occupation_numbers=occupation_numbers)

                           DO imo = 1 , n_mo(ispin)
                              CALL DCOPY(n_ao,mo(1,imo),1,mo_buffer(1,imo,ispin),1)
                              eig_buffer(imo,ispin) = eigenvalues(imo)
                              occ_buffer(imo,ispin) = occupation_numbers(imo)
                           END DO
                        END DO
                     END IF
                  END DO  ! imol_local
               END DO  ! imolecule_kind

               CALL mp_sum(imol_found,group)

               IF(imol_found==0 ) THEN
                  CALL cp_assert(.FALSE.,cp_failure_level,cp_assertion_failed,routineP,&
                    " WRITE RESTART : at least one molecule has not been found on any proc. "//&
CPSourceFileRef,&
               only_ionode=.TRUE.)
               ELSEIF(imol_found >=2 ) THEN
                  CALL cp_assert(.FALSE.,cp_failure_level,cp_assertion_failed,routineP,&
                    " WRITE RESTART : at least one molecule is found on more than one proc. "//&
CPSourceFileRef,&
               only_ionode=.TRUE.)
               ELSE
                  CALL mp_sum(mo_buffer,group)
                  CALL mp_sum(eig_buffer,group)
                  CALL mp_sum(occ_buffer,group)
                  DO ispin = 1,nspins
                     IF((ires>0).AND.(n_mo(ispin)>0)) THEN
                        WRITE (ires) imol_global , ispin
                        WRITE (ires) n_mo(ispin),n_ao,nelectron_spin(ispin)
                        WRITE (ires) eig_buffer(1:n_mo(ispin),ispin)
                        WRITE (ires) occ_buffer(1:n_mo(ispin),ispin)
                        DO imo = 1,n_mo(ispin)
                           WRITE (ires) mo_buffer(1:n_ao,imo,ispin)
                        END DO
                     END IF
                  END DO
               END IF
            END DO ! imol_global

            DEALLOCATE(mo_buffer,STAT=istat)
            CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
            DEALLOCATE(eig_buffer,STAT=istat)
            CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
            DEALLOCATE(occ_buffer,STAT=istat)
            CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

            CALL cp_print_key_finished_output(ires,logger,force_env_section,&
                 TRIM(keys(ikey)), error=error)
         END IF
      END DO
   END IF
      
   CALL timestop(handle)
   
 END SUBROUTINE write_mo_mol_restart

! *****************************************************************************
!> \brief Read  the RESTART file for the KG_GPW calculation
!>      The MO are written in a list of blocks, each one corresponding
!>      to one molecule
!>      Since the molecule are distributed over the projectors, some commuication
!>      is required to write the list in the correct order (order of the molecule)
!> \param kg_fm containes all the info abut the molecules and the MOof each molecule
!> \param nspin s
!> \param id_nr give an index to the restart if =/ 1
!> \param erro r
!> \note
!>      The KG_GPW RESTART file has a special form. It should be read and written only
!>      in a KG_GPW calculation. However the defoult name is still RESTART.
!>      An initial flag identifies this as a KG_GPW RESTART.
!> \author MI
! *****************************************************************************
  SUBROUTINE read_mo_mol_restart(kg_fm,para_env,nspins,id_nr,force_env_section,error)

    TYPE(kg_fm_p_type)                       :: kg_fm
    TYPE(cp_para_env_type), POINTER          :: para_env
    INTEGER, INTENT(IN)                      :: nspins, id_nr
    TYPE(section_vals_type), POINTER         :: force_env_section
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(LEN=*), PARAMETER :: routineN = 'read_mo_mol_restart', &
      routineP = moduleN//':'//routineN

    CHARACTER(LEN=6)                         :: method
    CHARACTER(LEN=default_path_length)       :: file_name
    INTEGER :: group, handle, imo, imol_found, imol_global, imol_global_read, &
      imol_local, imol_proc, imolecule_kind, ispin, ispin_read, istat, n_ao, &
      n_ao_read, n_mo(2), n_mo_read(2), nao_max, natom, nelectron_global, &
      nelectron_global_read, nelectron_spin(2), nmo_max, nmo_min, &
      nmol_kind_global, nmolecule_global, nmolecule_global_read, &
      nmolecule_kind, nmolecule_kind_read, nmolecule_local, nspins_read, &
      restart_unit, source
    INTEGER, ALLOCATABLE, DIMENSION(:)       :: nao_molkind_read, &
                                                natom_molkind_read, &
                                                nmol_molkind_read
    INTEGER, ALLOCATABLE, DIMENSION(:, :)    :: nel_molkind_read
    LOGICAL                                  :: exist, failure
    REAL(dp), ALLOCATABLE, DIMENSION(:, :)   :: eig_buffer, occ_buffer
    REAL(dp), ALLOCATABLE, &
      DIMENSION(:, :, :)                     :: mo_buffer
    REAL(dp), DIMENSION(:), POINTER          :: eigenvalues, &
                                                occupation_numbers
    REAL(dp), DIMENSION(:, :), POINTER       :: mo
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(fm_mol_blocks_type), DIMENSION(:), &
      POINTER                                :: mol_blocks
    TYPE(fm_mol_blocks_type), POINTER        :: mol_block
    TYPE(kg_fm_mol_set_type), POINTER        :: fm_mol_set
    TYPE(mol_mo_set_p_type), DIMENSION(:), &
      POINTER                                :: mos
    TYPE(mol_mo_set_type), POINTER           :: mo_set
    TYPE(section_vals_type), POINTER         :: dft_section

    CALL timeset(routineN,handle)

    failure = .FALSE.

    group = para_env%group
    source = para_env%source
    logger => cp_error_get_logger(error)

    method ="KG_GPW"
    nmolecule_kind = SIZE(kg_fm%kg_fm_mol_set)
    nmolecule_global = kg_fm%nmolecule_global
    nelectron_global = kg_fm%nelectron_global
    nao_max = kg_fm%nao_max
    nmo_max = kg_fm%nmo_max

    IF (para_env%ionode) THEN

      dft_section=>section_vals_get_subs_vals(force_env_section,"DFT",error=error)
      CALL wfn_restart_file_name(file_name,exist,dft_section,logger,error=error)
      IF (id_nr/=1) THEN
        file_name(LEN_TRIM(file_name):LEN(file_name)) = "-"//&
                    ADJUSTL(cp_to_string(id_nr))
      END IF

      CALL open_file(file_name=file_name,&
                     file_action="READ",&
                     file_form="UNFORMATTED",&
                     file_status="OLD",&
                     unit_number=restart_unit)

!     *** read some info about the molecules ***

      ALLOCATE(nmol_molkind_read(nmolecule_kind),STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
      ALLOCATE(nao_molkind_read(nmolecule_kind),STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
      ALLOCATE(natom_molkind_read(nmolecule_kind),STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
      ALLOCATE(nel_molkind_read(2,nmolecule_kind),STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

      READ (restart_unit) method, nmolecule_kind_read, nmolecule_global_read, nelectron_global_read, nspins_read

      CALL cp_assert(method=="KG_GPW",cp_failure_level,cp_assertion_failed,routineP,&
              "READ RESTART: this is not a KG_GPW restart file"//&
              CPSourceFileRef,&
              only_ionode=.TRUE.)
      CALL cp_assert(nmolecule_kind_read  == nmolecule_kind ,cp_failure_level,cp_assertion_failed,routineP,&
              "READ RESTART: different number of molecule kinds "//&
              CPSourceFileRef,&
              only_ionode=.TRUE.)
      CALL cp_assert(nmolecule_global_read == nmolecule_global_read,cp_failure_level,cp_assertion_failed,routineP,&
              "READ RESTART: different number of molecules "//&
              CPSourceFileRef,&
              only_ionode=.TRUE.)
      CALL cp_assert(nelectron_global_read == nelectron_global_read ,cp_failure_level,cp_assertion_failed,routineP,&
              "READ RESTART: different number of electrons "//&
              CPSourceFileRef,&
              only_ionode=.TRUE.)
      CALL cp_assert( nspins_read == nspins,cp_failure_level,cp_assertion_failed,routineP,&
             "READ RESTART: nspin is not equal (program this !!)"//&
              CPSourceFileRef,&
              only_ionode=.TRUE.)

      READ (restart_unit) nmol_molkind_read
      READ (restart_unit) nao_molkind_read
      READ (restart_unit) natom_molkind_read
      READ (restart_unit) nel_molkind_read(1,1:nmolecule_kind)
      READ (restart_unit) nel_molkind_read(2,1:nmolecule_kind)

      DO imolecule_kind = 1,nmolecule_kind
        NULLIFY(fm_mol_set)
        fm_mol_set => kg_fm%kg_fm_mol_set(imolecule_kind)
        CALL  get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_set,&
                                nmol_kind_global=nmol_kind_global,&
                                n_ao = n_ao,&
                                natom = natom, &
                                nelectron_spin=nelectron_spin)
        CPPostcondition(nmol_kind_global==nmol_molkind_read(imolecule_kind),cp_warning_level,routineP,error,failure)
        CPPostcondition(n_ao==nao_molkind_read(imolecule_kind),cp_warning_level,routineP,error,failure)
        CPPostcondition(natom==natom_molkind_read(imolecule_kind),cp_warning_level,routineP,error,failure)
        IF(failure) THEN
          CALL cp_assert(.FALSE.,cp_warning_level,cp_assertion_failed,routineP,&
                  " READ RESTART : molecule kind is not consistent "//&
CPSourceFileRef,&
               only_ionode=.TRUE.)
        END IF
      END DO

      DEALLOCATE(nmol_molkind_read,STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
      DEALLOCATE(nao_molkind_read,STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
      DEALLOCATE(natom_molkind_read,STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
      DEALLOCATE(nel_molkind_read,STAT=istat)
      CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

    END IF  ! ionode

    ALLOCATE(mo_buffer(nao_max,nmo_max,nspins),STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE(eig_buffer(nmo_max,nspins),STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE(occ_buffer(nmo_max,nspins),STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

    DO imol_global = 1,nmolecule_global
      imol_found = 0
      imol_proc = 0
      CALL dcopy(nao_max*nmo_max*nspins,0.0_dp,0,mo_buffer(1,1,1),1)
      CALL dcopy(nmo_max*nspins,0.0_dp,0,eig_buffer(1,1),1)
      CALL dcopy(nmo_max*nspins,0.0_dp,0,occ_buffer(1,1),1)

      IF(para_env%ionode) THEN
        DO ispin = 1,nspins
          READ (restart_unit) imol_global_read , ispin_read
          READ (restart_unit) n_mo_read(ispin),n_ao_read,&
                                nelectron_spin(ispin)
          READ (restart_unit) eig_buffer(1:n_mo_read(ispin),ispin)
          READ (restart_unit) occ_buffer(1:n_mo_read(ispin),ispin)
          DO imo = 1,n_mo_read(ispin)
           READ (restart_unit) mo_buffer(1:n_ao_read,imo,ispin)
          END DO
        END DO
      END IF
      CALL mp_bcast(n_mo_read,source,group)
      CALL mp_bcast(n_ao_read,source,group)

      CALL mp_bcast(mo_buffer,source,group)
      CALL mp_bcast(occ_buffer,source,group)
      CALL mp_bcast(eig_buffer,source,group)

      DO imolecule_kind = 1,nmolecule_kind
        NULLIFY(fm_mol_set,mol_blocks,mos, mo_set)
        NULLIFY(mo,eigenvalues,occupation_numbers)
        fm_mol_set => kg_fm%kg_fm_mol_set(imolecule_kind)
        CALL get_kg_fm_mol_set(kg_fm_mol_set=fm_mol_set,  &
                               nmolecule_local = nmolecule_local,&
                               n_ao=n_ao, n_mo=n_mo,nelectron_spin=nelectron_spin,&
                               fm_mol_blocks=mol_blocks)
        DO imol_local = 1,nmolecule_local
           mol_block => mol_blocks(imol_local)
           IF(imol_global == mol_block%imol_global) THEN
             ! Molecule found, now take the read  MO etc
             ! check consistency
             ! fill the correct mo_set one spin at a time
             ! and ciaociao
             imol_found = 1
             imol_proc = para_env%mepos

             CALL cp_assert(n_ao == n_ao_read,cp_failure_level,cp_assertion_failed,routineP,&
               "READ RESTART: different number of AO"//&
                CPSourceFileRef,&
                only_ionode=.TRUE.)

             CALL get_fm_mol_block(fm_mol_block=mol_block, mos = mos)

             DO ispin = 1,nspins
                mo_set => mos(ispin)%mo_set
                CALL get_mol_mo_set(mo_set=mo_set,mo=mo,eigenvalues=eigenvalues,&
                                    occupation_numbers=occupation_numbers)
                nmo_min = MIN( n_mo(ispin),n_mo_read(ispin))

                DO imo = 1 , nmo_min
                  CALL DCOPY(n_ao,mo_buffer(1,imo,ispin),1,mo(1,imo),1)
                  eigenvalues(imo) = eig_buffer(imo,ispin)
                  occupation_numbers(imo) = occ_buffer(imo,ispin)
                END DO
                DO imo =  nmo_min+1 , n_mo(ispin)
                  CALL DCOPY(n_ao,0.0_dp,0,mo(1,imo),1)
                  eigenvalues(imo) = 0.0_dp
                  occupation_numbers(imo) = 0.0_dp
                END DO
             END DO
           END IF

        END DO  ! imol_local

      END DO  ! imolecule_kind

    END DO  ! imol_global

    DEALLOCATE(mo_buffer,STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
    DEALLOCATE(eig_buffer,STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)
    DEALLOCATE(occ_buffer,STAT=istat)
    CPPostcondition(istat==0,cp_failure_level,routineP,error,failure)

    IF (para_env%ionode) CALL close_file(unit_number=restart_unit)

    CALL timestop(handle)

  END SUBROUTINE read_mo_mol_restart

END MODULE kg_gpw_fm_mol_utils

