!-----------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations         !
!   Copyright (C) 2000 - 2010  CP2K developers group                          !
!-----------------------------------------------------------------------------!

! *****************************************************************************
!> \brief kg_energy minim routine
!> \author gloria,30.12.2003
! *****************************************************************************
MODULE kg_energy
  USE atomic_kind_types,               ONLY: atomic_kind_type
  USE cell_types,                      ONLY: cell_type
  USE cp_control_types,                ONLY: dft_control_type
  USE cp_para_types,                   ONLY: cp_para_env_type
  USE dynamical_coeff_types,           ONLY: dyn_coeff_set_type,&
                                             dyn_coeff_type,&
                                             get_dyn_coeff
  USE f77_blas
  USE global_types,                    ONLY: global_environment_type
  USE input_cp2k_restarts,             ONLY: write_restart
  USE input_section_types,             ONLY: section_vals_get_subs_vals,&
                                             section_vals_type
  USE kg_core,                         ONLY: calculate_ecore_rspace,&
                                             calculate_eself
  USE kg_density,                      ONLY: calculate_density,&
                                             calculate_epc_density,&
                                             calculate_p_density,&
                                             calculate_vp_rspace_forces
  USE kg_energy_types,                 ONLY: init_kg_energy,&
                                             kg_energy_type
  USE kg_energy_utils,                 ONLY: diis_info_create,&
                                             diis_info_destroy,&
                                             diis_info_type,&
                                             do_diis_step
  USE kg_environment_methods,          ONLY: kg_qs_env_update
  USE kg_environment_types,            ONLY: get_kg_env,&
                                             kg_environment_type
  USE kg_intra,                        ONLY: calculate_ebond_corr
  USE kg_kxc,                          ONLY: calculate_kxc_derivatives
  USE kg_neighbor_lists,               ONLY: build_kg_neighbor_lists
  USE kg_ppl,                          ONLY: calculate_drho_ppl,&
                                             calculate_rho0_ppl
  USE kg_rho_types,                    ONLY: kg_rho_get,&
                                             kg_rho_type
  USE kg_rspw_types,                   ONLY: kg_rspw_get,&
                                             kg_rspw_type
  USE kg_scf,                          ONLY: kg_qs_scf
  USE kinds,                           ONLY: dp
  USE machine,                         ONLY: m_flush,&
                                             m_walltime
  USE message_passing,                 ONLY: mp_max
  USE pw_methods,                      ONLY: pw_axpy,&
                                             pw_copy,&
                                             pw_integrate_function,&
                                             pw_transfer,&
                                             pw_zero
  USE pw_poisson_methods,              ONLY: pw_poisson_solve
  USE pw_poisson_types,                ONLY: pw_poisson_type
  USE pw_pool_types,                   ONLY: pw_pool_create_pw,&
                                             pw_pool_give_back_pw,&
                                             pw_pool_type
  USE pw_types,                        ONLY: COMPLEXDATA1D,&
                                             REALDATA3D,&
                                             REALSPACE,&
                                             RECIPROCALSPACE,&
                                             pw_p_type,&
                                             pw_release
  USE qs_core_hamiltonian,             ONLY: build_core_hamiltonian_matrix
  USE qs_environment_types,            ONLY: qs_environment_type
  USE qs_neighbor_lists,               ONLY: build_qs_neighbor_lists
  USE scf_control_types,               ONLY: scf_control_type
  USE termination,                     ONLY: external_control
  USE timings,                         ONLY: timeset,&
                                             timestop
  USE xc,                              ONLY: xc_vxc_pw_create1
#include "cp_common_uses.h"

  IMPLICIT NONE
  CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'kg_energy'
  PRIVATE
  PUBLIC :: kg_energies , kg_qs_energies

CONTAINS

  ! *****************************************************************************
  SUBROUTINE kg_energies ( kg_env, root_section, globenv, error )

    TYPE(kg_environment_type), POINTER       :: kg_env
    TYPE(section_vals_type), POINTER         :: root_section
    TYPE(global_environment_type), POINTER   :: globenv
    TYPE(cp_error_type), INTENT(inout)       :: error

    TYPE(dft_control_type), POINTER          :: dft_control

    NULLIFY ( dft_control )

    CALL get_kg_env(kg_env=kg_env, dft_control=dft_control,error=error)

    IF ( dft_control % qs_control % method == "KG_GPW" ) THEN

       CALL kg_qs_energies ( kg_env, globenv, error)

    ELSE IF ( dft_control % qs_control % method == "KG_POL" ) THEN

       CALL kg_std_energies ( kg_env, root_section, globenv, error )

    ELSE

       CALL kg_only_energies ( kg_env,  kg_env%input, error )

    END IF

  END SUBROUTINE kg_energies

! *****************************************************************************
!> \brief Calculates the total potential energy
!>      for the standard kg method
!> \author gt
! *****************************************************************************
  SUBROUTINE kg_only_energies ( kg_env, force_env_section, error )

    TYPE(kg_environment_type), POINTER       :: kg_env
    TYPE(section_vals_type), POINTER         :: force_env_section
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_only_energies', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle, i, output_unit, stat
    LOGICAL                                  :: calculate_forces, failure
    REAL(KIND=dp)                            :: dvol, total_rho0_rspace
    TYPE(cell_type), POINTER                 :: cell
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(cp_para_env_type), POINTER          :: para_env
    TYPE(dft_control_type), POINTER          :: dft_control
    TYPE(kg_energy_type), POINTER            :: energy
    TYPE(kg_rho_type), POINTER               :: rho
    TYPE(kg_rspw_type), POINTER              :: rspw
    TYPE(pw_p_type), DIMENSION(:), POINTER   :: my_rho_g, my_rho_r, my_tau, &
                                                my_vxc_r, my_vxc_tau
    TYPE(pw_p_type), POINTER                 :: rho_core, rho_g, rho_r, &
                                                v_gspace
    TYPE(pw_poisson_type), POINTER           :: poisson_env
    TYPE(pw_pool_type), POINTER              :: auxbas_pw_pool
    TYPE(section_vals_type), POINTER         :: xc_section

    CALL timeset(routineN,handle )
    failure=.FALSE.
    para_env=>kg_env%para_env
    logger => cp_error_get_logger(error)
    output_unit=cp_logger_get_default_io_unit(logger)
    NULLIFY(energy,dft_control,cell,rho)
    NULLIFY(rspw,auxbas_pw_pool,v_gspace,poisson_env)
    NULLIFY(rho_r,rho_g,rho_core, my_vxc_r, my_vxc_tau, my_rho_g, my_tau)
    xc_section => section_vals_get_subs_vals(force_env_section,"DFT%XC", error=error)

    CALL get_kg_env(kg_env=kg_env,&
         dft_control=dft_control, energy=energy, cell=cell,&
         rho=rho,rspw=rspw,error=error)
    CPPrecondition(ASSOCIATED(rho),cp_failure_level,routineP,error,failure)
    CPPrecondition(ASSOCIATED(rspw),cp_failure_level,routineP,error,failure)
    IF(.NOT.failure) THEN
       CALL kg_rho_get(kg_rho=rho, rho_r=rho_r, rho_g=rho_g, &
            rho_core=rho_core,error=error)
       CALL kg_rspw_get(kg_rspw=rspw,auxbas_pw_pool=auxbas_pw_pool,&
            poisson_env=poisson_env,error=error)
    END IF
    calculate_forces=.FALSE.
    stat = 0
    dvol=auxbas_pw_pool%pw_grid%dvol

    CALL build_kg_neighbor_lists(kg_env,para_env,error)

    ! initialize energies

    CALL init_kg_energy(energy)

    ! calculate the constant parts of the energy
    ! self energy
    CALL calculate_eself ( kg_env, energy % core_self ,error=error)

    ! Calculate the rspace energy of the core charge distribution
    ! and its force contributions

    CALL calculate_ecore_rspace(kg_env,para_env,calculate_forces,error)

    ! calculate intramolecular bonded_correction energy

    CALL calculate_ebond_corr(kg_env,energy%bond_corr,calculate_forces,error=error)

    ! Calculate the local pseudopotential energy

    CALL calculate_rho0_ppl(kg_env,calculate_forces,error)

    ! Calculate the density generated by the core charges

    CALL calculate_epc_density(rho_core,rho%total_rho_core_rspace, kg_env,error=error)

    ! calculate the frozen electronic pseudo density

    CALL calculate_density (rho_r, rho_g, total_rho0_rspace, kg_env,error=error)
    rho%total_rho_gspace = pw_integrate_function(rho_g%pw,error=error)

    ! calculate the total density
    rho%total_rho_rspace = total_rho0_rspace+ rho%total_rho_core_rspace

    ! from the frozen rho, calculate the KE     and xc potentials
    ALLOCATE(my_rho_r(1))
    my_rho_r(1)%pw => rho_r%pw
    CALL xc_vxc_pw_create1(my_vxc_r,my_vxc_tau,my_rho_r,my_rho_g,my_tau,&
         energy%exc,xc_section,cell,auxbas_pw_pool,error=error)
    DO i = 1, SIZE(my_vxc_r)
       CALL pw_release(my_vxc_r(i)%pw,error)
    END DO
    DEALLOCATE(my_vxc_r,stat=stat)
    CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)
    NULLIFY(my_rho_r(1)%pw)
    DEALLOCATE(my_rho_r)
    NULLIFY(my_rho_r)

    ! add the frozen electronic density to the core density
    CALL pw_axpy(rho_core%pw,rho_g%pw,error=error)

    ALLOCATE(v_gspace,stat=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    IF (.NOT.failure) THEN
       CALL pw_pool_create_pw(auxbas_pw_pool, v_gspace%pw, &
            use_data = COMPLEXDATA1D,&
            in_space = RECIPROCALSPACE, error=error)
       CALL pw_zero(v_gspace%pw,error=error)
    END IF

    ! calculate electrostatic potential

    CALL pw_poisson_solve(poisson_env,rho_g%pw, energy % hartree,v_gspace%pw,error=error)

    CALL pw_pool_give_back_pw(auxbas_pw_pool,v_gspace%pw,error=error)
    DEALLOCATE(v_gspace,stat=stat)
    CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)

    ! add up all the potential energies

    energy % total = energy % core_self + energy % core_overlap + energy % exc + &
         energy % hartree + energy % pseudo + energy%bond_corr

    ! print
    IF (output_unit>0) THEN
       WRITE (UNIT=output_unit,FMT="(/,(T3,A,T40,F20.10))")&
            "Total core charge density (r-space):",&
            rho%total_rho_core_rspace
       WRITE (UNIT=output_unit,FMT="(T3,A,T60,F20.10)")&
            "Total charge density (r-space):     ",&
            rho%total_rho_rspace
       WRITE (UNIT=output_unit,FMT="(/,(T3,A,T55,F25.14))")&
            "Overlap energy of the core charge distribution:",energy%core_overlap,&
            "Bond correction energy:                        ",energy%bond_corr,&
            "Self energy of the core charge distribution:   ",energy%core_self,&
            "Pseudopotential  energy:                       ",energy%pseudo,&
            "Hartree energy:                                ",energy%hartree,&
            "Exchange-correlation energy:                   ",energy%exc,&
            "Total energy:                                  ",energy%total
       CALL m_flush(output_unit)
    END IF

    CALL timestop(handle)

  END SUBROUTINE kg_only_energies

! *****************************************************************************
!> \brief Calculates the total potential energy, total force
!>      for the standard kg method
!> \author gt
! *****************************************************************************
  SUBROUTINE kg_std_energies ( kg_env, root_section, globenv, error )

    TYPE(kg_environment_type), POINTER       :: kg_env
    TYPE(section_vals_type), POINTER         :: root_section
    TYPE(global_environment_type), POINTER   :: globenv
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_std_energies', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle, handle2, iter, &
                                                output_unit, stat
    LOGICAL                                  :: should_stop
    REAL(KIND=dp)                            :: maxerr, maxgrad, t1, t2
    TYPE(atomic_kind_type), DIMENSION(:), &
      POINTER                                :: atomic_kind_set
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(cp_para_env_type), POINTER          :: para_env
    TYPE(diis_info_type), POINTER            :: diis_info
    TYPE(dyn_coeff_set_type), POINTER        :: dyn_coeff_set
    TYPE(kg_rspw_type), POINTER              :: rspw
    TYPE(pw_p_type), POINTER                 :: vxc_r
    TYPE(pw_pool_type), POINTER              :: auxbas_pw_pool
    TYPE(scf_control_type), POINTER          :: scf_control

    CALL timeset(routineN,handle )

    NULLIFY(atomic_kind_set,dyn_coeff_set,diis_info,scf_control)
    NULLIFY(rspw,auxbas_pw_pool,vxc_r)
    para_env=>kg_env%para_env
    logger => cp_error_get_logger(error)
    output_unit=cp_logger_get_default_io_unit(logger)
    stat = 0
    iter = 0

    t1 = m_walltime()

    CALL get_kg_env(kg_env=kg_env,&
         atomic_kind_set=atomic_kind_set,&
         dyn_coeff_set=dyn_coeff_set,&
         rspw=rspw,&
         scf_control=scf_control,error=error)

    CALL kg_rspw_get(kg_rspw=rspw,auxbas_pw_pool=auxbas_pw_pool,error=error)

    CALL build_kg_neighbor_lists(kg_env,para_env,error)

    CALL diis_info_create(diis_info,scf_control%max_diis,dyn_coeff_set)

    CALL init_energy_calculation(kg_env,vxc_r,force_env_section=kg_env%input,error=error)

    CALL calc_c_gradients(kg_env,vxc_r,maxgrad,force_env_section=kg_env%input,error=error)

    IF (output_unit>0) THEN
       WRITE (UNIT=output_unit,&
            FMT="(/,T3,A,T16,A,T49,A,T68,A,/,T3,A)")&
            "Step","maxerr","Time","Convergence","Total energy",&
            REPEAT("-",77)
    END IF

    scf_loop: DO

       CALL timeset(routineN,handle2)
       IF (output_unit>0) CALL m_flush(output_unit)
       iter = iter + 1

       CALL do_diis_step(kg_env,diis_info,maxerr,error=error)
       CALL calc_c_gradients(kg_env,vxc_r,maxgrad,force_env_section=kg_env%input,&
            error=error)

       t2 = m_walltime()

       IF (output_unit>0) THEN
          WRITE (UNIT=output_unit,&
               FMT="(T2,I5,2X,E10.2,T32,F8.2,T40,2F20.10)")&
               iter,maxerr,t2 - t1,maxgrad,kg_env%energy%total
       END IF

       ! ** convergence check
       CALL external_control(should_stop,"KG_SCF",globenv=globenv,error=error)
       IF (maxgrad < scf_control%eps_scf) THEN
          IF (output_unit>0) THEN
             WRITE(UNIT=output_unit,FMT="(/,T3,A,I5,A/)")&
                  "*** SCF run converged in", iter, "steps ***"
          END IF
          CALL timestop(handle2)
          EXIT scf_loop
       ELSE IF (should_stop.OR.&
            iter == scf_control%max_scf) THEN
          IF (output_unit>0) THEN
             WRITE(UNIT=output_unit,FMT="(/,T3,A,/)")&
                  "*** SCF run NOT converged ***"
          END IF
          CALL timestop(handle2)
          EXIT
       END IF

       t1 = m_walltime()

       CALL timestop(handle2)

    END DO scf_loop

    CALL pw_pool_give_back_pw(auxbas_pw_pool,vxc_r%pw,error=error)
    DEALLOCATE(vxc_r,stat=stat)
    CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)

    CALL write_restart(kg_env=kg_env, root_section=root_section,&
         error=error)

    CALL diis_info_destroy(diis_info)

    CALL timestop(handle)

  END SUBROUTINE kg_std_energies

! *****************************************************************************
!> \brief Calculates the total potential energy, total force
!>      using the KG GPW method. This implies the calculation of the
!>      GPW KS energy  and the KS matrix via qs_ks_build_kohn_sham_matrix
!>      where the KE energy correction is added to be able to treat the
!>      molecule separatedly in the construction of the density.
!>      In the SCF iteration the optimization of the wavefunctions is
!>      done by the standard diagonalization of the diagonal blocks
!>      of the MO matrix. Each block is handled independently
!>      This is possible because the wavefunctions located
!>      on different molecules are orthogonal by construction.
!> \param kg_env Kg environment which contains the qs envirement as subenvironment
!> \param globenv global environment
!> \author JGH
! *****************************************************************************
  SUBROUTINE kg_qs_energies ( kg_env, globenv, error )

    TYPE(kg_environment_type), POINTER       :: kg_env
    TYPE(global_environment_type), POINTER   :: globenv
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'kg_qs_energies', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle
    TYPE(cp_para_env_type), POINTER          :: para_env
    TYPE(qs_environment_type), POINTER       :: qs_env

    CALL timeset(routineN,handle)

    NULLIFY ( qs_env )
    para_env=>kg_env%para_env
    CALL get_kg_env(kg_env=kg_env,&
         sub_qs_env=qs_env,error=error)
    CALL build_qs_neighbor_lists(qs_env,para_env,.TRUE.,qs_env%input,error=error)

    ! Calculate the overlap and the core Hamiltonian integral matrix
    CALL build_core_hamiltonian_matrix(qs_env=qs_env,&
         globenv=globenv,&
         calculate_forces=.FALSE., error=error)
    CALL kg_qs_env_update(kg_env,qs_env,error=error)

    ! Perform a SCF run
    CALL kg_qs_scf(kg_env,globenv,error=error)

    CALL timestop(handle)

  END SUBROUTINE kg_qs_energies

  ! *****************************************************************************
  SUBROUTINE init_energy_calculation ( kg_env, vxc_r, force_env_section, error)

    TYPE(kg_environment_type), POINTER       :: kg_env
    TYPE(pw_p_type), POINTER                 :: vxc_r
    TYPE(section_vals_type), POINTER         :: force_env_section
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'init_energy_calculation', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle, output_unit, stat
    LOGICAL                                  :: calculate_forces, failure
    REAL(KIND=dp)                            :: dvol, total_rho0_rspace
    TYPE(atomic_kind_type), DIMENSION(:), &
      POINTER                                :: atomic_kind_set
    TYPE(cell_type), POINTER                 :: cell
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(cp_para_env_type), POINTER          :: para_env
    TYPE(dft_control_type), POINTER          :: dft_control
    TYPE(dyn_coeff_set_type), POINTER        :: dyn_coeff_set
    TYPE(kg_energy_type), POINTER            :: energy
    TYPE(kg_rho_type), POINTER               :: rho
    TYPE(kg_rspw_type), POINTER              :: rspw
    TYPE(pw_p_type), DIMENSION(:), POINTER   :: my_rho_g, my_rho_r, my_tau, &
                                                my_vxc_r, my_vxc_tau
    TYPE(pw_p_type), POINTER                 :: rho_core, rho_g, rho_r
    TYPE(pw_pool_type), POINTER              :: auxbas_pw_pool
    TYPE(scf_control_type), POINTER          :: scf_control
    TYPE(section_vals_type), POINTER         :: xc_section

    CALL timeset(routineN,handle )
    failure=.FALSE.
    para_env=>kg_env%para_env
    logger=>cp_error_get_logger(error)
    output_unit=cp_logger_get_default_io_unit(logger)

    NULLIFY(auxbas_pw_pool,energy,atomic_kind_set,dft_control,&
         cell,rho,rspw,scf_control,dyn_coeff_set)
    NULLIFY(rho_r,rho_g,rho_core, my_vxc_r, my_vxc_tau, my_rho_g, my_tau)

    xc_section => section_vals_get_subs_vals(force_env_section,"DFT%XC",error=error)
    CALL get_kg_env( kg_env=kg_env, atomic_kind_set=atomic_kind_set,&
         dft_control=dft_control, energy=energy, cell=cell,&
         rho=rho,rspw=rspw,dyn_coeff_set=dyn_coeff_set,&
         scf_control=scf_control,error=error)

    CPPrecondition(ASSOCIATED(rho),cp_failure_level,routineP,error,failure)
    CPPrecondition(ASSOCIATED(rspw),cp_failure_level,routineP,error,failure)
    IF(.NOT.failure) THEN
       CALL kg_rho_get(kg_rho=rho, rho_r=rho_r, rho_g=rho_g, &
            rho_core=rho_core,error=error)
       CALL kg_rspw_get(kg_rspw=rspw,auxbas_pw_pool=auxbas_pw_pool,error=error)
    END IF

    calculate_forces=.FALSE.
    stat = 0
    dvol=auxbas_pw_pool%pw_grid%dvol

    CALL build_kg_neighbor_lists(kg_env,para_env,error)

    ! initialize energies

    CALL init_kg_energy(energy)

    ! calculate the constant parts of the energy
    ! self energy
    CALL calculate_eself ( kg_env, energy % core_self ,error=error)

    ! Calculate the rspace energy of the core charge distribution
    ! and its force contributions

    CALL calculate_ecore_rspace(kg_env,para_env,calculate_forces,error)

    ! calculate intramolecular bonded_correction energy

    CALL calculate_ebond_corr(kg_env,energy%bond_corr,calculate_forces,error=error)

    ! Calculate the local pseudopotential energy

    CALL calculate_rho0_ppl(kg_env,calculate_forces,error)

    ! Calculate the density generated by the core charges

    CALL calculate_epc_density(rho_core,rho%total_rho_core_rspace, kg_env,error=error)

    ! calculate the frozen electronic pseudo density

    CALL calculate_density (rho_r, rho_g, total_rho0_rspace, kg_env,error=error)

    ! calculate the total density
    rho%total_rho_rspace = total_rho0_rspace+ rho%total_rho_core_rspace

    ! from the frozen rho, calculate the thomas-fermi and xc potentials
    !  CALL xc_calculate_pw1 (rho_r%pw, rho_g%pw, vxc_r%pw, energy%exc, &
    !                         dft_control,auxbas_pw_pool)
    ALLOCATE(my_rho_r(1))
    my_rho_r(1)%pw => rho_r%pw
    CALL xc_vxc_pw_create1(my_vxc_r,my_vxc_tau,my_rho_r,my_rho_g,my_tau,&
         energy%exc,xc_section,cell,auxbas_pw_pool,error=error)
    ALLOCATE(vxc_r)
    vxc_r%pw => my_vxc_r(1)%pw
    NULLIFY(my_rho_r(1)%pw, my_vxc_r(1)%pw)
    DEALLOCATE(my_rho_r, my_vxc_r)
    NULLIFY(my_rho_r, my_vxc_r)

    vxc_r%pw%cr3d(:,:,:) =dvol*vxc_r%pw%cr3d(:,:,:)
    ! add the frozen electronic density to the core density
    CALL pw_axpy(rho_g%pw,rho_core%pw,error=error)

    CALL timestop(handle)

  END SUBROUTINE init_energy_calculation

  ! *****************************************************************************
  SUBROUTINE calc_c_gradients ( kg_env, vxc_r, maxgrad, force_env_section, error )

    TYPE(kg_environment_type), POINTER       :: kg_env
    TYPE(pw_p_type), POINTER                 :: vxc_r
    REAL(KIND=dp), INTENT(inout)             :: maxgrad
    TYPE(section_vals_type), POINTER         :: force_env_section
    TYPE(cp_error_type), INTENT(inout)       :: error

    CHARACTER(len=*), PARAMETER :: routineN = 'calc_c_gradients', &
      routineP = moduleN//':'//routineN

    INTEGER                                  :: handle, ikind, nkind, &
                                                output_unit, stat
    LOGICAL                                  :: calculate_forces, failure
    REAL(KIND=dp)                            :: dvol, e1_xc, e2_xc, &
                                                total_rhop_rspace
    REAL(KIND=dp), DIMENSION(:, :), POINTER  :: forces
    TYPE(atomic_kind_type), DIMENSION(:), &
      POINTER                                :: atomic_kind_set
    TYPE(cell_type), POINTER                 :: cell
    TYPE(cp_logger_type), POINTER            :: logger
    TYPE(cp_para_env_type), POINTER          :: para_env
    TYPE(dft_control_type), POINTER          :: dft_control
    TYPE(dyn_coeff_set_type), POINTER        :: dyn_coeff_set
    TYPE(dyn_coeff_type), POINTER            :: local_coeffs
    TYPE(kg_energy_type), POINTER            :: energy
    TYPE(kg_rho_type), POINTER               :: rho
    TYPE(kg_rspw_type), POINTER              :: rspw
    TYPE(pw_p_type), POINTER                 :: rho_core, rho_g, rho_r, &
                                                rhop_g, rhop_r, v1xc_r, &
                                                v2xc_r, v_gspace, v_rspace
    TYPE(pw_poisson_type), POINTER           :: poisson_env
    TYPE(pw_pool_type), POINTER              :: auxbas_pw_pool
    TYPE(section_vals_type), POINTER         :: xc_section

    failure = .FALSE.
    CALL timeset(routineN,handle )
    para_env=>kg_env%para_env
    logger=>cp_error_get_logger(error)
    output_unit=cp_logger_get_default_io_unit(logger)
    calculate_forces=.FALSE.

    NULLIFY(energy,atomic_kind_set,cell,dft_control,&
         dyn_coeff_set,rho,rspw,local_coeffs,forces)
    NULLIFY(rho_r,rho_g,rhop_r,rhop_g,rho_core,v_gspace,v_rspace,&
         v1xc_r,v2xc_r, poisson_env)

    xc_section => section_vals_get_subs_vals(force_env_section,"DFT%XC", error=error)
    CALL get_kg_env( kg_env=kg_env, atomic_kind_set=atomic_kind_set,&
         dft_control=dft_control, energy=energy, cell=cell,&
         rho=rho,rspw=rspw,dyn_coeff_set=dyn_coeff_set,error=error)
    CPPrecondition(ASSOCIATED(rho),cp_failure_level,routineP,error,failure)
    CPPrecondition(ASSOCIATED(rspw),cp_failure_level,routineP,error,failure)
    IF(.NOT.failure) THEN
       CALL kg_rho_get(kg_rho=rho, rho_r=rho_r, rho_g=rho_g, rhop_r=rhop_r,&
            rhop_g=rhop_g,rho_core=rho_core,error=error)
       CALL kg_rspw_get(kg_rspw=rspw,auxbas_pw_pool=auxbas_pw_pool,&
            poisson_env=poisson_env,error=error)
    END IF

    ALLOCATE(v_gspace,stat=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE(v_rspace,stat=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE(v1xc_r,stat=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)
    ALLOCATE(v2xc_r,stat=stat)
    CPPostcondition(stat==0,cp_failure_level,routineP,error,failure)

    IF (.NOT.failure) THEN
       CALL pw_pool_create_pw(auxbas_pw_pool, v_gspace%pw, &
            use_data = COMPLEXDATA1D,&
            in_space = RECIPROCALSPACE, error=error)
       CALL pw_zero(v_gspace%pw,error=error)
       CALL pw_pool_create_pw(auxbas_pw_pool, v_rspace%pw,&
            use_data = REALDATA3D,&
            in_space = REALSPACE, error=error)
       CALL pw_zero(v_rspace%pw,error=error)
       CALL pw_pool_create_pw(auxbas_pw_pool, v1xc_r%pw,&
            use_data = REALDATA3D,&
            in_space = REALSPACE, error=error)
       CALL pw_zero(v1xc_r%pw,error=error)
       CALL pw_pool_create_pw(auxbas_pw_pool, v2xc_r%pw,&
            use_data = REALDATA3D,&
            in_space = REALSPACE, error=error)
       CALL pw_zero(v2xc_r%pw,error=error)
    END IF

    dvol=v_rspace%pw%pw_grid%dvol
    nkind=SIZE(dyn_coeff_set%coeffs_of_kind)
    ! reinitialize forces
    DO ikind= 1, nkind
       local_coeffs => dyn_coeff_set%coeffs_of_kind(ikind)%coeffs
       IF(ASSOCIATED(local_coeffs)) THEN
          CALL get_dyn_coeff(coeffs=local_coeffs,forces=forces,error=error)
          forces(:,:)=0.0_dp
       END IF
    END DO
    CALL pw_zero(rho_g%pw,error=error)
    CALL pw_copy(rho_core%pw,rho_g%pw,error=error)
    rho%total_rho_gspace = pw_integrate_function(rho_g%pw,error=error)

    CALL calculate_drho_ppl(kg_env,calculate_forces,error)

    CALL calculate_p_density(rhop_r, rhop_g, total_rhop_rspace, kg_env,error=error)

    CALL calculate_kxc_derivatives(auxbas_pw_pool, rho_r, rho_g, rhop_r, vxc_r, v1xc_r,&
         v2xc_r, e1_xc, e2_xc, xc_section, calculate_forces, error)

    ! add the polarization density to get the total density
    CALL pw_axpy(rhop_g%pw,rho_g%pw,error=error)

    ! calculate electrostatic potential

    CALL pw_poisson_solve(poisson_env,rho_g%pw, energy % hartree,v_gspace%pw,error=error)

    CALL pw_transfer(v_gspace%pw,v_rspace%pw,error=error)
    CALL pw_pool_give_back_pw(auxbas_pw_pool,v_gspace%pw,error=error)
    DEALLOCATE(v_gspace,stat=stat)
    CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)

    v_rspace%pw%cr3d(:,:,:) = dvol*v_rspace%pw%cr3d(:,:,:)
    v_rspace%pw%cr3d(:,:,:) =-1.0_dp*(vxc_r%pw%cr3d(:,:,:) +&
         v1xc_r%pw%cr3d(:,:,:)+ v_rspace%pw%cr3d(:,:,:))

    CALL calculate_vp_rspace_forces(v_rspace,kg_env, calculate_forces,error=error)

    ! deallocate work storage
    CALL pw_pool_give_back_pw(auxbas_pw_pool,v_rspace%pw,error=error)
    DEALLOCATE(v_rspace,stat=stat)
    CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)
    CALL pw_pool_give_back_pw(auxbas_pw_pool,v1xc_r%pw,error=error)
    DEALLOCATE(v1xc_r,stat=stat)
    CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)
    CALL pw_pool_give_back_pw(auxbas_pw_pool,v2xc_r%pw,error=error)
    DEALLOCATE(v2xc_r,stat=stat)
    CPPostconditionNoFail(stat==0,cp_warning_level,routineP,error)

    ! add up all the potential energies

    energy % total = energy % core_self + energy % core_overlap + energy % exc + &
         e1_xc + e2_xc+ energy % hartree + energy % pseudo + energy%ppseudo + &
         energy%bond_corr

    ! get the maximum coefficent gradient

    maxgrad=0._dp
    DO ikind= 1, nkind
       local_coeffs => dyn_coeff_set%coeffs_of_kind(ikind)%coeffs
       IF(ASSOCIATED(local_coeffs)) THEN
          CALL get_dyn_coeff(coeffs=local_coeffs,forces=forces,error=error)
          maxgrad=MAX(maxgrad,MAXVAL(ABS(forces)))
       END IF
    END DO
    CALL mp_max(maxgrad,kg_env%para_env%group)

    ! print
    IF (output_unit>0) THEN
       WRITE (UNIT=output_unit,FMT="(/,(T3,A,T40,F20.10))")&
            "Total core charge density (r-space):",&
            rho%total_rho_core_rspace
       WRITE (UNIT=output_unit,FMT="(T3,A,T60,F20.10)")&
            "Total charge density (r-space):     ",&
            rho%total_rho_rspace
       WRITE (UNIT=output_unit,FMT="(T3,A,T60,F20.10)")&
            "Total rho_coefs charge density (r-space):     ",&
            total_rhop_rspace
       WRITE (UNIT=output_unit,FMT="(T3,A,T60,F20.10)")&
            "Maximum coefficent gradient         ",&
            maxgrad
       WRITE (UNIT=output_unit,FMT="(/,(T3,A,T55,F25.14))")&
            "Overlap energy of the core charge distribution:",energy%core_overlap,&
            "Bond correction energy:                        ",energy%bond_corr,&
            "Self energy of the core charge distribution:   ",energy%core_self,&
            "Pseudopotential  energy:                       ",energy%pseudo,&
            "Pseudopotential  energy  due to coefs:         ",energy%ppseudo,&
            "Hartree energy:                                ",energy%hartree,&
            "Exchange-correlation energy:                   ",energy%exc,&
            "1st order correction to xc energy coefs        ",e1_xc,&
            "2nd order correction to xc energy coefs        ",e2_xc,&
            "Total energy:                                  ",energy%total
       CALL m_flush(output_unit)
    END IF

    CALL timestop(handle)

  END SUBROUTINE calc_c_gradients

END MODULE kg_energy
