/*
 * Compiz cube model plugin
 *
 * cubemodel.cpp
 *
 * This plugin displays wavefront (.obj) 3D mesh models inside of
 * the transparent cube.
 *
 * Copyright : (C) 2008 by David Mikos
 * E-mail    : infiniteloopcounter@gmail.com
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*
 * Model loader code based on cubemodel/cubefx plugin "cubemodelModel.c.in"
 * code - originally written by Joel Bosvel (b0le).
 */

#include "cubemodel.h"

COMPIZ_PLUGIN_20090315 (cubemodel, CubemodelPluginVTable);

void
CubemodelScreen::initWorldVariables ()
{
    mHsize = screen->vpSize ().height () * cubeScreen->nOutput ();

    mArcAngle = 360.0f / mHsize;
    mRadius = cubeScreen->distance () / cosf (0.5 * (mArcAngle * toRadians));
    mTopDistance = cubeScreen->distance ();

    if (optionGetRescaleWidth ())
	mRatio = (float) screen->width () / (float) screen->height ();
    else
	mRatio = 1;

    mSideDistance = mTopDistance * mRatio;
}

void
CubemodelScreen::updateModel (int start,
			      int end)
{
    unsigned int	      i;
    CompOption::Value::Vector modelScale         = optionGetModelScaleFactor ();
    CompOption::Value::Vector modelX             = optionGetModelXOffset ();
    CompOption::Value::Vector modelY             = optionGetModelYOffset ();
    CompOption::Value::Vector modelZ             = optionGetModelZOffset ();
    CompOption::Value::Vector modelRotationPlane = optionGetModelRotationPlane ();
    CompOption::Value::Vector modelRotationRate  = optionGetModelRotationRate ();
    CompOption::Value::Vector modelAnimation     = optionGetModelAnimation ();
    CompOption::Value::Vector modelFps           = optionGetModelFps ();

    start = MAX (start, 0);
    end   = MIN (end,  (int ) mModels.size ());

    for (i = start; i < (unsigned int ) end; i++)
    {
	if (!mModels[i] || !mModels[i]->finishedLoading)
	    continue;

	if (modelScale.size () > i)
	    mModels[i]->scaleGlobal = modelScale.at (i).f ();

	if (modelX.size () > i)
	    mModels[i]->translate[0] = modelX.at (i).f () * mRatio;
	if (modelY.size () > i)
	    mModels[i]->translate[1] = modelY.at (i).f ();
	if (modelZ.size () > i)
	    mModels[i]->translate[2] = modelZ.at (i).f () * mRatio;

	if (modelRotationPlane.size () > i)
	{
	    int rot = modelRotationPlane.at (i).i ();

	    switch (rot % 3) {
	    case 0:
		mModels[i]->rotate[1] = 0;
		mModels[i]->rotate[2] = 1;
		mModels[i]->rotate[3] = 0;
	    	break;
	    case 1:
		mModels[i]->rotate[1] = 1;
		mModels[i]->rotate[2] = 0;
		mModels[i]->rotate[3] = 0;
	    	break;
	    case 2:
		mModels[i]->rotate[1] = 0;
		mModels[i]->rotate[2] = 0;
		mModels[i]->rotate[3] = 1;
	    	break;
	    }

	    if (rot / 3 != 0)
	    {
		mModels[i]->rotate[1] *= -1;
		mModels[i]->rotate[2] *= -1;
		mModels[i]->rotate[3] *= -1;
	    }

	}
	if (modelRotationRate.size () > i)
	    mModels[i]->rotateSpeed = modelRotationRate.at (i).f ();

	if (modelFps.size () > i)
	{
	    mModels[i]->fps = modelFps.at (i).i ();

	    if (modelAnimation.size () > i && modelAnimation.at (i).i () == 2)
		mModels[i]->fps *= -1;
	}
    }
}

void
CubemodelScreen::initCubemodel ()
{
    unsigned int   numModels;
    float translate[] = { 0, 0, 0 };
    float rotate[]    = { 0, 0, 0, 0 };
    float scale[]     = { 1, 1, 1, 1 };
    float color[]     = { 1, 1, 1, 1 };
    float rotateSpeed = 0;
    bool  status, animation = false;
    int   fps = 3;

    CompOption::Value::Vector  modelFilename      = optionGetModelFilename ();
    CompOption::Value::Vector  modelScale         = optionGetModelScaleFactor ();
    CompOption::Value::Vector  modelX             = optionGetModelXOffset ();
    CompOption::Value::Vector  modelY             = optionGetModelYOffset ();
    CompOption::Value::Vector  modelZ             = optionGetModelZOffset ();
    CompOption::Value::Vector  modelRotationPlane = optionGetModelRotationPlane ();
    CompOption::Value::Vector  modelRotationRate  = optionGetModelRotationRate ();
    CompOption::Value::Vector  modelAnimation     = optionGetModelAnimation ();
    CompOption::Value::Vector  modelFps           = optionGetModelFps ();

    numModels = modelFilename.size ();

    if (modelScale.size () < numModels)
	numModels = modelScale.size ();

    if (modelX.size () < numModels)
	numModels = modelX.size ();
    if (modelY.size () < numModels)
	numModels = modelY.size ();
    if (modelZ.size () < numModels)
	numModels = modelZ.size ();
    if (modelRotationPlane.size () < numModels)
	numModels = modelRotationPlane.size ();
    if (modelRotationRate.size () < numModels)
	numModels = modelRotationRate.size ();
    if (modelAnimation.size () < numModels)
	numModels = modelAnimation.size ();
    if (modelFps.size () < numModels)
	numModels = modelFps.size ();
	
    while (mModels.size ())
    {
	CubemodelObject *m = mModels.back ();
	delete m;
	mModels.pop_back ();
    }
    
    mModelFilename.clear ();

    mModelFilename.resize (numModels);

    for (unsigned int i = 0; i < numModels; i++)
    {
	mModels.push_back (new CubemodelObject);
	
	animation = false;
	if (modelAnimation.at (i).i () > 0)
	    animation = true;

	status = false;

	if (modelFilename.size () > i)
	    status = addModelObject (mModels[i],
				     modelFilename.at (i).s (),
				     translate, rotate,
				     rotateSpeed, scale,
				     color, animation, fps);

	if (!status)
	    mModelFilename.at (i) = CompString ();
	else
	    mModelFilename.at (i) = CompString (modelFilename.at (i).s ());
    }

    updateModel (0, mModels.size ());

    initWorldVariables ();
}

void
CubemodelScreen::freeCubemodel ()
{
    if (mModels.size ())
    {
	foreach (CubemodelObject *model, mModels)
	{
	    if (model)
	    {
		deleteModelObject (model);
		delete model;
	    }
	}
	mModels.clear ();
    }
    
    mModelFilename.clear ();
}

void
CubemodelScreen::updateCubemodel ()
{
    freeCubemodel ();
    initCubemodel ();
}

void
CubemodelScreen::loadingOptionChanged (CompOption	     *opt,
				       CubemodelOptions::Options num)
{
    unsigned int              i, numModels, fps = 3;
    CompOption::Value::Vector modelFilename, modelAnimation;
    float         translate[] = { 0, 0, 0 };
    float         rotate[]    = { 0, 0, 0, 0 };
    float         scale[]     = { 1, 1, 1, 1 };
    float         color[]     = { 1, 1, 1, 1 };
    float         rotateSpeed = 0;

    modelFilename  = optionGetModelFilename ();
    modelAnimation = optionGetModelAnimation ();
    numModels      = modelAnimation.size ();

    if (modelAnimation.size () < numModels)
	numModels = modelAnimation.size ();

    if (!mModels.size () || !mModelFilename.size ())
    {
	updateCubemodel ();
	return;
    }


    if (numModels != mModels.size ())
    {
        unsigned int oldNumModels (mModels.size ());

	/* Delete excess models */
	for (i = numModels; i < mModels.size (); i++)
	{
	    deleteModelObject (mModels[i]);

	    if (mModels[i])
		delete mModels[i];
	}

	mModels.resize (numModels);
	mModelFilename.resize (numModels);

	/* Create new models */
	if (numModels > 0)
	{
	    for (i = numModels; i < oldNumModels; i++)
	    {
		mModels.at (i) = new CubemodelObject ();

		mModelFilename.at (i) = CompString ();
	    }
	}
    }

    for (i = 0; i < numModels; i++)
    {
	bool animation, fileDiff = true;

	if (!modelFilename.size ())
	    continue;

	if (modelFilename.size () <= i)
	    continue;

	if (!mModelFilename.at (i).size ())
	    continue;

	if (modelFilename.at (i).s ().c_str ())
	    fileDiff = strcmp (mModelFilename.at (i).c_str (),
			       modelFilename.at (i).s ().c_str ());

	animation = (modelAnimation.at (i).i () > 0);

	if (animation != mModels[i]->animation || fileDiff)
	{
	    bool status;

	    deleteModelObject (mModels[i]);
	    if (mModelFilename.at (i).c_str ())
		mModelFilename.at (i) = CompString ();

	    status = addModelObject (mModels[i],
				     modelFilename.at (i).s (),
				     translate, rotate, rotateSpeed,
				     scale, color, animation, fps);
	    if (!status)
		mModelFilename.at (i) = CompString ();
	    else
		mModelFilename.at (i) = CompString (modelFilename.at (i).s ());
	}
    }

    updateModel (0, mModels.size ());
}

void
CubemodelScreen::optionChanged (CompOption *opt,
				CubemodelOptions::Options num)
{
    if (!mModels.size ())
    {
	updateCubemodel ();
	return;
    }

    updateModel (0, mModels.size ());
}


void
CubemodelScreen::cubeClearTargetOutput (float xRotate,
					float vRotate)
{
    cubeScreen->cubeClearTargetOutput (xRotate, vRotate);

    glClear (GL_DEPTH_BUFFER_BIT);
}

void
CubemodelScreen::setLightPosition (GLenum light)
{
    float angle = optionGetLightInclination () * toRadians;
    float position[] = { 0.0, 0.0, 1.0, 0.0 };

    if (optionGetRotateLighting ())
	angle = 0;

    position[1] = sinf (angle);
    position[2] = cosf (angle);

    glLightfv (light, GL_POSITION, position);
}

void
CubemodelScreen::cubePaintInside (const GLScreenPaintAttrib &attrib,
				  const GLMatrix	    &transform,
				  CompOutput		    *output,
				  int			    size)
{
    unsigned int       i;
    static const float matShininess[] = { 60.0 };
    static const float matSpecular[] = { 0.6, 0.6, 0.6, 1.0 };
    static const float matDiffuse[] = { 1.0, 1.0, 1.0, 1.0 };
    static const float matAmbient[] = { 0.8, 0.8, 0.9, 1.0 };

    static const float lmodelLocalviewer[] = { 0.0 };
    static       float lmodelTwoside[] = { 0.0 };
    static       float lmodelAmbient[] = { 0.4, 0.4, 0.4, 0.4 };
    static       float lmodelDiffuse[] = { 1.0, 1.0, 1.0, 1.0 };
    static       float lmodelSpecular[]= { 0.6, 0.6, 0.6, 0.6 };

    GLScreenPaintAttrib  sA  (attrib);
    GLMatrix             mT (transform);
    bool               enabledCull;
    int                cull;
    float              scale, outputRatio = 1.0f;
    
    mDamage = false;

    if (mHsize != screen->vpSize ().width () * cubeScreen->nOutput ())
    {
	initWorldVariables ();
	updateModel (0, mModels.size ());
    }
    if (mModels.empty ())
    {
	cubeScreen->cubePaintInside (attrib, transform, output, size);
	return;
    }

    sA.yRotate += cubeScreen->invert () * (360.0f / size) *
		  (cubeScreen->xRotations () - (screen->vp ().x () * cubeScreen->nOutput ()));

    gScreen->glApplyTransform (sA, output, &mT);

    glPushMatrix ();

    if (optionGetRotateLighting ())
	setLightPosition (GL_LIGHT1);

    glLoadMatrixf (mT.getMatrix ());

    if (!optionGetRotateLighting ())
	setLightPosition (GL_LIGHT1);


    glTranslatef (cubeScreen->outputXOffset (), -cubeScreen->outputYOffset (), 0.0f);
    glScalef (cubeScreen->outputXScale (), cubeScreen->outputYScale (), 1.0f);

    glPushAttrib (GL_COLOR_BUFFER_BIT | GL_TEXTURE_BIT |
		  GL_LIGHTING_BIT     | GL_DEPTH_BUFFER_BIT);

    glEnable (GL_BLEND);
    glColorMaterial (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);

    lmodelAmbient[0]  = optionGetLightAmbient ();
    lmodelDiffuse[0]  = optionGetLightDiffuse ();
    lmodelSpecular[0] = optionGetLightSpecular ();

    for (i = 1; i < 4; i++)
    {
	lmodelAmbient[i]  = lmodelAmbient[0];
	lmodelDiffuse[i]  = lmodelDiffuse[0];
	lmodelSpecular[i] = lmodelSpecular[0];
    }

    lmodelTwoside[0] = (optionGetRenderFrontAndBack () ? 1.0f : 0.0f);


    glLightModelfv (GL_LIGHT_MODEL_LOCAL_VIEWER, lmodelLocalviewer);
    glLightModelfv (GL_LIGHT_MODEL_TWO_SIDE, lmodelTwoside);
    glLightModelfv (GL_LIGHT_MODEL_AMBIENT,  lmodelAmbient);
    glLightfv (GL_LIGHT1, GL_DIFFUSE,   lmodelDiffuse);
    glLightfv (GL_LIGHT1, GL_SPECULAR,  lmodelSpecular);

    enabledCull = glIsEnabled (GL_CULL_FACE);

    glGetIntegerv (GL_CULL_FACE_MODE, &cull);
    glEnable (GL_CULL_FACE);

    glCullFace (~cull & (GL_FRONT | GL_BACK));
    glCullFace (cull);

    glPushMatrix ();

    glColor4usv (defaultColor);

    glMaterialfv (GL_FRONT_AND_BACK, GL_SHININESS, matShininess);
    glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, matSpecular);
    glMaterialfv (GL_FRONT_AND_BACK, GL_DIFFUSE, matDiffuse);
    glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, matAmbient);

    glEnable (GL_NORMALIZE);
    glEnable (GL_DEPTH_TEST);
    glEnable (GL_COLOR_MATERIAL);
    glEnable (GL_LIGHTING);
    glEnable (GL_LIGHT1);
    glDisable (GL_LIGHT0);

    glDepthFunc (GL_LEQUAL); /* for transparency maps */
    glShadeModel(GL_SMOOTH);

    glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    scale = cubeScreen->distance ();

    if (optionGetRescaleWidth ())
    {
	if (cubeScreen->multioutputMode () == CubeScreen::Automatic &&
	    (unsigned int )cubeScreen->nOutput () < 
	     screen->outputDevs ().size ())
	    outputRatio = (float) screen->width () / (float) screen->height ();
	else
	    outputRatio = (float) output->width () / (float) output->height ();
    }

    glScalef (scale / outputRatio, scale, scale / outputRatio);

    glPushMatrix ();

    glColor4f (1.0, 1.0, 1.0, 1.0);

    for (i = 0; i < mModels.size (); i++)
    {
	glPushMatrix ();
	drawModelObject (mModels[i],
	                 optionGetGlobalModelScaleFactor ());
	glPopMatrix ();

    }
    glPopMatrix ();

    glPopMatrix ();

    glDisable (GL_LIGHT1);
    glDisable (GL_NORMALIZE);

    if (!gScreen->lighting ())
	glDisable (GL_LIGHTING);

    glDisable (GL_DEPTH_TEST);

    if (enabledCull)
	glDisable (GL_CULL_FACE);

    glPopMatrix ();

    glPopAttrib ();

    mDamage = true;

    cubeScreen->cubePaintInside (attrib, transform, output, size);
}

void
CubemodelScreen::preparePaint (int ms)
{
    unsigned int i;

    for (i = 0; i < mModels.size (); i++)
    {
	if (!mModels[i]->finishedLoading)
	    continue;

	if (mModels[i]->updateAttributes)
	{
	    updateModel (i, i + 1);
	    mModels[i]->updateAttributes = false;
	}

	updateModelObject (mModels[i],  ms / 1000.0f);
    }

    cScreen->preparePaint (ms);
}

void
CubemodelScreen::donePaint ()
{
    if (mDamage)
	cScreen->damageScreen ();
	
    cScreen->donePaint ();
}


CubemodelScreen::CubemodelScreen (CompScreen *screen) :
    PluginClassHandler <CubemodelScreen, CompScreen> (screen),
    cScreen (CompositeScreen::get (screen)),
    gScreen (GLScreen::get (screen)),
    cubeScreen (CubeScreen::get (screen)),
    mDamage (false)
{
    CompositeScreenInterface::setHandler (cScreen);
    GLScreenInterface::setHandler (gScreen);
    CubeScreenInterface::setHandler (cubeScreen);

    initCubemodel ();

    optionSetModelFilenameNotify      (boost::bind (&CubemodelScreen::loadingOptionChanged, this, _1, _2));
    optionSetModelAnimationNotify     (boost::bind (&CubemodelScreen::loadingOptionChanged, this, _1, _2));

    optionSetModelScaleFactorNotify   (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
    optionSetModelXOffsetNotify       (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
    optionSetModelYOffsetNotify       (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
    optionSetModelZOffsetNotify       (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
    optionSetModelRotationPlaneNotify (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
    optionSetModelRotationRateNotify  (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
    optionSetModelFpsNotify           (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
    optionSetRescaleWidthNotify       (boost::bind (&CubemodelScreen::optionChanged, this, _1, _2));
}

CubemodelScreen::~CubemodelScreen ()
{
    freeCubemodel ();
}

bool
CubemodelPluginVTable::init ()
{
    if (!CompPlugin::checkPluginABI ("core", CORE_ABIVERSION) ||
    	!CompPlugin::checkPluginABI ("composite", COMPIZ_COMPOSITE_ABI) ||
    	!CompPlugin::checkPluginABI ("opengl", COMPIZ_OPENGL_ABI) ||
    	!CompPlugin::checkPluginABI ("cube", COMPIZ_CUBE_ABI))
	return false;

    return true;
}
