/*------------------------------------------------------------------------------------

  File        : inrcast.cpp

  Description : 1D/2D/3D Image converter and visualizer based on command line arguments

  Copyright  : David Tschumperle - http://www.greyc.ensicaen.fr/~dtschump/

  This software is governed by the CeCILL  license under French law and
  abiding by the rules of distribution of free software.  You can  use,
  modify and/ or redistribute the software under the terms of the CeCILL
  license as circulated by CEA, CNRS and INRIA at the following URL
  "http://www.cecill.info".

  As a counterpart to the access to the source code and  rights to copy,
  modify and redistribute granted by the license, users are provided only
  with a limited warranty  and the software's author,  the holder of the
  economic rights,  and the successive licensors  have only  limited
  liability.

  In this respect, the user's attention is drawn to the risks associated
  with loading,  using,  modifying and/or developing or reproducing the
  software by the user in light of its specific status of free software,
  that may mean  that it is complicated to manipulate,  and  that  also
  therefore means  that it is reserved for developers  and  experienced
  professionals having in-depth computer knowledge. Users are therefore
  encouraged to load and test the software's suitability as regards their
  requirements in conditions enabling the security of their systems and/or
  data to be ensured and,  more generally, to use and operate it in the
  same conditions as regards security.

  The fact that you are presently reading this means that you have had
  knowledge of the CeCILL license and that you accept its terms.

  -----------------------------------------------------------------------------------*/

#include "../CImg.h"
using namespace cimg_library;
// The undef below is necessary when using a non-standard compiler.
#ifdef cimg_use_visualcpp6
#define std
#endif

//---------------------------------------------
// Convert to output image, save and visualize
//---------------------------------------------
template<typename T,typename t> void output_file(int argc, char **argv, CImgList<T>& src,
						 const CImgList<char>& filename, const t&) {

  // Read command line arguments
  const char* typei = cimg_option("-i","float","Input pixel type (can be : char,uchar,ushort,short,uint,int,ulong,long,float,double)");
  cimg_option("-t",typei,"Output pixel type (can be : char,uchar,ushort,short,uint,int,ulong,long,float,double)");
  const bool a_orient   = cimg_option("-orient",false,"Compute vector orientations");
  const bool a_norm     = cimg_option("-norm",false,"Compute vector norms");
  const float a_rotate  = cimg_option("-angle",0.0f,"Rotate image(s) (angle in degrees)");
  const bool a_transpose  = cimg_option("-transpose",false,"Transpose image(s)");
  const char  a_mirror  = cimg_option("-mirror",'\0',"Mirror image(s) along axis (can be 'x','y','z' or 'v')");
  const float a_blur    = cimg_option("-blur",0.0f,"Blur image(s) (variance)");
  const float a_noiseg  = cimg_option("-ng",0.0f,"Add gaussian noise (variance)");
  const float a_noiseu  = cimg_option("-nu",0.0f,"Add uniform noise (variance)");
  const float a_noises  = cimg_option("-ns",0.0f,"Add salt&pepper noise (percentage)");
  const char *a_cut     = cimg_option("-cut",(const char*)NULL,"Pixel value cutting (ex : '0,255')");
  const char *a_vnorm   = cimg_option("-val",(const char*)NULL,"Pixel value normalization (ex : '0,255')");
  const int a_quantize  = cimg_option("-quantize",0,"Uniform quantization (number of levels)");
  const char *a_bcrop   = cimg_option("-beg",(const char*)NULL,"Upper-left coordinates crop (2D,3D or 4D coordinates)");
  const char *a_ecrop   = cimg_option("-end",(const char*)NULL,"Lower-right coordinates crop (2D,3D or 4D coordinates");
  const char *a_resize  = cimg_option("-g",(const char*)NULL,"Resize image(s) (ex : '100x100' or '50%x30%')");
  const int a_interp    = cimg_option("-interp",3,"Resizing interpolation type (0=none, 1=bloc, 2=mosaic, 3=linear, 4=grid, 5=cubic");
  const bool a_equalize = cimg_option("-equalize",false,"Equalize histogram");
  const float a_mul     = cimg_option("-mul",1.0f,"Multiply pixel values by a float constant");
  const float a_add     = cimg_option("-add",0.0f,"Add a float constant to pixel values");
  const int a_erode     = cimg_option("-erode",0,"Erode image(s) (number of steps)");
  const int a_dilate    = cimg_option("-dilate",0,"Dilate image(s) (number of steps)");
  const int a_median    = cimg_option("-median",0,"Apply a median filter with specified size");
  const float a_smooth  = cimg_option("-smooth",0.0f,"Smooth the image anisotropically with specified strength");
  const bool a_falsecolors = cimg_option("-falsecolors",false,"Compute false colors");
  const int a_histo     = cimg_option("-histogram",0,"Compute image histogram (number of levels)");
  const char *a_break   = cimg_option("-b",(const char*)NULL,"Break image along axis (can be 'z', 'x3', 'y2', etc...)");
  const char a_append   = cimg_option("-a",'\0',"Append images along axis (can be 'x','y','z' or 'v')");
  const bool a_ucrop    = cimg_option("-crop",false,"User-selected crop");
  const bool a_log      = cimg_option("-log",false,"Apply logarithm");
  const bool a_sqrt     = cimg_option("-sqrt",false,"Apply square root");
  const bool a_sqr      = cimg_option("-sqr",false,"Apply square");
  const char a_unroll   = cimg_option("-unroll",'\0',"Unroll axis");
  const char *a_permute = cimg_option("-permute",(const char*)NULL,"Permute image axes");
  const char *a_annot   = cimg_option("-annotate",(char*)NULL,"Add annotation to image(s)");
  const bool a_reverse  = cimg_option("-r",false,"Reverse image list");
  const char *file_o    = cimg_option("-o",(const char*)NULL,"Output file");
  const bool a_visu     = cimg_option("-visu",false,"Image visualization");
  const bool a_get_size = cimg_option("-get_size",false,"Get image size");
  const bool a_get_width = cimg_option("-get_width",false,"Get image width");
  const bool a_get_height = cimg_option("-get_height",false,"Get image height");
  const bool a_get_depth = cimg_option("-get_depth",false,"Get image depth");
  const bool a_get_dim = cimg_option("-get_dim",false,"Get image dim");
  if (!src.size) {
    std::fprintf(stderr,"\n** You must specify at least one input image\n** Try '%s -h' to get help\n**\n",cimg::basename(argv[0]));
    std::exit(0);
  }

  // Special commands : Get image dimensions
  if (a_get_size)   { std::printf("%d %d %d %d",src[0].width,src[0].height,src[0].depth,src[0].dim); std::exit(0); }
  if (a_get_width)  { std::printf("%d",src[0].width);  std::exit(0); }
  if (a_get_height) { std::printf("%d",src[0].height); std::exit(0); }
  if (a_get_depth)  { std::printf("%d",src[0].depth);  std::exit(0); }
  if (a_get_dim)    { std::printf("%d",src[0].dim);    std::exit(0); }

  // Display image statistics
  std::fprintf(stderr,"\n- Input image(s) statistics :\n");
  cimglist_for(src,l) {
    CImgStats stats(src(l));
    std::fprintf(stderr,"\t> [%d]='%s' : size=%dx%dx%dx%d, min=%g, mean=%g, max=%g, var=%g\n",
		 l,cimg::basename(filename(l).data),
		 src[l].dimx(),src[l].dimy(),src[l].dimz(),src[l].dimv(),
		 stats.min,stats.mean,stats.max,stats.variance
		 );
  }

  // Smooth the image anisotropically
  if (a_smooth>0) {
    std::fprintf(stderr,"\n- Smooth the image anisotropically with strength=%g\n",a_smooth); std::fflush(stderr);
    cimglist_for(src,l) src[l].blur_anisotropic(a_smooth);
  }

  // Vector orientation computation
  if (a_orient) {
    std::fprintf(stderr,"\n- Vector orientation computation\t"); std::fflush(stderr);
    cimglist_for(src,l) src[l].orientation_pointwise();
  }

  // Vector norm computation
  if (a_norm) {
    std::fprintf(stderr,"\n- Vector norm computation\t"); std::fflush(stderr);
    cimglist_for(src,l) src[l].norm_pointwise();
  }

  // Image rotation
  if (a_rotate!=0) {
    std::fprintf(stderr,"\n- Image rotation ( %g degrees )\t",a_rotate); std::fflush(stderr);
    cimglist_for(src,l) src[l].rotate(a_rotate,3);
  }

  // Image transpose
  if (a_transpose) {
    std::fprintf(stderr,"\n- Image transpose\t"); std::fflush(stderr);
    cimglist_for(src,l) src[l].transpose();
  }

  // Mirror Image
  if (a_mirror!='\0') {
    std::fprintf(stderr,"\n- Mirror image along axis '%c'\t",a_mirror); std::fflush(stderr);
    cimglist_for(src,l) src[l].mirror(a_mirror);
  }

  // Image blur
  if (a_blur!=0) {
    std::fprintf(stderr,"\n- Image blur ( sigma = %g )\t",a_blur); std::fflush(stderr);
    cimglist_for(src,l) src[l].blur(a_blur);
  }

  // Image noise
  if (a_noiseg) {
    std::fprintf(stderr,"\n- Add gaussian noise ( variance = %g )\t",a_noiseg); std::fflush(stderr);
    cimglist_for(src,l) src[l].noise(a_noiseg,0);
  }
  if (a_noiseu) {
    std::fprintf(stderr,"\n- Add uniform noise ( variance = %g )\t",a_noiseu); std::fflush(stderr);
    cimglist_for(src,l) src[l].noise(a_noiseu,1);
  }
  if (a_noises) {
    std::fprintf(stderr,"\n- Add uniform noise ( variance = %g )\t",a_noises); std::fflush(stderr);
    cimglist_for(src,l) src[l].noise(a_noises,2);
  }

  // Pixel value cutting
  if (a_cut) {
    std::fprintf(stderr,"\n- Pixel cutting in [%s]\t",a_cut); std::fflush(stderr);
    cimglist_for(src,l) {
      CImg<T>& img=src[l];
      CImgStats stats(img);
      double vmin=stats.min, vmax=stats.max;
      std::sscanf(a_cut,"%lg%*c%lg",&vmin,&vmax);
      img.cut((T)vmin,(T)vmax);
    }
  }

  // Pixel value normalization
  if (a_vnorm) {
    std::fprintf(stderr,"\n- Pixel normalization in [%s]\t",a_vnorm); std::fflush(stderr);
    cimglist_for(src,l) {
      CImg<T>& img=src[l];
      CImgStats stats(img);
      double vmin=stats.min, vmax=stats.max;
      std::sscanf(a_vnorm,"%lg%*c%lg",&vmin,&vmax);
      img.normalize((T)vmin,(T)vmax);
    }
  }

  // Image quantification
  if (a_quantize) {
    std::fprintf(stderr,"\n- Quantize image in %d levels\t",a_quantize); std::fflush(stderr);
    cimglist_for(src,l) src[l].quantize(a_quantize);
  }

  // Image cropping
  if (a_bcrop || a_ecrop) {
    std::fprintf(stderr,"\n- Crop image(s) to %s-%s\t",a_bcrop?a_bcrop:"*",a_ecrop?a_ecrop:"*"); std::fflush(stderr);
    cimglist_for(src,l) {
      CImg<T> &img=src[l];
      int x0=0,y0=0,z0=0,v0=0,x1=img.dimx()-1,y1=img.dimy()-1,z1=img.dimz()-1,v1=img.dimv()-1;
      char buf[32];
      if (a_bcrop) std::sscanf(a_bcrop,"%d%15[^0-9]%d%15[^0-9]%d%15[^0-9]%d%15[^0-9]",&x0,buf+0,&y0,buf+1,&z0,buf+2,&v0,buf+3);
      if (a_ecrop) std::sscanf(a_ecrop,"%d%15[^0-9]%d%15[^0-9]%d%15[^0-9]%d%15[^0-9]",&x1,buf+4,&y1,buf+5,&z1,buf+6,&v1,buf+7);
      if (buf[0]=='%') x0 = x0*(img.dimx()-1)/100;
      if (buf[1]=='%') y0 = y0*(img.dimy()-1)/100;
      if (buf[2]=='%') z0 = z0*(img.dimz()-1)/100;
      if (buf[3]=='%') v0 = v0*(img.dimv()-1)/100;
      if (buf[4]=='%') x1 = x1*(img.dimx()-1)/100;
      if (buf[5]=='%') y1 = y1*(img.dimy()-1)/100;
      if (buf[6]=='%') z1 = z1*(img.dimz()-1)/100;
      if (buf[7]=='%') v1 = v1*(img.dimv()-1)/100;
      if (x0>x1) cimg::swap(x0,x1);
      if (y0>y1) cimg::swap(y0,y1);
      if (z0>z1) cimg::swap(z0,z1);
      if (v0>v1) cimg::swap(v0,v1);
      img.crop(x0,y0,z0,v0,x1,y1,z1,v1);
    }
  }

  // Image resizing
  if (a_resize) {
    std::fprintf(stderr,"\n- Resize image(s) to '%s'\t",a_resize); std::fflush(stderr);
    int dimx=-100,dimy=-100,dimz=-100,dimv=-100;
    char buf[16];
    std::sscanf(a_resize,"%d%15[^0-9]%d%15[^0-9]%d%15[^0-9]%d%15[^0-9]",&dimx,buf,&dimy,buf+1,&dimz,buf+2,&dimv,buf+3);
    if (buf[0]=='%') dimx=-dimx;
    if (buf[1]=='%') dimy=-dimy;
    if (buf[2]=='%') dimz=-dimz;
    if (buf[3]=='%') dimv=-dimv;
    cimglist_for(src,l) src[l].resize(dimx,dimy,dimz,dimv,a_interp);
  }

  if (a_equalize) {
    std::fprintf(stderr,"\n- Equalize histogram\t"); std::fflush(stderr);
    cimglist_for(src,l) src[l].equalize_histogram(4096);
  }

  // Multiplication of pixel values
  if (a_mul!=1) {
    std::fprintf(stderr,"\n- Multiply pixel values by %g\t",a_mul); std::fflush(stderr);
    cimglist_for(src,l) src[l]*=a_mul;
  }

  // Add a constant to pixel values
  if (a_add!=0) {
    std::fprintf(stderr,"\n- Add %g to pixel values\t",a_add); std::fflush(stderr);
    cimglist_for(src,l) src[l]+=(T)a_add;
  }

  // Erode image
  if (a_erode) {
    std::fprintf(stderr,"\n- Erode image(s) %d times",a_erode); std::fflush(stderr);
    cimglist_for(src,l) src[l].erode(a_erode);
  }

  // Dilate image
  if (a_dilate) {
    std::fprintf(stderr,"\n- Dilate image(s) %d times",a_dilate); std::fflush(stderr);
    cimglist_for(src,l) src[l].dilate(a_dilate);
  }

  // Apply a median filter
  if (a_median) {
    std::fprintf(stderr,"\n- Apply a median filter of size %d\n",a_median); std::fflush(stderr);
    cimglist_for(src,l) src[l].blur_median(a_median);
  }

  // Compute false colors
  if (a_falsecolors) {
    std::fprintf(stderr,"\n- Compute false colors\t"); std::fflush(stderr);
    cimglist_for(src,l) {
      CImg<unsigned char> tmp = src[l].get_normalize((T)0,(T)255);
      cimg_forXYZV(tmp,x,y,z,k) src[l](x,y,z,k) = (T)cimg::ror((unsigned char)tmp(x,y,z,k),4);
    }
  }

  // Compute image histogram
  if (a_histo) {
    std::fprintf(stderr,"\n- Compute histogram(s)\t"); std::fflush(stderr);
    cimglist_for(src,l) src[l] = src[l].get_histogram(a_histo);
  }

  // Image break
  if (a_break) {
    int nb = 0;
    std::sscanf(a_break,"%*[xyzvXYZV]%d",&nb);
    std::fprintf(stderr,"\n- Break image along axis '%c'\t",a_break[0]); std::fflush(stderr);
    CImgList<T> tmp;
    cimglist_for(src,l) tmp.insert(src[l].get_split(a_break[0],nb));
    src = tmp;
  }

  // Images append
  if (a_append!='\0') {
    std::fprintf(stderr,"\n- Append images along axis '%c'\t",a_append); std::fflush(stderr);
    src = CImgList<T>(src.get_append(a_append,'p'));
  }

  // User-selected crop
  if (a_ucrop) {
    std::fprintf(stderr,"\n- User crop : Please select the image region to crop.\t"); std::fflush(stderr);
    int selection[6] = {-1,-1,-1,-1,-1,-1};
    src[0].feature_selection(selection,2);
    if (selection[0]<0 || selection[1]<0 || selection[2]<0 ||
	selection[3]<0 || selection[4]<0 || selection[5]<0) std::fprintf(stderr,".. Aborted..");
    else {
      std::fprintf(stderr,"-> Crop [%d,%d,%d]-[%d,%d,%d]",
		   selection[0],selection[1],selection[2],
		   selection[3],selection[4],selection[5]);
      cimglist_for(src,l) src[l].crop(selection[0],selection[1],selection[2],selection[3],selection[4],selection[5]);
    }
  }

  // Apply logarithm, square root or square
  if (a_log) cimglist_for(src,l) (src[l]+=(T)1e-10f).log();
  if (a_sqrt) cimglist_for(src,l) src[l].sqrt();
  if (a_sqr) cimglist_for(src,l) src[l].pow(2.0);

  // Unroll image
  if (a_unroll) {
    std::fprintf(stderr,"\n- Unroll image along axis '%c'.\t",a_unroll); std::fflush(stderr);
    cimglist_for(src,l) src[l].unroll(a_unroll);
  }

  // Permute image axes
  if (a_permute) {
    std::fprintf(stderr,"\n- Permute axes with permutation '%s'.\t",a_permute); std::fflush(stderr);
    cimglist_for(src,l) src[l].permute_axes(a_permute);
  }

  // Annotate images
  if (a_annot) {
    std::fprintf(stderr,"\n- Annotate image with \"%s\".\t",a_annot); std::fflush(stderr);
    cimglist_for(src,l) {
      const CImg<T> col(src[l].dimv(),1,1,1,(T)CImgStats(src[l],false).max);
      src[l].draw_text(a_annot,5,5,col.ptr());
    }
  }

  // Reverse image list
  if (a_reverse) {
    std::fprintf(stderr,"\n- Reverse image list"); std::fflush(stderr);
    src.reverse();
  }

  // Convert image to destination type
  std::fprintf(stderr,"\n- Conversion to '%s' pixels\t",cimg::type<t>::id()); std::fflush(stderr);
  CImgList<t> dest(src);
  std::fprintf(stderr,"\n- Output images list : %d image(s)\n",dest.size);
  {cimglist_for(dest,l) {
    const CImgStats stats(dest[l]);
    std::fprintf(stderr,"\t> [%d] : size=%dx%dx%dx%d, min=%g, mean=%g, max=%g, var=%g\n",
		 l,dest[l].dimx(),dest[l].dimy(),dest[l].dimz(),dest[l].dimv(),
		 stats.min,stats.mean,stats.max,stats.variance);
  }}


  // Save destination image
  if (file_o) {
    std::fprintf(stderr,"\n- Save image(s) to '%s'\t",file_o); std::fflush(stderr);
    dest.save(file_o);
  }

  // Display destination image
  if (a_visu || !file_o) {
    std::fprintf(stderr,"\n- Visualize image(s)\n"); std::fflush(stderr);
    if ( dest.size==1 && (
	 ( dest[0].dimy()==1 && dest[0].dimz()==1  ) ||
	 ( dest[0].dimx()==1  && dest[0].dimz()==1  ) ||
	 ( dest[0].dimx()==1  && dest[0].dimy()==1 ) ||
	 ( dest[0].dimx()==1  && dest[0].dimy()==1 && dest[0].dimz()==1))) { // 1D plot representation
      CImg<t> resized_img;
      const CImg<t>& img = dest[0].dimv()<=8?dest[0]:(resized_img=dest[0].get_resize(-100,-100,-100,8,0));
      CImgStats stats(img,false);
      CImg<unsigned char> palette = CImg<unsigned char>(3,img.dimv(),1,1,0).noise(300,1);
      const unsigned char gray[3]={128,128,128},white[3]={255,255,255};
      unsigned long trame = 0x0F0F0F0F;
      char message[1024];
      palette[0]=255; palette[1]=0; palette[2]=0;
      if (img.dimv()>1) { palette[3]=0; palette[4]=255; palette[5]=0; }
      if (img.dimv()>2) { palette[6]=0; palette[7]=0; palette[8]=255; }
      CImgDisplay disp(500,300,"1D plot of the output image",0);
      CImg<unsigned char> visu0;
      do {
	if (!visu0.data || disp.is_resized) { // plot graphics
	  disp.resize();
	  visu0 = CImg<unsigned char>(disp.dimx(),disp.dimy(),1,3,0);
	  if (img.dimv()==1) visu0.draw_graph(img,palette.ptr(),1,stats.max,stats.min,0.4f);
	  cimg_forV(img,k) visu0.draw_graph(img.get_shared_channel(k),&palette(0,k),0,stats.max,stats.min);
	  visu0.draw_axis(CImg<unsigned int>::sequence(1+visu0.width/60,0,(int)img.size()-1),
			  CImg<double>::sequence(1+visu0.height/60,stats.max,stats.min),
			  gray);
	}
	CImg<unsigned char> visu(visu0);
	if (disp.mouse_x>=0) {
	  visu.draw_line(disp.mouse_x,0,disp.mouse_x,visu.dimy()-1,gray,trame,0.5);
	  trame = cimg::rol(trame);
	  const unsigned x = disp.mouse_x*img.size()/(img.dimv()*disp.dimx());
	  std::sprintf(message,"x=%d\n[ ",x);
	  cimg_forV(img,k) std::sprintf(message+std::strlen(message),"%s%g%s",k==0?"":"  ",
					(float)img(x,0,0,k),k==img.dimv()-1?"":"\n");
	  std::sprintf(message+std::strlen(message)," %s]",dest[0].dimv()<=8?"":"\n...");
	  visu.draw_text(message,disp.mouse_x-20,10,white,NULL,1);
	} else visu=visu0;
	disp.display(visu).wait(40);
      } while (!disp.is_closed && !disp.key);
    }
    else dest.display("inrcast");
  }
}

//---------------------------------------------------------
// Read input image and do conversion to output pixel type
//---------------------------------------------------------
template<typename T> void input_file(int argc, char **argv, T& pixel_type) {
  pixel_type = 0;

  // Read input image list
  std::fprintf(stderr,"- Load image(s)...\t"); std::fflush(stderr);
  CImgList<T> src;
  CImgList<char> filename;
  bool opt = false;
  for (int k=1; k<argc; k++)
    if (argv[k][0]=='-' && argv[k][1]!='\0' && argv[k][1]!='.') opt=true;
    else {
      if (!opt ||
	  !cimg::strcasecmp(argv[k-1],"-h") ||
	  !cimg::strcasecmp(argv[k-1],"-orient") ||
	  !cimg::strcasecmp(argv[k-1],"-norm") ||
	  !cimg::strcasecmp(argv[k-1],"-equalize") ||
	  !cimg::strcasecmp(argv[k-1],"-crop") ||
	  !cimg::strcasecmp(argv[k-1],"-plot") ||
	  !cimg::strcasecmp(argv[k-1],"-visu") ||
	  !cimg::strcasecmp(argv[k-1],"-get_size") ||
	  !cimg::strcasecmp(argv[k-1],"-get_width") ||
	  !cimg::strcasecmp(argv[k-1],"-get_height") ||
	  !cimg::strcasecmp(argv[k-1],"-get_depth") ||
	  !cimg::strcasecmp(argv[k-1],"-get_dim")
	  ) {
	std::fprintf(stderr,"\n\t> Loading [%d]='%s'\t",k-1,argv[k]); std::fflush(stderr);

	CImgList<T> tmpl;
	const char *ext = cimg::filename_split(argv[k]);

	if (!cimg::strcasecmp(ext,"yuv")) {
	  unsigned int w=352,h=288;
	  const char *geom = cimg_option("-g","352x288",NULL);
	  if (geom) std::sscanf(geom,"%u%*c%u",&w,&h);
	  unsigned int frame_first=0;
	  int frame_last=-1;
	  const char *frames = cimg_option("-yuvframes",(const char*)NULL,"YUV range of frames");
	  if (frames) std::sscanf(frames,"%u%*c%d",&frame_first,&frame_last);
	  tmpl.load_yuv(argv[k],w,h,frame_first,frame_last,true);
	} else {
	  if (!cimg::strcasecmp(ext,"raw")) {
	    unsigned int w=0,h=1,d=1,v=1;
	    const char *geom = cimg_option("-g",(char*)NULL,NULL);
	    if (geom) std::sscanf(geom,"%u%*c%u%*c%u%*c%u",&w,&h,&d,&v);
	    else {
	      std::fprintf(stderr,"\ninrcast : You must specify an input geometry (option -g) for .raw images\n");
	      std::exit(0);
	    }
	    const bool multiplexed = cimg_option("-multiplexed",false,"Use multiplexed mode");
	    const bool endian_swap = cimg_option("-endian",false,"Big/Little endian mode");
	    tmpl = CImgList<T>(CImg<T>::get_load_raw(argv[k],w,h,d,v,multiplexed,endian_swap));
	  } else {
	    if (!cimg::strcasecmp(ext,"rgb")) {
	      unsigned int w=0,h=1;
	      const char *geom = cimg_option("-g",(char*)NULL,NULL);
	      if (geom) std::sscanf(geom,"%u%*c%u",&w,&h);
	      else {
		std::fprintf(stderr,"\ninrcast : You must specify an input geometry (option -g) for .rgb images\n");
		std::exit(0);
	      }
	      tmpl = CImgList<T>(CImg<T>::get_load_rgb(argv[k],w,h));
	    } else {
	      if (!cimg::strcasecmp(ext,"rgba")) {
		unsigned int w=0,h=1;
		const char *geom = cimg_option("-g",(char*)NULL,NULL);
		if (geom) std::sscanf(geom,"%u%*c%u",&w,&h);
		else {
		  std::fprintf(stderr,"\ninrcast : You must specify an input geometry (option -g) for .rgba images\n");
		  std::exit(0);
		}
		tmpl = CImgList<T>(CImg<T>::get_load_rgba(argv[k],w,h));
	      } else tmpl = CImgList<T>(argv[k]);
	    }
	  }
	}
	const CImg<char> name(argv[k],cimg::strlen(argv[k])+1);
	if (tmpl.size==1) filename.insert(name);
	else cimglist_for(tmpl,l) {
	  char s[1024];
	  std::sprintf(s,"%s [%d]",name.data,l);
	  filename.insert(CImg<char>(s,cimg::strlen(s)+1));
	}
	src.insert(tmpl);
      }
      opt=false;
    }
  std::fputc('\n',stderr);

  // Convert to output pixel type (if specified)
  const char *typei = cimg_option("-i","float",NULL);
  const char *typeo = cimg_option("-t",typei,NULL);
  bool saved = false;
  if (!saved && !cimg::strcasecmp(typeo,"bool"))   { bool t=false;       output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"char"))   { char t=0;           output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"uchar"))  { unsigned char t=0;  output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"ushort")) { unsigned short t=0; output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"short"))  { short t=0;          output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"uint"))   { unsigned int t=0;   output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"int"))    { int t=0;            output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"ulong"))  { unsigned long t=0;  output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"long"))   { long t=0;           output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"float"))  { float t=0;          output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved && !cimg::strcasecmp(typeo,"double")) { float t=0;          output_file(argc,argv,src,filename,t); saved=true; }
  if (!saved) { std::fprintf(stderr,"inrcast : Cannot save image with pixel type '%s' (see help)",typeo); std::exit(0); }
}

//----------------
// Main procedure
//----------------
int main(int argc,char **argv) {
  cimg_usage("1D/2D/3D Image(s) converter and visualizer.\n\t\tusage : inrcast [options] image_in1 [image_in2] [...] [-o image_out]");
  if (argc==1) { std::fprintf(stderr,"Try '%s -h' to get help\n",cimg::basename(argv[0])); std::exit(0); }

  // Read input image and call converter
  bool casted = false;
  const char *typei = cimg_option("-i","float",NULL);
  if (!casted && !cimg::strcasecmp(typei,"bool"))   { bool t=false;       input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"char"))   { char t=0;           input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"uchar"))  { unsigned char t=0;  input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"ushort")) { unsigned short t=0; input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"short"))  { short t=0;          input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"uint"))   { unsigned int t=0;   input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"int"))    { int t=0;            input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"ulong"))  { unsigned long t=0;  input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"long"))   { long t=0;           input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"float"))  { float t=0;          input_file(argc,argv,t); casted=true; }
  if (!casted && !cimg::strcasecmp(typei,"double")) { float t=0;          input_file(argc,argv,t); casted=true; }
  if (!casted) { std::fprintf(stderr,"Cannot load image pixel type '%s' (see help)",typei); std::exit(0); }

  // Return to shell
  std::fprintf(stderr,"Done !\n");
  std::exit(0);
  return 0;
}
