/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <locale.h> /* setlocale() */
#include <sys/ioctl.h>

/* open */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "config.h"
#include "cdw_logging.h"
#include "gettext.h"
#include "cdw_drive.h"
#include "cdw_utils.h"
#include "cdw_debug.h"



void after_event(const char *title, __attribute__((unused)) int adddic)
{
	cdw_logging_display_log_conditional(title);
#ifdef HAVE_LIBMYSQLCLIENT
	if ((strcmp(config.autodic, "1") == 0) && (adddic == 1))
		add_to_dic();
#endif
#ifdef HAVE_LIBSQLITE
	if ((strcmp(config.autodic, "1") == 0) && (adddic == 1))
		add_to_dic();
#endif
	/* disabled in version 0.4.0, ejecting after write operation is
	handled by conditional_eject_after_write(); even if this eject is
	required by other operation, they are unsupported in 0.4.0 */
	/*
	if (strcmp(config.eject, "1") == 0) {
		eject_tray(config.cdrw_device);
	}
	*/

	return;
}





/**
   \brief Process main(argc, argv[]) arguments

   \return -1 if at least one argument was incorrect
   \return 1 if user wants to turn support for DVD+R/DL on
   \return 0 otherwise
*/
int process_commandline_args(int argc, char *argv[])
{
#ifdef HAVE_LIBMYSQLCLIENT
	if ((strcmp((char *) argv[1], "--catalog") == 0) || (strcmp((char *) argv[1], "-c") == 0)) {
		initscr();
		start_color();
		cbreak();
		noecho();
		keypad(stdscr, TRUE);
		/* colors */
		init_curses_colors();
		lines = LINES - 2;
		if ((LINES < 24) || (COLS < 79)) {
			/* 2TRANS: error message displayed when terminal requirements are not met */
			fprintf(stderr, _("Needed min 80x25 terminal!"));
			exit(EXIT_FAILURE);
		}
		cddb_window();
		exit(EXIT_SUCCESS);
	}
#endif

	bool support_dvd_rp_dl = false;

	/* flags used to prevent displaying the same content when short and long
	   option are used in the same command line */
	bool displayed_help = false;
	bool displayed_version = false;

	for (int i = 1; i < argc; i++) {
		cdw_sdm ("INFO: processing option argv[%d] = \"%s\"\n", i, argv[i]);
		if ((strcmp((char *) argv[i], "--version") == 0) || (strcmp((char *) argv[i], "-v") == 0)) {
			if (displayed_version) {
				;
			} else {
				/* 2TRANS: this is message printed in terminal,
				   first %s is program name, second %s is
				   program version */
				printf(_("%s-%s Copyright (C) 2002 - 2003 Balazs Varkonyi\n"), PACKAGE, VERSION);
				/* 2TRANS: this is message printed in terminal,
				   first %s is program name, second %s is
				   program version */
				printf(_("%s-%s Copyright (C) 2007 - 2010 Kamil Ignacak\n\n"), PACKAGE, VERSION);

				displayed_version = true;
			}
		} else if ((strcmp((char *) argv[i], "--help") == 0) || (strcmp((char *) argv[i], "-h") == 0)) {
			if (displayed_help) {
				;
			} else {
				/* 2TRANS: this is help text displayed in console;
				   %s will be replaced by program name, don't change its
				   position */
				printf(_("Usage: %s [options]\n\n"), argv[0]);
				/* 2TRANS: this is help text displayed in console */
				printf(_("Options:\n\n"));
#ifdef HAVE_LIBMYSQLCLIENT
				/* 2TRANS: this is help text displayed in console;
				   don't localize anything before colon */
				printf(_("  --catalog | -c            : start in Disk Catalog mode\n"));
#endif
				/* 2TRANS: this is help text displayed in console;
				   don't localize anything before colon */
				printf(_("  --help    | -h            : show this screen\n"));
				/* 2TRANS: this is help text displayed in console;
				   don't localize anything before colon */
				printf(_("  --version | -v            : show version\n"));
				/* 2TRANS: this is help text displayed in console;
				   don't localize anything before colon */
				printf(_("  --enable-dvd-rp-dl        : enable support for DVD+R DL (buggy, with dvd+rw-tools only)\n\n"));

				displayed_help = true;
			}
		} else if (strcmp((char *) argv[i], "--enable-dvd-rp-dl") == 0) {
			support_dvd_rp_dl = true;
		} else {
			/* 2TRANS: first %s is program name, second - argument given to the program */
			printf(_("%s: invalid option: %s\n"), argv[0], argv[i]);
			/* 2TRANS: %s is program name */
			printf(_("Try \"%s --help\"\n"), argv[0]);
			return -1;
		}
	}

	if (support_dvd_rp_dl) {
		return 1;
	} else {
		return 0;
	}
}





/*
 * Simple wrapper for locale setting functions
 */
int cdw_locale_init(void)
{
	setlocale(LC_ALL, "");
	textdomain(PACKAGE);
	bindtextdomain(PACKAGE, LOCALEDIR);

	return 0;
}




#if 0
/* unused at this moment, replaced with registering functions with atexit() */
/**
 * Deallocate all previously allocated resources, terminate ncurses
 *
 * Prepare application for closing:close all files, deallocate memory,
 * destroy custom and ncurses widgets. Exit ncurses. Try to do all this
 * in reverse order to the one found in main().
 */
void clean_before_cdw_exit(void)
{
	cdw_config_module_clean(); /* cdw configuration module */

	/* file manager has to clean up 'selected files' subwindow
	   view before cdw_ui_main_clean() can remove parent window;
	   FIXME: perhaps cdw_ui_main should call file manager's
	   function to destroy the view */
	cdw_file_manager_clean(); /* cdw file selector */

	cdw_main_ui_clean(); /* main application window */

	cdw_logging_clean();

	cdw_ext_tools_clean();

	cdw_fs_clean();

	cdw_ncurses_clean();

	cdw_cdio_clean();

	return;
}
#endif




#if 0
/* currently unused, disabling to suppress compiler warnings */
/**
 * Read raw CD sector using Linux ioctl(2) - just a non-portable tool
 *
 * \param sector - valid sector number of sector to be read
 * \param buffer - char table of size 2352
 *
 * \return 0
 */
int raw_read_sector(lsn_t sector, char *_buffer)
{
	int fd = open(config.cdrw_device, O_RDONLY | O_NONBLOCK);
	if (fd == -1) {
		/* debug code */
		/* fprintf(stderr, "\n\n raw_read failed\n\n\n"); */
		return -1;
	}


	union {
		struct cdrom_msf msf;   /* input WARNING: make sure which data structure is used as input, read comments for CDROMREAD* in ioctl.txt */
		char buffer[2352];      /* return */
	} arg;

	msf_t p_msf;
	cdio_lsn_to_msf(3000, &p_msf);
	arg.msf.cdmsf_min0 = p_msf.m;     /* start minute */
	arg.msf.cdmsf_sec0 = p_msf.s;     /* start second */
	arg.msf.cdmsf_frame0 = p_msf.f;   /* start frame */

	int rv0 = ioctl(fd, CDROMREADRAW, &arg);

	/* debug code */
	/* perror("1 -- ");
	fprintf(stderr, "rv0 = %d\n", rv0);

	fprintf(stderr, ">>RAW1>>");
	int i = 0;
	for (i =  0; i< 40; i++) {
	fprintf(stderr, " %x ", arg.buffer[i]);
}
	fprintf(stderr, "<<\n");
	*/


	/* ************************************* */


	char internal_buffer[2352];
	struct cdrom_read arg2;
	arg2.cdread_lba = 3000; /* FIXME: LBA != lsn */
	arg2.cdread_bufaddr = (char *) malloc(2352);
	arg2.cdread_buflen = 2352;

	int rv = ioctl(fd, CDROMREADRAW, &arg2);

	/* debug code */
	/*
	perror("2 -- ");
	fprintf(stderr, "rv = %d\n", rv);

	fprintf(stderr, ">>RAW2>>");
	for (i =  0; i< 40; i++) {
	fprintf(stderr, " %x ", internal_buffer[i]);
}
	fprintf(stderr, "<<\n");
	*/

	close(fd);


	/* int disc_status = ioctl(fd, CDROM_DISC_STATUS, 0); */
	/* debug code */
	/*
	if (disc_status == CDS_NO_INFO) {
	fprintf(stderr, ">>>>>>>>> CDS_NO_INFO <<<<<<<<<<<\n");
} else if (disc_status == CDS_AUDIO) {
	fprintf(stderr, ">>>>>>>>> CDS_AUDIO <<<<<<<<<<<\n");
} else if (disc_status == CDS_MIXED) {
	fprintf(stderr, ">>>>>>>>> CDS_MIXED <<<<<<<<<<<\n");
} else if (disc_status == CDS_XA_2_2) {
	fprintf(stderr, ">>>>>>>>> CDS_XA_2_2 <<<<<<<<<<<\n");
} else if (disc_status == CDS_XA_2_1) {
	fprintf(stderr, ">>>>>>>>> CDS_XA_2_1 <<<<<<<<<<<\n");
} else if (disc_status == CDS_DATA_1) {
	fprintf(stderr, ">>>>>>>>> CDS_DATA_1 <<<<<<<<<<<\n");
} else {
	fprintf(stderr, ">>>>>>>>> ???? <<<<<<<<<<<\n");
}
	*/

	return 0;
}





int raw_read_sector2(lsn_t sector)
{
	struct cdrom_msf msf;
	char buffer[2352];

	msf_t p_msf;

	int fd = open(config.cdrw_device, O_RDWR | O_NONBLOCK);
	if (fd == -1) {
		/* debug code */
		/* fprintf(stderr, "bledny desk. pliku\n"); */
		return -1;

	}

	cdio_lsn_to_msf(sector, &p_msf);
	msf.cdmsf_min0 = p_msf.m;     /* start minute */
	msf.cdmsf_sec0 = p_msf.s;     /* start second */
	msf.cdmsf_frame0 = p_msf.f;   /* start frame */

	msf.cdmsf_min0 = 1;     /* start minute */
	msf.cdmsf_sec0 = 1;     /* start second */
	msf.cdmsf_frame0 = 1;   /* start frame */

	memcpy(buffer, &msf, sizeof(msf));
	int rv = ioctl(fd, CDROMREADRAW, buffer);
	/* debug code */
	/*
	perror("UNO: -- ");
	fprintf(stderr, ": %d\n", rv);
	*/
	close(fd);

	return 0;
}
#endif





cdw_rv_t get_cds_mixed_with_ioctl(void)
{
	const char *drive = cdw_drive_get_drive_fullpath();
	int fd = open(drive, O_RDONLY);
	if (fd != -1) {
		int ioctl_mode = ioctl(fd, CDROM_DISC_STATUS, 0);
		close(fd);
		if (ioctl_mode != -1) {
			if (ioctl_mode == CDS_MIXED) {
				return CDW_OK;
			} else {
				return CDW_NO;
			}
		} else {
			return CDW_GEN_ERROR;
		}
	} else {
		return CDW_GEN_ERROR;
	}
}







/**
   \brief Read one line of text, but no longer than 10k of chars

   Function ends the line with '\0';

   \param file - open file that you want to read from

   \return pointer to freshly allocated space with characters read
   \return NULL if reading failed
*/
char *my_readline_10k(FILE *file)
{
	char *buffer = NULL;
	size_t total_len = 0;
	while(1) {
		char fgets_buffer[100];
		int fgets_buffer_size = 100; /* reasonable initial value */

		char *r = fgets(fgets_buffer, fgets_buffer_size, file);
		if (r == fgets_buffer) { /* some content was read to buffer */

			size_t fgets_len = strlen(fgets_buffer);
			total_len += fgets_len;
			char *tmp = (char *) realloc(buffer, total_len + 1);

			if (tmp != buffer) { /* buffer was re-allocated AND moved */
				buffer = tmp;
			}

			char *end = buffer + total_len - fgets_len;
			/* +1 for ending '\0' that is always added by fgets(); */
			strncpy(end, fgets_buffer, fgets_len + 1);

			/* '\n' from file is also read by fgets() and put into string */
			if (*(buffer + total_len - 1) == '\n') {
				// *(buffer + total_len - 1) = '\0';
				/* this also means that last fragment of line was read */
				break;
			}

			if (total_len >= 10000) {
				*(buffer + total_len) = '\0';
				/* I can't imagine any option longer than 10k chars */
				break;
			}


		} else { /* r == NULL: EOF or error */
			break;
		}
	}

	return buffer;
}









/**
   \brief Function inspecting given table of id,label pairs and returning
   label matching given id.

   First argument is table of elements of cdw_id_label_t type.
   Last element of the table must be {-1, "unknown value"}, where -1 serves
   as guard and "unknown value" string that is displayed when there is no
   other match.

   Second argument is id / key of searched string.

   \param table - table of id,label pairs
   \param id -  id to be matched against values in table

   \return pointer to const string that matches given id,
   \return fall back error string if no id matches
*/
const char *cdw_utils_id_label_table_get_label(cdw_id_label_t *table, long int id)
{
	cdw_assert (table != (cdw_id_label_t *) NULL, "ERROR: input table is NULL\n");

	/* this function uses while() loop for traversing table with id/label
	   pairs, but since:
	   a) this function is called only in debug builds
	   b) table sizes are usually very small (usually less than 10 elements)

	   then longer time of lookup (sequential rather than direct access)
	   does not influence performance */

#ifndef NDEBUG
	int j = 0;
	/* this piece of code is used to detect potential incorrectly built
	   tables without final guard string */
	while (table[j].label != (char *) NULL) {
		__attribute__((unused)) size_t len = strlen(table[j].label);
		j++;
	}
#endif

	int i = 0;
	/* it turns out that there may be valid IDs == -1,
	   so I'm using (char *) NULL as guard */
	while (table[i].label != (char *) NULL
	       && table[i].id != id) {

		i++;
	}

	return table[i].label; /* may be (char *) NULL */
}







static cdw_id_label_t cdw_rv_t_table[] = {
	{ CDW_OK,             "CDW_OK"            },
	{ CDW_CANCEL,         "CDW_CANCEL"        },
	{ CDW_NO,             "CDW_NO"            },
	{ CDW_GEN_ERROR,      "CDW_GEN_ERROR"     },
	{ CDW_FILE_INVALID,   "CDW_FILE_INVALID"  },
	{ CDW_MEM_ERROR,      "CDW_MEM_ERROR"     },
	{ CDW_SYS_ERROR,      "CDW_SYS_ERROR"     },
	{ CDW_ARG_ERROR,      "CDW_ARG_ERROR"     },
	{ CDW_FAILSAFE_MODE,  "CDW_FAILSAFE_MODE" },
	{ -1,                 "ERROR: unimplemented return value" }}; /* -1 = guard */


const char *cdw_utils_get_crv_label(cdw_rv_t crv)
{
	return cdw_utils_id_label_table_get_label(cdw_rv_t_table, crv);
}





cdw_id_label_t cdw_bool_type_labels[] = {
	{ CDW_UNKNOWN, "CDW_UNKNOWN" },
	{ CDW_TRUE,    "CDW_TRUE" },
	{ CDW_FALSE,   "CDW_FALSE" },
	{ 0,           (char *) NULL }};


const char *cdw_utils_get_cdw_bool_type_label(int val)
{
	return cdw_utils_id_label_table_get_label(cdw_bool_type_labels, val);
}





/**
   \brief Helper function for qsort() - compare two integer values

   Function that is used as compar() argument to qsort() function.
   It returns -1, 0 or 1  if first argument is less than, equal or
   larger than second.

   Passing this function to qsort() results in table of ints sorted
   from lowest to highest.

   \param int1 - first integer to compare
   \param int2 - second integer to compare

   \return -1, 0 or 1  if first int is less than, equal or larger than second
 */
int cdw_utils_compare_ints(const void *int1, const void *int2)
{
	const int *i1 = (const int *) int1;
	const int *i2 = (const int *) int2;

	if (*i1 < *i2) {
		return -1;
	} else if (*i1 > *i2) {
		return 1;
	} else {
		return 0;
	}
}






/**
   \brief Calculate estimated time of accomplishment for a task, convert this time to string

   You can calculate \p eta like this:
    - eta = amount left / speed
    - speed = amount done / time from beginning of process
    - eta = (amount left / amount done) * time from beginning of process

    or:
    int eta = (int) (((1.0 * todo_size) / done_size) * time_from_start);

    \param eta_string - output string with ETA in hh:mm:ss format
    \param eta - input eta value (most probably in seconds ;) )
    \param n - size of output buffer (does not include space for ending '\0')
*/
void cdw_utils_eta_calculations(char *eta_string, int eta, size_t n)
{
	cdw_sdm ("ETA input value: %d\n", eta);
	if (eta < 0) { /* it may happen at the beginning of process */
		/* 2TRANS: ETA is Expected Time of Arrival, time to finish a task;
		   three "??" values are number of hours, minutes seconds left,
		   "??" are used when the time cannot be calculated.
		   The string will be displayed in progress dialog window. Keep short. */
		snprintf(eta_string, n + 1, _("ETA: ??:??:??"));
	} else {
		/* C99 truncates result of division towards zero (C: A Reference manual, 5th ed., 7.6.1.) */

		int eta_hour = (eta / 60) / 60;
		/* extract number of seconds in full hours from total number of seconds */
		int eta_min = (eta - ((eta_hour * 60) * 60)) / 60;
		/* extract number of seconds in full hours and in full minutes from total number of seconds */
		int eta_sec = eta - (((eta_hour * 60) * 60) + (eta_min * 60));

		/* 2TRANS: ETA is Expected Time of Arrival, time to finish a task;
		   three "%d" values are number of hours, minutes seconds left. The
		   string will be displayed in progress dialog window. Keep short. */
		snprintf(eta_string, n + 1, _("ETA: %d:%02d:%02d"), eta_hour, eta_min, eta_sec);
	}
	cdw_sdm ("ETA output value: %s\n", eta_string);

	return;
}





/* *********************** */
/* *** unit tests code *** */
/* *********************** */

//#define CDW_UNIT_TEST_CODE /* definition used during development of unit tests code */
#ifdef CDW_UNIT_TEST_CODE



static void test_cdw_utils_eta_calculations(void);


void cdw_utils_run_tests(void)
{
	fprintf(stderr, "testing cdw_utils.c\n");

	test_cdw_utils_eta_calculations();

	fprintf(stderr, "done\n\n");

	return;
}



void test_cdw_utils_eta_calculations(void)
{
	fprintf(stderr, "\ttesting cdw_utils_eta_calculations()... ");

	char buf[50 + 1];

	struct {
		size_t buf_len; /* this test may use subbuffers of various lengths */
		int input_eta;
		const char *expected_result;
	} test_data[] = {
		{  20,  0, "ETA: 0:00:00" },
		{  20,  1, "ETA: 0:00:01" },
		{  20,  2, "ETA: 0:00:02" },
		{  20,  3, "ETA: 0:00:03" },
		{  20, 10, "ETA: 0:00:10" },
		{  20, 60, "ETA: 0:01:00" },
		{  20, 61, "ETA: 0:01:01" },
		{  20, 20 * 60 + 4,                    "ETA: 0:20:04" },
		{  20, (17 * 60 * 60) + (21 * 60) + 4, "ETA: 17:21:04" },
		{  10, (17 * 60 * 60) + (21 * 60) + 4, "ETA: 17:21" },

		{   0,  0, (char *) NULL }};

	int i = 0;
	while (test_data[i].expected_result != (char *) NULL) {
		cdw_utils_eta_calculations(buf, test_data[i].input_eta, test_data[i].buf_len);
		int rv = strcmp(test_data[i].expected_result, buf);
		cdw_assert_test (rv == 0, "ERROR: failed test #%d, eta = %d\n", i, test_data[i].input_eta);
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}


#endif /* #ifdef CDW_UNIT_TEST_CODE */

