/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <nl_types.h>
#include <time.h>
#include <libintl.h>

#include "main.h"
#include "options.h"
#include "gettext.h"
#include "log.h"

#include "pipe_regexp.h"
#include "thread.h" /* PIPE_BUFFER_SIZE */


int stdin_pipe[2];
int stdout_pipe[2];
int stderr_pipe[2];


FILE *logfi;

/* this is THREAD.c, so avoid conflicts */
pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

/* main cdw configuration data structure */
extern struct conf config;

int lasttrack = 0;

/* time captured at the beginning of process */
time_t time0;

/* buffers for data read/written through pair of UNIX sockets */
char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];




/*
 * print_stdout() and print_stderr() are two functions reading from
 * stdout (stdout_pipe[0]) and stderr (stderr_pipe[0])
 * of other process - this process creates image, writes disc or does sth else.
 * Purpose of this two functions is to intercept output of this process,
 * either from stdout or stderr file descriptors associated with child process.
 *
 * Information read from pipes is then processed using regexp functions - see
 * pipe_regexp.c for details.
 *
 * Creation of process and connection of sockets is done in run_command().
 * run_command() is main function in this file, hiding all threads-related
 * defails from the user.
 */


/* I see CDrecorder version 0.1 source, and I write this code after this!!!
 * THX to Sharad Mittal */
void *print_stdout(void *dummy)
{
	int data_processed = 0;

	/* initialize regexp variables */
	stdout_regexp_prepare();

	/* get some data from pipe */
	while (1) {
		memset(stdout_pipe_buffer, '\0', sizeof(stdout_pipe_buffer));

		/* read console output of some child process */
		data_processed = read(stdout_pipe[0], stdout_pipe_buffer, PIPE_BUFFER_SIZE);
		pthread_mutex_lock(&mutex);
		if (data_processed != -1) {
			fprintf(logfi, stdout_pipe_buffer);
			fflush(logfi);
			stdout_regexp_execute();
		} else if (data_processed == 0) { /* EOF */
			break;
		}
		pthread_mutex_unlock(&mutex);
		if (data_processed == 0)
			break;
		usleep(10);
	}


	stdout_regexp_destroy();

	return NULL;
}



/*
 * please see comments before print_stdout()
 */
void *print_stderr(void *dummy)
{
	int data_processed_err = 0;

	/* initialize regexp variables */
	stderr_regexp_prepare();

	while (1) {
		memset(stderr_pipe_buffer, '\0', sizeof(stderr_pipe_buffer));

		data_processed_err = read(stderr_pipe[0], stderr_pipe_buffer, PIPE_BUFFER_SIZE);
		pthread_mutex_lock(&mutex);
		if (data_processed_err != -1) {
			fprintf(logfi, stderr_pipe_buffer);
			fflush(logfi);
			stderr_regexp_execute();
		} else if (data_processed_err == 0) { // EOF
			break;
		}
		pthread_mutex_unlock(&mutex);
		if (data_processed_err == 0)
			break;
		usleep(10);
	}

	stderr_regexp_destroy();

	return NULL;
}



/**
 * Run external command
 *
 * Create pairs of sockets to new process and launch given program
 * as new child process. The process will inherit created sockets.
 *
 * \param char *command - program name + it's arguments
 *
 * \returns 1 on success
 */
int run_command(const char *command)
{
	if ((logfi = fopen(config.logfile, "w")) == NULL) {
		clean_before_cdw_exit();
		/* 2TRANS: this is message displayed in console when log file is not found and cdw will exit */
		fprintf(stderr, _("Cannot open logfile...\n"));
		exit(-1);
	}
	lasttrack = 0;

	time0 = time(NULL);
	/* 2TRANS: this is time stamp in printed to log file; %d is an int representing UNIX time */
	fprintf(logfi, _("Time: %d\n"), (int)time0);
	fprintf(logfi, "%s\n", command);
	if ((socketpair(AF_UNIX, SOCK_STREAM, 0, stdin_pipe) == 0)
		    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stdout_pipe) == 0)
		    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stderr_pipe) == 0)) {

		int fork_result;

		fcntl(stdout_pipe[1], F_SETFL, O_ASYNC);
		fcntl(stderr_pipe[1], F_SETFL, O_ASYNC);
		fcntl(stderr_pipe[0], F_SETFL, O_NONBLOCK);

		fork_result = fork();

		if (fork_result == -1) {
			clean_before_cdw_exit();
			/* 2TRANS: this is string displayed in console when
			  something went very badly and cdw cannot create child
			  process - this is serious error and cdw will exit */
			fprintf(stderr, _("Fork failure\n"));
			exit(-1);
		} else if (fork_result == 0) { // we are child process
			/* we are child process - don't write to stdout and stderr;
			* instead we should write to socked inherited from parent process,
			* so parent process can know what we (child process) are doing
			*
			* dup() will copy it's argument descriptors to lowest available
			* descriptors; let's make stdin, stdout and stderr
			* 'lowest avaliable' by closing them before calling dup();
			*
			 * let's also close original descriptor after copying it:
			* we have copy, original is no longer needed; we don't need
			* the other end of pipe here either (that is why we close both
			* *_pipe[0] and *_pipe[1]), but parent will keep the other ends
			* open to read from it as we write to our new stdout and stderr
			*/
			close(0);
			dup(stdin_pipe[0]);
			close(stdin_pipe[0]);
			close(stdin_pipe[1]);

			close(1);
			dup(stdout_pipe[1]);
			close(stdout_pipe[0]);
			close(stdout_pipe[1]);

			close(2);
			dup(stderr_pipe[1]);
			close(stderr_pipe[0]);
			close(stderr_pipe[1]);

			/* below a copy of cdw turns into e.g. mkisofs */

			char *cmd[] = { "sh" , "-c", command, (char *)0 };
			char *env[] = { "LC_ALL=C", "TERM=xterm", (char *)0 };

			/* I'm using execve, because it allows me to set
			   LC_ALL variable. After setting LC_ALL to C
			   I am sure that any command will be running with
			   English locale and will produce English messages
			   passed via pipe to regexp code. This way I don't
			   have to worry that some day wodim or genisoimage
			   developers will localize their software. */
			int exec_ret = execve ("/bin/sh", cmd, env);

			/* exec*() functions don't return */
			clean_before_cdw_exit();

			/* 2TRANS: this is message displayed in console when
			   something went very badly and exec() function
			   returned after a call */
			perror(_("Error: return of exec() function\n"));
			/* 2TRANS: this is second part of message displayed
			   in console when something went very badly and
			   exec() function returned after a call; %d is
			   value returned by exec(); cdw will exit after
			   displaying this message */
			fprintf(stderr, _("exec() returned with value %d\n"), exec_ret);
			exit(0);
		} else { // we are still in parent process
			int rv;
			pthread_t pout, perr;

			close(stdin_pipe[0]);
			close(stdin_pipe[1]);
			close(stderr_pipe[1]);
			close(stdout_pipe[1]);

			rv = pthread_create(&pout, NULL, &print_stdout, NULL);
			if (rv != 0) {
				/* 2TRANS: this is debug message displayed in
				   console when thread cannot be created */
				fprintf(stderr, _("Thread stdout creation error"));
			}
			rv = pthread_create(&perr, NULL, &print_stderr, NULL);
			if (rv != 0) {
				/* 2TRANS: this is debug message displayed in
				   console when thread cannot be created */
				fprintf(stderr, _("Thread stderr creation error"));
			}

			/* threads are created; now we must wait for them to complete
			* their job and return (both of them);
			* when thead functions end we can execute pthread_join()s */
			pthread_join(perr, NULL);
			pthread_join(pout, NULL);

			/* nothing more to read from these two */
			close(stderr_pipe[0]);
			close(stdout_pipe[0]);

			fclose(logfi);
		}
	} else { /* creating socket pairs failed */
		exit(-1);
	}
	return 1;
}
