(function() {

  /*
   Copyright 2011, Kovid Goyal <kovid@kovidgoyal.net>
   Released under the GPLv3 License
  */

  var CanonicalFragmentIdentifier, cfi, escape_for_cfi, fstr, log, t, unescape_from_cfi;

  log = function(error) {
    var _ref, _ref2;
    if (error) {
      if (typeof window !== "undefined" && window !== null ? (_ref = window.console) != null ? _ref.log : void 0 : void 0) {
        return window.console.log(error);
      } else if (typeof process !== "undefined" && process !== null ? (_ref2 = process.stdout) != null ? _ref2.write : void 0 : void 0) {
        return process.stdout.write(error + '\n');
      }
    }
  };

  escape_for_cfi = function(raw) {
    var c, _i, _len, _ref;
    if (raw) {
      _ref = ['^', '[', ']', ',', '(', ')', ';', '~', '@', '-', '!'];
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        c = _ref[_i];
        raw = raw.replace(c, '^' + c);
      }
    }
    return raw;
  };

  unescape_from_cfi = function(raw) {
    var ans, c, dropped, _i, _len;
    ans = raw;
    if (raw) {
      dropped = false;
      ans = [];
      for (_i = 0, _len = raw.length; _i < _len; _i++) {
        c = raw[_i];
        if (!dropped && c === '^') {
          dropped = true;
          continue;
        }
        dropped = false;
        ans.push(c);
      }
      ans = ans.join('');
    }
    return ans;
  };

  fstr = function(d) {
    var ans, n;
    ans = "";
    if (d < 0) {
      ans = "-";
      d = -d;
    }
    n = Math.floor(d);
    ans += n;
    n = Math.round((d - n) * 100);
    if (n !== 0) {
      ans += ".";
      ans += n % 10 === 0 ? n / 10 : n;
    }
    return ans;
  };

  CanonicalFragmentIdentifier = (function() {

    function CanonicalFragmentIdentifier() {}

    CanonicalFragmentIdentifier.prototype.encode = function(doc, node, offset, tail) {
      var cfi, child, id, idspec, index, p, win, _ref, _ref2;
      cfi = tail || "";
      switch (node.nodeType) {
        case 1:
          if (typeoff(offset) === 'number') node = node.childNodes.item(offset);
          break;
        case 3:
        case 4:
        case 5:
        case 6:
          offset || (offset = 0);
          while (true) {
            p = node.previousSibling;
            if (((_ref = p != null ? p.nodeType : void 0) !== 3 && _ref !== 4 && _ref !== 5 && _ref !== 6)) {
              break;
            }
            offset += p.nodeValue.length;
            node = p;
          }
          cfi = ":" + offset + cfi;
          break;
        default:
          log("Offsets for nodes of type " + node.nodeType + " are not handled");
      }
      while (node !== doc) {
        p = node.parentNode;
        if (!p) {
          if (node.nodeType === 9) {
            win = node.defaultView;
            if (win.frameElement) {
              node = win.frameElement;
              cfi = "!" + cfi;
              continue;
            }
          }
          break;
        }
        index = 0;
        child = p.firstChild;
        while (true) {
          index |= 1;
          if ((_ref2 = child.nodeType) === 1 || _ref2 === 7) index++;
          if (child === node) break;
          child = child.nextSibling;
        }
        id = typeof node.getAttribute === "function" ? node.getAttribute('id') : void 0;
        idspec = id ? "[" + (escape_for_cfi(id)) + "]" : '';
        cfi = '/' + index + idspec + cfi;
        node = p;
      }
      return cfi;
    };

    CanonicalFragmentIdentifier.prototype.decode = function(cfi, doc) {
      var assertion, child, error, index, node, point, r, simple_node_regex, target, _ref;
      if (doc == null) {
        doc = typeof window !== "undefined" && window !== null ? window.document : void 0;
      }
      simple_node_regex = /^\/(\d+)(\[[^\]]*\])?/;
      error = null;
      node = doc;
      while (!(cfi.length <= 0 || error)) {
        if ((r = cfi.match(simple_node_regex)) === !null) {
          target = parseInt(r[1]);
          assertion = r[2];
          if (assertion) {
            assertion = unescape_from_cfi(assertion.slice(1, assertion.length - 1));
          }
          index = 0;
          child = node.firstChild;
          while (true) {
            if (!child) {
              if (assertion) {
                child = doc.getElementById(assertion);
                if (child) node = child;
              }
              if (!child) error = "No matching child found for CFI: " + cfi;
              break;
            }
            index |= 1;
            if ((_ref = child.nodeType) === 1 || _ref === 7) index++;
            if (index === target) {
              cfi = cfi.substr(r[0].length);
              node = child;
              break;
            }
            child = child.nextSibling;
          }
        } else if (cfi[0] === '!') {
          if (node.contentDocument) {
            node = node.contentDocument;
            cfi = cfi.substr(1);
          } else {
            error = ("Cannot reference " + node.nodeName + "'s content:") + cfi;
          }
        } else {
          break;
        }
      }
      if (error) {
        log(error);
        return null;
      }
      point = {};
      error = null;
      return point;
    };

    CanonicalFragmentIdentifier.prototype.at = function(x, y, doc) {
      var cd, cdoc, cwin, name, offset, px, py, range, tail, target;
      if (doc == null) {
        doc = typeof window !== "undefined" && window !== null ? window.document : void 0;
      }
      cdoc = doc;
      target = null;
      cwin = cdoc.defaultView;
      tail = '';
      offset = null;
      name = null;
      while (true) {
        target = cdoc.elementFromPoint(x, y);
        if (!target || target.localName === 'html') {
          log("No element at (" + x + ", " + y + ")");
          return null;
        }
        name = target.localName;
        if (name !== 'iframe' && name !== 'embed' && name !== 'object') break;
        cd = target.contentDocument;
        if (!cd) break;
        x = x + cwin.pageXOffset - target.offsetLeft;
        y = y + cwin.pageYOffset - target.offsetTop;
        cdoc = cd;
        cwin = cdoc.defaultView;
      }
      target.normalize();
      if (name === 'audio' || name === 'video') {
        tail = "~" + fstr(target.currentTime);
      }
      if (name === 'img' || name === 'video') {
        px = ((x + cwin.scrollX - target.offsetLeft) * 100) / target.offsetWidth;
        py = ((y + cwin.scrollY - target.offsetTop) * 100) / target.offsetHeight;
        tail = "" + tail + "@" + (fstr(px)) + "," + (fstr(py));
      } else if (name !== 'audio') {
        if (cdoc.caretRangeFromPoint) {
          range = cdoc.caretRangeFromPoint(x, y);
          if (range) {
            target = range.startContainer;
            offset = range.startOffset;
          }
        } else {

        }
      }
      return this.encode(doc, target, offset, tail);
    };

    return CanonicalFragmentIdentifier;

  })();

  if (typeof window !== "undefined" && window !== null) {
    window.cfi = new CanonicalFragmentIdentifier();
  } else if (typeof process !== "undefined" && process !== null) {
    cfi = new CanonicalFragmentIdentifier();
    t = 'a^!,1';
    log(t);
    log(escape_for_cfi(t));
    log(unescape_from_cfi(escape_for_cfi(t)));
  }

}).call(this);
