# Loom, a plugin for bzr to assist in developing focused patches.
# Copyright (C) 2006 - 2008 Canonical Limited.
# Copyright (C) 2009 Aaron Bentley
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

from bzrlib import bzrdir
from bzrlib.branch import Branch
from bzrlib.commands import Command
from bzrlib.option import RegistryOption
from bzrlib.transport import get_transport
import bzrlib.urlutils
from bzrlib.plugins.pipeline.pipeline import PipeStorage

class cmd_import_loom(Command):
    """Import a loom into a pipeline."""

    hidden = True

    takes_args = ['pipeline', 'loom?']

    takes_options = [
        RegistryOption('format',
            help='Specify a format for this branch. '
            'See "help formats".',
            lazy_registry=('bzrlib.bzrdir', 'format_registry'),
            converter=lambda name: bzrdir.format_registry.make_bzrdir(name),
            value_switches=True,
            title="Branch Format",
            )]

    def run(self, pipeline, loom='.', format=None):
        root_transport = get_transport(pipeline)
        loom_branch = Branch.open(loom)
        import_threads(loom_branch, root_transport, format=format)


def import_threads(loom_branch, root_transport, format=None):
    """Import the threads in this loom into pipes.

    :param loom_branch: A LoomBranch
    :param root_transport: Transport for the directory to place branches
        under.  Defaults to branch root transport.
    """
    threads = loom_branch.get_loom_state().get_threads()
    last_branch = None
    for thread_name, thread_revision, _parents in reversed(threads):
        thread_transport = root_transport.clone(thread_name)
        user_location = bzrlib.urlutils.unescape_for_display(
            thread_transport.base, 'utf-8')
        try:
            control_dir = bzrdir.BzrDir.open(thread_transport.base,
                possible_transports=[thread_transport])
            tree, branch = control_dir._get_tree_branch()
        except bzrlib.errors.NotBranchError:
            bzrlib.trace.note('Creating branch at %s' % user_location)
            branch = bzrdir.BzrDir.create_branch_convenience(
                thread_transport.base, force_new_tree=False,
                possible_transports=[thread_transport], format=format)
            tree, branch = branch.bzrdir.open_tree_or_branch(
                thread_transport.base)
        else:
            if thread_revision == branch.last_revision():
                bzrlib.trace.note('Skipping up-to-date branch at %s'
                                  % user_location)
                continue
            else:
                bzrlib.trace.note('Updating branch at %s' % user_location)
        if tree is not None:
            tree.pull(loom_branch, stop_revision=thread_revision)
        else:
            branch.pull(loom_branch, stop_revision=thread_revision)
        if last_branch is not None:
            PipeStorage.connect(branch, last_branch)
        last_branch = branch
