/*
 * Main application class
 *
 * Copyright (C) 2003--2010  Enrico Zini <enrico@enricozini.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "BuffyWindow.h"

#include "Environment.h"

#include <gtkmm/stock.h>
#include <gtkmm/fileselection.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/toggleaction.h>
#include <gtkmm/spinbutton.h>

#include <map>

using namespace std;
using namespace buffy;


BuffyWindow::BuffyWindow()
	: updateButton("Update"),
	  prefDialog(*this)
{
	config::Config& conf = Environment::get().config();

	set_title("Buffy Folder List");
	//set_border_width(5);
	config::Section prefs = conf.application("buffy");
	prefs.addDefault("winw", "500");
	prefs.addDefault("winh", "500");

	int saved_w = prefs.getInt("winw");
	int saved_h = prefs.getInt("winh");
	
	set_default_size(saved_w, saved_h);

	add(mainVBox);

	// Create the actions used by Buffy

	actionGroup = Gtk::ActionGroup::create("Actions");
	actionGroup->add(Gtk::Action::create("FileMenu", "_File"));
	actionGroup->add(Gtk::Action::create("ViewMenu", "_View"));
	actionGroup->add(Gtk::Action::create("Quit", Gtk::Stock::QUIT, "_Quit", "Quit"),
			sigc::mem_fun(*this, &BuffyWindow::on_quit));
	
	actionGroup->add(actionPreferences = Gtk::Action::create("Preferences", "_Preferences..."),
			sigc::bind<BuffyWindow*>(
				sigc::mem_fun(prefDialog, &PrefDialog::do_dialog), this));

	actionGroup->add(actionRescanFolders = Gtk::Action::create("RescanFolders", "_Update folder list", ""),
			sigc::mem_fun(*this, &BuffyWindow::rescanFolders));
	actionGroup->add(actionViewEmpty = Gtk::ToggleAction::create("ViewAll", "View _all folders", "", conf.view().empty()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));
	actionGroup->add(actionViewRead = Gtk::ToggleAction::create("ViewNonEmpty", "View all _non empty folders", "", conf.view().read()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));
	actionGroup->add(actionViewImportant = Gtk::ToggleAction::create("ViewImportant", "View all folders with _important messages", "", conf.view().important()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));

	uimanager = Gtk::UIManager::create();
	uimanager->insert_action_group(actionGroup);
	add_accel_group(uimanager->get_accel_group());

	// Layout the actions in the main menubar
	try
	{
		Glib::ustring ui_info = 
			"<ui>"
			"  <menubar name='MenuBar'>"
			"    <menu action='FileMenu'>"
			"      <menuitem action='RescanFolders'/>"
			"      <separator/>"
			"      <menuitem action='Preferences'/>"
			"      <separator/>"
			"      <menuitem action='Quit'/>"
			"    </menu>"
			"    <menu action='ViewMenu'>"
			"      <menuitem action='ViewAll'/>"
			"      <menuitem action='ViewNonEmpty'/>"
			"      <menuitem action='ViewImportant'/>"
			"    </menu>"
			"  </menubar>"
			"</ui>";
		uimanager->add_ui_from_string(ui_info);
	}
	catch(const Glib::Error& ex)
	{
		warning("Building menus failed: %s\n", ex.what().c_str());
	}

	Gtk::Widget* menuBar = uimanager->get_widget("/MenuBar") ;
	mainVBox.pack_start(*menuBar, Gtk::PACK_SHRINK);

	folderList.configureFilter(conf);
	folderList.signal_open_folder().connect(sigc::mem_fun(*this, &BuffyWindow::on_open_folder));
	mainVBox.pack_start(folderList, Gtk::PACK_EXPAND_WIDGET);

	mainVBox.pack_start(buttonBox, Gtk::PACK_SHRINK);


	/*
	menuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("_File", fileMenu));
	fileMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::QUIT,
				sigc::mem_fun(*this, &BuffyWindow::on_quit)));
	*/

	/*
	menuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("_View", viewMenu));
	viewMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::QUIT,
				sigc::mem_fun(*this, &BuffyWindow::on_quit)));
*/

	buttonBox.pack_start(updateButton, Gtk::PACK_SHRINK);
	updateButton.signal_clicked().connect(
			sigc::mem_fun(folderList, &FolderList::do_update));

	set_timer(conf.general().interval());

	Environment::get().signal_child_exit().connect(sigc::mem_fun(*this, &BuffyWindow::on_program_exited));

	show_all_children();
}

void BuffyWindow::rescanFolders()
{
	folderList.clear();
	Environment::get().enumerateFolders(folderList);
}

void BuffyWindow::on_program_exited(int pid, int status)
{
	debug("Notified of exit of pid %d, status %d\n", pid, status);
	folderList.do_update();
}

bool BuffyWindow::on_configure_event(GdkEventConfigure* c)
{
	bool res = Gtk::Window::on_configure_event(c);
	int x, y;
	Gtk::Window::get_position(x, y);

	config::Config& conf = Environment::get().config();
	config::Section prefs = conf.application("buffy");

	//conf.set_application_data("buffy", "winx", fmt(c->x));
	//conf.set_application_data("buffy", "winy", fmt(c->y));
	prefs.setInt("winx", x);
	prefs.setInt("winy", y);
	prefs.setInt("winw", c->width);
	prefs.setInt("winh", c->height);

	return res;
}

void BuffyWindow::on_show()
{
	Gtk::Window::on_show();

	config::Config& conf = Environment::get().config();
	config::Section prefs = conf.application("buffy");
	if (prefs.isSet("winx") && prefs.isSet("winy"))
	{
		string saved_x = prefs.get("winx");
		string saved_y = prefs.get("winy");
		move(prefs.getInt("winx"), prefs.getInt("winy"));
	}

	if (folderList.size() == 0)
		rescanFolders();
}

void BuffyWindow::on_open_folder(MailFolder folder)
{
	debug("OOF activated %s\n", folder.path().c_str());

	config::MailProgram m = Environment::get().config().selectedMailProgram();
	verbose("Running %s\n", m.command("gui").c_str());
	m.run(folder, "gui");

	/*
	//MailProgram m("Mutt", "x-terminal-emulator -e mutt -f %p");
	if (m)
	{
	}
	else
		warning("No mail programs are currently selected.  This is probably due to hand-editing the configuration file and forgetting to add selected=\"true\" to one mail program.\n");
	*/
}

void BuffyWindow::on_quit()
{
	hide();
}

void BuffyWindow::on_filter_changed()
{
	config::Config& conf = Environment::get().config();

	bool empty = actionViewEmpty->get_active();
	bool read = actionViewRead->get_active();
	bool important = actionViewImportant->get_active();
	verbose("Filter changed %s %s %s\n",
			empty ? "empty" : "",
			read ? "read" : "",
			important ? "important" : "");

	conf.view().setEmpty(empty);
	conf.view().setRead(read);
	conf.view().setImportant(important);

	folderList.configureFilter(conf);
}

void BuffyWindow::set_timer(int interval)
{
	if (timer_connection.connected())
		timer_connection.disconnect();

	// interval is given in milliseconds
	timer_connection = Glib::signal_timeout().connect(
			sigc::mem_fun(folderList, &FolderList::do_timer_tick),
			interval*1000);
}

// vim:set ts=4 sw=4:
