/***************************************************************************
                   xbelwrite.cpp  -  XbelWrite Implementation
                             -------------------
    begin                : Sun Sep 15 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include <QFile>

#include "xbelwrite.h"
#include "xmlparser.h"

/***************************************************************************
 *                                                                         *
 *   Function Prototypes                                                   *
 *                                                                         *
 ***************************************************************************/

static	void	Indent(QTextStream &t, int spaces);
static	void	WriteBookmark(QTextStream &t, const BkBookmark &bookmark, int level, BridgeCfg &cfg)
					throw(BkException);
static	void	WriteBrowserOrds(QTextStream &t, const BrowserOrdLst &lst, int level, BridgeCfg &cfg);
static	void	WriteDocumentX(QTextStream &t, const BkFolder &folder, int level, BridgeCfg &cfg)
					throw(BkException);

/***************************************************************************
 *                                                                         *
 *   Indent()                                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextstream &t                                                     *
 *      int spaces                                                         *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function writes the requested number of spaces, given in the  *
 *      second parameter to the file given in the first parameter.  It is  *
 *      used for indention in the XML file.                                *
 *                                                                         *
 ***************************************************************************/

static void Indent(QTextStream &t, int spaces)
{
	for(int i=0; i<spaces; i++)
		t << ' ';
}

/***************************************************************************
 *                                                                         *
 *   writeBookmark()                                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &t                                                     *
 *      const BkBookmark &bookmark                                         *
 *      int level                                                          *
 *      BridgeCfg &cfg                                                     *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function saves a bookmark, given in the first parameter to    *
 *      file given in the first parameter.  The bookmark elements,         *
 *      including children elements are indented using the third parameter.*
 *                                                                         *
 ***************************************************************************/

static void WriteBookmark(
	QTextStream &t, const BkBookmark &bookmark, int level, BridgeCfg &cfg)
	throw(BkException)
{
	auto	QString	tempstr;

	Indent(t, level);
	EncodeString(bookmark.url(), tempstr);
	t
		<< "<bookmark href=\""
		<< tempstr
		<< "\">"
		<< endl;

	if(bookmark.isFieldValid(BKVALID_TITLE))
	{
		EncodeString(bookmark.title(), tempstr);
		Indent(t, level+1);
		t
			<< "<title>"
			<< tempstr
			<< "</title>"
			<< endl;
	}

	if(bookmark.isFieldValid(BKVALID_DESC))
	{
		EncodeString(bookmark.desc(), tempstr);
		Indent(t, level+1);
		t
			<< "<desc>"
			<< tempstr
			<< "</desc>"
			<< endl;
	}

	Indent(t, level+1);
	t << "<info>" << endl;
	Indent(t, level+2);
	t
		<< "<metadata owner=\""
		<< BRIDGE_URI
		<< "\">"
		<< endl;
	WriteBrowserOrds(t, bookmark.browserFound(), level+3, cfg);
	if(bookmark.ignore())
	{
		Indent(t, level+3);
		t << "<ignore/>" << endl;
	}
	Indent(t, level+2);
	t << "</metadata>" << endl;
	Indent(t, level+1);
	t << "</info>" << endl;

	Indent(t, level);
	t << "</bookmark>" << endl;
}

/***************************************************************************
 *                                                                         *
 *   WriteBrowserOrds()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &t                                                     *
 *      const BrowserOrdLst &nodeLst                                       *
 *      int level                                                          *
 *      BridgeCfg &cfg                                                     *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function writes a list of browsers, specified in the second   *
 *      argument to the XBEL file specified in the first argument.  This   *
 *      list is in the form of <browser></browser> tags.                   *
 *                                                                         *
 ***************************************************************************/

static void WriteBrowserOrds(
	QTextStream &t, const BrowserOrdLst &nodeLst, int level, BridgeCfg &cfg)
{
	BrowserOrdLst::const_iterator it;
	BrowserOrdLst lst(nodeLst);
	extern	bool	gNoDates;

	if(gNoDates)
		std::sort(lst.begin(), lst.end());
	
	std::vector<BrowserCfg>::const_iterator bit;

	for(it=lst.begin(); it!=lst.end(); ++it)
	{
		bit = std::find(cfg.browsers.begin(), cfg.browsers.end(), *it);
		if(bit == cfg.browsers.end())
			continue;

		Indent(t, level);
		t
			<< "<browser num=\""
			<< bit->browserNum()
			<< "\">"
			<< *it
			<< "</browser>"
			<< endl;
	}
}

/***************************************************************************
 *                                                                         *
 *   WriteDate()                                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &t                                                     *
 *      time_t t                                                           *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function writes the date and time, given in the second        *
 *      parameter to the file given in the first parameter.                *
 *                                                                         *
 ***************************************************************************/

void WriteDate(QTextStream &stream, time_t t)
{
	struct tm *tm;

	tm = gmtime(&t);

	QString	temp;
	temp.sprintf("%d-%02d-%02dT%02d:%02d:%02dZ",
		tm->tm_year+1900,
		tm->tm_mon+1,
		tm->tm_mday,
		tm->tm_hour,
		tm->tm_min,
		tm->tm_sec);

	stream << temp;
}

/***************************************************************************
 *                                                                         *
 *   WriteXbelDocument()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      const char *filename                                               *
 *      const BkFolder &root                                               *
 *      BridgeCfg &cfg                                                     *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function saves the bookmark tree, specified in the first      *
 *      parameter to the file specified in the second parameter.           *
 *                                                                         *
 ***************************************************************************/

void WriteXbelDocument(const QString &filename, const BkFolder &Root, BridgeCfg &cfg)
		throw(BkException)
{
	QFile	out;

	// open the file for writing

	out.setFileName(filename);
	if(out.open(QIODevice::WriteOnly | QIODevice::Text) == false)
	{
		QString msg;

		msg = "Unable to Open \"" + filename + "\" for Writing";
		BKEXCEPT(msg);
	}

	QTextStream t(&out);
	
	t.setCodec("UTF-8");

	t << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
	t << "<xbel folded=\"no\">" << endl;

	WriteDocumentX(t, Root, 1, cfg);
	t << "</xbel>" << endl;
}

/***************************************************************************
 *                                                                         *
 *   WriteDocumentX()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &t                                                     *
 *      const BkFolder &folder                                             *
 *      int level                                                          *
 *      BridgeCfg &cfg                                                     *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function saves the folder, given in the second parameter to   *
 *      the file given in the first parameter.  It is indented using the   *
 *      third parameter.                                                   *
 *                                                                         *
 ***************************************************************************/

static void WriteDocumentX(
	QTextStream &t, const BkFolder &folder, int level, BridgeCfg &cfg)
	throw(BkException)
{
	auto	BookmarkLst::const_iterator bit;
	auto	FolderLst::const_iterator fit;

	for(bit = folder.BookmarksBegin();
		bit != folder.BookmarksEnd();
		++bit)
    {
		if(bit->isDeleted())
			continue;

		WriteBookmark(t, *bit, level+1, cfg);
	}

	for(fit = folder.FoldersBegin();
		fit != folder.FoldersEnd();
		++fit)
    {
		if(fit->isDeleted())
			continue;

		Indent(t, level+1);
		t << "<folder folded=\"no\">" << endl;

		if(fit->isFieldValid(BKVALID_TITLE))
		{
			QString tempstr;

			EncodeString(fit->title(), tempstr);
			Indent(t, level+2);
			t
				<< "<title>"
				<< tempstr
				<< "</title>"
				<< endl;
		}

		if(fit->isFieldValid(BKVALID_DESC))
		{
			QString tempstr;

			EncodeString(fit->desc(), tempstr);
			Indent(t, level+2);
			t
				<< "<desc>"
				<< tempstr
				<< "</desc>"
				<< endl;
		}

		Indent(t, level+2);
		t << "<info>" << endl;
		Indent(t, level+3);
		t
			<< "<metadata owner=\""
			<< BRIDGE_URI
			<< "\">"
			<< endl;
		WriteBrowserOrds(t, fit->browserFound(), level+4, cfg);
		if(fit->ignore())
		{
			Indent(t, level+4);
			t << "<ignore/>" << endl;
		}
		Indent(t, level+3);
		t << "</metadata>" << endl;
		Indent(t, level+2);
		t << "</info>" << endl;

		WriteDocumentX(t, *fit, level+1, cfg);

		Indent(t, level+1);
		t << "</folder>" << endl;
	}
}
