/***************************************************************************
                xbelread.cpp  -  XbelRead Class Implementation
                             -------------------
    begin                : Sun Sep 15 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include <ctype.h>
#include <stdlib.h>
#include <stack>

#include "xmlparser.h"
#include "xbelread.h"
#include "browserlist.h"

/***************************************************************************
 *                                                                         *
 *   Constants                                                             *
 *                                                                         *
 ***************************************************************************/

#define SAXSTATE_XBEL			(SAXSTATE_USER+1)
#define SAXSTATE_FOLDER			(SAXSTATE_USER+2)
#define SAXSTATE_BOOKMARK		(SAXSTATE_USER+3)
#define SAXSTATE_TITLE			(SAXSTATE_USER+4)
#define SAXSTATE_INFO			(SAXSTATE_USER+5)
#define SAXSTATE_METADATA		(SAXSTATE_USER+6)
#define SAXSTATE_BRIDGESAVED	(SAXSTATE_USER+7)
#define SAXSTATE_BROWSER		(SAXSTATE_USER+8)
#define SAXSTATE_IGNORE			(SAXSTATE_USER+9)
#define SAXSTATE_DESC			(SAXSTATE_USER+10)

/***************************************************************************
 *                                                                         *
 *   XML Parser Call-back Functions                                        *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartXbel(void *, const xmlChar *, const xmlChar **, short);
static bool xmlStartBookmark(void *, const xmlChar *, const xmlChar **, short);
static bool xmlStartFolder(void *, const xmlChar *, const xmlChar **, short);
static bool xmlStartMetadata(void *, const xmlChar *, const xmlChar **, short);
static void xmlEndBrowser(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndFolder(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndBookmark(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndIgnore(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndTitle(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndDesc(void *, const xmlChar *, const xmlChar *, short, short);

static ELEMHANDLER xmlHandlers[] = {
	{
		SAXSTATE_XBEL,
		"xbel",
		xmlStartXbel,
		NULL,
	},{
		SAXSTATE_FOLDER,
		"folder",
		xmlStartFolder,
		xmlEndFolder,
	},{
		SAXSTATE_BOOKMARK,
		"bookmark",
		xmlStartBookmark,
		xmlEndBookmark,
	},{
		SAXSTATE_TITLE,
		"title",
		NULL,
		xmlEndTitle,
	},{
		SAXSTATE_INFO,
		"info",
		NULL,
		NULL,
	},{
		SAXSTATE_METADATA,
		"metadata",
		xmlStartMetadata,
		NULL,
	},{
		SAXSTATE_BROWSER,
		"browser",
		NULL,
		xmlEndBrowser,
	},{
		SAXSTATE_IGNORE,
		"ignore",
		NULL,
		xmlEndIgnore,
	},{
		SAXSTATE_DESC,
		"desc",
		NULL,
		xmlEndDesc,
	},
	// this element marks the end of the list
	{ 0, NULL, NULL, NULL }
};

/***************************************************************************
 *                                                                         *
 *   Structure Definitions                                                 *
 *                                                                         *
 ***************************************************************************/

typedef struct {
	std::stack<BkFolder *>	*folders;
	BkBookmark	*bookmark;
} SAXDATA;

/***************************************************************************
 *                                                                         *
 *   Function Prototypes                                                   *
 *                                                                         *
 ***************************************************************************/

static	void	xmlStartDocument(void *);

/***************************************************************************
 *                                                                         *
 *   xmlStartDocument()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called by the SAX2 parsing engine before any      *
 *      others.  It uses the opportunity to initialize state data.  It     *
 *      also pushes the current state on to a state stack.  This stack is  *
 *      pushed when a new XML element is encountered and popped at the end *
 *      of the element.  This allows the callback functions to always      *
 *      know where they are in the XML document.                           *
 *                                                                         *
 ***************************************************************************/

static void xmlStartDocument(void *ctx)
{
	SAXDATA *data = (SAXDATA *)ctx;

	data->folders->push(new BkFolder);
	data->bookmark = NULL;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartXbel()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *              (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar **    (not used)                                     *
 *      short prevState                                                    *
 *   Return:                                                               *
 *      true if there was an error, false otherwise                        *
 *   Description:                                                          *
 *      This function is called when the <xbel> element is encountered.    *
 *      All it really does is verify the <xbel> element is occuring in the *
 *      correct part of the XML document.                                  *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartXbel(void *, const xmlChar *, const xmlChar **, short prevState)
{
	if(prevState != SAXSTATE_STARTEND)
		return true;

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartBookmark()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar **atts                                               *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      true if there was an error, false otherwise                        *
 *   Description:                                                          *
 *      This function is called when the <bookmark> element is             *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartBookmark(void *ctx, const xmlChar *, const xmlChar **atts, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	BkBookmark bookmark;

	data->bookmark = &data->folders->top()->addBookmark(bookmark);

	// save some attributes in the element

	for(int i=0; atts[i]; i+=2)
	{
		if(!xmlStrcmp(atts[i], (const xmlChar *)"href"))
		{
			data->bookmark->setUrl(QString::fromUtf8((const char *)atts[i+1]));
			data->bookmark->setValidField(BKVALID_URL);
		}
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartFolder()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar **    (not used)                                     *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      true if there was an error, false otherwise                        *
 *   Description:                                                          *
 *      This function is called when the <folder> element is encountered.  *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartFolder(void *ctx, const xmlChar *, const xmlChar **, short)
{
	SAXDATA	*data = (SAXDATA *)ctx;

	BkFolder folder, *child;
	child = &data->folders->top()->addFolder(folder);
	data->folders->push(child);

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartMetadata()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *              (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar **atts                                               *
 *      short prevState                                                    *
 *   Return:                                                               *
 *      true if there was an error, false otherwise                        *
 *   Description:                                                          *
 *      This function is called when the <metadata> element is             *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartMetadata(void *, const xmlChar *, const xmlChar **atts, short prevState)
{
	int	i;

	if(prevState != SAXSTATE_INFO)
		return true;

	// verify the owner of the meta data

	for(i=0; atts[i]; i+=2)
	{
		if(!xmlStrcmp(atts[i], (const xmlChar *)"owner"))
			if(!xmlStrcmp(atts[i+1], (const xmlChar *)BRIDGE_URI))
				break;
	}

	if(atts[i])
		return false;
	else
		return true;
}

/***************************************************************************
 *                                                                         *
 *   xmlEndBrowser()                                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar *pChars                                              *
 *      short state                                                        *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <browsers> element is  *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndBrowser(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short)
{
	Q_ASSERT(state == SAXSTATE_BROWSER);

	SAXDATA	*data = (SAXDATA *)ctx;
	BkNode	*pNode;

	if(data->bookmark)
		pNode = data->bookmark;
	else
		pNode = data->folders->top();
	Q_ASSERT(pNode != NULL);

	pNode->setBrowserSaved(atol((const char *)pChars));
}

/***************************************************************************
 *                                                                         *
 *   xmlSAXendFolder()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      short state                                                        *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <folder> element is    *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndFolder(void *ctx, const xmlChar *, const xmlChar *, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	Q_ASSERT(state == SAXSTATE_FOLDER);
	Q_ASSERT(data->folders->size() > 0);

	data->folders->pop();
}

/***************************************************************************
 *                                                                         *
 *   xmlEndBookmark()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      short state                                                        *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <bookmark> element is  *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndBookmark(void *ctx, const xmlChar *, const xmlChar *, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	Q_ASSERT(state == SAXSTATE_BOOKMARK);

	data->bookmark = NULL;
}

/***************************************************************************
 *                                                                         *
 *   xmlEndIgnore()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      short state                                                        *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <ignore/> element is   *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndIgnore(void *ctx, const xmlChar *, const xmlChar *, short state, short)
{
	Q_ASSERT(state == SAXSTATE_IGNORE);

	SAXDATA	*data = (SAXDATA *)ctx;
	BkNode	*pNode;

	if(data->bookmark)
		pNode = data->bookmark;
	else
		pNode = data->folders->top();
	Q_ASSERT(pNode != NULL);

	pNode->setIgnore(true);
}

/***************************************************************************
 *                                                                         *
 *   xmlEndTitle()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const char *pChars                                                 *
 *      short               (not used)                                     *
 *      short prevState                                                    *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <title> element is     *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndTitle(void *ctx, const xmlChar *, const xmlChar *pChars, short, short parentState)
{
	SAXDATA *data = (SAXDATA *)ctx;

	if(pChars)
	{
		if(parentState == SAXSTATE_FOLDER)
		{
			Q_ASSERT(data->folders->size() > 0);
			data->folders->top()->setTitle(
				QString::fromUtf8((const char *)pChars));
			data->folders->top()->setValidField(BKVALID_TITLE);
		}
		else if(parentState == SAXSTATE_BOOKMARK)
		{
			Q_ASSERT(data->bookmark != NULL);
			data->bookmark->setTitle(
				QString::fromUtf8((const char *)pChars));
			data->bookmark->setValidField(BKVALID_TITLE);
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   xmlEndDesc()                                                          *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const char *pChars                                                 *
 *      short               (not used)                                     *
 *      short prevState                                                    *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <desc> element is      *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndDesc(void *ctx, const xmlChar *, const xmlChar *pChars, short, short parentState)
{
	SAXDATA *data = (SAXDATA *)ctx;

	if(pChars)
	{
		if(parentState == SAXSTATE_FOLDER)
		{
			Q_ASSERT(data->folders->size() > 0);
			data->folders->top()->setDesc(
				QString::fromUtf8((const char *)pChars));
			data->folders->top()->setValidField(BKVALID_DESC);
		}
		else if(parentState == SAXSTATE_BOOKMARK)
		{
			Q_ASSERT(data->bookmark != NULL);
			data->bookmark->setDesc(
				QString::fromUtf8((const char *)pChars));
			data->bookmark->setValidField(BKVALID_DESC);
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   ParseXbelDocument()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      const char *filename                                               *
 *      BkFolder **root                                                    *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function begins the parsing process for an XML file.  The     *
 *      first parameter is the full path and filename of the file.  It     *
 *      allocates the state data (SAXDATA) needed by the callback          *
 *      functions and sets up the handler structure given to libxml        *
 *      containing pointers to each of the callback functions.             *
 *                                                                         *
 ***************************************************************************/

void ParseXbelDocument(const QString &filename, BkFolder **root) throw(BkException)
{
	SAXDATA		data;

	data.folders = new std::stack<BkFolder *>;

	// parse the file

	ParseXmlDocument(
		filename,
		&data,
		xmlHandlers,
		xmlStartDocument,
		NULL);

	if(data.folders->size() != 1)
		BKEXCEPT("XML Parser has Inconsistent Internal State");
	*root = data.folders->top();
	data.folders->pop();

	delete data.folders;
}
