/**
 * Copyright Copr. INRIA/INRA
 * Contact :  biomaj_AT_genouest.org
 * 
 * BioMAJ is a workflow engine dedicated to biological bank management. 
 * The Software automates the update cycle and the supervision of the locally 
 * mirrored bank repository. The project is a collaborative effort between two 
 * French Research Institutes INRIA (Institut National de Recherche en
 * Informatique 
 * et en Automatique) & INRA (Institut National de la Recherche Agronomique).
 *
 * Inter Deposit Digital Number : IDDN ...................................
 *
 * This software is governed by the CeCILL-A license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info". 
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability. 
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or 
 * data to be ensured and,  more generally, to use and operate it in the 
 * same conditions as regards security. 
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-A license and that you accept its terms.
 */


package org.inria.biomaj.exe.main;

import java.io.File;
import java.io.FileInputStream;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.cli2.CommandLine;
import org.apache.commons.cli2.DisplaySetting;
import org.apache.commons.cli2.Group;
import org.apache.commons.cli2.Option;
import org.apache.commons.cli2.builder.ArgumentBuilder;
import org.apache.commons.cli2.builder.DefaultOptionBuilder;
import org.apache.commons.cli2.builder.GroupBuilder;
import org.apache.commons.cli2.commandline.Parser;
import org.apache.commons.cli2.util.HelpFormatter;
import org.apache.commons.cli2.validation.EnumValidator;
import org.apache.tools.ant.Project;
import org.inria.biomaj.exe.bank.BankFactory;
import org.inria.biomaj.exe.bank.BiomajBank;
import org.inria.biomaj.exe.options.BioMAJLogger;
import org.inria.biomaj.exe.options.BiomajCleanDB;
import org.inria.biomaj.exe.options.BiomajImport;
import org.inria.biomaj.exe.options.BiomajMoveProductionDirectories;
import org.inria.biomaj.exe.options.BiomajRemoveBank;
import org.inria.biomaj.exe.options.ChangeDbName;
import org.inria.biomaj.exe.options.IndexXmlDescription;
import org.inria.biomaj.exe.options.StatusInformation;
import org.inria.biomaj.exe.workflow.WorkflowEngine;
import org.inria.biomaj.exe.workflow.WorkflowEngineFactory;
import org.inria.biomaj.singleton.BiomajInformation;
import org.inria.biomaj.singleton.BiomajLogger;
import org.inria.biomaj.utils.BiomajBuildException;
import org.inria.biomaj.utils.BiomajConst;
import org.inria.biomaj.utils.BiomajException;

/**
 * 
 * Biomaj Main / Used Commons Jakarta CLI2 lib to handle Biomaj options.  
 * @author ofilangi
 * @version Biomaj 0.9
 * @since Biomaj 0.9.2 
 */
public class Biomaj {
	/**
	 * Current version of biomaj
	 */
	public static final String VERSION = "1.1.0";

	/**
	 * Name of Environment variable to get the root application
	 */

	public static final String ENV_BIOMAJ = "BIOMAJ_ROOT";

	/**
	 * Builder for general option
	 */
	final static DefaultOptionBuilder obuilder = new DefaultOptionBuilder();

	/**
	 * Argument builder
	 */
	final static ArgumentBuilder abuilder = new ArgumentBuilder();

	/**
	 * Group builder
	 */
	final static GroupBuilder gbuilder = new GroupBuilder();

	/**
	 * Sub option builder
	 */
	final static DefaultOptionBuilder subobuilder = new DefaultOptionBuilder();

	/**
	 * Group Builder to construc dependances
	 */
	final static GroupBuilder childbuilder = new GroupBuilder();

	/**
	 * SubOption Definition 
	 */
	private static Option optionStage  ;
	private static Option optionDebug  ;
	private static Option optionVerbose;
	private static Option optionCheckCurrentProcess  ;
	private static Option optionConsole  ;
	private static Option optionNew  ;
	private static Option optionFromScratch  ;

	private static Option optionKeepDir  ;
	private static Option optionPaths;
	private static Option optionAllDirs;
	private static Option optionStatusFilterType;
	private static Option optionStatusOnline;
	private static Option optionStatusUpdating;
	
	private static Option processRebuild;
	private static Option metaRebuild;
	private static Option blockRebuild;

	public static void main (String args []) {
		
		createOptionBase();
//		------------------------------------------------------------------------------------
		//Option pour Import
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("import");
		obuilder.withShortName("I");
		obuilder.withDescription("Populate database for a bank according to its properties file and production directory. " +
		"Data tree directory must be in compliance with the bank properties files and biomaj tree directory organisation\n");

//		bank argument
		abuilder.reset();
		abuilder.withName("bank");
		abuilder.withDescription("bank name");
		abuilder.withMinimum(1);
		abuilder.withMaximum(1);
		obuilder.withArgument(abuilder.create());

		Option optionImport = obuilder.create();
		
//		------------------------------------------------------------------------------------
		//Option pour Index
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("index");
		obuilder.withShortName("i");
		obuilder.withDescription("Generate an index of all release include in repository\nby default index filename is located in : ${statefiles.dir}/index.xml\n");
		obuilder.withRequired(false);
//		bank argument
		abuilder.reset();
		abuilder.withName("file");
		abuilder.withDescription("Xml name file (saved in statefile directory)");
		abuilder.withMinimum(0);
		abuilder.withMaximum(1);
		obuilder.withArgument(abuilder.create());

		Option optionIndex = obuilder.create();

		
//		------------------------------------------------------------------------------------
//		------------------------------------------------------------------------------------
		//Option pour Change-dbname
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("change-dbname");
		obuilder.withDescription("Change the name of the workflow on the repository\n ");
		obuilder.withRequired(false);
//		bank argument
		abuilder.reset();
		abuilder.withName("dbname");
		abuilder.withDescription("");
		abuilder.withMinimum(2);
		abuilder.withMaximum(2);
		obuilder.withArgument(abuilder.create());
		//abuilder.withName("newDbname");
		//obuilder.withArgument(abuilder.create());
		

		Option optionChangeDbName = obuilder.create();

//		------------------------------------------------------------------------------------

//		------------------------------------------------------------------------------------
		//Option pour Rebuild
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("rebuild");
		obuilder.withDescription("Rebuild bank named dbname (downgrade the current version checks data and rerun post-processing)\n");
		obuilder.withRequired(false);

//		bank argument
		abuilder.reset();
		abuilder.withName("dbname");
		abuilder.withDescription("bank name");
		abuilder.withMinimum(1);
		abuilder.withMaximum(1);
		obuilder.withArgument(abuilder.create());
	
		abuilder.reset();
		abuilder.withName("name");
		//abuilder.withDescription();
		abuilder.withMinimum(1);
		abuilder.withSubsequentSeparator(',');
		
		//Argument a = ;
		
		subobuilder.reset();
		subobuilder.withLongName("process");
		subobuilder.withArgument(abuilder.create());
		subobuilder.withRequired(false);
		subobuilder.withDescription("Force the execution of process");
		processRebuild = subobuilder.create();
		
		//Option processRebuild = 

		subobuilder.reset();
		subobuilder.withLongName("meta");
		subobuilder.withArgument(abuilder.create());
		subobuilder.withRequired(false);
		subobuilder.withDescription("Force the execution of metaprocess");
		metaRebuild = subobuilder.create();
		
		subobuilder.reset();
		subobuilder.withLongName("block");
		subobuilder.withArgument(abuilder.create());
		subobuilder.withRequired(false);
		subobuilder.withDescription("Force the execution of block");
		blockRebuild = subobuilder.create();
		
		childbuilder.reset();
		childbuilder.withOption(processRebuild);
		childbuilder.withOption(metaRebuild);
		childbuilder.withOption(blockRebuild);
		childbuilder.withOption(optionConsole);
		childbuilder.withOption(optionVerbose);
		childbuilder.withOption(optionDebug);

		obuilder.withChildren(childbuilder.create());

		Option optionRebuild = obuilder.create();
//		------------------------------------------------------------------------------------

//		------------------------------------------------------------------------------------
		//Option pour Version
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("version");
		obuilder.withDescription("Version information");
		obuilder.withRequired(false);

		Option optionVersion = obuilder.create();
//		------------------------------------------------------------------------------------
//		------------------------------------------------------------------------------------
		//Option pour Clean
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("clean-database");
		obuilder.withDescription("Delete files that no longer exist from database.\n");
		obuilder.withRequired(false);

		Option optionCleanDb = obuilder.create();
//		------------------------------------------------------------------------------------
		//Option pour ViewLogger
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("view-log");
		obuilder.withDescription("view logs\n");
		obuilder.withRequired(false);

		Option optionViewLog = obuilder.create();
//		------------------------------------------------------------------------------------
		//Option pour Help
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("help");
		obuilder.withDescription("Display current help message");
		obuilder.withRequired(false);

		Option optionHelp = obuilder.create();
//		------------------------------------------------------------------------------------
		//Option pour Move-production-directories
		//------------------
		obuilder.reset(); 
		obuilder.withLongName("move-production-directories");
		obuilder.withDescription("Export all production directories to a new version directory construct by the properties data.dir and version.dir.\n");
		obuilder.withRequired(false);

		abuilder.reset();
		abuilder.withName("dbname");
		abuilder.withDescription("bank name");
		abuilder.withMinimum(1);
		abuilder.withMaximum(1);
		obuilder.withArgument(abuilder.create());
		
		Option optionMoveProductionDirectories = obuilder.create();
//		------------------------------------------------------------------------------------
		
		gbuilder.reset();
		gbuilder.withName("MAIN OPTIONS:");
		gbuilder.withMinimum(1);
		gbuilder.withMaximum(1);
		gbuilder.withDescription("General descriptions....");
		Option optionUpdate = createOptionUpdate();gbuilder.withOption(optionUpdate);
		Option optionStatus = createOptionStatus();gbuilder.withOption(optionStatus);
		gbuilder.withOption(optionRebuild);
		Option optionRemove = createOptionRemove();gbuilder.withOption(optionRemove);
		gbuilder.withOption(optionImport);
		gbuilder.withOption(optionMoveProductionDirectories);
		gbuilder.withOption(optionIndex);
		gbuilder.withOption(optionChangeDbName);
		gbuilder.withOption(optionCleanDb);
		//Pour l instant on ne traite pas cette option qui n est pas implement completement....
		gbuilder.withOption(optionViewLog);
		gbuilder.withOption(optionVersion);
		gbuilder.withOption(optionHelp);

		gbuilder.withMinimum(1);
		gbuilder.withMaximum(1);

		Group mutuallyExclusiveOption = gbuilder.create();

		HelpFormatter hf = new HelpFormatter(" "," "," ",150);
		hf.setShellCommand("biomaj");
		hf.setGroup(mutuallyExclusiveOption);
		
		hf.getFullUsageSettings().remove(DisplaySetting.DISPLAY_GROUP_NAME);
		hf.getFullUsageSettings().remove(DisplaySetting.DISPLAY_GROUP_ARGUMENT);
		hf.getFullUsageSettings().add(DisplaySetting.DISPLAY_GROUP_EXPANDED);
		
		hf.getLineUsageSettings().remove(DisplaySetting.DISPLAY_PROPERTY_OPTION);
		hf.getLineUsageSettings().add(DisplaySetting.DISPLAY_PARENT_ARGUMENT);
		hf.getLineUsageSettings().add(DisplaySetting.DISPLAY_ARGUMENT_BRACKETED);

		hf.getLineUsageSettings().remove(DisplaySetting.DISPLAY_GROUP_NAME);
		//hf.getFullUsageSettings().remove(DisplaySetting.DISPLAY_GROUP_EXPANDED);
		//hf.getLineUsageSettings().remove(DisplaySetting.DISPLAY_GROUP_ARGUMENT);

		hf.setDisplaySettings(HelpFormatter.DEFAULT_DISPLAY_USAGE_SETTINGS);
		hf.setDivider(HelpFormatter.DEFAULT_GUTTER_RIGHT);

		hf.setFooter("Biomaj Software is a workflows motor engine dedicated to biological bank managenemt."+
				"This software is governed by the CeCILL license under French law and abiding by " +
				"the rules of distribution of free software.  You can  use, modify and/ or redistribute " +
				"the software under the terms of the CeCILL license as circulated by CEA, CNRS and INRIA at " +
				"the following URL \"http://www.cecill.info\"."+"\nCopyright or Copr. INRIA/INRA\n");


		//hf.getFullUsageSettings().remove(DisplaySetting.DISPLAY_GROUP_);
		// parse the command line arguments
		Parser parser = new Parser();
		parser.setGroup(mutuallyExclusiveOption);
		parser.setHelpFormatter(hf);
		parser.setHelpTrigger("--help");

		CommandLine cl = parser.parseAndHelp(args);

		if (cl==null) {
			System.exit(-2);
		}


		//Test de la config general !
		try {
			BiomajInformation.getInstance();
		} catch (Exception e) {
			System.err.println(e.getLocalizedMessage());
			System.exit(-3);
		}
		
//		BiomajSQLQuerier.initConnection(true);

		try {
			/**********************
			 * UPDATE AND REBUILD
			 *********************/
			if (cl.hasOption(optionUpdate)||cl.hasOption(optionRebuild)) {
				String[] listP = null, listM= null,listB = null;
				List<String> banks ;
				int command = WorkflowEngine.TARGET_ALL;
				if (cl.hasOption(optionUpdate))
					banks = cl.getValues(optionUpdate);
				else
				{
					command =  WorkflowEngine.TARGET_REBUILD;
					banks = cl.getValues(optionRebuild);
					
					if (cl.hasOption(processRebuild)||cl.hasOption(metaRebuild)||cl.hasOption(blockRebuild)) {
						
						listP = new String[cl.getValues(processRebuild).size()];

						for (int i=0;i<cl.getValues(processRebuild).size();i++)
							listP[i] = (String)cl.getValues(processRebuild).get(i);

						listM = new String[cl.getValues(metaRebuild).size()];

						for (int i=0;i<cl.getValues(metaRebuild).size();i++)
							listM[i] = (String)cl.getValues(metaRebuild).get(i);

						listB = new String[cl.getValues(blockRebuild).size()];

						for (int i=0;i<cl.getValues(blockRebuild).size();i++)
							listB[i] = (String)cl.getValues(blockRebuild).get(i);
					}
				}

				if (cl.hasOption(optionStage)) {
					command = getCommand((String)cl.getValue(optionStage));
				}
				int mode = Project.MSG_INFO;
				if (cl.hasOption(optionVerbose)) {
					mode = Project.MSG_VERBOSE;
				}
				if (cl.hasOption(optionDebug)) {
					mode = Project.MSG_DEBUG;
				}

				processBanks(banks,command,cl.hasOption(optionConsole),mode,cl.hasOption(optionCheckCurrentProcess),cl.hasOption(optionNew),cl.hasOption(optionFromScratch),listP,listM,listB);
			} 
			/*********************
			 * REMOVE
			 *********************/
			else if (cl.hasOption(optionRemove)) {
				BiomajRemoveBank brb = new BiomajRemoveBank();
				int mode = Project.MSG_INFO;
				if (cl.hasOption(optionVerbose)) {
					mode = Project.MSG_VERBOSE;
				}
				if (cl.hasOption(optionDebug)) {
					mode = Project.MSG_DEBUG;
				}
				brb.setShowConsole(cl.hasOption(optionConsole), mode);
				brb.setBankName((String)cl.getValue(optionRemove));
				brb.setKeepProductionDirectory(cl.hasOption(optionKeepDir));
				if (cl.hasOption(optionAllDirs))
					brb.executeWithoutUserInteraction(null, true);
				else if (cl.hasOption(optionPaths))
					brb.executeWithoutUserInteraction(cl.getValues(optionPaths), false);
				else
					brb.execute();
			}
			/**********************
			 * STATUS
			 *********************/
			else if (cl.hasOption(optionStatus)) {


				if ((cl.getValue(optionStatus)==null)||(cl.hasOption(optionStatusFilterType)||cl.hasOption(optionStatusUpdating)||cl.hasOption(optionStatusOnline))) {
					String filterType = "";
					if (cl.hasOption(optionStatusFilterType)) 
						filterType = (String)cl.getValue(optionStatusFilterType);
					StatusInformation.getStatus(filterType,cl.hasOption(optionStatusUpdating) ,cl.hasOption(optionStatusOnline));
				} else {
					StatusInformation.getStatus((String)cl.getValue(optionStatus));
				}

			}
			/**********************
			 * CLEAN-DATABASE
			 *********************/
			else if (cl.hasOption(optionCleanDb)) {
				BiomajCleanDB.clean();
			} 
			/**********************
			 * LOG
			 *********************/
			else if (cl.hasOption(optionViewLog)) {
				//ViewLogger v =  new ViewLogger();
				//ServiceLogger v  = new ServiceLogger();
				//v.execute();
				System.out.println("Please wait...");
				BioMAJLogger v = new BioMAJLogger();
				
				v.execute();
			}
			/**********************
			 * INDEX
			 *********************/
			else if (cl.hasOption(optionIndex)){
				IndexXmlDescription gen = new IndexXmlDescription();
				gen.createXmlGeneralFile((String)cl.getValue(optionIndex));
			}
			/**********************
			 * CHANGE-DBNAME
			 *********************/
			else if (cl.hasOption(optionChangeDbName)) {
				ChangeDbName.changeDbName((String)cl.getValues(optionChangeDbName).get(0), (String)cl.getValues(optionChangeDbName).get(1));
			} 

			/**********************
			 * IMPORT
			 *********************/
			else if (cl.hasOption(optionImport)) {
				BiomajImport.importIntoDB((String)cl.getValue(optionImport));
//				BiomajSQLQuerier.closeConnection();
			}
			/**********************
			 * MOVE PRODUCTION VERSION
			 *********************/
			else if (cl.hasOption(optionMoveProductionDirectories)) {
				BiomajMoveProductionDirectories move = new BiomajMoveProductionDirectories();
				move.moveProductionDirectories((String)cl.getValue(optionMoveProductionDirectories));
			} 
			/**********************
			 * VERSION
			 *********************/
			else if (cl.hasOption(optionVersion)) {	
				System.out.println("BioMAJ version "+VERSION);
			} 
			
		} catch (Exception be) {
			System.err.println(be.getLocalizedMessage());
			System.err.println("** BioMAJ stopped. **");
			BiomajLogger.getInstance().log(be.getLocalizedMessage());
			//BiomajLogger.getInstance().close();
		}
	}

	protected static void processBanks(List<String> banks,int command,boolean withConsole,int mode,boolean checkCurrent,boolean newUpdate,boolean fromScratch,
			String[] listP,String[] listM,String[] listB) {
		if (banks.size() == 0) {
			System.out.println("There are no bank to process.");
			return;
		}

		/*
		for (String b : banks)
			System.out.println("process +"+b+" "+command+Boolean.toString(withConsole)+"-"+mode+" "+Boolean.toString(checkCurrent)+"-"+Boolean.toString(newUpdate)+"-");
		 */
		setProxyEnvironment();
		BankFactory bf = new BankFactory();
		List<Thread> running = new ArrayList<Thread>();
		for (String bank : banks) {
			try {
//				BiomajCleanStatefile.corrigeCompatibiliteVersion(bank);
				BiomajBank bb = bf.createBank(bank,true);
				// System.out.println("name:"+bb.getDbName());
				WorkflowEngineFactory wef = new WorkflowEngineFactory();
				WorkflowEngine we ;
				we = wef.createWorkflow(bb);
				we.setModeConsole(withConsole, mode);
				we.setWorkWithCurrentDirectory(checkCurrent);
				we.setForceForANewUpdate(newUpdate);
				we.setFromScratch(fromScratch);
				we.setCommand(command);
				we.setListBlockProcessToRebuild(listB);
				we.setListMetaProcessToRebuild(listM);
				we.setListProcessToRebuild(listP);
				we.start();
				running.add(we);

			} catch (BiomajException e) {
				// Une biomajException est affichée sur la sortie erreur par defaut!
				System.err.println("Biomaj has detected an error! for bank ["+ bank+"]");
				System.err.println(e.getLocalizedMessage());
				
			} catch (BiomajBuildException bbe) {
				System.err
				.println("Biomaj has detected an error during the execution of workflow for the bank :"
						+ bank);
			} catch (Exception e) {
				e.printStackTrace();
				System.err.println(e.getLocalizedMessage());
			}
		}
		
		// Wait for all the threads to end before closing connection
		for (Thread t : running) {
			try {
				t.join();
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
//		BiomajSQLQuerier.closeConnection();
	}

	public static void createOptionBase() {
		/**
		 * Option console
		 */
		subobuilder.reset(); 
		subobuilder.withLongName("console");
		subobuilder.withShortName("c");
		subobuilder.withDescription("Open java console including logs information on workflow session");
		subobuilder.withRequired(false);

		optionConsole = subobuilder.create();

		/**
		 * Option verbose
		 */
		subobuilder.reset(); 
		subobuilder.withLongName("verbose");
		subobuilder.withShortName("v");
		subobuilder.withDescription("Verbose mode");
		subobuilder.withRequired(false);

		optionVerbose = subobuilder.create();

		/**
		 * Option debug
		 */
		subobuilder.reset(); 
		subobuilder.withLongName("debug");
		subobuilder.withShortName("V");
		subobuilder.withDescription("Extra verbose mode\n\n");
		subobuilder.withRequired(false);

		optionDebug = subobuilder.create();
	}

	/**
	 * Create update option and dependances options
	 * @since 0.9.2.0
	 * @return
	 */
	public static Option createOptionUpdate() {
		obuilder.reset(); 
		obuilder.withLongName("update");
		obuilder.withShortName("d");
		obuilder.withDescription("Update specified bank (dbname)");

		//bank argument
		abuilder.reset();
		abuilder.withName("dbname");
		abuilder.withDescription("Bank name defined in property file");
		abuilder.withMinimum(1);
		abuilder.withSubsequentSeparator(',');

		/*****************************
		 * Creation des options dependantes de update
		 */
		//On peut valider si le properties existe par exemple !!!
		//abuilder.withValidator(arg0);
		obuilder.withArgument(abuilder.create());

		//obuilder.reset(); 
		//obuilder.withShortName("stage");
		//obuilder.withDescription("starting point can be precised");
		//stag argument

		childbuilder.reset();
		childbuilder.withName("Update options:\n");
		//childbuilder.withDescription("");
		childbuilder.withMinimum(0);


		/**
		 * Option update/stage
		 */
		subobuilder.reset(); 
		subobuilder.withLongName("stage");
		subobuilder.withDescription("Ending point can be specified. possible stage value : preprocess|sync|postprocess|deployment");
		subobuilder.withRequired(false);

		abuilder.reset();
		abuilder.withName("task");
		abuilder.withDescription("Possible value: preprocess|sync|postprocess|deployment\n");

		Set<String> enumSet = new TreeSet<String>();
		enumSet.add("preprocess");
		enumSet.add("sync");
		enumSet.add("postprocess");
		enumSet.add("deployment");

		EnumValidator validator = new EnumValidator( enumSet );
		abuilder.withValidator(validator);

		abuilder.withMinimum(1);
		abuilder.withMaximum(1);
		abuilder.withInitialSeparator('=');
		subobuilder.withArgument(abuilder.create());

		optionStage = subobuilder.create();
		childbuilder.withOption(optionStage);

		childbuilder.withOption(optionConsole);
		childbuilder.withOption(optionVerbose);
		childbuilder.withOption(optionDebug);

		/**
		 * Option update/check current process
		 */
		/*
		subobuilder.reset(); 
		subobuilder.withLongName("check-current-postprocess");
		subobuilder.withShortName("p");
		subobuilder.withDescription("restart process if data have been changed or removed in the current production directory");
		subobuilder.withRequired(false);

		optionCheckCurrentProcess = subobuilder.create();
		childbuilder.withOption(optionCheckCurrentProcess);
		 */
		/**
		 * Option update/new update : force l'update de la banque
		 */
		subobuilder.reset(); 
		subobuilder.withLongName("new");
		subobuilder.withShortName("N");
		subobuilder.withDescription("Start a new update cycle even if a unclosed update cycle is open (ps:previous sessions are ignored)\n");
		subobuilder.withRequired(false);

		optionNew = subobuilder.create();
		childbuilder.withOption(optionNew);

		
		/**
		 * Option update fromscratch : force le retelechargement de la derniere release en erreur ou non.
		 */
		subobuilder.reset();
		subobuilder.withLongName("fromscratch");
		subobuilder.withDescription("Force update from scratch of the bank by skipping the repository verification\n");
		subobuilder.withRequired(false);
		
		optionFromScratch = subobuilder.create();
		childbuilder.withOption(optionFromScratch);
		
		/**
		 * On ajoute les fils a l option update
		 */
		obuilder.withChildren(childbuilder.create());

		return obuilder.create();
	}

	/**
	 * 
	 * @return
	 */
	public static Option createOptionStatus() {
		obuilder.reset(); 
		obuilder.withLongName("status");
		obuilder.withShortName("S");
		obuilder.withDescription("Print out bank status. If dbname is specified, detailed profile will be print out");

//		bank argument
		abuilder.reset();
		abuilder.withName("dbname");
		abuilder.withDescription("Bank name defined in property file");
		abuilder.withMinimum(0);
		abuilder.withMaximum(1);
		obuilder.withArgument(abuilder.create());

		/**
		 * Option status db-type : affiche seulement les bank de type <type>
		 */

		childbuilder.reset();
		childbuilder.withName("status options:");
		subobuilder.reset(); 
		subobuilder.withLongName("dbtype");
		subobuilder.withDescription("Filter bank list using dbtype specified");
		subobuilder.withRequired(false);

		abuilder.reset();
		abuilder.withName("dbtype");
		abuilder.withDescription("Filter bank listing using dbtype specifed");

		abuilder.withMinimum(1);
		abuilder.withMaximum(1);
		abuilder.withInitialSeparator('=');
		subobuilder.withArgument(abuilder.create());

		optionStatusFilterType = subobuilder.create();
		childbuilder.withOption(optionStatusFilterType);

		/**
		 * Option status online : filtre sur les banque seulement online
		 */

		subobuilder.reset(); 
		subobuilder.withLongName("online");
		subobuilder.withDescription("Filter from bank list online release");
		subobuilder.withRequired(false);

		optionStatusOnline = subobuilder.create();
		childbuilder.withOption(optionStatusOnline);

		/**
		 * Option status updating : filtre sur les banque seulement online
		 */
		subobuilder.reset(); 
		subobuilder.withLongName("updating");
		subobuilder.withDescription("Filter in bank list, updating session\n");
		subobuilder.withRequired(false);

		optionStatusUpdating = subobuilder.create();
		childbuilder.withOption(optionStatusUpdating);


		/**
		 * On ajoute les fils a l option update
		 */
		obuilder.withChildren(childbuilder.create());

		return obuilder.create();
	}


	/**
	 * 
	 * @return
	 */
	public static Option createOptionRemove() {
		obuilder.reset(); 
		obuilder.withLongName("remove");
		obuilder.withDescription("Delete all productions directories (use with care!)");

//		bank argument
		abuilder.reset();
		abuilder.withName("dbname");
		abuilder.withDescription("Bank name defined in property file");
		abuilder.withMinimum(1);
		abuilder.withMaximum(1);
		obuilder.withArgument(abuilder.create());



		childbuilder.reset();
		childbuilder.withName("options:");
		//childbuilder.withDescription("");
		childbuilder.withMinimum(0);

		/*
		 * Option remove/keep-dir-prod : si true n efface pas les repertoire de prod
		 */
		subobuilder.reset(); 
		subobuilder.withLongName("keep-dir-prod");
		subobuilder.withDescription("Delete/keep production directory\n\n");
		subobuilder.withRequired(false);	

		optionKeepDir = subobuilder.create();
		
		/*
		 * Option to directly specify the paths of the directories to be deleted
		 */
		subobuilder.reset();
		subobuilder.withLongName("paths");
		subobuilder.withDescription("Directly specify the directories paths to be deleted");
		subobuilder.withRequired(false);
		abuilder.reset();
		abuilder.withName("path");
		abuilder.withDescription("Production directory paths");
		abuilder.withMinimum(1);
		subobuilder.withArgument(abuilder.create());
		
		optionPaths = subobuilder.create();
		
		subobuilder.reset();
		subobuilder.withLongName("all");
		subobuilder.withDescription("Delete all directories");
		subobuilder.withRequired(false);
		optionAllDirs = subobuilder.create();
		
		childbuilder.withOption(optionPaths);
		childbuilder.withOption(optionAllDirs);
		childbuilder.withOption(optionKeepDir);
		childbuilder.withOption(optionConsole);
		childbuilder.withOption(optionVerbose);
		childbuilder.withOption(optionDebug);
		
		/*
		 * On ajoute les fils a l option update
		 */
		obuilder.withChildren(childbuilder.create());

		return obuilder.create();
	}

	/**
	 * Set proxy
	 * author:ofilangi
	 * 
	 *  a tester
	 */

	private static void setProxyEnvironment() {
		final Properties prop = new Properties();
		try {
			prop.load(new FileInputStream(new File(BiomajInformation.getInstance().getProperty(BiomajInformation.WORKFLOWSDIR)+"/"+BiomajConst.globalProperties)));
		} catch (Exception e) {
			System.err.println(e.getMessage());
			System.exit(-1);
		}
		/**
		 * AUTRE METHODE : http://floatingsun.net/articles/java-proxy.html
		 */

		/*
		if (prop.containsKey(BiomajConst.httpsProxyHost)||prop.containsKey(BiomajConst.httpsProxyPassword)||
				prop.containsKey(BiomajConst.httpsProxyPort)||prop.containsKey(BiomajConst.httpsProxyUser)) {

			System.setProperty("java.protocol.handler.pkgs","com.sun.net.ssl.internal.www.protocol");
			System.out.println("values for proxy https...");

			if (prop.containsKey(BiomajConst.httpsProxyHost))
			{
				System.setProperty("https.proxyHost",prop.getProperty(BiomajConst.httpsProxyHost));
				System.out.println("https.proxyHost:"+prop.getProperty(BiomajConst.httpsProxyHost));
			}

			if (prop.containsKey(BiomajConst.httpsProxyPort))
			{
				System.setProperty("https.proxyPort",prop.getProperty(BiomajConst.httpsProxyPort));
				System.out.println("https.proxyPort"+prop.getProperty(BiomajConst.httpsProxyPort));
			}

			if (prop.containsKey(BiomajConst.httpsProxyUser))
			{
				System.setProperty("https.proxyUser",prop.getProperty(BiomajConst.httpsProxyUser));
				System.out.println("https.proxyUser"+prop.getProperty(BiomajConst.httpsProxyUser));
			}

			if (prop.containsKey(BiomajConst.httpsProxyPassword))
			{
				System.setProperty("https.proxyPassword",prop.getProperty(BiomajConst.httpsProxyPassword));
				System.out.println("https.proxyPassword"+prop.getProperty(BiomajConst.httpsProxyPassword));
			}

			if (prop.containsKey(BiomajConst.httpsProxyUser)&&prop.containsKey(BiomajConst.httpsProxyPassword))
				Authenticator.setDefault( new HttpAuthenticateProxy(prop.getProperty(BiomajConst.httpsProxyUser),prop.getProperty(BiomajConst.httpsProxyPassword)) );

		} else */
		if (prop.containsKey(BiomajConst.proxyHost)||prop.containsKey(BiomajConst.proxyPassword)||
				prop.containsKey(BiomajConst.proxyPort)||prop.containsKey(BiomajConst.proxyUser)) {
			
			System.getProperties().put("proxySet", "true");

			if (prop.containsKey(BiomajConst.proxyHost)) {
				System.getProperties().put( "socksProxyHost" ,prop.getProperty(BiomajConst.proxyHost));
			}

			if (prop.containsKey(BiomajConst.proxyPort)) {
				System.getProperties().put( "socksProxyPort",prop.getProperty(BiomajConst.proxyPort));
			}
			if (prop.containsKey(BiomajConst.proxyUser)&&prop.containsKey(BiomajConst.proxyPassword)) {
				// Authenticate to the proxy
				Authenticator.setDefault(new Authenticator() {
					@Override
					protected PasswordAuthentication getPasswordAuthentication() {
						return new PasswordAuthentication(prop.getProperty(BiomajConst.proxyUser),
								prop.getProperty(BiomajConst.proxyPassword).toCharArray());
					}
				});
				System.getProperties().put("socksProxyUser", prop.getProperty(BiomajConst.proxyUser));
				System.getProperties().put("socksProxyPassword", prop.getProperty(BiomajConst.proxyPassword));
			}
		}
	}

	protected static Integer getCommand(String c) {

		if (c == null)
			return WorkflowEngine.TARGET_ALL;

		if (c.compareTo("preprocess") == 0)
			return WorkflowEngine.TARGET_UNTIL_PREPROCESS;
		if ((c.compareTo("sync") == 0) || (c.compareTo("mirror") == 0))
			return WorkflowEngine.TARGET_UNTIL_MIRROR;
		if (c.compareTo("postprocess") == 0)
			return WorkflowEngine.TARGET_UNTIL_POSTPROCESS;
		if (c.compareTo("deploy") == 0)
			return WorkflowEngine.TARGET_UNTIL_DEPLOY;

		// System.err.println(c+" unknown option!");
		return WorkflowEngine.TARGET_ALL;
	}

}
