/*
 * Biloba
 * Copyright (C) 2004-2008 Guillaume Demougeot, Colin Leroy
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

/**
 * Biloba - Q1 2005
 * Game by Guillaume Demougeot <dmgt@wanadoo.fr>
 * Code by Colin Leroy <colin@colino.net>
 *
 * This file contains the network server code.
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h> 
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pthread.h>
#include <sys/time.h>
#include <time.h>
#include <signal.h>
#include "netops.h"
#include "llist.h"
#include "game.h"

static FILE *logfp = NULL;
/* redefinitions to avoid sdl dependancy */
typedef enum {
	INPUT_LOCAL,
	INPUT_NETWORK,
	INPUT_AI,
	NUM_INPUT_SYSTEMS
} InputSystemMethod;

static char *input_system_names(InputSystemMethod i) {
	char *name[3] = {
			"LOCAL",
			"NETWORK",
			"AI" };
	if (i < 0 || i >= NUM_INPUT_SYSTEMS)
		return "WRONG TYPE";

	return name[i];
};

typedef enum {
	PAWN_ORANGE = 0,
	PAWN_BLUE,
	PAWN_RED,
	PAWN_GREEN,
	PAWN_NUM_COLORS
} PawnColor;

static unsigned int cur_id = 0;
pthread_mutex_t gamelist_mutex;
static LList *gamelist = NULL;

typedef enum {
	CONN_NEW,
	CONN_SETUP_NEW,
	CONN_SETUP_LISTGAME,
	CONN_SETUP_PRELJOIN,
	CONN_SETUP_JOIN,
	CONN_WAIT_PLAYERS,
	CONN_START_PLAY,
	CONN_PLAY,
	CONN_PLAY_SEND_MOVE,
	CONN_PLAY_SEND_QUIT,
	CONN_PLAY_RECEIVE,
	CONN_KILL,
	CONN_EXIT,
	NUM_CONN_STATES
} ConnState;

typedef struct _ConnData {
	int fd;
	char *addr;
	ConnState state;

	Game *game;
	int previously_reserved;
	int player_num;
	int move_id;
	int am_initiator;
	int last_sent_join;
} ConnData;

static ConnData *conn_data_init(int fd, const char *addr)
{
	ConnData *cd = malloc(sizeof(ConnData));
	int optOn = 1;

	if (cd == NULL)
		return NULL;

	setsockopt(fd, SOL_SOCKET, SO_REUSEADDR,
		&optOn, sizeof(optOn));

	cd->fd = fd;
	cd->addr = strdup(addr);
	cd->state = CONN_NEW;
	cd->game = NULL;
	cd->previously_reserved = -1;
	cd->player_num = -1;
	cd->move_id = 0;
	cd->am_initiator = 0;
	cd->last_sent_join = -1;

	return cd;
}

static char state_buf[32];
static char *state_to_str(ConnState state)
{
	switch(state) {
	case CONN_NEW:			return "CONN_NEW";
	case CONN_SETUP_NEW:		return "CONN_SETUP_NEW";
	case CONN_SETUP_LISTGAME:	return "CONN_SETUP_LISTGAME";
	case CONN_SETUP_PRELJOIN:	return "CONN_SETUP_PRELJOIN";
	case CONN_SETUP_JOIN:		return "CONN_SETUP_JOIN";
	case CONN_WAIT_PLAYERS:		return "CONN_WAIT_PLAYERS";
	case CONN_START_PLAY:		return "CONN_START_PLAY";
	case CONN_PLAY:			return "CONN_PLAY";
	case CONN_PLAY_SEND_MOVE:	return "CONN_PLAY_SEND_MOVE";
	case CONN_PLAY_SEND_QUIT:	return "CONN_PLAY_SEND_QUIT";
	case CONN_PLAY_RECEIVE:		return "CONN_PLAY_RECEIVE";
	case CONN_KILL:			return "CONN_KILL";
	case CONN_EXIT:			return "CONN_EXIT";
	default:			
		snprintf(state_buf, sizeof(state_buf), "%d", state);
		return state_buf;
	}
}

static void do_log(ConnData *cd, const char *format, ...)
{
	va_list args;
	struct timeval tv;
	struct tm tm;
	time_t t;

	gettimeofday(&tv, NULL);
	t = time(NULL);
	localtime_r(&t, &tm);

        fprintf(logfp, "%02d/%02d/%4d %02d:%02d:%02d.%04d [%s(%d)] - ", 
		    tm.tm_mday,
		    tm.tm_mon +1,
		    tm.tm_year + 1900,
		    tm.tm_hour,
		    tm.tm_min,
		    tm.tm_sec,
		    (int)(tv.tv_usec / 1000),
		    cd->addr,
		    cd->fd);

        va_start(args, format);
        vfprintf(logfp, format, args);
        va_end(args);
	
	fflush(logfp);
}

static void free_game(Game *game)
{
	int i = 0;
	
	if (!game)
		return;

	free(game->name);
	for (i = 0; i < 4; i++)
		free(game->player_name[i]);
	
	pthread_cond_destroy(&game->cond_wait);
	pthread_cond_destroy(&game->cond_move);
	pthread_cond_destroy(&game->cond_quit);
	free(game);
}

/* must be called with gamelist_mutex held */
static LList *get_games(int num_players, int max)
{
	LList *cur = gamelist;
	LList *results = NULL;
	int i = 0;
	
	while (cur) {
		Game *game = (Game *)cur->data;
		if (game->num_players == num_players &&
		    game->first_avail_spot > 0 &&
		    game->started == 0 &&
		    game->killed == 0) {
			results = llist_append(results, game);
			i++;
		}
		if (i >= max) 
			break;
		cur = cur->next;
	}
	
	return results;
}

static Game *get_game(int id)
{
	LList *cur = gamelist;
	
	while (cur) {
		Game *game = (Game *)cur->data;
		if (game->id == id)
			return game;

		cur = cur->next;
	}
	return NULL;
}

static int get_nb_missing_players(Game *game)
{
	int i = 0;
	int r = 0;
	if (game == NULL)
		return 0;
	for (i = 0; i < game->num_players; i++) {
		if (game->player_name[i] == NULL)
			r++;
	}
	return r;
}

static void set_timeouts(ConnData *cd, int seconds)
{
	struct timeval timeo;
	timeo.tv_sec = seconds;
	timeo.tv_usec = 0;
	if (setsockopt(cd->fd, SOL_SOCKET, SO_RCVTIMEO, &timeo, sizeof(timeo)) < 0)
		perror("setsockopt");
	if (setsockopt(cd->fd, SOL_SOCKET, SO_SNDTIMEO, &timeo, sizeof(timeo)) < 0)
		perror("setsockopt");
}

int server_setup_game(ConnData *cd, Game **new_game)
{
	char *p_name;
	int gameid = 1;
	char buf[256];
	unsigned char len = 0, i, p;
	int numplayers = 0;

	*new_game = malloc(sizeof(struct _Game));
	if (*new_game == NULL)
		return -ENOMEM;

	memset(*new_game, 0, sizeof(struct _Game));

	if (send_msg(cd->fd, "OK", 3) < 0)
		goto error;

	/* read game name */
	if (read_msg(cd->fd, buf, 255, &len) < 0) 
		goto error;

	buf[254] = '\0';

	do_log(cd, " > game name '%s'\n", buf);

	(*new_game)->name = strdup(buf);

	cur_id++;
	gameid = htonl(cur_id);
	if (send_msg(cd->fd, &gameid, 4) < 0)
		goto error;

	do_log(cd, " < game ID %d\n", cur_id);

	(*new_game)->id = cur_id;
	for (i = 0; i < 4; i++) {
		(*new_game)->player_name[i] = NULL;
	}
	/* read player info */
	if (read_msg(cd->fd, buf, 255, &len) < 0)
		goto error;

	buf[254] = '\0';

	i = 0;
	numplayers = (int) buf[i]; i++;

	do_log(cd, " > %d players\n", numplayers);
	if (numplayers < 1 || numplayers > 4) {
		send_msg(cd->fd, "ERROR", 6);
		do_log(cd, "   wrong number of players\n");
		goto error;
	}

	(*new_game)->num_players = numplayers;
	(*new_game)->first_avail_spot = -1;


	for (p = 0; p < numplayers; p++) {

		(*new_game)->player_type[p] = (InputSystemMethod)buf[i];
		
		if ((*new_game)->player_type[p] < 0 ||
		    (*new_game)->player_type[p] >= NUM_INPUT_SYSTEMS) {
			send_msg(cd->fd, "ERROR", 6);
			do_log(cd, "   wrong player type %d for player %d\n",
				(*new_game)->player_type[p], p);
			goto error;
		}

		i++; p_name = buf+i;
		do_log(cd, " > player %d (%s) is named %s\n", 
			p,
			input_system_names((*new_game)->player_type[p]),
			p_name);

		(*new_game)->player_name[p] = strdup(p_name);
		i+=strlen(p_name)+1;
	}

	(*new_game)->num_remote = 0;
	(*new_game)->first_avail_spot = -1;
	for (i = 0; i < numplayers; i++) {
		if ((*new_game)->player_type[i] == INPUT_NETWORK) {
			free((*new_game)->player_name[i]);
			(*new_game)->player_name[i] = NULL;
			(*new_game)->num_remote++;
			do_log(cd, "   cleared player %d's name (net player)\n", i);
			if ((*new_game)->first_avail_spot == -1) {
				(*new_game)->first_avail_spot = i + 1;
					do_log(cd, "   first spot for another player is %d\n",
						(*new_game)->first_avail_spot);
			}
		}
	}

	if ((*new_game)->first_avail_spot == -1) {

		do_log(cd, "   ERROR: could not get a spot for another player\n");
		send_msg(cd->fd, "ERROR", 6);
		goto error;
	}

	if (send_msg(cd->fd, "READY",6) < 0)
		goto error;

	do_log(cd, "   < game %s (%d) fully set up\n", (*new_game)->name,
		(*new_game)->id);

	pthread_cond_init(&(*new_game)->cond_wait, NULL);
	pthread_cond_init(&(*new_game)->cond_move, NULL);
	pthread_cond_init(&(*new_game)->cond_quit, NULL);

	pthread_mutex_lock (&gamelist_mutex);
	gamelist = llist_append(gamelist, (*new_game));
	pthread_mutex_unlock (&gamelist_mutex);
	return 0;
error:
	free_game(*new_game);
	*new_game = NULL;
	return -1;
}

static int server_get_requested_num_players(ConnData *cd)
{
	int nump;
	unsigned char len;

	if (send_msg(cd->fd, "NUMP", 5) < 0) {
		return -1;
	}

	if (read_msg(cd->fd, &nump, 4, &len) < 0) {
		return -1;
	}

	if (len > 4) {
		return -1;
	}
	return ntohl(nump);
}

static int server_send_games(ConnData *cd, int nump)
{
	LList *avail = NULL, *cur;
	int num, err;

	pthread_mutex_lock (&gamelist_mutex);

	avail = get_games(nump, 6);

	num = htonl(llist_length(avail));

	if (send_msg(cd->fd, &num, 4) < 0) {
		err = -1;
		goto error;
	}

	cur = avail;

	do_log(cd, " < sending %d games available\n", llist_length(avail));

	/* send up to 6 games: id, name */
	while (cur) {
		Game *game = cur->data;
		num = htonl(game->id);
		if (send_msg(cd->fd, &num, 4) < 0) {
			err = -1;
			goto error;
		}


		if (send_msg(cd->fd, game->name, strlen(game->name) + 1) < 0) {
			err = -1;
			goto error;
		}

		do_log(cd, "   sent game ID %d, name '%s'\n", game->id, game->name);

		cur = cur->next;
	}
	err = 0;
error:
	pthread_mutex_unlock (&gamelist_mutex);
	llist_free(avail);
	return err;
}

static int server_send_player_names(ConnData *cd, Game *game)
{
	int j;
	for (j = 0; j < game->num_players; j++) {
		const char *name = game->player_name[j] ? game->player_name[j] : "";
		if (send_msg(cd->fd, name, strlen(name) + 1) < 0)
			return -1;

		do_log(cd, " > sent player %d name '%s'\n", j,
			game->player_name[j] ? game->player_name[j] : "");
	}
	return 0;
}

static int server_get_first_spot(ConnData *cd, Game *game)
{
	int tmp = htonl(game->first_avail_spot);
	if (send_msg(cd->fd, &tmp, 4) < 0)
		return -1;

	do_log(cd, " < sent %d (first available spot)\n", game->first_avail_spot);

	game->first_avail_spot++;

	/* No more free spot, game would be able to begin if player joins */
	if (game->first_avail_spot > game->num_players)
		game->first_avail_spot = -1;
	
	return 0;
}

static void server_put_first_spot_back(ConnData *cd, Game *game)
{
	if (game != NULL) {
		/* put back in pool */
		game->first_avail_spot--;
		if (game->first_avail_spot < 0)
			game->first_avail_spot = game->num_players;
		do_log(cd, "   un-reserved spot on game %d (%s)\n", game->id, game->name);
	}
}			
static Game *server_do_reserve_game(ConnData *cd, int game_id)
{
	Game *game;

	if (cd->previously_reserved > 0) {
		do_log(cd, "   de-reserving spot in previously reserved game %d\n",
			cd->previously_reserved);
		server_put_first_spot_back(cd, get_game(cd->previously_reserved));
		cd->previously_reserved = -1;
	}

	game = get_game(game_id);
	if (game == NULL) {
		do_log(cd, "   game %d does not exist\n", game_id);
		send_msg(cd->fd, "ERROR1", 7);
		game = NULL;
		goto error;
	} else
		do_log(cd, "   game %d is '%s'\n", game_id, game->name);


	if (send_msg(cd->fd, "OK", 3) < 0) {
		game = NULL;
		goto error;
	}

	if (server_send_player_names(cd, game) < 0) {
		game = NULL;
		goto error;
	}

	if (server_get_first_spot(cd, game) < 0) {
		game = NULL;
		goto error;
	}

error:
	return game;
}

int server_get_game_id(ConnData *cd)
{
	unsigned char len;
	int game_id;

	if (read_msg(cd->fd, &game_id, 4, &len) < 0) {
		return -1;
	}
	game_id = ntohl(game_id);

	return game_id;
}

static int server_get_new_playernum(ConnData *cd)
{
	int mynump;
	unsigned char len;

	if (read_msg(cd->fd, &mynump, 4, &len) < 0) {
		return -1;
	}

	mynump = ntohl(mynump);


	if (send_msg(cd->fd, "OK", 3) < 0) {
		return -1;
	}

	return mynump;
}

static char *server_get_new_playername(ConnData *cd)
{
	char buf[256];
	unsigned char len;

	if (read_msg(cd->fd, buf, 255, &len) < 0) {
		return NULL;
	}
	buf[254] = '\0';

	return strdup(buf);
}	

static int server_wait_for_ping(ConnData *cd)
{
	char buf[256];
	unsigned char len;

	if (read_msg(cd->fd, buf, 255, &len) < 0)
		return -1;

	buf[254] = '\0';

	if (strcmp("READY?", buf) != 0) {
		do_log(cd, "protocol error (got %s)\n", buf);
		return -1;
	}
	return 0;
}

static int server_send_waiting_for_players(ConnData *cd, int nb_missing_players)
{
	int tmp = htonl(nb_missing_players);
	if (send_msg(cd->fd, &tmp, 4) < 0)
		return -1;

	return 0;
}

static int server_send_last_join(ConnData *cd, int player_num, const char *player_name)
{
	const char *name = player_name ? player_name : "";
	int tmp = htonl(player_num);

	if (send_msg(cd->fd, &tmp, 4) < 0)
		return -1;

	if (send_msg(cd->fd, name, strlen(name)+1) < 0)
		return -1;
	
	return 0;
}

static ConnState do_transition(ConnData *cd, ConnState newstate)
{
	if (cd->state != newstate)
		do_log(cd, "%s => %s\n", state_to_str(cd->state),
		       state_to_str(newstate));
	cd->state = newstate;
	return cd->state;
}

static ConnState get_state_from_command(ConnData *cd, const char *command)
{
	/* New connection, asking to create a game */
	if (cd->state == CONN_NEW && !strcmp(command, "NEWGAME"))
		return do_transition(cd, CONN_SETUP_NEW);
	
	/* New connection, asking to list games */
	if (cd->state == CONN_NEW && !strcmp(command, "LISTGAME"))
		return do_transition(cd, CONN_SETUP_LISTGAME);
	
	/* Games listed, doing preliminary join (game info, spot reserve) */
	if (cd->state == CONN_SETUP_LISTGAME && !strcmp(command, "PRELJOIN"))
		return do_transition(cd, CONN_SETUP_PRELJOIN);
	
	/* Prel join done, doing new preliminary join on another game */
	if (cd->state == CONN_SETUP_PRELJOIN && !strcmp(command, "PRELJOIN"))
		return do_transition(cd, CONN_SETUP_PRELJOIN);
	
	/* Prel join done, finalising join */
	if (cd->state == CONN_SETUP_PRELJOIN && !strcmp(command, "JOIN"))
		return do_transition(cd, CONN_SETUP_JOIN);
	
	/* New game setup done, now waiting for players */
	if (cd->state == CONN_SETUP_NEW && !strcmp(command, "WAIT_PLAYERS"))
		return do_transition(cd, CONN_WAIT_PLAYERS);

	/* Game join done, now waiting for players */
	if (cd->state == CONN_SETUP_JOIN && !strcmp(command, "WAIT_PLAYERS"))
		return do_transition(cd, CONN_WAIT_PLAYERS);

	/* Wait for players done, still waiting */
	if (cd->state == CONN_WAIT_PLAYERS && !strcmp(command, "WAIT_PLAYERS"))
		return do_transition(cd, CONN_WAIT_PLAYERS);

	/* Wait for players done, now starting the game */
	if (cd->state == CONN_WAIT_PLAYERS && !strcmp(command, "START_PLAY"))
		return do_transition(cd, CONN_START_PLAY);

	/* Game start done, now playing the game */
	if (cd->state == CONN_START_PLAY && !strcmp(command, "PLAY"))
		return do_transition(cd, CONN_PLAY);

	return do_transition(cd, CONN_KILL);
}

static ConnState get_state_from_play_command(ConnData *cd, const char command)
{
	/* Playing, going to send a move */
	if (cd->state == CONN_PLAY && command == 'p')
		return do_transition(cd, CONN_PLAY_SEND_MOVE);
	
	/* Playing, going to quit */
	if (cd->state == CONN_PLAY && command == 'q')
		return do_transition(cd, CONN_PLAY_SEND_QUIT);
	
	/* Playing, going to send a move */
	if (cd->state == CONN_PLAY && command == 'r')
		return do_transition(cd, CONN_PLAY_RECEIVE);

	return do_transition(cd, CONN_KILL);	
}

static ConnState server_handle_conn_new(ConnData *cd)
{
	unsigned char len = 0;
	char buf[256];

	set_timeouts(cd, 10);

	if (read_msg(cd->fd, buf, 255, &len) < 0)
		return CONN_KILL;

	if (len == 0)
		return CONN_KILL;

	return get_state_from_command(cd, buf);
}

static ConnState server_handle_conn_setup_new(ConnData *cd)
{
	set_timeouts(cd, 500);

	if (server_setup_game(cd, &cd->game) < 0)
		return CONN_KILL;

	cd->am_initiator = 1;
	cd->player_num = 0;
	return get_state_from_command(cd, "WAIT_PLAYERS");
}

static ConnState server_handle_conn_setup_listgame(ConnData *cd)
{
	unsigned char len = 0;
	char buf[256];

	int nump = 0;

	set_timeouts(cd, 500);

	nump = server_get_requested_num_players(cd);
	do_log(cd, " > requested games with %d players\n", nump);

	if (nump < 1 || nump > 4)
		return CONN_KILL;

	if (server_send_games(cd, nump) < 0)
		return CONN_KILL;

	if (read_msg(cd->fd, buf, 255, &len) < 0)
		return CONN_KILL;

	return get_state_from_command(cd, buf);
}

static ConnState server_handle_conn_setup_preljoin(ConnData *cd)
{
	/* preliminary join. user clicked on game.
	 * let him fill his name, bump first_avail_spot
	 */
	unsigned char len = 0;
	char buf[256];
	int game_id;
	Game *game;

	game_id = server_get_game_id(cd);
	if (game_id < 0)
		return CONN_KILL;

	do_log(cd, " > pre-reserving game ID %d\n", game_id);

	pthread_mutex_lock (&gamelist_mutex);
	game = server_do_reserve_game(cd, game_id);
	if (game)
		cd->previously_reserved = game->id;
	pthread_mutex_unlock(&gamelist_mutex);

	/* Reservation should have worked */
	if (cd->previously_reserved < 0)
		return CONN_KILL;

	if (read_msg(cd->fd, buf, 255, &len) < 0)
		return CONN_KILL;

	return get_state_from_command(cd, buf);
}

static ConnState server_handle_conn_setup_join(ConnData *cd)
{
	/* player joins reserved game. */
	int mynump;

	pthread_mutex_lock (&gamelist_mutex);
	cd->game = get_game(cd->previously_reserved);

	if (cd->game == NULL) {
		do_log(cd, " error: no pre-reserved game.\n");
		pthread_mutex_unlock (&gamelist_mutex);
		return CONN_KILL;
	}
	cd->previously_reserved = -1;

	do_log(cd, " join on pre-reserved game %d (%s)\n",
		cd->game->id, cd->game->name);

	mynump = server_get_new_playernum(cd);

	do_log(cd, "  > player number %d\n", mynump);

	if (mynump < 0) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	cd->game->player_name[mynump] = server_get_new_playername(cd);

	do_log(cd, " > player name '%s'\n", 
		cd->game->player_name[mynump] 
			? cd->game->player_name[mynump] 
			: "NULL");

	if (cd->game->player_name[mynump] == NULL) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	/* All others but me are remote players */
	cd->game->num_remote = cd->game->num_players - 1;
	cd->game->last_joined = mynump;

	pthread_mutex_unlock (&gamelist_mutex);
	cd->player_num = mynump;

	/* see if we can start game */
	return get_state_from_command(cd, "WAIT_PLAYERS");
}

static ConnState server_handle_conn_wait_players(ConnData *cd)
{
	/* The game is now setup and we just have to wait for all players to
	 * be there. */
	int nb_missing_players;

	pthread_mutex_lock(&gamelist_mutex);

	if (cd->game->killed) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	nb_missing_players = get_nb_missing_players(cd->game);

	do_log(cd, " waiting for %d players on game %d (%s)...\n",
		nb_missing_players, cd->game->id, cd->game->name);

	if (server_wait_for_ping(cd) < 0) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	if (server_send_waiting_for_players(cd, nb_missing_players) < 0) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	if (server_send_last_join(cd, cd->game->last_joined,
		cd->game->player_name[cd->game->last_joined]) < 0) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	if (cd->game->last_joined != cd->last_sent_join)
		do_log(cd, " < %d new joins (last '%s')\n",
			cd->game->last_joined,
			cd->game->player_name[cd->game->last_joined]);

	cd->last_sent_join = cd->game->last_joined;

	if (nb_missing_players == 0) {
		do_log(cd, " all players arrived on game %d (%s)\n",
			cd->game->id, cd->game->name);

		cd->game->started = 1;

		pthread_mutex_unlock(&gamelist_mutex);

		return get_state_from_command(cd, "START_PLAY");
	}

	pthread_mutex_unlock(&gamelist_mutex);
	sleep(1);

	return get_state_from_command(cd, "WAIT_PLAYERS");
}

static ConnState server_handle_conn_start_play(ConnData *cd)
{
	int i;

	pthread_mutex_lock(&gamelist_mutex);
	cd->game->move_id = 0;
	cd->game->cur_x = -1;
	cd->game->cur_y = -1;
	cd->game->quit_players = 0;
	cd->game->killed = 0;

	for (i = 0; i < cd->game->num_players; i++) {
		if (send_msg(cd->fd, cd->game->player_name[i], 
				strlen(cd->game->player_name[i])+1) < 0) {
			pthread_mutex_unlock(&gamelist_mutex);
			return CONN_KILL;
		}
	}
	pthread_mutex_unlock(&gamelist_mutex);
	return get_state_from_command(cd, "PLAY");
}

static ConnState server_handle_conn_play(ConnData *cd)
{
	unsigned char len = 0;
	char action;

	if (read_msg(cd->fd, &action, 1, &len) < 0) {
		return CONN_KILL;
	}

	return get_state_from_play_command(cd, action);
}

static ConnState server_handle_conn_play_receive(ConnData *cd)
{
	int playercolor, x, y;
	unsigned char len = 0;

	if (read_msg(cd->fd, &playercolor, 4, &len) < 0)
		return CONN_KILL;

	playercolor = ntohl(playercolor);

	pthread_mutex_lock(&gamelist_mutex);

	do_log(cd, " > game %d: player %d waiting for player %d move\n",
		cd->game->id, cd->player_num, playercolor);


	while (cd->game->move_id <= cd->move_id) {
		pthread_cond_wait(&cd->game->cond_move, &gamelist_mutex);
		if (cd->game->killed)
			break;
	}

	if (!cd->game->killed) {
		x = htonl(cd->game->cur_x);
		y = htonl(cd->game->cur_y);
	} else {
		x = htonl(-2);
		y = htonl(-2);
	}

	if (send_msg(cd->fd, &x, 4) < 0) {
		cd->game->waiting_acks--;
		pthread_cond_broadcast(&cd->game->cond_wait);
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}
	if (send_msg(cd->fd, &y, 4) < 0) {
		cd->game->waiting_acks--;
		pthread_cond_broadcast(&cd->game->cond_wait);
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}
	do_log(cd, " < game %d: pushed move ID %d"
		   " (x %d, y %d) from player %d to player %d\n",
		cd->game->id, cd->game->move_id,
		cd->game->cur_x, cd->game->cur_y,
		playercolor, cd->player_num);
	cd->move_id++;

	cd->game->waiting_acks--;
	pthread_cond_broadcast(&cd->game->cond_wait);
	pthread_mutex_unlock(&gamelist_mutex);

	return CONN_PLAY;
}

static void kill_game(Game *game)
{
	if (game->killed == 1)
		return;

	game->killed = 1;
	pthread_cond_broadcast(&game->cond_wait);
	pthread_cond_broadcast(&game->cond_move);
}

static ConnState server_handle_conn_play_send(ConnData *cd, int quit)
{
	int playercolor, x, y;
	unsigned char len = 0;

	pthread_mutex_lock(&gamelist_mutex);
	while (cd->game->waiting_acks > 0) {
		/* wait for the previous move to be received by 
		 * all */
		pthread_cond_wait(&cd->game->cond_wait, &gamelist_mutex);
		if (cd->game->killed) {
			pthread_mutex_unlock(&gamelist_mutex);
			return CONN_KILL;
		}
	}

	if (read_msg(cd->fd, &playercolor, 4, &len) < 0) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	playercolor = ntohl(playercolor);
	if (playercolor != cd->player_num) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	if (!quit) {
		if (read_msg(cd->fd, &x, 4, &len) < 0) {
			pthread_mutex_unlock(&gamelist_mutex);
			return CONN_KILL;
		}
		if (read_msg(cd->fd, &y, 4, &len) < 0) {
			pthread_mutex_unlock(&gamelist_mutex);
			return CONN_KILL;
		}

		x = ntohl(x);
		y = ntohl(y);
	} else {
		x = -2;
		y = -2;
	}

	cd->game->cur_player = playercolor;
	cd->game->cur_x = x;
	cd->game->cur_y = y;
	cd->game->move_id++;
	cd->game->waiting_acks = cd->game->num_remote;

	pthread_cond_broadcast(&cd->game->cond_move);

	do_log(cd, " < game %d: player %d sends move ID %d (x %d, y %d)\n",
		cd->game->id, playercolor, cd->game->move_id, x, y);

	if (quit) {
		pthread_mutex_unlock(&gamelist_mutex);
		return CONN_KILL;
	}

	/* wait here that everyone gets the move. 
	 * this avoids deadlocks */
	while (cd->game->waiting_acks > 0) {
		pthread_cond_wait(&cd->game->cond_wait, &gamelist_mutex);
		if (cd->game->killed) {
			pthread_mutex_unlock(&gamelist_mutex);
			return CONN_KILL;
		}
	}

	pthread_mutex_unlock(&gamelist_mutex);
	return CONN_PLAY;
}

static ConnState server_handle_conn_kill(ConnData *cd)
{
	do_log(cd, "Connection done\n", cd);

	pthread_mutex_lock(&gamelist_mutex);

	if (cd->previously_reserved > 0)
	{
		Game *reserved_game = get_game(cd->previously_reserved);
		server_put_first_spot_back(cd, reserved_game);

		/* Don't let clients not joining after a PRELJOIN kill
		 * games */
		if (reserved_game == cd->game)
			cd->game = NULL;
	}

	if (cd->game) {
		cd->game->quit_players++;
		pthread_cond_broadcast(&cd->game->cond_quit);

		if (!cd->game->killed) {
			kill_game(cd->game);
		}

		if (cd->game->quit_players == 1) {
			/* It's us who quit the game */
			do_log(cd, "killing game %d (%d players)\n",
				cd->game->id, cd->game->num_players,
				get_nb_missing_players(cd->game));
		} else {
			do_log(cd, "did quit the game.\n");
		}

		if (cd->am_initiator)
		{
			/* Only the initiator finalizes the game kill, for
			 * obvious double-free reasons */
			do_log(cd, "waiting for %d quit acks before finalizing "
				"game %d kill\n",
				cd->game->num_players 
				- get_nb_missing_players(cd->game)
				- cd->game->quit_players,
				cd->game->id);
			while (cd->game->quit_players < cd->game->num_players 
				- get_nb_missing_players(cd->game))
			{
				/* Wait for all others to be gone */
				pthread_cond_wait(&cd->game->cond_quit, &gamelist_mutex);
			}
			do_log(cd, "finalizing game %d kill\n", cd->game->id);

			gamelist = llist_remove(gamelist, cd->game);
			free_game(cd->game);
			cd->game = NULL;
		}
	}
	pthread_mutex_unlock(&gamelist_mutex);
	return CONN_EXIT;
}

static ConnState server_handle_state(ConnData *cd)
{
	switch (cd->state)
	{
	case CONN_NEW:
		return server_handle_conn_new(cd);
	case CONN_SETUP_NEW:
		return server_handle_conn_setup_new(cd);
	case CONN_SETUP_LISTGAME:
		return server_handle_conn_setup_listgame(cd);
	case CONN_SETUP_PRELJOIN:
		return server_handle_conn_setup_preljoin(cd);
	case CONN_SETUP_JOIN:
		return server_handle_conn_setup_join(cd);
	case CONN_WAIT_PLAYERS:
		return server_handle_conn_wait_players(cd);
	case CONN_START_PLAY:
		return server_handle_conn_start_play(cd);
	case CONN_PLAY:
		return server_handle_conn_play(cd);
	case CONN_PLAY_RECEIVE:
		return server_handle_conn_play_receive(cd);
	case CONN_PLAY_SEND_MOVE:
		return server_handle_conn_play_send(cd, 0);
	case CONN_PLAY_SEND_QUIT:
		return server_handle_conn_play_send(cd, 1);
	case CONN_KILL:
		return server_handle_conn_kill(cd);
	default:
		return CONN_EXIT;
	}
}

static void *client_conn_handler(void *data)
{
	ConnData *cd = (ConnData *)data;

	do_log(cd, "New connection\n", cd);

	while (cd->state != CONN_EXIT)
		cd->state = server_handle_state(cd);
	
	do_log(cd, "Shutting down connection\n");
	shutdown(cd->fd, SHUT_RDWR);
	close(cd->fd);
	free(cd->addr);
	free(cd);

	pthread_exit(NULL);
}

int main(int argc, char *argv[])
{
	int portno = 8000;
	int sockfd, clifd;
	unsigned int clilen;
	struct sockaddr_in serv_addr, cli_addr;
	pthread_t threadid;
	int optOn = 1;
	
	if (argc > 1) {
		logfp = fopen(argv[1], "wb");
		if (logfp == NULL) {
			printf("can't open %s\n", argv[1]);
			logfp = stdout;
		}
	} else
		logfp = stdout;
	
	sockfd = socket(AF_INET, SOCK_STREAM, 0);
	if (sockfd < 0) {
        	perror("Error on opening socket");
		exit(1);
	}
	memset((char *) &serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = INADDR_ANY;
	serv_addr.sin_port = htons(portno);
	setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR,
			&optOn, sizeof(optOn));
	if (bind(sockfd, (struct sockaddr *) &serv_addr,
		sizeof(serv_addr)) < 0) {
		perror("Error on binding");
		exit(1);
	}
	
	listen(sockfd, 1000);
	
	pthread_mutex_init(&gamelist_mutex, NULL);
	
	clilen = sizeof(cli_addr);
	while ((clifd = accept(sockfd,
			      (struct sockaddr *)&cli_addr,
			      &clilen)) >= 0) {

		ConnData *cd = conn_data_init(clifd, inet_ntoa(cli_addr.sin_addr));

		if (cd != NULL) {
			pthread_create(&threadid, NULL, client_conn_handler,
				       (void *)cd);
			pthread_detach(threadid);
		}

		clilen = sizeof(cli_addr);	      
	}
	close(sockfd);
	pthread_mutex_destroy(&gamelist_mutex);
	return 0;
}
