/*
 * Copyright (c) 2005-2009 Jeremy Erickson
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of the Software, to omit the above copyright notice and
 * permission notice, but to nonetheless exercise the rights granted
 * therein, in copies or substantial portions of the Software which are
 * solely in the form of machine-executable object code generated by a
 * source language processor.
 */

#ifdef Q_WS_WIN
    #define DECORATION extern "C" __declspec(dllexport)
#else
    #define DECORATION extern "C"
#endif

#define TRUE 1
#define FALSE 0

/** The testaments, as used by uBookType.
  */
enum Testament{
    OLD_TESTAMENT = 1,
    NEW_TESTAMENT = 64
};

/** Used for uBooks or "Universal Books".
  * The purpose of uBooks is to make plugins
  * interact with each other, and to allow
  * sorting of the books.
  */
typedef struct{
    int testament;
    int book;
} uBookType;

/** Perform any initialization.
  * This function must be called
  * before a plugin is used.
  */
DECORATION void initialize();

/** Clean up and/or finalize
  * This function must be called
  * when a plugin is unloaded.
  */
DECORATION void finalize();

/** Gets the list of books.
  * @param numBooks Pointer to an integer which
  * will be set to the number of books.
  * @param translation The translation to obtain
  * the list for (e.g. one might be NT-only or
  * have apocryphal books).  Should be in UTF-8.
  * An empty string should return a default list.
  * @return An array of char arrays with the books
  * encoded in UTF-8.
  */
DECORATION const char * const * getBooks(int * numBooks, 
        const char * translation);

/** Gets whether the books changed between two translations.
  * @param oldTranslation The translation last used when
  * obtaining books.  Should be in UTF-8.
  * @param newTranslation The next translation to check for
  * a change in book list.  Should be in UTF-8.
  */
DECORATION int booksChanged(const char * oldTranslation,
    const char * newTranslation);

/** Converts a uBook to a book name.
  * @param uBook The uBook to convert.
  * @return The name of the book, in UTF-8.
  */
DECORATION const char * book(uBookType uBook);

/** Converts a book name to a uBook.
  * @param book The name of the book to convert, in UTF-8.
  * @return The equivalent uBook.
  */
DECORATION const uBookType uBook(const char * book);

/** Queries whether the module provides a list of available translations.
  * This is likely to be the case when there is a verse loader, even
  * if there is only one translation.
  * @return 1 if a list is available, 0 if not.
  */
DECORATION int translationsAvailable();

/** Gets the list of translations.
  * @param numTrans Pointer to an integer which will be set to
  * the number of translations.
  * @return An array of char arrays in UTF-8 naming available
  * translations, or an empty array if no translation list is 
  * available.
  */
DECORATION const char * const * getTranslations(int * numTrans);

/** Gets the number of chapters in a book.
 *  Only one parameter needs to be provided; the
 *  plugin should do something sensible.
 *  @param book The name of the book, in UTF-8.
 *  @param uBook The uBook.
 *  @return The number of chapters in the book.
 */
DECORATION int getNumChapters(const char * book, uBookType uBook);

/** Queries whether verses can be loaded through the module.
  * @return 1 if a verse loader is available, 0 if not.
  */
DECORATION int verseLoaderAvailable();

/** Obtains the text of a verse.
  * @param book The name of the book, encoded in UTF-8.
  * May be blank or invalid if a uBook is provided.
  * @param uBook The uBook.  May be blank or invalid if
  * a book is provided.
  * @param chapter The chapter number of the verse.
  * @param verses The range or list of verses within the chapter.
  * @param translation The translation to load from.
  * @return The text of the verse, encoded in UTF-8.
  */
DECORATION const char * getVerse(const char * book, uBookType uBook,
                                const char * chapter, const char * verses,
                                const char * translation);

/** Quieries whether the text of a particular verse can be obtained.
  * @param book The name of the book, encoded in UTF-8.
  * May be blank or invalid if a uBook is provided.
  * @param uBook The uBook.  May be blank or invalid if
  * a book is provided.
  * @param chapter The chapter number of the verse.
  * @param verses The range or list of verses within the chapter.
  * @param translation The translation to load from.
  * @return 1 if the verse is available, 0 if not.
  */
DECORATION int verseAvailable(const char * book, uBookType uBook,
                    const char * chapter, const char * verses,
                    const char * translation);

/** Quieries how many verses will be returned by the current request.
  * (A return value of 101 means >100)
  * @param book The name of the book, encoded in UTF-8.
  * May be blank or invalid if a uBook is provided.
  * @param uBook The uBook.  May be blank or invalid if
  * a book is provided.
  * @param chapter The chapter number of the verse.
  * @param verses The range or list of verses within the chapter.
  * @param translation The translation to load from.
  * @return Number of verses to return, or 101 for >100, or 0 if not available.
  */
DECORATION int verseCount(const char * book, uBookType uBook,
                    const char * chapter, const char * verses,
                    const char * translation);
