/****************************************************************
**
** Attal : Lords of Doom
**
** game.cpp
** Manages the whole game
**
** Version : $Id: game.cpp,v 1.230 2008/03/03 22:50:10 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo - Forest Darling
**
** Date : 17/08/2000
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "game.h"

// generic include files
#include <assert.h>

// include files for QT
#include <QHBoxLayout>
#include <QHostAddress>
#include <QInputDialog>
#include <QLabel>
#include <QLayout>
#include <QMessageBox>
#include <QPixmap>
#include <QVBoxLayout>

// application specific includes
#include "conf.h"
#include "libCommon/artefactManager.h"
#include "libCommon/attalSettings.h"
#include "libCommon/attalSocket.h"
#include "libCommon/calendar.h"
#include "libCommon/dataTheme.h"
#include "libCommon/genericEvent.h"
#include "libCommon/genericInsideBuilding.h"
#include "libClient/graphicalPath.h"
#include "libCommon/pathFinder.h"
#include "libCommon/technic.h"
#include "libCommon/unit.h"
#include "libCommon/priceMarket.h"
#include "libCommon/attalTemplates.h"

//#include "libFight/fightUnit.h"

#include "libClient/askChest.h"
#include "libClient/askDialog.h"
#include "libClient/bonus.h"
#include "libClient/building.h"
#include "libClient/cell.h"
#include "libClient/chatWidget.h"
#include "libClient/chest.h"
#include "libClient/displayLord.h"
#include "libClient/displayBase.h"
#include "libClient/event.h"
#include "libClient/gainLevel.h"
#include "libClient/gameInfo.h"
#include "libClient/graphicalArtefact.h"
#include "libClient/gui.h"
#include "libClient/imageTheme.h"
#include "libClient/lord.h"
#include "libClient/lordExchange.h"
#include "libClient/mapCreature.h"
#include "libClient/mapView.h"
#include "libClient/miniMap.h"


extern QString DATA_PATH;
extern QString IMAGE_PATH;
extern QString VERSION;
extern TechnicList techList;
extern Map * theMap;

Game::Game( QWidget * parent , const char * name )
  : QWidget( parent, Qt::MSWindowsFixedSizeDialogHint ),
	  GraphicalGameData()
{
	setWindowTitle( QString ( name ));
	theMap = new Map( this );
	_map = (GenericMap *)theMap;
	_isPlaying = false;
	_socket = NULL;
	_dispLord = NULL;
	_currentCell = NULL;
	_lordExchange = NULL;
	_popup = NULL;
	_period = 40;
	_idTime = -1;

	initWidgets();

	_player = new Player( this, _map );
	_player->setConnectionName( AttalSettings::getInstance()->getStrategyModeSettings().connectionName );
	_control->setPlayer( _player );
	_scrLord->setPlayer(_player);
	_scrBase->setPlayer(_player);
	_scrLord->reinit();
	_scrBase->reinit();

	GraphicalGameData::reinit();
	_control->disableGame();
	_scrLord->setEnabled( false );
	_scrBase->setEnabled( false );
	_control->reinit();

	connect( _chat, SIGNAL( sig_message( QString ) ), SLOT( slot_message( QString ) ) );
	connect( this, SIGNAL( sig_newMessage( QString ) ), _chat, SLOT( slot_displayMessage( QString ) ) );

	connect( _scrLord, SIGNAL( sig_lord() ), SLOT( slot_displayLord() ) );
	connect( _scrLord, SIGNAL( sig_lordSelected() ), SLOT( slot_lordSelected() ) );
	connect( this, SIGNAL( sig_lordReinit() ), _scrLord , SLOT( slot_reinit() ) );

	connect( _scrBase, SIGNAL( sig_base() ), SLOT( slot_displayBase() ) );
	connect( _scrBase, SIGNAL( sig_baseSelected() ), SLOT( slot_baseSelected() ) );
	connect( this, SIGNAL( sig_baseReinit() ), _scrBase ,SLOT( slot_reinit() ) );

	connect( _control, SIGNAL( sig_endTurn() ), SLOT( slot_endTurn() ) );
	connect( _control, SIGNAL( sig_endTurn() ), SIGNAL( sig_endTurn() ) );

	connect( _view , SIGNAL( sig_mouseMoved  ( GenericCell * ) ), SLOT( slot_mouseMoved( GenericCell * ) ) );
	connect( _view , SIGNAL( sig_mouseLeftPressed( GenericCell * ) ), SLOT( slot_mouseLeftPressed( GenericCell * ) ) );
	connect( _view , SIGNAL( sig_mouseRightPressed( GenericCell * ) ), SLOT( slot_mouseRightPressed( GenericCell * ) ) );
	connect( _view, SIGNAL( sig_mouseReleased() ), SLOT( slot_mouseReleased() ) );
	connect( _view, SIGNAL( sig_viewportResized( double, double ) ), _miniMap, SLOT( slot_mapviewResized( double, double ) ) );
	connect( _view, SIGNAL( sig_viewportScrolled( double, double ) ), _miniMap, SLOT( slot_mapviewScrolled( double, double ) ) );
	connect(  this , SIGNAL( sig_Center( int , int ) ), _view , SLOT( slot_Center( int ,int ) ) );
	
	connect( theMap , SIGNAL( sig_cellChanged( int, int )), _miniMap, SLOT( slot_redrawCell( int , int ) ) );
	connect( this , SIGNAL( sig_cellChanged( int, int )), _miniMap, SLOT( slot_redrawCell( int , int ) ) );
	connect( _miniMap , SIGNAL( sig_mouseReleasedMinimap( double, double ) ), _view, SLOT( slot_Center( double, double ) ) );
}

Game::~Game()
{
	TRACE("~Game");

	delete _gameInfo;
	delete _baseInfo;
	delete _lordInfo;
	delete _player;
	
	if( _layControlH1 ) {
		delete _layControlH1;
	}
	if( _layControlV1 ) {
		delete _layControlV1;
	}

	if( _dispLord ) {
		delete _dispLord;
	}
	
	if( _lordExchange ) {
		delete _lordExchange;
	}
}

void Game::reinit()
{
	TRACE("Game::reinit");

	stopTimer();
	/* disable controls */
	_control->disableGame();
	_scrLord->setEnabled( false );
	_scrBase->setEnabled( false );
	_scrLord->deselect();
	_scrBase->deselect();

	/* clean up player data */
	_player->cleanData();
	/* redraw minimap */
	_miniMap->redrawMap( theMap );
	_state = MS_NOTHING;
	/* redraw various widgets  */
	if( _dispLord ) {
		delete _dispLord;
		_dispLord = NULL;
	}
	
	if( _lordExchange ) {
		delete _lordExchange;
		_lordExchange = NULL;
	}
	
	GraphicalGameData::reinit();
	
	_stackInfo->raiseInfo();
	/* these are to remove icon from buttons */
	_scrLord->reinit();
	_scrBase->reinit();
	emit sig_statusBar();

	ImageTheme.endMusic();

}

void Game::initWidgets()
{
	theMap->setSceneRect( 0,0, 4000, 3000 );

	_view = new MapView( theMap, this );
	_control = new GameControl( this );
	_scrLord = new ScrollLord( false, 4, this );
	_scrBase = new ScrollBase( false, 4, this );

	_layControlH1 = 0;
	
	_layControlV1 = new QVBoxLayout();	
	_layControlV1->addWidget( _scrLord, 1, Qt::AlignHCenter );
	_layControlV1->addWidget( _control, 0, Qt::AlignHCenter );
	_layControlV1->addWidget( _scrBase, 1, Qt::AlignHCenter );

	_layH1 = new QHBoxLayout();
	_layH1->addWidget( _view, 1 );
	_layH1->addLayout( _layControlV1 );

	_chat = new ChatWidget( this );

	//these 4 widget will be eliminated
	_gameInfo = new GameInfo( _calendar, 0 );
	_lordInfo = new LordInfo( 0 );
	_baseInfo = new BaseInfo( 0 );
	_stackInfo = new InfoStack( this );
	_stackInfo->init( _gameInfo, _lordInfo, _baseInfo );
	_stackInfo->hide();

	_miniMap = new MiniMap( theMap, this );

	QHBoxLayout * layH2 = new QHBoxLayout();
	layH2->addWidget( _chat, 1 );
	//layH2->addWidget( _stackInfo, 1 );
	layH2->addWidget( _miniMap );

	_layout = new QVBoxLayout( this );
	_layout->addLayout( _layH1 );
	_layout->addLayout( layH2 );
	_layout->activate();
}

void Game::setPlayerName( QString name ) 
{
	AttalSettings::StrategyModeSettings settings = AttalSettings::getInstance()->getStrategyModeSettings();
	settings.connectionName = name;
	_player->setConnectionName( name );
	AttalSettings::getInstance()->setStrategyModeSettings( settings );
}

void Game::slot_mouseMoved( GenericCell *cell )
{
	QString msg;
	
	if( cell != _currentCell ) {
		if( cell->getLord() ) {
			msg = tr( "Lord " ) + cell->getLord()->getName();
			setCursor( Qt::WaitCursor );
		} else if( cell->getBase() ) { 
			msg = cell->getBase()->getBaseDescription();
			setCursor( Qt::WaitCursor );
		} else if( cell->getBuilding() ) {
			msg = cell->getBuilding()->getName()+ QString(tr(" - ")) + DataTheme.buildings.at(cell->getBuilding()->getType())->getDescription();
			setCursor( Qt::WaitCursor );
		} else if( cell->getEvent() ) {
			setCursor( Qt::WaitCursor );
			switch(cell->getEvent()->getType()){
				case GenericEvent::EventNone:
				break;
				case GenericEvent::EventArtefact:
					msg =  tr("Artefact: ") + cell->getEvent()->getArtefact()->getName();
				break;
				case GenericEvent::EventBonus: {
					GenericBonus * bonus = cell->getEvent()->getBonus();
					msg = bonus->getBonusDescription();
				} break;
				case GenericEvent::EventChest:
					msg =  tr("Chest ");
				break;
			default:
				break;
			}
		} else if( cell->getCreature() ) {
			msg =  tr("About ") + QString::number(DataTheme.getRandomInCategory(cell->getCreature()->getCategoryNumber()))  + "  " + cell->getCreature()->getCreature()->getName();
			setCursor( Qt::WaitCursor );
		} else {
#ifdef QT_DEBUG
			setCursor( Qt::ArrowCursor );
			msg = QString(tr("Cell: row %1, col %2, coordinates x %3, y %4 ")).arg(cell->getRow()).arg(cell->getCol()).arg(cell->getCol() * DataTheme.tiles.getWidth()).arg( cell->getRow() * DataTheme.tiles.getHeight());
#endif
		}
		emit sig_statusMsg( msg );
	}
}


void Game::slot_mouseLeftPressed( GenericCell * cell )
{
	if( _popup && _popup->isVisible() ) {
		_popup->hide();
		_popup->setType();
		return;
	}

	if( !_player ) {
		return;
	}

	switch( _state ) {
	case MS_NOTHING:
		handleClickNothing( cell );
		break;
	case MS_LORD:
		handleClickLord( cell );
		break;
	case MS_BASE:
		handleClickBase( cell );
		break;
	case MS_TECHNIC:
		handleClickTechnic( cell );
		break;
	}
}

void Game::setState( MapState state )
{
	_state = state;
}

void Game::handleClickNothing( GenericCell * cell )
{
	GenericLord * lord = cell->getLord();
	
	if( lord ) {
		if( _player->hasLord( lord ) ) {
			_player->setSelectedLord( lord );
			slot_lordSelected();
		}
		return;
	}

	GenericBase * base = cell->getBase();
		
	if( base ) {
		if( _player->hasBase( base ) ) {
			_player->setSelectedBase( base );
			slot_baseSelected();
		}
		return;
	}
}

void Game::handleClickLord( GenericCell * cell )
{
	TRACE("Game::handleClickLord row %d col %d", cell->getRow(), cell->getCol() );

	GenericLord * selectedLord = _player->getSelectedLord();
	GraphicalPath * gpath = theMap->getGraphicalPath();
		
	if( selectedLord && ( cell->getCoeff() >= 0 ) ) {
		if(cell->getLord()){
			emit sig_Center( cell->getRow(), cell->getCol() );
		}
		if( ((Lord*)selectedLord)->getDestination() != cell ) {
			 gpath->computePath( selectedLord->getCell(), cell, selectedLord->getCharac( MOVE ) );
			 selectedLord->setDestination( cell );
		} else {
			if( selectedLord->getCell()->getBuilding() ) {
				selectedLord->getCell()->getBuilding()->out( selectedLord );
			}
			QList<GenericCell *> list = gpath->followPath( selectedLord->getCell(), selectedLord->getCharac( MOVE )) ;
			_socket->sendMvts( selectedLord->getId(), list );
		}
	}		
}

void Game::handleClickBase( GenericCell * cell )
{
	GenericBase * base = cell->getBase();
		
	if( base ) {
		if( _player->hasBase( base ) ) {
			if( base == _player->getSelectedBase() ) {
				emit sig_base( base );
				return;
			} else {
				_player->setSelectedBase( base );
				emit sig_baseReinit();
				emit sig_lordReinit();
				return;
			}
		} else {
			return;
		}
	}	
	
	handleClickNothing( cell );
}

void Game::handleClickTechnic( GenericCell * /*cell*/ )
{
	/// XXX: TODO
	logEE( "not yet implemented" );
}


void Game::slot_mouseRightPressed( GenericCell * cell )
{
	if( !_player ) {
		return;
	}

	if( !_popup ) {
		_popup = new AttalPopup( this );
	}
	
	
	QPoint point = (QPoint)(_view->mapFromScene(cell->getCol() * DataTheme.tiles.getWidth(), cell->getRow() * DataTheme.tiles.getHeight()));
	
	GenericLord * lord = cell->getLord();

	if( lord ) {
		_popup->setLord( lord );
	}

	GenericBuilding * build = cell->getBuilding();

	if( build ) {
		_popup->setBuilding( build );
	}

	GenericBase * base = cell->getBase();

	if( base ) {
		_popup->setBase( base );
	}

	GenericEvent * event = cell->getEvent();

	if( event ) {
		_popup->setEvent( event );
	}

	GenericMapCreature * crea = cell->getCreature();

	if( crea ) {
		_popup->setCreature( crea );
	}

	_popup->move( point );
	_popup->show();
}

void Game::slot_mouseReleased()
{

}

void Game::enter( GenericLord * /*lord*/, GenericBuilding * /*building*/ )
{
	/// XXX: TODO
	//logEE( "not yet implemented" );
}

void Game::enter( GenericLord * lord, GenericBase * base )
{
	TRACE("Game::enter lord %p base %p", lord, base);

	if( base ) {
		base->enter( lord );
		emit sig_base( base );
	}
}

void Game::beginTurn()
{
	_isPlaying = true;
	emit sig_statusBar();
	_gameInfo->nothing();
	_gameInfo->setStatePlayer( true );
	if( _calendar->getDay() == 1 ) {
		QMessageBox::information( this, tr( "Monday" ), tr( "It is " ) + _calendar->getDayName() + tr(", a new week is beginning"));
	}
	_player->newTurn();
	ImageTheme.playSound( AttalSound::SND_NEWTURN );
	_state = MS_NOTHING;

	if( _scrLord->getListCount() > 0 ) {
		_scrLord->select( 0 );
	} else if( _scrBase->getListCount() > 0 ) {
		_scrBase->select( 0 );
	}
	emit sig_beginTurn();
}

void Game::playerActive( char num )
{
	_gameInfo->waitPlayer( num );
}

void Game::nextLord()
{ 
	_player->nextLord(); 
	slot_lordSelected();
}
 
void Game::nextBase()
{ 
	_player->nextBase();
	slot_baseSelected(); 
}

void Game::endTurn()
{
	assert( _socket );
	if( _isPlaying ) {
		if( _player->shouldEnd() ) {
			_socket->sendTurnEnd();
			_isPlaying = false;
			_gameInfo->setStatePlayer( _isPlaying );
		} else {
			QMessageBox msb( tr("Are you sure ?"), tr( "One or more heroes may still move. Are you sure you want to end your turn ?"), QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
			if ( msb.exec() == QMessageBox::Yes ) {
				_socket->sendTurnEnd();
				_isPlaying = false;
				_gameInfo->setStatePlayer( _isPlaying );
			}
		}
	}
}

void Game::beginGame( int nb )
{
	TRACE("Game::beginGame int nb %d", nb);

	restartTimer();
	setPlayerNumber( nb );
	_control->enableGame();
	_scrLord->enableGame();
	_scrBase->enableGame();
	ImageTheme.playMusicMap();
}

void Game::endGame()
{
	TRACE("Game::endGame");

	reinit();
}

void Game::handleSocket()
{
	switch( _socket->getCla1() ) {
	case SO_MSG:
		socketMsg();
		break;
	case SO_GAME:
		socketGame();
		break;
	case SO_TURN:
		socketTurn();
		break;
	case SO_MODIF:
		socketModif();
		break;
	case SO_QR:
		socketQR();
		break;
	case SO_MVT:
		socketMvt();
		break;
	case SO_TECHNIC:
		break;
	case SO_EXCH:
		socketExchange();
		break;
	case SO_CONNECT:
		socketConnect();
		break;
	case SO_FIGHT:
		socketFight();
		break;
	default:
		logEE( "Unknown socket_class" );
	}
}

void Game::socketMsg()
{
	/*int cla2 = */ _socket->getCla2();
	QString msg;
	uchar len = _socket->readChar();
	for( uint i = 0; i < len; i++ ) {
		msg[i] = _socket->readChar();
	}
	emit sig_newMessage( msg );
}

void Game::slot_message( QString msg )
{
	if( _socket ) {
		_socket->sendMessage( _player->getConnectionName() + " : " + msg );
	} else {
		emit sig_newMessage( "(Not connected) : " + msg );
	}
}

void Game::socketGame()
{
	switch( _socket->getCla2() ) {
	case C_GAME_BEGIN:
		beginGame( _socket->readChar() );
		/// XXX: clear old stuff if necessary
		break;
	case C_GAME_LOST:
		socketGameLost();
		break;
	case C_GAME_WIN:
		socketGameWin();
		break;
	case C_GAME_END:
		emit sig_endGame();
		break;
	case C_GAME_INFO:
		socketGameInfo();
		break;
	case C_GAME_CALENDAR:
		socketGameCalendar();
		break;
	case C_GAME_TAVERN:
		break;
	default:
		logEE( "case not handled" );
		break;
	}
}

void Game::socketGameLost()
{
	QString text;
	AttalMessage msg;

	int nb = _socket->readChar();
	if( nb == _player->getNum() ) {
		msg.setWindowTitle( tr( "You lose") );
		msg.addText(tr( " You lose") );
		msg.addPixmap( ImageTheme.getFlag( nb ) );
		msg.exec();
		emit sig_result( false );
	} else {
		text = tr("Player ") + QString::number( nb ) + tr(" has lost.");
		msg.setWindowTitle( tr("A player has lost.") );
		msg.addText( text );
		msg.addPixmap( ImageTheme.getFlag( nb ) );
		msg.exec();
	}
}

void Game::socketGameWin()
{
	QString text;
	AttalMessage msg;

	int nb = _socket->readChar();

	msg.setWindowTitle( tr("A player has Win.") );
	
	if( nb == _player->getNum() ) {
		msg.addText( tr("You  win !!") );
		msg.addPixmap( ImageTheme.getFlag( nb ) );
		msg.exec();
		emit sig_result( true );
	} else {
		text = tr("Player ") + QString::number( nb ) + tr(" has win.");
		msg.addText( text + tr(" You lose") );
		msg.addPixmap( ImageTheme.getFlag( nb ) );
		msg.exec();
		//emit sig_result( false );
	}
}

void Game::socketGameInfo()
{
	switch( _socket->getCla3() ) {
	case C_INFOPLAYER_TEAM: 
		socketGameInfoTeam();
		break;
	case C_INFOPLAYER_NAME:
		break;
	}
}

void Game::socketGameInfoTeam()
{
	uchar player = _socket->readChar();	
	uchar teamId = _socket->readChar();
	
	TRACE("Game::GameInfoTeam player %d, teamId %d", player, teamId);

	if( getPlayer( player ) ) {
		getPlayer( player )->setTeam( teamId );
	}
	if( player == _player->getNum() ) {
		_player->setTeam( teamId );
	}
}

void Game::socketGameCalendar()
{
	_calendar->setDateByType( 0 , _socket->readInt());
	_calendar->setDateByType( 1 , _socket->readInt());
	_calendar->setDateByType( 2 , _socket->readInt());
	_calendar->setDateByType( 3 , _socket->readInt());
	_calendar->setDateByType( 4 , _socket->readInt());
	
	emit sig_statusBar();
}

void Game::socketTurn()
{
	switch( _socket->getCla2() ) {
	case C_TURN_PLAY:
		socketTurnPlay();
		break;
	case C_TURN_LORD:
		logEE( "Should not happen (Client : SO_TURN/C_TURN_LORD)" );
		break;
	case C_TURN_PLORD:
		logEE( "Should not happen (Client : SO_TURN/C_TURN_PLORD)" );
		break;
	case C_TURN_END:
		logEE( "Should not happen (Client : SO_TURN/C_TURN_END)" );
		break;
	}
}

void Game::socketTurnPlay()
{
	TRACE("Game::socketTurnPlay");

	uint num = _socket->readChar();
	if( num == (uint)_player->getNum() ) { // casting from int to uint -- getNum,setNum, and _num should all be using uint's
		beginTurn();
	} else {
		playerActive( num );
	}
}

void Game::socketMvt()
{
	// XXX: not finished at all
					
	TRACE("Game::socketMvt");

	if( _socket->getCla2() == C_MVT_ONE ) {
		uchar lord = _socket->readChar();
		int row = _socket->readInt();
		int col = _socket->readInt();
					
		TRACE("Game::socketMvt lord %d, row %d, col %d", lord, row, col);

		if(!_map->inMap(row,col)) {
			return;
		}

		GenericLord * theLord = _lords.at( lord );
		if( theLord ) {
			GenericCell * oldCell = theLord->getCell();

			theLord->setCell( _map->at( row, col ) );

			if( oldCell ) {
				int oldRow = oldCell->getRow();
				int oldCol = oldCell->getCol();
				emit sig_cellChanged( oldRow, oldCol );
			} else {
				logEE("Should be a cell");
			}

			emit sig_Center( row, col );

			if( _map->at( row, col )->getBuilding() != 0 ) {
				if( _lords.at( lord )->getOwner() == _player ) {
					enter( _player->getSelectedLord(), _map->at( row, col )->getBuilding() );
				}
			} else if( _map->at( row, col )->getBase() != 0 ) {
				if( theLord->getOwner() == _player ) {
					TRACE("Game::socketMvt: EnterBase");
					enter( _player->getSelectedLord(), _map->at( row, col )->getBase() );
				}
			}
			emit sig_cellChanged( row, col );
		} else {
			logEE( "Try to move a non-existent lord %d to cell (%d, %d)", lord, row, col );
		}
	} else {
		logEE( "Should not happen" );
	}
}

void Game::socketExchange()
{
	switch( _socket->getCla2() ) {
	case C_EXCH_START: 
		exchangeStart();
		break;
	case C_EXCH_UNIT:
		exchangeUnits();
		break;
	case C_EXCH_ARTEFACT:
		exchangeArtefact();
		break;
	case C_EXCH_BASEUNITCL:
		exchangeBaseUnits();
		break;
	default:
		break;
	}
}

void Game::exchangeStart()
{
	uchar idLord1 = _socket->readChar();
	uchar idLord2 = _socket->readChar();
	GenericLord * lord1 = (GenericLord * ) _lords.at( idLord1 );
	GenericLord * lord2 = (GenericLord * ) _lords.at( idLord2 );

	if( _lordExchange == NULL ) {
		_lordExchange = new LordExchange( this, _socket );
	}
	_lordExchange->initLords( lord1, lord2 );
	_lordExchange->show();

}

void Game::exchangeUnits()
{
	/*uchar idLord1 = */_socket->readChar();
	/*uchar idUnit1 = */_socket->readChar();
	/*uchar idLord2 = */_socket->readChar();
	/*uchar idUnit2 = */_socket->readChar();

	logEE("Should not happen");
}

void Game::exchangeArtefact()
{
	uchar idLord1 = _socket->readChar();
	int item = _socket->readInt();
	uchar idLord2 = _socket->readChar();
	
	exchangeArtefactLord( idLord1, idLord2, item);

	updateWindows();
}

void Game::exchangeBaseUnits()
{
	/*int row = */_socket->readInt();
	/*int col = */_socket->readInt();
	/*uchar idUnit1 = */_socket->readChar();
	/*uchar idLord = */_socket->readChar();
	/*uchar idUnit2 =*/ _socket->readChar();
}



void Game::socketModif()
{
	/// XXX: check number of args below... (?)

	switch( _socket->getCla2() ) {
	case C_MOD_MAP:
		socketModifMap();
		break;
	case C_MOD_CELL:
		socketModifCell();
		break;
	case C_MOD_LORD:
		socketModifLord();
		break;
	case C_MOD_PLAYER:
		socketModifPlayer();
		break;
	case C_MOD_BASE:
		socketModifBase();
		break;
	case C_MOD_BUILD:
		socketModifBuilding();
		break;
	case C_MOD_ARTEFACT:
		socketModifArtefact();
		break;
	case C_MOD_CREATURE:
		socketModifCreature();
		break;
	case C_MOD_EVENT:
		socketModifEvent();
		break;
	}
}

void Game::socketModifMap()
{
	int h = _socket->readInt();
	int w = _socket->readInt();
	TRACE("Game::socketModifMap h/w %d/%d", h, w );
	_map->newUnknownMap( h, w );
	_miniMap->redrawMap( theMap );
	_miniMap->slot_mapviewResized( _view->width() / theMap->width(), _view->height() / theMap->height() ); // HACK
	layout()->update();
}

void Game::socketModifCell()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	int type = _socket->readInt();
	uchar diversification = _socket->readChar();
	int a4 = _socket->readInt();
	int a5 = _socket->readInt();
	int a6 = _socket->readInt();
	int a7 = _socket->readInt();
	//TRACE("Game::socketModifCell row %d, col %d, type %d, diversification %d, a4 %d, a5 %d, a6 %d, a7 %d", row, col, type, diversification, a4, a5, a6, a7 );
	_map->changeCell( row, col, type, a4, a5, a6, a7 , diversification );
}

void Game::socketModifLord()
{
	switch( _socket->getCla3() ) {
	case C_LORD_VISIT:
		socketModifLordVisit();
		break;
	case C_LORD_NEW:
		socketModifLordNew();
		break;
	case C_LORD_CHARAC:
		socketModifLordCharac();
		break;
	case C_LORD_UNIT:
		socketModifLordUnit();
		break;
	case C_LORD_REMOVE:
		socketModifLordRemove();
		break;
	case C_LORD_GARRISON:
		socketModifLordGarrison();
		break;
	case C_LORD_MACHINE:
		socketModifLordMachine();
		break;
	}
}

void Game::socketModifLordVisit()
{
	/// XXX: not finished, we should use 'num' for the player color
	uchar num = _socket->readChar();
	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar id = _socket->readChar();
	uchar present = _socket->readChar();

	Lord * lord = (Lord *) _lords.at( id );
	if( present == 1 ) {
		lord->setAnimated( true );
		lord->setEnabled( true );
		lord->setSelected( true );
		//lord->setActive( true );
		/// XXX: change
		lord->setOwner( getPlayer( num ) );
		lord->setCell( _map->at( row, col ) );
	} else {
		if( lord->getCell() ) {
			lord->getCell()->setLord(NULL);
		}
	}
}

void Game::socketModifLordNew()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar id = _socket->readChar();

	TRACE("Game::socketModifLordNew row %d, col %d, id %d", row, col ,id );

	Lord * lord = (Lord *) _lords.at( id );
	lord->setAnimated( true );
	lord->setEnabled( true );
	lord->setVisible( true );
	lord->setSelected( true );
	//lord->setActive( true );
	lord->setCell( _map->at( row, col ) );
	lord->setOwner( _player );
	_player->addLord( lord );

	GenericBase * base = _map->at( row, col )->getBase();
	if( base && base->getOwner() == _player ) {
		enter( lord , base );
	}
	emit sig_lordReinit();
	emit sig_cellChanged( row, col );
}

void Game::socketModifLordUnit()
{
	uchar id = _socket->readChar();
	uchar pos = _socket->readChar();
	uchar race = _socket->readChar();
	uchar level = _socket->readChar();
	int nb = _socket->readInt();
	uchar move = _socket->readChar();
	int health = _socket->readInt();

	updateLordUnit( id , pos ,race ,level, nb,move, health);
	
	updateWindows();
	emit sig_exchange();
}

void Game::socketModifLordRemove()
{
	int idLord = _socket->readChar();
	TRACE("Game::socketModifLordRemove idLord %d", idLord );
	Lord * lord = ( Lord *)_lords.at(  idLord );
	if(lord) {
		lord->removeFromGame();
		emit sig_lordReinit();
		GraphicalPath * gpath = theMap->getGraphicalPath();
		gpath->clearPath();	
		lord->setVisible( false );
	}
	if(_player->getSelectedLord()){
		_lordInfo->init( _player->getSelectedLord() );
	}

}
		
void Game::socketModifLordCharac()
{
	char lord = _socket->readChar();
	char charac = _socket->readChar();
	int nb = _socket->readInt();
	
	if(_lords.at( lord )) {
		_lords.at( lord )->setBaseCharac( (LordCharac) charac, nb );
	}
}

void Game::socketModifLordGarrison()
{
	uchar lord = _socket->readChar();
	uchar state = _socket->readChar();
	if(_lords.at( lord )) {
		_lords.at( lord )->setVisible( state != 1 );
	}

	emit sig_lordReinit();
}

void Game::socketModifLordMachine()
{
	uchar lord = _socket->readChar();
	uchar id = _socket->readChar();
	if(_lords.at( lord )) {
		_lords.at( lord )->addMachine( id );
	}
}

void Game::socketModifPlayer()
{
	switch( _socket->getCla3() ) {
	case C_PLAY_RESS: {
		socketModifRess( _player , _socket);
		emit sig_statusBar();
										}
		break;
	case C_PLAY_PRICE: {
		uchar ress = _socket->readChar();
		uint price = _socket->readInt();
		_player->getPriceMarket()->setResourcePrice( ress, price);
		}
		break;
		/*
	case C_PLAY_POPUL:
		socketModifPlayerPopulation();
		break;
	*/
	}
}

/*
void Game::socketModifPlayerPopulation()
{
	uint population = _socket->readInt();

	TRACE("Game::socketModifPlayer population = %d", population);

	_player->setPopulation( population );
	emit sig_statusBar();

}
*/

void Game::socketModifBase()
{
	switch( _socket->getCla3() ) {
	case C_BASE_NEW:
		socketModifBaseNew();
		break;
	case C_BASE_OWNER:
		socketModifBaseOwner();
		break;
	case C_BASE_NAME:
		socketModifBaseName();
		break;
	case C_BASE_BUILDING:
		socketModifBaseBuilding();
		break;
	case C_BASE_UNIT:
		socketModifBaseUnit();
		break;
	case C_BASE_POPUL:
		socketModifBasePopulation();
		break;
	case C_BASE_RESS:
		socketModifBaseResources();
		break;
	case C_BASE_PRODUCTION:
		socketModifBaseProduction();
		break;
	}
}

void Game::socketModifBaseNew()
{
	uchar race = _socket->readChar();
	int row = _socket->readInt();
	int col = _socket->readInt();
	int population = _socket->readInt();
	uchar id = _socket->readChar();
	int nb = _socket->readChar();
	QList<uchar> forbidlist;
	
	for( int i = 0; i < nb; i++ ) {
		forbidlist.append( _socket->readChar() );
	}

	getNewBase(	race, row, col, population,	id,	nb,	forbidlist);
}

void Game::socketModifBaseOwner()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar playerNum = _socket->readChar();
	GenericBase * base = _map->at( row, col )->getBase();
	
	TRACE("Game::socketModifBaseOwner row %d, col  %d, player num %d", row,col,playerNum);

	if( playerNum == _player->getNum() ) {
		_player->addBase( base );
		base->setOwner( _player );
	} else {
		/// XXX: improve management of base of other player
		if( base->getOwner() == _player ) {
			_player->removeBase( base );
		}
		base->setOwner( getPlayer( playerNum ) );
	}
	emit sig_baseReinit();
}

void Game::socketModifBaseName()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uint length = _socket->readInt();
	QString name;
	
	for( uint i = 0; i < length; i++ ) {
		name[i] = _socket->readChar();
	}
	
	GenericBase * base = _map->at( row, col )->getBase();
	base->setName( name );
}

void Game::socketModifBaseBuilding()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar level = _socket->readChar();
	bool create = (bool)_socket->readChar();

	updateBaseBuilding( row, col, level,create );

}

void Game::socketModifBaseUnit()
{
	TRACE("Game::socketModifBaseUnit");

	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar pos = _socket->readChar();
	uchar race = _socket->readChar();
	uchar level = _socket->readChar();
	int number = _socket->readInt();

	if( _map->at( row, col )->getBase() ) {
		GenericBase * base = _map->at( row, col )->getBase();
		base->addUnit( pos, race, level, number );
	}
	emit sig_exchange();
}

void Game::socketModifBasePopulation()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uint popul = _socket->readInt();
	GenericBase * base = _map->at( row, col )->getBase();

	if(base){
		base->setPopulation(popul);	
	}
	if(_player->getSelectedBase()){
		_baseInfo->init( _player->getSelectedBase() );
	}

}

void Game::socketModifBaseResources()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	GenericBase * base = _map->at( row, col )->getBase();
	
	TRACE("Game::socketModifBaseResources row %d, col  %d", row,col );
	
	socketModifRess( base , _socket);

	emit sig_statusBar();
}

void Game::socketModifBaseProduction()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar race = _socket->readChar();
	uchar level = _socket->readChar();
	int number = _socket->readInt();
	
	TRACE("Game::socketModifBaseProduction row %d, col  %d, race %d, level %d, number %d", row,col, race, level, number );
	

	if( _map->at( row, col )->getBase() ) {
		TRACE("base");
		GenericBase * base = _map->at( row, col )->getBase();
		Creature * creature = DataTheme.creatures.at( race, level );
		base->setCreatureProduction( creature, number );
	}

}



void Game::socketModifBuilding()
{
	switch( _socket->getCla3() ) {
	case C_BUILD_NEW: 
		socketModifBuildingNew();
		break;
	case C_BUILD_OWNER: 
		socketModifBuildingOwner();
		break;
	case C_BUILD_RESS:
		socketModifBuildingResources();
		break;
	}
	
}

void Game::socketModifBuildingNew()
{
	uchar type = _socket->readChar();
	int id = _socket->readInt();
	int row = _socket->readInt();
	int col = _socket->readInt();
	
	getNewBuilding( type, id , row, col);
}

void Game::socketModifBuildingOwner()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	int playNum = _socket->readInt();
	
	TRACE("Game::socketModifBuildingOwner row %d, col  %d,player num %d", row,col,playNum);

	Building * build = (Building*)_map->at( row, col )->getBuilding();
	if(  playNum == _player->getNum() ) {
		if( build ) {
			_player->addBuilding( (GenericBuilding *) build );
			build->setOwner( _player );
		}
	} else {
		if( build ) {
			if(build->getOwner() == _player){
				_player->removeBuilding( (GenericBuilding *) build );
			}
			if( playNum == GenericPlayer::NO_PLAYER ) {
				build->setOwner( 0 );
			} else {
				build->setOwner( getPlayer(playNum) );
			}
		}
	}
}

void Game::socketModifBuildingResources()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	
	TRACE("Game::socketModifBuildingOwner row %d, col  %d", row,col);
	
	Building * build = (Building*)_map->at( row, col )->getBuilding();
	
	socketModifRess( build , _socket);
}

void Game::socketModifArtefact()
{
	switch( _socket->getCla3() ) {
		case C_ART_DELLORD: {
			uint type = _socket->readInt();
			char lord = _socket->readChar();
			getLord( lord )->getArtefactManager()->removeArtefactByType( type );
		}
		break;
		case C_ART_ADDLORD: {
			ImageTheme.playSound( AttalSound::SND_GOOD );
			uint type = _socket->readInt();
			char lord = _socket->readChar();
			if( ! getLord( lord )->getArtefactManager()->hasArtefactType( type ) ) {
				getLord( lord )->getArtefactManager()->addArtefact( type );
			}
		}
		break;
	}
}

void Game::socketModifCreature()
{
	switch( _socket->getCla3() ) {
	case C_CRE_NEW: 
		socketModifCreatureNew();
		break;
	case C_CRE_UPDATE:
		socketModifCreatureUpdate();
		break;
	case C_CRE_DEL:
		socketModifCreatureRemove();
		break;
	case C_CRE_RESS: 
		socketModifCreatureRess();
		break;
	}
}
		
void Game::socketModifCreatureNew()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar race = _socket->readChar();
	uchar level = _socket->readChar();
	int nb = _socket->readInt();
	bool looking = (bool) _socket->readChar();
	
	getNewMapCreature( row, col, race, level, nb, looking);

}

void Game::socketModifCreatureUpdate()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	int nb = _socket->readInt();
	GenericMapCreature * creature = _map->at( row, col )->getCreature();
	if( creature ) {
		creature->setCategoryNumber( nb );
	}
}

void Game::socketModifCreatureRemove()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	
	removeMapCreature( row, col );
}
	
void Game::socketModifCreatureRess()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	GenericMapCreature * creature = _map->at( row, col )->getCreature();
	
	socketModifRess( creature , _socket);
}

void Game::socketModifEvent()
{
	switch( _socket->getCla3() ) {
	case C_EVENT_NEW:
		socketEventNew();
		break;
	case C_EVENT_DEL: 
		socketEventDel();
		break;
	}
}

void Game::socketEventNew()
{
	uint i;
	int row = _socket->readInt();
	int col = _socket->readInt();
	GenericEvent::EventType type = (GenericEvent::EventType) _socket->readChar();

	GenericEvent * event = NULL;

	if( type == GenericEvent::EventArtefact ) {
		int id = _socket->readInt();
		uchar typeArtefact = _socket->readChar();

		event = getNewArtefact( id );
		GenericArtefact * artefact = event->getArtefact();
		artefact->setType( typeArtefact );
	} else if( type == GenericEvent::EventBonus ) {
		uchar typeBonus = _socket->readChar();
		uchar nbParam = _socket->readChar();

		event = getNewBonus();
		GenericBonus * bonus = event->getBonus();
		bonus->setType( (GenericBonus::BonusType) typeBonus );
		for( i = 0; i < nbParam; i++ ) {
			bonus->addParam( _socket->readInt() );
		}
		bonus->setupBonus();
	} else if( type == GenericEvent::EventChest ) {
		uchar nbParam = _socket->readChar();

		event = getNewChest();
		GenericChest * chest = event->getChest();
		for( i = 0; i < nbParam; i++ ) {
			chest->addParam( _socket->readInt() );
		}
		dynamic_cast<Chest*>(chest)->setupChest();
	}

	event->setCell( _map->at( row, col ) );
	_map->at( row, col )->setEvent( event );
}

void Game::socketEventDel()
{
	int row = _socket->readInt();
	int col = _socket->readInt();

	removeEvent( row, col );
}

void Game::socketConnect()
{
	switch( _socket->getCla2() ) {
	case C_CONN_OK:
		socketConnectOk();
		break;
	case C_CONN_ID:
		socketConnectId();
		break;
	case C_CONN_NAME:
		socketConnectName();
		break;
	case C_CONN_PLAYER:
		break;
	}
}

void Game::socketConnectOk()
{
	TRACE("Game::socketConnectOk");

	_socket->sendConnectionVersion( VERSION  );
	_socket->sendConnectionName( _player->getConnectionName() );
	emit sig_newMessage( QString( tr("Connection established, %1") ).arg( _player->getConnectionName() ) );
	emit sig_newMessage( QString( tr("Host address %1") ).arg( _socket->peerAddress().toString()));
	emit sig_newMessage( QString( tr("Host port %1,") ).arg( _socket->peerPort())+ QString( tr(" Our port %1") ).arg( _socket->peerPort() ) );
}

void Game::socketConnectId()
{
	uchar tmp;

	tmp=_socket->readChar();
	_player->setNum( tmp );
	emit sig_newMessage( QString( tr("Connection ID, %1") ).arg( _player->getNum() ) );
}

void Game::socketConnectName()
{	
	QString res;
	uint len = _socket->readChar();
	for( uint i = 0; i < len; i++) {
		res.append( _socket->readChar() );
	}
	_player->setConnectionName(res);
	emit sig_newMessage( QString( tr("Name of player: %1") ).arg( res ) );
}

/*!

*/

void Game::slot_lordSelected()
{
	emit sig_Center(_player->getSelectedLord()->getCell()->getRow(), _player->getSelectedLord()->getCell()->getCol());
	_scrBase->deselect();
	_lordInfo->init( _player->getSelectedLord() );
	_stackInfo->raiseLord();
}
	

void Game::slot_baseSelected()
{
	emit sig_Center(_player->getSelectedBase()->getCell()->getRow(), _player->getSelectedBase()->getCell()->getCol());
	_scrLord->deselect();
	_baseInfo->init( _player->getSelectedBase() );
	_stackInfo->raiseBase();
}

void Game::slot_displayLord()
{
	emit sig_Center(_player->getSelectedLord()->getCell()->getRow(), _player->getSelectedLord()->getCell()->getCol());
	if( _player->numLord() > 0 ) {
		if( !_dispLord ) {
			_dispLord = new DisplayLordTab( _player, _socket, this );
		}
		if( _isPlaying ) {
			_dispLord->exec();
		}
	}
}

void Game::slot_displayBase()
{
	if(_player->getSelectedBase()){
		emit sig_Center(_player->getSelectedBase()->getCell()->getRow(), _player->getSelectedBase()->getCell()->getCol());
		if( _isPlaying ) {
			emit sig_base( _player->getSelectedBase() );
		}
	}
}

void Game::socketQR()
{
	switch( _socket->getCla2() ) {
		case C_QR_MSG_NEXT: 
			socketQRMsgNext();
			break;
		case C_QR_MSG_END: 
			socketQRMsgEnd();
			break;
		case C_QR_LEVEL: 
			socketQRLevel();
			break;
		case C_QR_CHEST:
			socketQRChest();
			break;
		case C_QR_CREATURE_FLEE: 
			socketQRCreature(QRFLEE);
	  break;
	case C_QR_CREATURE_MERCENARY: 
		socketQRCreature(QRMERCENARY);
	  break;
	case C_QR_CREATURE_JOIN:
		socketQRCreature(QRJOIN);
	  break;
	case C_QR_ANSWER:
		logEE( "Should not happen" );
		break;
	}
}

void Game::socketQRMsgNext()
{
	uchar len = _socket->readChar();
	for( uint i = 0; i < len; i++ ) {
		_msg.append( _socket->readChar() );
	}
}

void Game::socketQRMsgEnd()
{		
	//QMessageBox::information( this, "Information", QString( buf+(3*sizeof(char)) ), 0 );
	uchar type = _socket->readChar();
	socketQRMsgNext();
	if( type  == QR_SCEN_DESC ) {
		emit sig_scenInfo( _msg );
	}
	QMessageBox::information( this, tr("Information"), _msg );
	_msg = "";
}
	
void Game::socketQRLevel()
{
	GainLevel * level = new GainLevel( this );
	level->reinit();
	level->exec();
	_socket->sendAnswer( level->getChoice() );
	delete level;
}

void Game::socketQRChest()
{
	AskChest * chest = new AskChest();
	chest->exec();
	_socket->sendAnswerEnum( chest->getResult() );
	delete chest;
}

void Game::socketQRCreature(QRCreature state)
{
	AskDialog dialog;
	QString text;
	/// XXX: to improve with name/category of creature

	switch(state) {
		case QRFLEE:
			text = tr("The creatures are fleeing. Do you want to let them flee ?") ;
			break;
		case QRMERCENARY:
			text = tr("You can buy these creatures. Do you want to buy them ?") ;
			break;
		case QRJOIN:
			text = tr("The creatures want to join. Do you accept them ?") ;
			break;
		default:
			break;
	}

	dialog.setText( text );
	dialog.setYesNo();

	if( dialog.exec() ) {
		_socket->sendAnswerYesNo( true );
	} else {
		_socket->sendAnswerYesNo( false );
	}

}

void Game::socketFight()
{
	switch( _socket->getCla2() ) {
	case C_FIGHT_INIT: 
		socketFightInit();
		break;
	case C_FIGHT_LORD:
		//logEE( "Should not happen (FIGHT_LORD)" );
		break;
	case C_FIGHT_UNIT:
		//logEE( "Should not happen (FIGHT_UNIT)" );
		break;
	case C_FIGHT_END:
		socketFightEnd();
		break;
	default:
		//logEE( "Should not happen (FIGHT)" );
		break;
	}
}

void Game::socketFightInit()
{
	uchar cla = _socket->readChar();
	uchar lord = _socket->readChar();
	emit sig_fight( getLord( lord ), (CLASS_FIGHTER)cla );
	stopTimer();
}

void Game::socketFightEnd()
{
	restartTimer();
}

void Game::displayMiniMap( bool state )
{
	if( _miniMap ) {
		if( state ) {
			_miniMap->show();
			_miniMap->redrawMap( theMap );
		} else {
			_miniMap->hide();
		}
	}
}

void Game::displayFullScreen( bool state )
{
	if( _stackInfo && _chat && _miniMap) {
		if( state ) {
			//_stackInfo->show();
			_chat->show();
			_miniMap->show();
		} else {
			//_stackInfo->hide();
			_chat->hide();
			_miniMap->hide();
		}
	}
}

void Game::updateOptions()
{
	updateAnimations();
	updateDispositionMode();
}

void Game::updateAnimations()
{
	AttalSettings::StrategyModeSettings settings = AttalSettings::getInstance()->getStrategyModeSettings();
	if( settings.isAnimationEnabled ) {
		restartTimer();
	} else {
		stopTimer();
	}
}

void Game::updateDispositionMode()
{
	AttalSettings::DispositionMode mode = AttalSettings::getInstance()->getDispositionMode();
 
	_layH1->removeWidget( _view );
	if( _layControlV1 ) {
		_layH1->removeItem( _layControlV1 );
	}

	if( _layControlH1 ) {
		delete _layControlH1;
		_layControlH1 = NULL;
	}
	_layControlH1 = new QHBoxLayout();

	if( _layControlV1 ) {
		delete _layControlV1;
		_layControlV1 = NULL;
	}
	_layControlV1 = new QVBoxLayout();

	switch( mode ) {
		case AttalSettings::DM_VERYCOMPACT:	
			_scrLord->repaintButtons( 2 );
			_scrBase->repaintButtons( 2 );
			/*
			_layControlH1->addStretch( 1 );
			_layControlH1->addWidget( _scrLord );
			_layControlH1->addStretch( 1 );
			_layControlH1->addWidget( _scrBase );
			_layControlH1->addStretch( 1 );

			_layControlV1->addLayout( _layControlH1, 1  );
			_layControlV1->addWidget( _control, 0, Qt::AlignHCenter );	
		*/	
			_layControlV1->addWidget( _scrLord, 1, Qt::AlignHCenter );
			_layControlV1->addWidget( _control, 0, Qt::AlignHCenter );
			_layControlV1->addWidget( _scrBase, 1, Qt::AlignHCenter );
			break;
		case AttalSettings::DM_COMPACT:	
			_scrLord->repaintButtons( 3 );
			_scrBase->repaintButtons( 3 );
			
			_layControlV1->addWidget( _scrLord, 1, Qt::AlignHCenter );
			_layControlV1->addWidget( _control, 0, Qt::AlignHCenter );
			_layControlV1->addWidget( _scrBase, 1, Qt::AlignHCenter );
			
			break;
		case AttalSettings::DM_FULL:
			_scrLord->repaintButtons( 4 );
			_scrBase->repaintButtons( 4 );
			
			_layControlV1->addWidget( _scrLord, 1, Qt::AlignHCenter );
			_layControlV1->addWidget( _control, 0, Qt::AlignHCenter );
			_layControlV1->addWidget( _scrBase, 1, Qt::AlignHCenter );
			
			break;
	}
	_layH1->addWidget( _view, 1 );
	_layH1->addLayout( _layControlV1 );
	
	emit sig_baseReinit();
	emit sig_lordReinit();
}

void Game::updateWindows()
{
	if( _dispLord ) {
		_dispLord->reupdate();
	}
	
	if( _lordExchange ) {
		_lordExchange->reinit();
	}
	
	if( _player->getSelectedLord() ) {
		_lordInfo->init( _player->getSelectedLord() );
	}
}

void Game::timerEvent(QTimerEvent * /* event */)
{
	stopTimer();
	advanceAnimations();
	restartTimer();
}

void Game::setAdvancePeriod( int period ) 
{
	stopTimer();
	_period =  period;
	restartTimer();
}

void Game::restartTimer() 
{
	if( _idTime == -1 ) {
		_idTime = startTimer(_period);
	}
}

void Game::stopTimer() 
{
	if( _idTime != -1 ) {
		killTimer( _idTime );
		_idTime = -1;
	}
}

