/** \file libassogiate/mime-package.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "mime-package.hh"
#include "mime-directory.hh"

#include <cerrno>
#include <glib/gstdio.h>
#include <glibmm/fileutils.h>
#include <glibmm/miscutils.h>
#include <libxml/tree.h>
#include <libxml++/parsers/domparser.h>
#include <libgnomevfsmm/init.h>

/******************************************************************************/
/* Globals                                                                    */
/******************************************************************************/

/******************************************************************************/
/* This class works around DomParser deleting its document on destruction. */
class TemporaryDomParser : public xmlpp::DomParser {
/******************************************************************************/

public:

	TemporaryDomParser() throw()
	:	xmlpp::DomParser()
	{}

	explicit
	TemporaryDomParser(const Glib::ustring& filename, bool validate = false)
	:	xmlpp::DomParser(filename, validate)
	{} 

	xmlpp::Document*
	disown_document() throw()
	{
		xmlpp::Document* result = doc_;
		doc_ = NULL; return result;
	}

}; /* class TemporaryDomParser */

/******************************************************************************/
/* This class works around a bug in libxml++ prior to 2.14. */
class NsSafeDocument : public xmlpp::Document {
/******************************************************************************/

public: 

	xmlpp::Element*
	create_root_node(const Glib::ustring& name,
		const Glib::ustring& ns_uri = Glib::ustring(),
		const Glib::ustring& ns_prefix = Glib::ustring())
	{
#ifdef CREATE_ROOT_NODE_WORKS
		return xmlpp::Document::create_root_node(name, ns_uri, ns_prefix);
#else
		xmlNode *node = xmlNewDocNode(cobj(), NULL,
			(const xmlChar*)name.c_str(), NULL);
		xmlDocSetRootElement(cobj(), node);

		xmlpp::Element *element = get_root_node();

		if (!ns_uri.empty()) {
			element->set_namespace_declaration(ns_uri, ns_prefix);
			xmlNs *ns = xmlSearchNs(cobj(), element->cobj(),
				(const xmlChar*)(ns_prefix.empty() ? NULL : ns_prefix.c_str()));
			if (ns)
				xmlSetNs(element->cobj(), ns);
			else
				throw xmlpp::exception("The namespace (" + ns_prefix +
					") has not been declared.");
		}

		return element;
#endif
	}

}; /* class NsSafeDocument */

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

/******************************************************************************/
/* class MimeDatabaseError and descendants                                    */
/******************************************************************************/

MimeDatabaseError::MimeDatabaseError(const ustring& description) throw()
:	std::runtime_error(description)
{}

MimeDatabaseLoadError::MimeDatabaseLoadError(const ustring& description) throw()
:	MimeDatabaseError(description)
{}

void
MimeDatabaseLoadError::unhandled() throw()
{
	try { signal_unhandled().emit(*this); } catch (...) {}
}

sigc::signal<void, MimeDatabaseLoadError>
MimeDatabaseLoadError::signal_unhandled() throw()
{
	static sigc::signal<void, MimeDatabaseLoadError> the;
	return the;
}

MimeDatabaseWriteError::MimeDatabaseWriteError(const ustring& description)
	throw()
:	MimeDatabaseError(description)
{}

MimeDatabaseUpdateError::MimeDatabaseUpdateError(const ustring& description)
	throw()
:	MimeDatabaseError(description)
{}

void
MimeDatabaseUpdateError::unhandled() throw()
{
	try { signal_unhandled().emit(*this); } catch (...) {}
}

sigc::signal<void, MimeDatabaseUpdateError>
MimeDatabaseUpdateError::signal_unhandled() throw()
{
	static sigc::signal<void, MimeDatabaseUpdateError> the;
	return the;
}

/******************************************************************************/
/* class MimePackage                                                          */
/******************************************************************************/

MimePackage::MimePackage(const std::string& path, Location type)
	throw(MimeDatabaseLoadError)
:	m_path(path), m_type(type), m_document(NULL), s_reloaded()
{	reload(); }

MimePackage::~MimePackage()
{	delete m_document; }

bool
MimePackage::has_contents() const throw()
{	return !m_document->get_root_node()->get_children("mime-type").empty(); }

void
MimePackage::reload() throw(MimeDatabaseLoadError)
{
	try {
		delete m_document;
		if (Glib::file_test(m_path, Glib::FILE_TEST_IS_REGULAR)) {
			TemporaryDomParser parser(m_path);
			m_document = parser.get_document()->get_root_node()
				? parser.disown_document() : create_empty_document();
		} else
			m_document = create_empty_document();
	} catch (xmlpp::exception& e) {
		throw MimeDatabaseLoadError(e.what());
	} catch (...) {
		throw MimeDatabaseLoadError(ustring());
	}

	try { s_reloaded.emit(); } catch (...) {}
}

sigc::signal<void>
MimePackage::signal_reloaded() throw()
{	return s_reloaded; }

void
MimePackage::extend_node_map(MimeNodeMap& node_map) throw()
{
	xmlpp::Node::NodeList types =
		m_document->get_root_node()->get_children("mime-type");

	FOREACH(xmlpp::Node::NodeList, types, i) {
		if cast(*i, xmlpp::Element, el) {
			xmlpp::Attribute *type = el->get_attribute("type");
			if (type != NULL)
				node_map.insert(MimeNodeMap::value_type(type->get_value(),
					MimeNodeMap::value_type::second_type(el, m_type)));
		}
	}
}

xmlpp::Document*
MimePackage::create_empty_document() throw(xmlpp::exception)
{
	NsSafeDocument* result = new NsSafeDocument();
	result->create_root_node("mime-info",
		"http://www.freedesktop.org/standards/shared-mime-info");
	return result;
}

/******************************************************************************/
/* class WritableMimePackage                                                  */
/******************************************************************************/

WritableMimePackage::WritableMimePackage(const std::string& path,
	const MimeDirectory& dir, Location type) throw(MimeDatabaseLoadError)
:	MimePackage(path, type),
	m_ignore_events(0), m_monitor(), m_theme_monitor(),
	m_theme_dir(), m_icons_dir(), m_mime_dir(dir.get_mime_dir()),
	s_icons_changed()
{
	std::list<std::string> icons_path;
	icons_path.push_back(dir.get_data_dir());
	icons_path.push_back("icons");
	icons_path.push_back("hicolor");
	m_theme_dir = Glib::build_filename(icons_path);

	icons_path.push_back("48x48");
	icons_path.push_back("mimetypes");
	m_icons_dir = Glib::build_filename(icons_path);

	Gnome::Vfs::init();
	m_monitor.add(m_path, Gnome::Vfs::MONITOR_FILE,
		sigc::mem_fun(*this, &WritableMimePackage::on_monitor_event));
	m_theme_monitor.add(Glib::build_filename(m_theme_dir, "icon-theme.cache"),
		Gnome::Vfs::MONITOR_FILE,
		sigc::mem_fun(*this, &WritableMimePackage::on_monitor_event));
}

void
WritableMimePackage::add_type(const MimeType& type) throw(MimeTypeExistsError,
	MimeDatabaseError)
{
	xmlpp::Node::NodeList types =
		m_document->get_root_node()->get_children("mime-type");

	FOREACH(xmlpp::Node::NodeList, types, i) {
		if cast(*i, xmlpp::Element, el) {
			xmlpp::Attribute *type_a = el->get_attribute("type");
			if (type_a != NULL && type_a->get_value() == type.get_full_name())
				throw MimeTypeExistsError();
		}
	}

	type.output(*m_document->get_root_node()->add_child("mime-type"));
	write();
}

void
WritableMimePackage::replace_type(const MimeType& type, ustring old_name)
	 throw(MimeDatabaseError)
{
	if (old_name.empty()) old_name = type.get_full_name();
	xmlpp::Element *new_el = NULL;

	xmlpp::Node::NodeList types =
		m_document->get_root_node()->get_children("mime-type");
	FOREACH(xmlpp::Node::NodeList, types, i) {
		if cast(*i, xmlpp::Element, el) {
			xmlpp::Attribute *type_a = el->get_attribute("type");
			if (type_a == NULL || type_a->get_value() != old_name)
				continue;
			if (new_el == NULL)
				new_el = el;
			else
				m_document->get_root_node()->remove_child(el);
		}
	}

	if (new_el == NULL)
		new_el = m_document->get_root_node()->add_child("mime-type");

	type.output(*new_el);
	write();
}

void
WritableMimePackage::remove_type(const ustring& type) throw(MimeDatabaseError)
{
	xmlpp::Node::NodeList types =
		m_document->get_root_node()->get_children("mime-type");

	FOREACH(xmlpp::Node::NodeList, types, i) {
		if cast(*i, xmlpp::Element, el) {
			xmlpp::Attribute *type_a = el->get_attribute("type");
			if (type_a != NULL && type_a->get_value() == type)
				m_document->get_root_node()->remove_child(el);
		}
	}

	write();
}

void
WritableMimePackage::clear_types() throw(MimeDatabaseError)
{
	if (g_unlink(m_path.data()) != 0)
		throw MimeDatabaseWriteError(Glib::strerror(errno));

	reload();
	update_mime_database();
}

void
WritableMimePackage::import_types(const ustring& from) throw(xmlpp::exception,
	MimeDatabaseError)
{
	xmlpp::DomParser parser(from);
	xmlpp::Element *root = parser.get_document()->get_root_node();
	if (root != NULL) {
		if (root->get_name() != "mime-info" || root->get_namespace_uri() !=
			"http://www.freedesktop.org/standards/shared-mime-info")
			throw xmlpp::exception("Not a shared-mime-info file.");

		xmlpp::Node::NodeList types = root->get_children("mime-type");
		FOREACH(xmlpp::Node::NodeList, types, i)
			m_document->get_root_node()->import_node(*i, true /*recursive*/);

		write();
	}
}

void
WritableMimePackage::export_types(const ustring& to) const
	throw(xmlpp::exception)
{	m_document->write_to_file(to); }

void
WritableMimePackage::update_mime_database() throw(MimeDatabaseUpdateError)
{
	ustring program = Glib::find_program_in_path("update-mime-database");
	if (program.empty())
		throw MimeDatabaseUpdateError("update-mime-database not in path");

	std::list<std::string> argv;
	argv.push_back(program);
	argv.push_back(m_mime_dir);

	try {
		Glib::Pid pid;
		Glib::spawn_async(m_mime_dir, argv, Glib::SPAWN_STDOUT_TO_DEV_NULL,
			sigc::slot<void>(), &pid);
		Glib::signal_child_watch().connect
			(sigc::ptr_fun(&WritableMimePackage::on_mime_update_exited), pid);
	} catch (Glib::Exception& e) {
		throw MimeDatabaseUpdateError(e.what());
	} catch (...) {
		throw MimeDatabaseUpdateError("unknown error");
	}
}

bool
WritableMimePackage::has_default_icon(const MimeType& type) const throw()
{
	try {
		get_default_icon(type);
		return true;
	}
	catch (std::invalid_argument) {
		return false;
	}
}

std::string
WritableMimePackage::get_default_icon(const MimeType& type) const
	throw(std::invalid_argument)
{
	std::string icon = Glib::build_filename(m_icons_dir,
		type.m_type + "-" + type.m_subtype);
	
	if (Glib::file_test(icon + ".svg", Glib::FILE_TEST_EXISTS))
		icon += ".svg";
	else if (Glib::file_test(icon + ".png", Glib::FILE_TEST_EXISTS))
		icon += ".png";
	else if (Glib::file_test(icon + ".xpm", Glib::FILE_TEST_EXISTS))
		icon += ".xpm";
	else
		throw std::invalid_argument(type.get_full_name());
	
	GError *error = NULL;
	const Glib::ScopedPtr<gchar> buf(g_file_read_link(icon.data(), &error));
	if (error == NULL)
		return buf.get();
	else
		throw std::invalid_argument(Glib::Error(error).what());
}

void
WritableMimePackage::set_default_icon(const MimeType& type,
	const std::string& path)
	throw(std::invalid_argument, MimeDatabaseError)
{
	std::string icon = Glib::build_filename(m_icons_dir,
		type.m_type + "-" + type.m_subtype);

	remove_icons(icon);

	if (Glib::str_has_suffix(path, ".svg"))
		icon += ".svg";
	else if (Glib::str_has_suffix(path, ".png"))
		icon += ".png";
	else if (Glib::str_has_suffix(path, ".xpm"))
		icon += ".xpm";
	else
		throw std::invalid_argument(path);

	if (!Glib::file_test(m_icons_dir, Glib::FILE_TEST_IS_DIR))
		if (g_mkdir_with_parents(m_icons_dir.data(), 0755) != 0)
			throw MimeDatabaseWriteError(Glib::strerror(errno));

	if (symlink(path.data(), icon.data()) != 0)
		throw MimeDatabaseWriteError(Glib::strerror(errno));
	
	update_icon_cache();
}

void
WritableMimePackage::unset_default_icon(const MimeType& type)
	throw(MimeDatabaseError)
{
	if (remove_icons(Glib::build_filename(m_icons_dir,
		type.m_type + "-" + type.m_subtype)));
		update_icon_cache();
}

void
WritableMimePackage::update_icon_cache() throw(MimeDatabaseUpdateError)
{
	ustring program = Glib::find_program_in_path("gtk-update-icon-cache");
	if (program.empty())
		throw MimeDatabaseUpdateError("gtk-update-icon-cache not in path");

	std::list<std::string> argv;
	argv.push_back(program);
	argv.push_back("--force");
	argv.push_back("--ignore-theme-index");
	argv.push_back(m_theme_dir);

	try {
		Glib::Pid pid;
		Glib::spawn_async(m_theme_dir, argv, Glib::SPAWN_STDOUT_TO_DEV_NULL,
			sigc::slot<void>(), &pid);
		Glib::signal_child_watch().connect
			(sigc::ptr_fun(&WritableMimePackage::on_theme_update_exited), pid);
	} catch (Glib::Exception& e) {
		throw MimeDatabaseUpdateError(e.what());
	} catch (...) {
		throw MimeDatabaseUpdateError("unknown error");
	}
}

sigc::signal<void>
WritableMimePackage::signal_icons_changed() throw()
{	return s_icons_changed; }

void
WritableMimePackage::write() throw(MimeDatabaseError)
{
	m_ignore_events += Glib::file_test(m_path, Glib::FILE_TEST_EXISTS) ? 1 : 2;

	try {
		m_document->write_to_file(m_path);
	} catch (xmlpp::exception& e) {
		reload();
		throw MimeDatabaseWriteError(e.what());
	}

	reload();
	update_mime_database();
}

bool
WritableMimePackage::remove_icons(const ustring& path)
	throw(MimeDatabaseWriteError)
{
	std::list<std::string> targets;
	
	if (Glib::file_test(path + ".svg", Glib::FILE_TEST_EXISTS))
		targets.push_back(path + ".svg");
	if (Glib::file_test(path + ".png", Glib::FILE_TEST_EXISTS))
		targets.push_back(path + ".png");
	if (Glib::file_test(path + ".xpm", Glib::FILE_TEST_EXISTS))
		targets.push_back(path + ".xpm");

	FOREACH(std::list<std::string>, targets, target)
		if (g_unlink(target->data()) != 0)
			throw MimeDatabaseWriteError(Glib::strerror(errno));

	return !targets.empty();
}

void
WritableMimePackage::on_monitor_event(const Gnome::Vfs::MonitorHandle& monitor,
	const ustring&, const ustring&, Gnome::Vfs::MonitorEventType type) throw()
{
	if (monitor.gobj() == m_theme_monitor.gobj()) {
		try { s_icons_changed.emit(); } catch (...) {}
		return;
	}

	if (m_ignore_events > 0) { --m_ignore_events; return; }

	switch (type) {
	case Gnome::Vfs::MONITOR_EVENT_CREATED:
	case Gnome::Vfs::MONITOR_EVENT_DELETED:
	case Gnome::Vfs::MONITOR_EVENT_CHANGED:
		try {
			reload();
		} catch (MimeDatabaseLoadError& e) {
			e.unhandled();
		}
	default: break;
	}
}

void
WritableMimePackage::on_mime_update_exited(GPid, int result) throw()
{
	if (result != 0)
		MimeDatabaseUpdateError(compose::ucompose("update-mime-database "
			"failed, returning %1", result)).unhandled();
}

void
WritableMimePackage::on_theme_update_exited(GPid, int result) throw()
{
	if (result != 0)
		MimeDatabaseUpdateError(compose::ucompose("gtk-update-icon-cache "
			"failed, returning %1", result)).unhandled();
}

} /* namespace assoGiate */
