/* $Id: HFCollision.cpp,v 1.7 2003/03/19 09:25:06 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "HFWorld.h"
#include "HFQuadtree.h"
#include <math.h>

namespace Ark
{

   /*
    * Tests wether a collision occurs between the given bbox and the rest of
    * the world (terrain and entities).
    */
   bool
   HeightField::TestCollision (const BBox &box,
			       int8 colflags, 
			       std::vector<Collision> &collisions)
   {
      Collision worldcol;

      Vector3 vs[4];
      // the index of the bottom vertex colliding
      int n = -1;

      // Test a collision with the world
      for (int i = 0; i < 4; i++)
      {
	 vs[i] = box.m_Min;

	 if (i & 1) vs[i].X = box.m_Max.X;
	 if (i & 2) vs[i].Z = box.m_Max.Z;

	 if (vs[i].X > 0 && vs[i].X < m_SizeX &&
	     vs[i].Z > 0 && vs[i].Z < m_SizeZ)
	 {
	    vs[i].Y = GetHeight (vs[i].X, vs[i].Z);
	    if (box.m_Min.Y < vs[i].Y)
	    {
	       worldcol.m_Flags |= Collision::COLLIDE;
	       n = i;
	    }
	 }
	 else
	    return false;
      }

      if (worldcol.m_Flags & Collision::COLLIDE)
      {
	 if (colflags & Collision::PLANE)
	 {
	    // Compute collision plane
	    Vector3 tri[3];

	    if (n == 0)
	    {
	       tri[0] = vs[0];
	       tri[1] = vs[1];
	       tri[2] = vs[2];
	    }
	    else if (n == 3)
	    {
	       tri[0] = vs[1];
	       tri[1] = vs[2];
	       tri[2] = vs[3];
	    }
	    else
	    {
	       tri[0] = vs[n];
	       tri[1] = vs[n-1];
	       tri[2] = vs[n+1];
	    }

	    worldcol.m_Plane = Plane::GetTriPlane (tri[0], tri[1], tri[2]);
	    worldcol.m_Flags |= Collision::PLANE;
	 }

	 worldcol.m_Pos = vs[n];
	 worldcol.m_Flags |= Collision::POSITION;

	 // Set ground type, if possible
	 size_t g = GetGround (int(vs[n].X / m_Scale),
			       int(vs[n].Z / m_Scale));
	 if (g < m_Grounds.size())
	 {
	    worldcol.m_Flags |= Collision::WORLD;
	    worldcol.m_Material = &*m_Grounds[g];
	 }
	 else
	    worldcol.m_Material = NULL;

	 collisions.push_back (worldcol);
	 return true;
      }

      return false;
   }

   /*
    * Try to trace a ray and tell if there is a collision (stop after the first
    * one)
    */
   bool
   HeightField::RayTrace (const Ray &ray,
			  int8 colflags,
			  std::vector <Collision> &collisions)
   {      
      Collision col;
      bool test_ents = (colflags & Collision::ENTITY) ? true : false;
      bool res = m_Quadtree->RayTrace (ray, col, test_ents);
      
      if (res)
      {
	 collisions.push_back (col);
	 return true;
      }
      else
	 return false;
   }

   /// FIXME: create the HFEntityData structure...
   void
   HeightField::Add (Entity *entity)
   {
      World::Add (entity);
      entity->m_WorldData = (void*) new HFEntityData;
   }


   void
   HeightField::Remove (Entity *entity)
   {
      assert (entity != NULL);
      assert (entity->m_WorldData != NULL);

      // Remove the entity from the quadtree.
      m_Quadtree->RemoveEntity (entity, (HFEntityData *) entity->m_WorldData);

      delete ((HFEntityData*) entity->m_WorldData);
      entity->m_WorldData = NULL;

      World::Remove (entity);
   }

   /*
    * Detect collisions between all entities in the world
    */
   void
   HeightField::DetectCollisions()
   {
      assert (m_Quadtree != NULL);
      // Update each entity bounding boxes...

      std::vector<Entity *>::iterator i;
      for (i = m_Entities.begin(); i != m_Entities.end(); i++)
      {
	 HFEntityData *data = (HFEntityData *) (*i)->m_WorldData;

	 if (data->m_OldBbox.m_Min != (*i)->m_BBox.m_Min ||
	     data->m_OldBbox.m_Max != (*i)->m_BBox.m_Max ||
	     data->m_Patches.size() == 0)
	 {
	    m_Quadtree->UpdateEntity (*i, data);
	    data->m_OldBbox.m_Min = (*i)->m_BBox.m_Min;
	    data->m_OldBbox.m_Max = (*i)->m_BBox.m_Max;
	 }
      }

      // Real collision detection
      ColSystem *cs = GetCache()->GetColSystem();
      if (cs == NULL)
	 return;

      ColliderList colliders;
      m_Quadtree->GetColliders(colliders);
      ColliderList::iterator pr;
      
      for (pr = colliders.begin(); pr != colliders.end(); pr++)
      {
	 // A carried object does not collide with its carrier.
	 // (well, in our simplified "rules"..).
	 if (pr->first.m_E1->m_AttachmentEnt == pr->first.m_E2 ||
	     pr->first.m_E2->m_AttachmentEnt == pr->first.m_E1)
	    continue;

	 ColPair p;
	 if (cs->TestCollision (pr->first.m_E1->m_MState,
				pr->first.m_E2->m_MState, p))
	 {
	    if (!(pr->first.m_E1->m_Flags & Entity::STATIC))
	       pr->first.m_E1->AddCollision (pr->first.m_E2, p, false);
	    ColPair p2;
	    p2.m_Triangle0 = p.m_Triangle1;
	    p2.m_Triangle1 = p.m_Triangle0;
	    p2.m_BodyPart0 = p.m_BodyPart1;
	    p2.m_BodyPart1 = p.m_BodyPart0;

	    if (!(pr->first.m_E2->m_Flags & Entity::STATIC))
	       pr->first.m_E2->AddCollision (pr->first.m_E1, p2, false);
	 }
	 else
	 {
	    if (!(pr->first.m_E1->m_Flags & Entity::STATIC))
	       pr->first.m_E1->AddCollision (pr->first.m_E2, p, true);
	    if (!(pr->first.m_E2->m_Flags & Entity::STATIC))
	       pr->first.m_E2->AddCollision (pr->first.m_E1, p, true);
	 }
      }

      return;
   }

   // Collider stuff
   // FIXME: move in Ark/ArkCollision (??)

   bool operator < (const Collider &b, const Collider &c)
   {
      if (b.m_E1 < c.m_E1)
	 return true;
      else if (b.m_E1 < c.m_E1)
	 return false;
      else
	 return b.m_E2 < c.m_E2;
   }

   bool operator == (const Collider &b, const Collider &c)
   {
      return (b.m_E1 == c.m_E1) && (b.m_E2 == c.m_E2);
   }


}
