/* $Id: ArkLoader3DS.cpp,v 1.44 2003/03/20 17:23:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) David Farrell <fdavid@cyberramp.net>
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
//  3D Studio object reader
#include <Ark/ArkLoader.h>
#include <Ark/ArkCache.h>
#include <Ark/ArkModelBuild.h>

#include <sstream>
#include <stdlib.h>

#include <vector>
#include "3ds.h"

#undef DEBUG_3DS

namespace Ark
{

#ifdef WORDS_BIGENDIAN
   inline void Swap16 (uint16 *src)
   {
      *src =  ((*src) & 0x00FF << 8) |
	      ((*src) & 0xFF00 >> 8);
   }

   inline void Swap32 (uint32 *src)
   {
      // Swap words
      *src =  ((*src) & 0x0000FFFF << 16) |
	 ((*src) & 0xFFFF0000 >> 16);

      uint16* src16 = reinterpret_cast<uint16*>(src);
      
      // Swap bytes
      Swap16(src16);
      Swap16(src16+1);
   }
#else
   inline void Swap16 (uint16 *src) {}
   inline void Swap32 (uint32 *src) {}
#endif

   class Loader3DS : public Loader
   {

	 String m_Name;
	 Stream *m_InFile;
	 H3dsScene *m_Scene;
	 int m_DoubleSided;
	 int alphablending;
	 int defaultmaterialwarning;

private:
	 // =====================================================
	 // Reads len number of bytes from the file into the memory pointed
	 // to by *dest
	 void dread(void *dest, int len)
	 {
	    m_InFile->read ((char *)dest, len);
	    
	    if ( m_InFile->fail() )
	       Error ("Error reading data...");

	    /* Endian Issues */
	    if (len == 2) Swap16 ((uint16*) dest);
	    if (len == 4) Swap32 ((uint32*) dest);
	 }

	 String dreadstring ()
	 {
	    char name[100];
	    size_t n = 0;

	    do
	       dread(&name[n++], 1);
	    while (name[n-1]!='\0' && n<sizeof(name));

	    name[n-1]='\0';
	    return String (name);
	 }

	 uint16 dreaduword ()
	 {
	    uint16 r;
	    dread (&r, 2);

	    return r;
	 }

	 uint16 dreadword ()
	 {
	    int16 r;
	    dread (&r, 2);
	    return r;
	 }

	 int32 dreaddword ()
	 {
	    int32 r;
	    dread (&r, 4);
	    return r;
	 }

	 // Retrieves the file position
	 int32 dgetpos()
	 {
	    long pos = m_InFile->tellg();
	    if (pos < 0)
	       Error ("Error getting file position...");

	    return pos;
	 }

	 // Retrieves the file position
	 void dsetpos(int32 pos)
	 {
	    // XXX if (!m_InFile->Seek(SM_SET, pos))
	    if (!m_InFile->seekg (pos, std::ios::beg))
	       Error ("Error setting file position...");
	 }

      private:
	 void Error (const String &text)
	 {
	    Ark::Sys()->Warning ("While reading file %s: %s", m_Name.c_str(),
				 text.c_str());
	 }

      private:
	 // =====================================================
	 // Reads in the list of vertices for the current mesh object.
	 void ReadVertList(H3dsMeshObj *meshobj)
	 {
	    meshobj->vertices.resize (dreadword());

	    for (size_t n = 0; n < meshobj->vertices.size(); n++)
	    {
	       Vector3 &v = meshobj->vertices[n];

	       dread(&v.X, sizeof(float32));
	       dread(&v.Y, sizeof(float32));
	       dread(&v.Z, sizeof(float32));
	    }

#ifdef DEBUG_3DS
	    printf(" Read Vertex List with %d vertices\n",
		   meshobj->vertices.size());
#endif
	 }

	 // Reads in the face list for the current mesh object.
	 // Each face is a set of 3 indexes into the mesh object's
	 // vertex list and a set of options.
	 int16 ReadFaceList(H3dsMeshObj *meshobj)
	 {
	    meshobj->facelist.resize(dreadword());
	    
	    for(size_t n = 0; n < meshobj->facelist.size(); n++)
	    {
	       short A = dreadword();
	       short B = dreadword();
	       short C = dreadword();
	       
	       meshobj->facelist[n].flags = dreadword();

	       /** Meaning of flags
		* From Autodesk' 3D Studio File Format" r0.91 : 
		*  " This number is is a binary number which expands to 3
		*    values. For example 0x0006 would expand to 110 binary.
		*    The value should be read as 1 1 0. This value can be found
		*    in 3d-studio ascii files as AB:1 BC:1 AC:0. Which probably
		*    indicated the order of the vertices. For example AB:1
		*    would be a normal line from A to B. But AB:0 would mean a
		*    line from B to A. "
		*/

	       bool AB, BC, AC;

	       AB = BC = AC = false;
	       if (meshobj->facelist[n].flags & 4) AB = true;
	       if (meshobj->facelist[n].flags & 2) BC = true;
	       if (meshobj->facelist[n].flags & 1) AC = true;

	       // We want points in triangles to be counter-clockwise.
	       // Reorder them.. (FIXME: it seems they are always in CCW
	       // 
	       if (AB && BC && !AC)
	       {
		  meshobj->facelist[n].p0 = A;
		  meshobj->facelist[n].p1 = B;
		  meshobj->facelist[n].p2 = C;
	       }
	       else if (!AB && !BC && !AC)
	       {
		  meshobj->facelist[n].p0 = A;
		  meshobj->facelist[n].p1 = B;
		  meshobj->facelist[n].p2 = C;
	       }
	       else
	       {
		  meshobj->facelist[n].p0 = A;
		  meshobj->facelist[n].p1 = B;
		  meshobj->facelist[n].p2 = C;
	       }
	    }

#ifdef DEBUG_3DS
	    printf(" Read Face List with %d faces\n",
		   meshobj->facelist.size());
#endif

	    return meshobj->facelist.size(); /* WG */
	 }

	 // Reads in the mapping list/texture coords. for the current mesh
	 // object
	 void ReadMapList(H3dsMeshObj *meshobj)
	 {
	    meshobj->texcoords.resize (dreadword());
	    for(size_t n = 0; n < meshobj->texcoords.size(); n++)
	    {
	       dread(&meshobj->texcoords[n].X, sizeof(float32));
	       dread(&meshobj->texcoords[n].Y, sizeof(float32));
	    }

#ifdef DEBUG_3DS
	    printf(" Read Map List with %d maps\n",
		   meshobj->texcoords.size());
#endif
	 }

	 // Reads in the translation matrix for the current mesh object.
	 // I'm not sure when this is used in a 3DS model.
	 void ReadTraMatrix(H3dsMeshObj *meshobj)
	 {
	    for (int i = 0; i < 12; i++)
	       dread(&meshobj->TraMatrix[i], sizeof(float32));

	    meshobj->matrix=1;
#ifdef DEBUG_3DS
	    float32 *t = meshobj->TraMatrix;
	    printf(" Read Translation Matrix\n");
	    printf(" %4.4f %4.4f %4.4f %4.4f\n", t[0], t[3], t[6], t[9]);
	    printf(" %4.4f %4.4f %4.4f %4.4f\n", t[1], t[4], t[7], t[10]);
	    printf(" %4.4f %4.4f %4.4f %4.4f\n", t[2], t[5], t[8], t[11]);
#endif
	 }

	 // Assigns the material in this chunk to the faces specified.
	 // Each material has a list of indexes into the face array to
	 // mark which faces use which material.
	 void ReadFaceMaterial(H3dsMeshObj *meshobj)
	 {
	    meshobj->bindings.resize(meshobj->bindings.size() + 1);

	    H3dsMatList &binding = meshobj->bindings.back ();
	    
	    // The material name is first
	    binding.name = dreadstring ();
	    
	    // Then the number of faces
	    binding.faces.resize (dreadword());
	    
	    // Bind the faces to the material
	    for (size_t n = 0; n < binding.faces.size(); n++)
	       binding.faces[n] = dreadword();
	    
#ifdef DEBUG_3DS
	    printf(" Read Material %s with %d faces\n",
		   binding.name.Get(),
		   binding.faces.size());
#endif
	 }

	 void ReadDefaultMaterial(H3dsMeshObj *meshobj, int16 nf)
	 {
	    H3dsMatList binding;
	    
	    // The material name is first
	    binding.name = "WG_DEFAULT";

	    // Then the number of faces
	    binding.faces.resize (nf);
	    
	    // Bind the faces to the material
	    for (int n = 0; n < nf; n++)
	       binding.faces[n] = n;

#ifdef DEBUG_3DS
	    printf(" Faked Material %s with %d faces\n", "WG_DEFAULT", nf);
#endif

	    // Add created binding to list
	    meshobj->bindings.push_back( binding );
	 }

	 // Eats a triangle mesh, which is a combination of XYZ vertices,
	 // faces, and bindings of material to faces.
	 void ReadTriMeshBlocks(int32 p, const String &name)
	 {
	    int16 fl = 0;
	    int32 pc;
	    bool found_facemat;

	    H3dsMeshObj meshobj;
	    
	    found_facemat = 0;
	    meshobj.name = name;

	    while((pc=dgetpos()) < p)
	    {
	       int id = dreadword ();
	       int32 len = dreaddword ();
	       
	       switch(id)
	       {
		  case H3DS_CHUNK_VERTLIST:
		     ReadVertList (&meshobj);
		     break;
		  case H3DS_CHUNK_FACELIST:
		     fl = ReadFaceList (&meshobj);
		     break;
		  case H3DS_CHUNK_MAPLIST:
		     ReadMapList  (&meshobj);
		     break;
		  case H3DS_CHUNK_TRMATRIX:
		     ReadTraMatrix(&meshobj);
		     break;
		  case H3DS_CHUNK_FACEMAT:
		     ReadFaceMaterial(&meshobj);
		     found_facemat = 1;
		     break;

		     /* case H3DS_CHUNK_SMOOLIST: */
		  default: dsetpos(pc+len);
	       }
	    }

	    if (!found_facemat)
	    {
	       defaultmaterialwarning = 1;
	       ReadDefaultMaterial(&meshobj, fl);
	    }
	    
	    // mesh is created, add it to the mesh list
	    m_Scene->meshobjs.push_back( meshobj );
	 }


	 // Reads in object blocks, which are cameras, meshes, or lights;
	 // this code only supports meshes and ignores the rest
	 void ReadObjBlocks(int32 p)
	 {
	    int32 pc;
	    
	    String name = dreadstring();
#ifdef DEBUG_3DS
	    printf("Read Object: %s\n", name.Get());
#endif

	    while((pc = dgetpos()) < p)
	    {
	       uint id = (uint) dreadword ();
	       int32 len = dreaddword ();

	       switch(id)
	       {
		  case H3DS_CHUNK_TRIMESH:
		     ReadTriMeshBlocks(pc+len, name);
		     break;
		     //case H3DS_CHUNK_LIGHT:
		     //case H3DS_CHUNK_CAMERA:
		  default: dsetpos(pc+len);
	       }
	    }
	    
	    dsetpos(p);
	 }


	 // Reads in the name of the material
	 void ReadMatChunk(int32 p, H3dsMat *mat)
	 {
	    mat->name = dreadstring();

#ifdef DEBUG_3DS
	    printf("Read Material Name: %s\n", mat->name.Get());
#endif
	 }


	 // Reads in an RGB FP triplet
	 void ReadRGBFloat(int32 p, Color *rgb)
	 {
	    dread(&rgb->R, sizeof(float32));
	    dread(&rgb->G, sizeof(float32));
	    dread(&rgb->B, sizeof(float32));

#ifdef DEBUG_3DS
	    printf(" Read in RGB FP triplet\n");
	    printf(" Red: %f\n", rgb->R);
	    printf(" Green: %f\n", rgb->R);
	    printf(" Blue: %f\n", rgb->B);
#endif
	 }

	 // Reads in an RGB byte triplet
	 void ReadRGBByte(int32 p, Color *rgb)
	 {
	    unsigned char red, green, blue;
	    
	    dread(&red, sizeof(red));
	    dread(&green, sizeof(green));
	    dread(&blue, sizeof(blue));

	    rgb->R = (float)(red)/256.0f;
	    rgb->G = (float)(green)/256.0f;
	    rgb->B = (float)(blue)/256.0f;

#ifdef DEBUG_3DS
	    printf(" Read in RGB byte triplet\n");
	    printf(" Red: %f\n", rgb->R);
	    printf(" Green: %f\n", rgb->G);
	    printf(" Blue: %f\n", rgb->B);
#endif
  }


	 // Reads in a color assignment
	 void ReadShadingChunk(int32 p, Color *rgb)
	 {
	    int32 pc;

	    while((pc = dgetpos()) < p)
	    {
	       uint id = (uint) dreadword ();
	       int32 len = dreaddword ();

	       switch(id)
	       {
		  case H3DS_CHUNK_RGB1: ReadRGBFloat(pc+len, rgb); break;
		  case H3DS_CHUNK_RGB2: ReadRGBByte (pc+len, rgb); break;
		  default: dsetpos(pc+len);
	       }
	    }
	 }


	 void ReadTextureNameChunk(int32 p, H3dsTexture *texture)
	 {
	    texture->texturename = dreadstring();
#ifdef DEBUG_3DS
	    printf(" Read Texture Name: %s\n", texture->texturename.Get());
#endif
	 }


	 
	 void ReadTextureChunk(int32 p, H3dsTexture *texture)
	 {
	    int32 pc;
	    
	    while((pc = dgetpos()) < p)
	    {
	       uint id = (uint) dreadword ();
	       int32 len = dreaddword ();
	       
	       switch(id)
	       {
		  case H3DS_CHUNK_MAPFILENAME:
		     ReadTextureNameChunk(pc+len, texture);
		     break;
		  default:
		     dsetpos(pc+len);
	       }
	    }
	 }

	 // I assume that if one material is doublesided, they all are,
	 // so turn on a global doublesided flag
	 void ReadDoublesidedChunk(int *material_doublesided)
	 {
	    *material_doublesided = 1;
	    m_DoubleSided = 1;
	    
#ifdef DEBUG_3DS
	    printf(" Doublesided Material\n");
#endif
	 }
	 
	 void ReadAmountChunk(int16 *amount)
	 {
	    int16 temp;
	    
	    dread(&temp, sizeof(temp));
	    *amount = temp;
	 }

	 // I interpret the transparency information as the alpha
	 // value for the entire material
	 void ReadAlphaChunk(int32 p, float32 *alpha)
	 {
	    int16 amount;
	    int32 pc;

	    while((pc = dgetpos()) < p)
	    {
	       uint id = (uint) dreadword ();
	       int32 len = dreaddword ();

	       switch(id)
	       {
		  case H3DS_CHUNK_AMOUNT:
		     ReadAmountChunk(&amount);
		     break;
		  default:
		     dsetpos(pc+len);
	       }
	    }

	    *alpha = (float32)amount / 100.0f;
	    if (*alpha < .01f)
	       *alpha = 1.0f;
	    else
	       alphablending = 1;
	    
#ifdef DEBUG_3DS
	    printf(" Alpha Channel: %f\n", *alpha);
#endif
	 }

	 // Fills out the current material with the information contained
	 // in these chunks
	 void ReadMaterialBlocks(int32 p)
	 {
	    H3dsMat materialobj;
	    int32 pc;
	    
	    while((pc = dgetpos()) < p)
	    {
	       uint id = (uint) dreadword ();
	       int32 len = dreaddword ();

	       switch(id)
	       {
		  case H3DS_CHUNK_MATNAME:
		     ReadMatChunk(pc+len, &materialobj);
		     break;
		     
		  case H3DS_CHUNK_AMBIENT:
		     ReadShadingChunk(pc+len, &materialobj.ambient);
		     break;
		  case H3DS_CHUNK_DIFFUSE:
		     ReadShadingChunk(pc+len, &materialobj.diffuse);
		     break;
		  case H3DS_CHUNK_SPECULAR:
		     ReadShadingChunk(pc+len, &materialobj.specular);
		     break;
		  case H3DS_CHUNK_TEXTURE:
		     ReadTextureChunk(pc+len, &materialobj.texture);
		     break;
		  case H3DS_CHUNK_DOUBLESIDED:
		     ReadDoublesidedChunk(&materialobj.doublesided);
		     break;
		  case H3DS_CHUNK_TRANSPARENCY:
		     ReadAlphaChunk(pc+len, &materialobj.alpha);
		     break;
		  default:
		     dsetpos(pc+len);
		     break;
	       }
	    }

	    m_Scene->materials.push_back (materialobj);
	 }
	 
	 // Reads in mesh objects and materials
	 void ReadObjMeshBlocks(int32 p)
	 {
	    int32 pc;
	    while((pc = dgetpos()) < p)
	    {
	       uint id = (uint) dreadword ();
	       int32 len = dreaddword ();
	       
	       switch(id)
	       {
		  case H3DS_CHUNK_MATERIAL:
		     ReadMaterialBlocks(pc+len);
		     break;
		     
		  case H3DS_CHUNK_OBJBLOCK:
		     ReadObjBlocks(pc+len);
		     break;
		     
		  default: dsetpos(pc+len);
	       }
	    }
	 }
	 

	 // Reads in objects; this would be the start of where keyframer
	 // support would be added.
	 void ReadMainBlocks(int32 p)
	 {
	    int32 pc;
	    
	    while((pc = dgetpos ()) < p)
	    {
	       uint id = (uint) dreadword ();
	       int32 len = dreaddword ();
	       
	       switch(id)
	       {
		  case H3DS_CHUNK_OBJMESH:
		     ReadObjMeshBlocks(pc+len);
		     break;
		     
		  case H3DS_CHUNK_KEYFRAMER:
		  default: dsetpos(pc+len);
	       }
	    }
	 }


	 // Reads in a H3dsScene from the disk
	 H3dsScene *HRead3dsScene(Stream *file)
	 {
	    m_InFile = file;
	    
	    int32 pc = dgetpos();
	    uint id = (uint) dreadword ();
	    int32 len = dreaddword ();
	    
	    if (id ==H3DS_CHUNK_MAIN) 
	    {
	       m_Scene = new H3dsScene();
	       ReadMainBlocks(pc+len);
	       
	       return m_Scene;
	    }
	    else
	    {
	       return NULL;
	    }
	 }

	 void CreateSubMesh
	   (Model &model,
	    SubModel &submodel,
	    H3dsMeshObj *meshobj,
	    const H3dsMatList &binding)
	 {
	    // If there's no triangle, don't add a submesh..
	    if (binding.faces.size() == 0)
	       return;

	    // Find material id;
	    int materialid = 0;
	    size_t i;
	    for (i = 0; i < m_Scene->materials.size(); i++)
	    {
	       if (m_Scene->materials[i].name == binding.name)
	       {
		  materialid = i;
		  break;
	       }
	    }

	    bool dblsided = false;
	    if (model.m_Skin.GetMaterial(materialid)->m_Flags &
		MATERIAL_IS_DOUBLESIDED)
	    {
	       dblsided = true;
	    }

	    // Convert triangle list
	    int ntris = binding.faces.size();

	    PrimitiveBlock pb;
	    pb.SetType (PRIM_TRIANGLES);
	    pb.Resize (ntris * 3);

	    for (int j = 0; j < ntris; j++)
	    {
	       H3dsFace &face = meshobj->facelist[binding.faces[j]];
	       pb[j*3] = face.p0;
	       pb[j*3+2] = face.p2;
	       pb[j*3+1] = face.p1;
	    }

	    submodel.Builder()->AddBlock (materialid, pb);


	    if (dblsided)
	    {
	       for (int j = 0; j < ntris; j++)
	       {
		  H3dsFace &face = meshobj->facelist[binding.faces[j]];
		  pb[j*3] = face.p0;
		  pb[j*3+2] = face.p1;
		  pb[j*3+1] = face.p2;
	       }

	       submodel.Builder()->AddBlock (materialid, pb);
	    }
	 }

	 // Create a model using our internal data representation.
	 bool ConvertToModel (const String &name,
			      const String &args,
			      Cache *vc,
			      Model *mdl)
	 {
	    int current;

	    // Start with the material properties
	    // If the model has default materials in it, substitute an
	    // all-white material
	    if (defaultmaterialwarning)
	    {
	       MaterialPtr mat;

	       vc->Get(V_MATERIAL, name + "?WG_DEFAULT", mat);
	       
	       mat->m_Flags = MATERIAL_HAS_LIGHTING;
	       mat->m_DecorName = "WG_DEFAULT";
	       mat->m_Ambient = Color (1.0f, 1.0f, 1.0f, 1.0f);
	       mat->m_Diffuse = Color (1.0f, 1.0f, 1.0f, 1.0f);
	       mat->m_Specular = Color (1.0f, 1.0f, 1.0f, 1.0f);

	       mdl->m_Skin.Insert(mat);
	    }

	    // Material directory : remove the .3ds extension
	    size_t i; //FIXME: use iterator, for_each algorithm
	    String matdir( name, 0, name.size() - 4);


	    /* Feed each material and its properties to OutGL */
	    for (i = 0; i < m_Scene->materials.size(); i++)
	    {
	       H3dsMat *imat = &m_Scene->materials[i];

	       String abs_mat_name = matdir + "/" + imat->name + ".mat";
	       MaterialPtr mat;

	       if (Sys()->FS()->IsFile (abs_mat_name))
	       {
			   vc->Get (V_MATERIAL, abs_mat_name, mat);
	       }
	       else
	       {
		  vc->Get (V_MATERIAL, name + "?" + imat->name, mat);

		  Sys()->Log("%s: Material '%s' inside\n", 
			     name.c_str(),imat->name.c_str());
	       
		  mat->m_Flags = MATERIAL_HAS_LIGHTING;
		  mat->m_DecorName = imat->name;
		  
		  mat->m_Ambient.R = imat->ambient.R * 2;
		  mat->m_Ambient.G = imat->ambient.G * 2;
		  mat->m_Ambient.B = imat->ambient.B * 2;
		  
		  mat->m_Diffuse = imat->diffuse;
		  mat->m_Specular = imat->specular;
		  
		  if (imat->texture.texturename != "")
		  {
		     TexturePtr texture;
		     if (vc->Get (V_TEXTURE, imat->texture.texturename, texture))
		     {
			 texture->Configure();

			 mat->m_Flags |= MATERIAL_HAS_PASS1;
			 mat->m_Passes[0].m_Texture = texture;
			 mat->m_Passes[0].m_Flags |= PASS_HAS_TEXTURE;
		     }
		  }
	       }

	       mdl->m_Skin.Insert (mat);
	    }

	    // Create sub model list with enough elements
	    mdl->m_SubModels.resize (m_Scene->meshobjs.size());

	    // Then do the vertices and polygons
	    // Order is important!
	    current = 0;

	    for (i = 0; i < m_Scene->meshobjs.size(); i++)
	    {
	       H3dsMeshObj *meshobj = &m_Scene->meshobjs[i];
	       SubModel &submodel = mdl->m_SubModels[i];

	       submodel.m_Name = meshobj->name;
	       std::vector<Vertex> vertices;

	       vertices.resize (meshobj->vertices.size());

	       // Swap y and z (.3ds wierdness)
	       for (size_t j = 0; j < vertices.size(); j++)
	       {
		  vertices[j].m_Coord
		     = Vector3 (-meshobj->vertices[j].Y,
				meshobj->vertices[j].Z,
				meshobj->vertices[j].X);
	       }

	       if (meshobj->texcoords.size() > 0)
	       {
		  size_t j;

		  /// Set texture coordinates
		  for (j = 0; j < meshobj->texcoords.size(); j++)
		  {
		     vertices[j].m_TexCoord =
			Vector2
			(meshobj->texcoords[j].X,
			 1.0f - meshobj->texcoords[j].Y);
		  }

		  /// Null texture coords
		  for (; j < vertices.size(); j++)
		     vertices[j].m_TexCoord = Vector2();

		  submodel.Builder()->SetFormat (VertexBuffer::VB_HAS_COORD | VertexBuffer::VB_HAS_UV0);
	       }
	       else
		  submodel.Builder()->SetFormat (VertexBuffer::VB_HAS_COORD);

	       submodel.Builder()->SetVertices
		  (&vertices[0], vertices.size());

	       // Apply this material's translation matrix to this
	       // material's vertices.
	       // This doesn't seem to work as a normal transformation
	       // matrix, so this part is commented out.
	       /*
	       if (meshobj->matrix)
	       {
		  m = meshobj->TraMatrix;

		  for (y=0; y<meshobj->NumVerts; y++)
		  {
		     prime.x = m[0]*meshobj->vertlist[y].x +
		     	       m[3]*meshobj->vertlist[y].y +
			       m[6]*meshobj->vertlist[y].z;
		     prime.y = m[1]*meshobj->vertlist[y].x +
			       m[4]*meshobj->vertlist[y].y +
			       m[7]*meshobj->vertlist[y].z;
		     prime.z = m[2]*meshobj->vertlist[y].x +
			       m[5]*meshobj->vertlist[y].y +
			       m[8]*meshobj->vertlist[y].z;

		     prime.x += m[ 9];
		     prime.y += m[10];
		     prime.z += m[11];

		     meshobj->vertlist[y].x = prime.x;
		     meshobj->vertlist[y].y = prime.y;
		     meshobj->vertlist[y].z = prime.z;
		  }
	       }
	       */

	       /// Add primitive blocks
	       for (size_t k = 0; k < meshobj->bindings.size(); k++)
		  CreateSubMesh (*mdl, submodel,
				 meshobj, meshobj->bindings[k]);

	       /// Compute normal & destroy builder
	       submodel.Build();
	    }

	    mdl->m_Skeleton = NULL;

	    if ( args.find( "center=true" ) != String::npos )
	       mdl->SetupBBox (true);
	    else
	       mdl->SetupBBox (false);

	    scalar height;
	    
	    if ( StringToScalar(args, height) )
	    {
	       if (height == 0.0)
		  return true;

	       mdl->SetRealHeight (height);
	    }

	    return true;
	 }

      public:
	 Loader3DS()
	 {
	    m_InFile = NULL;
	    m_Scene = NULL;
	 }

	 bool CanLoad (ObjectType type, Stream &file, const String& name, const String &args)
	 {
	    if (type != V_MODEL)
	       return false;

	    m_Name = name;
	    m_InFile = &file;
	    uint id = (uint) dreadword ();

	    bool res = (id == H3DS_CHUNK_MAIN);
	    m_InFile = NULL;

	    return res;
	 }

	 bool Load (Object *vis, Stream &file, const String &name, const String &args,
		    Cache *cache = NULL, Progress *progress = NULL,
		    int granularity = 0)
	 {
	    if (vis == NULL || vis->Type() != V_MODEL)
	       return false;

	    m_Name = name;
	    m_DoubleSided = defaultmaterialwarning = alphablending = 0;

	    /* Read in the file and place it in m_Scene */
	    if (HRead3dsScene(&file) == NULL)
	    {
	       Error ("Failed to parse");

	       delete m_Scene;
	       m_Scene = NULL;
	       m_InFile = NULL;

	       return false;
	    }

	    Model *mdl = (Model*) vis;
	    if (ConvertToModel(name, args, cache, mdl) == false)
	       Error ("Cannot convert 3DS model to Ark model");

	    delete m_Scene;
	    m_Scene = NULL;

	    if (defaultmaterialwarning)
	    {
	       Ark::Sys()->Warning
		  ("%s: This model uses default materials. 3D Studio saves\n"
		   "no information about default materials, so View3DS \n"
		   "substitutes an all-white material in its place. \n"
		   "Therefore,some of the model may appear as bright white.\n"
		   "To fix this, name each material in 3D Studio.\n",
		   name.c_str());
	    }

	    if (m_DoubleSided)
	    {
	       Ark::Sys()->Warning
		  ("%s: This model contains double-sided polygons.\n"
		   "The number of triangles in this model has now\n"
		   "DOUBLED. If possible, re-do the model in\n"
		   "3D Studio without double-sided materials.\n"
		   "Use a normal threshold of at least 90.\n"
		   "Also, stripification will not work properly.\n");
	    }

	    m_InFile = NULL;
	    return true;
	 }

	 String GetInformations()
	 {
	    return "Autodesk 3DS Model";
	 }

   }; // class Loader3DS

   void ark_Add3dsLoader (Loaders *loaders)
   {
      loaders->Add (new Loader3DS());
   }

} // namespace Ark
