/* $Id: ArkFileSys.cpp,v 1.31 2002/11/01 22:52:31 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/Ark.h>
#include <Ark/ArkStream.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkFileSys.h>


#include <stdio.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_ZLIB_H
#include <zlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h> // for mkdir
#endif

#ifdef WIN32
 #include <windows.h>
#endif

// STL map
#include <map>

namespace Ark
{

   String StripSlashes(const String& fname)
   {
#ifdef WIN32
      // under WIN32, replace /'s with \'s

      String ret(fname);

      // there is at least one slash
      for (unsigned int i = 0; i < ret.size(); i++)
      {
	  if (ret[i] == '/')
	      ret[i] = '\\';
      }
	  
      return ret;
#else
      return fname;
#endif
  }

   FileSystem::FileSystem() {}
   FileSystem::~FileSystem() {}

   String
   FileSystem::GetFileName (const String &name, bool err /* = true */)
   {
      assert (!name.empty());
      
      // name doesnt contain a mountpoint... so just return it !
      if (name[0] != '{')
      {
	 String fname = StripSlashes( name );
	 return fname;
      }
      
      String::size_type mpos = name.find( "}", 1 );
      
      if (mpos == String::npos)
      {
	 if (err == true)
	    Sys()->Error ("Bad mount-point specification in \"%s\"",
			  name.c_str());
	 
	 return name;
      }
      
      String mpn = name.substr(1, mpos - 1);
      
      std::map<String,String>::iterator it = m_MountPoints.find (mpn);
      String dir;
      
      if (it == m_MountPoints.end())
      {
	 if (err == true)
	    Sys()->Warning ("\"%s\" is not mounted (in \"%s\")", mpn.c_str(),
			    name.c_str());
	 
	 dir = "";
      }
      else
	 dir = it->second;
      
      String real( dir + name.substr(mpos + 1, String::npos) );
      return StripSlashes( real );
   }
   
   bool
   FileSystem::Mount (const String &name, const String &dir)
   {
      assert (name != "");
      assert (dir != "");
      
      m_MountPoints[name] = GetFileName (dir, true);
      
      if (Sys()->IsVerbose ())
	 Sys()->Log ("\"%s\" mounted on {%s}\n", dir.c_str(), name.c_str ());
      
      return true;
   }
   
   bool 
   FileSystem::UnMount (const String &name)
   {
      // for associative containr, call erase() to remove an element
      m_MountPoints.erase( name );
      
      return true;
   }
   
   bool
   FileSystem::IsFile (const String &name)
   {
      String real = GetFileName (name, false);
      FILE *file = fopen (real.c_str(), "r");
      
      if (file)
      {
	 fclose (file);
	 return true;
      }
      else
      {
#ifdef HAVE_ZLIB_H
	 real += ".gz";
	 gzFile file = gzopen (real.c_str(), "r");

	 if (file)
	 {
	    gzclose (file);
	    return true;
	 }
#endif
	 return false;
      }
   }
   
   bool
   FileSystem::IsDir (const String &name)
   {
#ifdef HAVE_DIRENT_H
      String real = GetFileName (name, false);
      DIR *dir = opendir (real.c_str());
      
      if (dir == NULL && errno == ENOTDIR)
	 return false;
      
      closedir (dir);
#endif
      //FIXME: Directory checking under Windows..
      
      return true;
   }
   
   bool 
   FileSystem::MakeDir (const String &path, int mode)
   {
      String real = GetFileName (path, false);
      
#ifdef WIN32
      return CreateDirectory(real.c_str(), NULL) == 0;
#else
      // mkdir() returns 0 on success
      return mkdir (real.c_str(), mode) == 0;
#endif
   }
}
