/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#ifndef __UTILS_HPP
#define __UTILS_HPP

#include "include.hpp"
#include "error.hpp"

/** Replace all occurences of a substring.
    @date Sat Mar 05 2005
    @author Jan Wedekind (wedesoft@users.sourceforge.net) */
std::string replaceAll( const std::string &text,
                        const std::string &toReplace,
                        const std::string &replaceBy );

/** Prepare string for use in an XML document.
    The prefix <TT><![CDATA[</TT>is added. The postfix <TT>]]></TT> is added.
    every occurence of <TT>]]></TT> is replaced by <TT>"]]]]><![CDATA[></TT>.
    @date Sat Mar 05 2005
    @author Jan Wedekind (wedesoft@users.sourceforge.net) */
std::string xmlText( const std::string &text );

/** Generate version-text to version number.
    @param version ( version * 1000 + subversion ) */
std::string versionText( int version );

/** Create temporary XML dictionary.
    Create a temporary XML file with translations, which is included by the
    XSL-file creating docbook-output from anymeal's internal XML format.

    Sourcecode for using \c mkstemp and \c std::fstream in combination was
    found in the bayonne-project.

    @return fileName */
std::string dictionaryFile(void) throw (Error);

/** Create temporary XML map for units.
    Create a temporary XML file for mapping units on Mealmaster abbreviations,
    which is included by the XSL-file converting anymeal's internal XML format
    to Mealmaster.

    Sourcecode for using \c mkstemp and \c std::fstream in combination was
    found in the bayonne-project.

    @return fileName */
std::string mealmasterMapFile(void) throw (Error);

/** Wrapper for finding a ressource with KDE.
    This is a wrapper for the following method-call:
    \code
    KGlobal::dirs()->findResource( type, fileName )
    \endcode
    @param type Type of ressource.
    @param fileName Relative filename of file to be located.
    @return The full filename on success. Otherwise the unmodified relative
    filename is returned. */
std::string findAnyMealFile( const char *type, const char *fileName );

/** Wrapper for getting current language from KDE or using getenv.
    This method will first try to call
    \code
    KGlobal::locale()->language()
    \endcode
    If there is no \c KLocale object, 'en' will be returned.
    @return The two-letter language code (ISO 639-1 standard). */
std::string anyMealLanguage(void);

/** Greatest common divisor.
    Implementation of the
    <A HREF="http://mathworld.wolfram.com/EuclideanAlgorithm.html">Euclidean
    algorithm</A> to find the greatest common divisor of two numbers.
    @param a natural number
    @param b natural number
    @return The greatest common divisor of a and b. */
int gcd( int a, int b );

/** Try to open file.
    This function can be used for testing, wether a directory or file
    exists.
    See manpage <A HREF="man:/open(2)">open(2)</A> for more details.
    @param fileName Name of file to test for.
    @return 0, if the file is readable or else the error number generated
    by the call to \c open. */
int testFile( const std::string &fileName );

/** Create directory and make parent directories as needed.
    Similar to doing
    \verbatim
    mkdir -p dirname
    \endverbatim
    on the command-line.
    
    Also see manpages <A HREF="man:/mkdir(1)">mkdir(1)</A> and
    <A HREF="man:/mkdir(3p)">mkdir(3p)</A>
    @param dirName Name of directory to create */
void createDirRecursively( const std::string &dirName ) throw (Error);

#endif
