/* Copyright (C) 2002, 2003, 2004, 2005, 2006 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include "include.hpp"
#include <boost/shared_ptr.hpp>
#include <fstream>
#include <iostream>
#include <kaboutdata.h>
#include <kapplication.h>
#include <kcmdlineargs.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <ksplashscreen.h>
#include <magic.h>
#include <mysql/mysql.h>
#include <qimage.h>
#include <qtextcodec.h>
#include <XPath/XPathInit.hpp>
#include <util/PlatformUtils.hpp>
#include "anyMeal.h"
#include "mealMasterCompiler.hpp"
#include "mysqlDatabase.hpp"
#include "mysqlNetwork.hpp"
#include "recoder.hpp"
#include "mealMasterStoreErroneousHandler.hpp"
#include "utils.hpp"
#include "xslCompiler.hpp"
#include "xsu.hpp"

using namespace boost;
using namespace std;

// Default-arguments didn't work for some reason.
static const KCmdLineOptions options[] = {
  { "i", 0, 0 },
  { "input <file>", I18N_NOOP( "Input file name (default: standard input)." ),
    0 },
  { "o", 0, 0 },
  { "output <file>", I18N_NOOP( "Output file name (default: standard "
                                "output)." ), 0 },
  { "x", 0, 0 },
  { "mm2xml", I18N_NOOP( "Select Mealmaster to anyMeal-XML compiler" ), 0 },
  { "e", 0, 0 },
  { "mm2xmlerr <file>", I18N_NOOP( "Specify name of file for appending "
                                   "erroneous Mealmaster recipes." ), 0 },
  { "z", 0, 0 },
  { "xsu", I18N_NOOP( "Select XML-SQL-utility as compiler." ), 0 },
  { "s", 0, 0 },
  { "server <name>", I18N_NOOP( "Server hosting database. [localhost]" ), 0 },
  { "d", 0, 0 },
  { "database <name>", I18N_NOOP( "Name of MySQL database." ), 0 },
  { "u", 0, 0 },
  { "user <name>", I18N_NOOP( "User account for accessing database. "
                              "[$USER]" ), 0 },
  { "p", 0, 0 },
  { "password <password>", I18N_NOOP( "Password of database user. []" ), 0 },
  { "q", 0, 0 },
  { "sqlin", I18N_NOOP( "Select XSL script for converting XML-queries to "
                        "SQL" ), 0 },
  { "r", 0, 0 },
  { "sqlout", I18N_NOOP( "Select XSL script for converting raw XML to "
                         "anyMeal-XML" ), 0 },
  { "b", 0, 0 },
  { "docbook", I18N_NOOP( "Select XSL script for converting anyMeal-XML "
                          "to Docbook" ), 0 },
  { "l", 0, 0 },
  { "recipeml", I18N_NOOP( "Select XSL script for converting anyMeal-XML "
                           "to RecipeML" ), 0 },
  { "t", 0, 0 },
  { "html", I18N_NOOP( "Select XSL script for converting Docbook to HTML" ),
    0 },
  { "f", 0, 0 },
  { "fo", I18N_NOOP( "Select XSL script for converting Docbook to Formatted "
                     "Objects XML" ), 0 },
  { "m", 0, 0 },
  { "xml2mm", I18N_NOOP( "Select XML to Mealmaster compiler" ), 0 },
  { "y", 0, 0 },
  { "rm2xml", I18N_NOOP( "Select RecipeML to anyMeal-XML compiler" ), 0 },
  { "+[input file]", I18N_NOOP("Input file name (default: standard input)."),
    0 },
  KCmdLineLastOption };

bool cli(void) throw (Error)
{
  bool guiDisabled = false;

  KCmdLineArgs::addCmdLineOptions( options );

  KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

  // Get file names and other data.
  const char
    *inputFileName      = ( args->count() > 0 ? args->arg( 0 ) :
                            args->getOption( "input" ).data() ),
    *outputFileName     = args->getOption( "output"    ).data(),
    *parseErrorFileName = args->getOption( "mm2xmlerr" ).data(),
    *server             = args->getOption( "server"    ).data(),
    *database           = args->getOption( "database"  ).data(),
    *user               = ( args->getOption( "user"      ).isEmpty() ?
                            getenv( "USER" ) :
                            args->getOption( "user"      ).data() ),
    *password           = args->getOption( "password"  ).data();

  // Select compiler.
  enum { NONE, MM2AnyMeal, SQL2RAW, XML2SQL, RAW2AnyMeal, AnyMeal2Docbook,
         AnyMeal2RecipeML, Docbook2HTML, Docbook2FO, AnyMeal2MM,
	 RecipeML2AnyMeal } compiler = NONE;
  if ( args->isSet( "mm2xml"   ) ) compiler = MM2AnyMeal;
  if ( args->isSet( "xsu"      ) ) compiler = SQL2RAW;
  if ( args->isSet( "sqlin"    ) ) compiler = XML2SQL;
  if ( args->isSet( "sqlout"   ) ) compiler = RAW2AnyMeal;
  if ( args->isSet( "docbook"  ) ) compiler = AnyMeal2Docbook;
  if ( args->isSet( "recipeml" ) ) compiler = AnyMeal2RecipeML;
  if ( args->isSet( "html"     ) ) compiler = Docbook2HTML;
  if ( args->isSet( "fo"       ) ) compiler = Docbook2FO;
  if ( args->isSet( "xml2mm"   ) ) compiler = AnyMeal2MM;
  if ( args->isSet( "rm2xml"   ) ) compiler = RecipeML2AnyMeal;

  if ( compiler != NONE || inputFileName != NULL || 
       outputFileName != NULL || parseErrorFileName != NULL ||
       server != NULL || database != NULL || password != NULL ||
       user != getenv( "USER" ) ) {

    guiDisabled = true;

    // The application object (no GUI)..
    KApplication app( true, false );

    // Tell Qt, that source-code is encoded in UTF-8.
    QTextCodec::setCodecForCStrings
      ( QTextCodec::codecForName( "UTF-8" ) );

    try {

      // Initialise recoder library
      // Recoder::outer = recode_new_outer( false );
      // ERRORMACRO( Recoder::outer != NULL, Error, ,
      //             "Failure initialising recoder library." );

      ERRORMACRO( args->count() <= args->getOption( "input" ).isNull() ? 1 : 0,
                  Error, ,
                  i18n( "Only one input file may be specified." ) );
      
      // Open input stream.
      shared_ptr< ifstream > inputFile;
      istream *input = &cin;
      if ( inputFileName ) {
        inputFile = shared_ptr< ifstream >( new ifstream( inputFileName,
                                                          ios::binary ) );
        input = inputFile.get();
      };

      // Open output stream.
      shared_ptr< ofstream > outputFile;
      ostream *output = &cout;
      if ( outputFileName ) {
        outputFile = shared_ptr< ofstream >( new ofstream( outputFileName,
                                                           ios::binary ) );
        output = outputFile.get();
      };

      switch ( compiler ) {
      case MM2AnyMeal: {
        shared_ptr< ostream > stream;
        MealMasterParseErrorHandlerPtr parseErrorHandler;

        if ( parseErrorFileName != NULL ) {
          stream = shared_ptr< ostream >
            ( new ofstream( parseErrorFileName, ios::binary ) );
          parseErrorHandler = MealMasterParseErrorHandlerPtr
            ( new MealMasterStoreErroneousHandler( stream.get(), parseErrorFileName ) );
        } else
          parseErrorHandler = MealMasterParseErrorHandlerPtr( new MealMasterParseErrorHandler );

        MealMasterCompiler( -1, parseErrorHandler ).translate( *input, *output );
        break;}
      case SQL2RAW: {
        ERRORMACRO( database != NULL, Error, ,
                    i18n( "No datasource selected." ) );
        assert( user != NULL );
        DatabasePtr
          connection( new MySQLDatabase
                      ( MySQLServerPtr
                        ( new MySQLNetwork
                          ( user, password == NULL ? "" : password,
                            server == NULL ? "localhost" : server,
                            MYSQL_PORT ) ),
                        database ) );
        XSU( connection ).translate( *input, *output );
        break;}
      case XML2SQL: {
        // I don't know, why I have to store it in a temporary variable first.
        // But otherwise it won't compile.
        string fileName( findAnyMealFile( "appdata", "scripts/mysqlIn.xsl" ) );
        XSLCompiler( fileName,
                     FormatterListener::OUTPUT_METHOD_TEXT, SQLENCODING, "" ).
          translate( *input, *output );
        break;}
      case RAW2AnyMeal: {
        string fileName( findAnyMealFile( "appdata", "scripts/mysqlOut.xsl" ) );
        XSLCompiler( fileName,
                     FormatterListener::OUTPUT_METHOD_XML, "UTF-8", "" ).
          translate( *input, *output );
        break;}
      case AnyMeal2Docbook: {
        string fileName( findAnyMealFile( "appdata",
                                          "scripts/recipeToDocbook.xsl" ) );
        XSLCompiler compiler
          ( fileName,
            FormatterListener::OUTPUT_METHOD_XML, HTMLENCODING,
            findAnyMealFile( "appdata", "scripts/anymeal.xsd" ) );
        // Relative path (quoted)!
        compiler.setParam( "lfile", string( "\"file://" ) + dictionaryFile() +
                           "\"" );
        compiler.setParam( "lname", string( "\"" ) + anyMealLanguage() +
                           "\"" );
        compiler.translate( *input, *output );
        break;}
      case AnyMeal2RecipeML: {
        string fileName( findAnyMealFile( "appdata",
                                          "scripts/recipeToRecipeML.xsl" ) );
        XSLCompiler( fileName,
                     FormatterListener::OUTPUT_METHOD_XML, HTMLENCODING,
                     findAnyMealFile( "appdata", "scripts/anymeal.xsd" ) ).
          translate( *input, *output );
        break;};      
      case Docbook2HTML: {
        XSLCompiler( DOCBOOK"/html/docbook.xsl",
                     FormatterListener::OUTPUT_METHOD_HTML, HTMLENCODING, "" ).
          translate( *input, *output );
        break;}
      case Docbook2FO: {
        string fileName( findAnyMealFile( "appdata",
                                          "scripts/docbookToFo.xsl" ) );
        XSLCompiler compiler
          ( fileName,
            FormatterListener::OUTPUT_METHOD_XML, HTMLENCODING, "" );
        compiler.setParam( "paper.type", "\"A4\"" );
        compiler.translate( *input, *output );
        break;}
      case AnyMeal2MM: {
        string fileName( findAnyMealFile( "appdata",
                                          "scripts/recipeToMealMaster.xsl" ) );
        XSLCompiler compiler
	  ( fileName,
	    FormatterListener::OUTPUT_METHOD_TEXT, MMENCODING,
	    findAnyMealFile( "appdata", "scripts/anymeal.xsd" ) );
        compiler.setParam( "mfile", string( "\"file://" ) +
			   mealmasterMapFile() + "\"" );
	compiler.translate( *input, *output );
        break;}
      case RecipeML2AnyMeal: {
        string fileName( findAnyMealFile( "appdata",
                                          "scripts/recipeMLToRecipe.xsl" ) );
        XSLCompiler( fileName,
                     FormatterListener::OUTPUT_METHOD_XML, "UTF-8",
		     findAnyMealFile( "appdata", "scripts/recipeML.xsd" ) ).
          translate( *input, *output );
	break;}
      default:
        ERRORMACRO( false, Error, , i18n( "No compiler was selected." ) );
      };

    } catch ( Error &e ) {

      cerr << i18n( "Error: " ) << e.what() << endl;
      
    };

    // Destruction of recoder crashes. Symbol clash with mysql library?
    // if ( Recoder::outer != NULL ) {
    //   recode_delete_outer( Recoder::outer );
    //   Recoder::outer = NULL;
    // };
    
  };
  
  return guiDisabled;
}

class AnyMealIcons : public QMimeSourceFactory
{
public:
  AnyMealIcons(void) {}
  const QMimeSource* data( const QString& abs_name ) const {
#ifndef NDEBUG
      cerr << "Request for icon \"" <<  abs_name << "\"." << endl;
#endif
    const QMimeSource* d = QMimeSourceFactory::data( abs_name );
    if ( d || abs_name.isNull() ) return d;
    QPixmap pixmap( iconLoader.loadIcon( abs_name.left
                                         ( abs_name.length() - 4 ),
                                         KIcon::Toolbar, 32,
                                         KIcon::DefaultState, 0, true ) );
    if ( !pixmap.isNull() ) {
      ((QMimeSourceFactory *)this)->setPixmap( abs_name, pixmap );
    } else {
#ifndef NDEBUG
      cerr << "Could not find icon \"" <<  abs_name
           << "\" in standard directory." << endl;
#endif
        if ( pixmap.load( QString( "icons/hi32-action-" ) + abs_name ) )
          ((QMimeSourceFactory *)this)->setPixmap( abs_name, pixmap );
        else if ( pixmap.load( QString( "icons/hi32-app-" ) + abs_name ) )
          ((QMimeSourceFactory *)this)->setPixmap( abs_name, pixmap );
    };
    return QMimeSourceFactory::data( abs_name );
  };
protected:
  KIconLoader iconLoader;
};

magic_t magic = NULL;

shared_ptr< KSplashScreen > splash;

int gui(void)
{
  int retVal;

  // The application object.
  KApplication app( true, true );


  // Mimesource-factory for loading icons. KApplication seems to spoil
  // initialisation-code generated by "uic -embed" :-(
  AnyMealIcons anyMealIcons;
  QMimeSourceFactory::defaultFactory()->addFactory( &anyMealIcons );

  // Tell Qt, that source-code is encoded in UTF-8.
  QTextCodec::setCodecForCStrings
    ( QTextCodec::codecForName( "UTF-8" ) );

  // Not a matter of taste. It's because the default style has an erroneous
  // palette for QComboTableItem.
  // app.setStyle( "windows" );

  try {
  
    // Open splash-screen.
    QPixmap pixmap( findAnyMealFile( "appdata", "pics/splash.jpg" ) );
    ERRORMACRO( !pixmap.isNull(), Error, ,
                "Could not load image \""
                << findAnyMealFile( "appdata", "pics/splash.jpg" ) << "\"." );
    splash = shared_ptr< KSplashScreen >( new KSplashScreen( pixmap ) );
    splash->show();
  
    // Initialise recoder-library
    splash->message( i18n( "Initialising recode-library ..." ) );
    Recoder::outer = recode_new_outer( false );
    ERRORMACRO( Recoder::outer != NULL, Error, ,
                "Failed to initialise recoder library." );

    // Initialise magic-library.
    splash->message( i18n( "Initialising magic-library ..." ) );
    magic = magic_open( MAGIC_NONE );
    ERRORMACRO( magic != NULL, Error, ,
                i18n( "Failed to initialise magic library." ) );
    ERRORMACRO( magic_load( magic, NULL ) == 0, Error, ,
                magic_error( magic ) );

    splash->message( i18n( "Reading icons ..." ) );
    AnyMeal anyMeal( NULL, "anyMeal mainwindow" );
    splash->message( i18n( "Opening main window ..." ) );
    app.setMainWidget( &anyMeal );
    anyMeal.show();
    splash.reset();
    retVal = app.exec();

  } catch ( Error &e ) {

    // Remove splash-screen, if it is open already.
    splash.reset();

    // This would be a serious software error.
    KMessageBox::error( NULL, e.what() );
    retVal = 1;

  };

  // Destruct magic library.
  if ( magic != NULL ) {
    magic_close( magic );
    magic = NULL;
  };
  // Destruction of recoder crashes. Symbol clash with mysql library?
  // if ( Recoder::outer != NULL ) {
  //   recode_delete_outer( Recoder::outer );
  //   Recoder::outer = NULL;
  // };

  // Remove factory.
  QMimeSourceFactory::defaultFactory()->removeFactory( &anyMealIcons );

  return retVal;
}

int main( int argc, char *argv[] )
{
  int retVal = 0;

  // Disable SuSE KDE integration, because it is breaking this application.
  // setenv( "QT_NO_KDE_INTEGRATION", "1", 1 );

  // Static initialisation of xerces must be done in main thread!
  XMLPlatformUtils::Initialize();
  // Static initialisation of xalan must be done in main thread!
  XalanTransformer::initialize();

  {

    // Static initalisation of XPath utilities.
    XPathInit theXPathInit;

    // Construct information about anymeal.
    KAboutData about
      ( "anymeal", I18N_NOOP( "AnyMeal" ), VERSION,
        I18N_NOOP( "recipe database application" ),
        KAboutData::License_GPL,
        I18N_NOOP( "Copyright (C) 2002, 2003, 2004, 2005, 2006, Jan Wedekind, "
                   "Sheffield, United Kingdom" ),
        I18N_NOOP( "AnyMeal comes with ABSOLUTELY NO WARRANTY.\n"
                   "This is free software, and you are welcome to "
                   "redistribute it\n"
                   "under certain conditions; see file 'COPYING' for "
                   "details.\n"
                   "See 'anymeal.lsm' file for contact information." ),
        "http://sourceforge.net/projects/anymeal/",
        "wedesoft@users.sourceforge.net" );
    about.addAuthor( "Jan Wedekind", I18N_NOOP( "Main developer" ),
                     "jan@wedesoft.de",
                     "http://www.wedesoft.de/" );
    about.addCredit( "Sandro Tosi",
                     I18N_NOOP( "Debian-package and man-page." ),
                     "matrixhasu@altervista.org",
                     "http://matrixhasu.altervista.org/" );
    about.addCredit( "Scott Welliver",
                     I18N_NOOP( "Project inspired from original "
                                "Mealmaster-software." ),
                     NULL,
                     "http://episoft.home.comcast.net/" );
    about.addCredit( "Roland Jesse",
                     I18N_NOOP( "Recipe-editor inspired from "
                                "mango-software." ),
                     "rjesse@users.sourceforge.net",
                     "http://mango.sf.net/" );
    about.addCredit( "Wikibooks Cookbook",
                     I18N_NOOP( "GFDL licensed photos." ),
                     NULL,
                     "http://en.wikibooks.org/wiki/Cookbook" );
    about.addCredit( "krecipes team",
                     I18N_NOOP( "Database-wizard inspired from "
                                "krecipes-software." ),
                     NULL,
                     "http://krecipes.sourceforge.net/" );
    about.addCredit( "Free Software Foundation",
                     I18N_NOOP( "The GNU/Linux platform." ),
                     "gnu@gnu.org",
                     "http://www.gnu.org/gnu/the-gnu-project.html" );
    about.addCredit( "GCC team",
                     I18N_NOOP( "The GNU-C++ compiler" ),
                     "gcc@gnu.org",
                     "http://www.gnu.org/software/gcc/" );
    about.addCredit( "Bram Moolenaar and others",
                     I18N_NOOP( "The vi editor" ),
                     NULL,
                     "http://www.vim.org/" );
    about.addCredit( "Dimitri van Heesch",
                     I18N_NOOP( "Doxygen documentation system" ),
                     "dimitri@stack.nl",
                     "http://www.stack.nl/~dimitri/doxygen/" );
    about.addCredit( "Apache Software Foundation",
                     I18N_NOOP( "Xalan-C and Xerces-C XML-libraries." ),
                     "apache@apache.org",
                     "http://xml.apache.org/" );
    about.addCredit( "MySQL Developers",
                     I18N_NOOP( "MySQL database software" ),
                     NULL,
                     "http://dev.mysql.com/" );
    about.addCredit( "Boost team",
                     I18N_NOOP( "Boost C++ library" ),
                     NULL,
                     "http://www.boost.org/" );
    about.addCredit( "KDE team",
                     I18N_NOOP( "KDE desktop" ),
                     NULL,
                     "http://www.kde.org/" );
    about.addCredit( "Matthias Kiefer",
                     I18N_NOOP( "kbabel" ),
                     "kiefer@kde.org",
                     "http://i18n.kde.org/tools/kbabel/" );
    about.addCredit( "Dia team",
                     I18N_NOOP( "Diagram editor Dia" ),
                     NULL,
                     "http://www.gnome.org/projects/dia/" );
    about.addCredit( "Graphviz team",
                     I18N_NOOP( "Graph visualization software Graphviz" ),
                     NULL,
                     "http://www.graphviz.org/" );
    about.addCredit( "Docbook team",
                     I18N_NOOP( "Docbook XSL stylesheets" ),
                     NULL,
                     "http://docbook.sourceforge.net/" );
    about.addCredit( "Nick Gorham",
                     I18N_NOOP( "unixODBC software" ),
                     "nick.gorham@easysoft.com",
                     "http://www.unixodbc.org/" );
    about.addCredit( "Trolltech",
                     I18N_NOOP( "Qt graphical user-interface library" ),
                     "info@trolltech.com",
                     "http://www.trolltech.com/" );
    about.addCredit( "Sourceforge",
                     I18N_NOOP( "Hosting and advertising for AnyMeal" ),
                     NULL,
                     "http://sourceforge.net/" );
    about.addCredit( "Noah Levitt",
                     I18N_NOOP( "GNOME Unicode character map" ),
                     "nlevitt@columbia.edu",
                     "http://gucharmap.sourceforge.net/" );
    about.addCredit( "Formatdata Company",
                     I18N_NOOP( "This product is RecipeML compatible." ),
                     "RecipeML@formatdata.com",
                     "http://www.formatdata.com/recipeml/" );
    about.setTranslator( I18N_NOOP("_: NAME OF TRANSLATORS\nYour names"),
                         I18N_NOOP("_: EMAIL OF TRANSLATORS\nYour emails") );

    KCmdLineArgs::init( argc, argv, &about );

    if ( !cli() ) retVal = gui();

  };

  // Terminate Xalan...
  XalanTransformer::terminate();
  // Terminate Xerces...
  XMLPlatformUtils::Terminate();
  // Clean up the ICU, if it's integrated...
  // XalanTransformer::ICUCleanUp();
  return retVal;
}
