//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qstring.h>
#include <qtextstream.h>
#include <qdom.h>

//Projectwide includes
#include "settinggroup.h"
#include "setting.h"

//==============================================
SettingGroup::SettingGroup(QString name)
{
  this->name = name;
  firstSetting = NULL;
  lastSetting = NULL;
  next = NULL;
}
//==============================================
SettingGroup::~SettingGroup()
{
  Setting* cur = firstSetting;
  while(cur != NULL)
  {
    Setting* t = cur->getNext();
    delete cur;
    cur = t;
  }
}
//==============================================
QString SettingGroup::getName()
{
  return name;
}
//==============================================
QString SettingGroup::getValue(QString key)
{
  Setting* cur = firstSetting;
  while(cur != NULL)
  {
    if(cur->getKey().compare(key) == 0)
    {
      return cur->getValue();
    }
    cur = cur->getNext();
  }
  return "-1";
}
//==============================================
void SettingGroup::resetSetting(QString key)
{
  Setting* cur = firstSetting;
  while(cur != NULL)
  {
    if(cur->getKey().compare(key) == 0)
    {
      cur->resetSetting();
    }
    cur = cur->getNext();
  }
}
//==============================================
void SettingGroup::setValue(QString key, QString value)
{
  Setting* cur = firstSetting;
  while(cur != NULL)
  {
    if(cur->getKey().compare(key) == 0)
    {
      cur->setValue(value);
      return;
    }
    cur = cur->getNext();
  }

  //setting not found, create new one and append to list
  cur = new Setting(key, value);
  if(firstSetting == NULL)
    firstSetting = cur;
  else
    lastSetting->setNext(cur);
  lastSetting = cur;
}
//==============================================
SettingGroup* SettingGroup::getNext()
{
  return next;
}
//==============================================
void SettingGroup::setNext(SettingGroup* next)
{
  this->next = next;
}
//==============================================
void SettingGroup::saveSettings(QTextStream& stream)
{
  stream << "  <group name=\"" << getName() << "\">\n";

  //iterate over every setting
  Setting* cur = firstSetting;
  while(cur != NULL)
  {
    stream << "    <setting key=\"" << cur->getKey() << "\" value=\"" << cur->getValue() << "\"/>\n";
    cur = cur->getNext();
  }

  stream << "  </group>\n";
}
//==============================================
void SettingGroup::loadSettings(QDomNode& root)
{
  //iterate over all children (Settings)
  QDomNode node = root.firstChild();
  QDomText val;
  while( !node.isNull() )
  {
    if( node.isElement() && node.nodeName() == "setting" )
    {
      //find key and value, if either is missing move on to next setting
      QDomNamedNodeMap attributes = node.attributes();
      if(attributes.namedItem("key").isNull() || attributes.namedItem("value").isNull())
      {
        node = node.nextSibling();
        continue;
      }


      QString k = attributes.namedItem("key").nodeValue();
      QString v = attributes.namedItem("value").nodeValue();

      //key and value found -> add new setting
      setValue( attributes.namedItem("key").nodeValue(),
                    attributes.namedItem("value").nodeValue() );
    }

    //move on to next setting
    node = node.nextSibling();
  }
}
//==============================================
