# -*- coding: utf-8 -*-

# Copyright (c) 2008 - 2010 Lukas Hetzenecker <LuHe@gmx.at>

import pickle
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from PyQt4.QtSql import *
import ui.ui_settings
import ui.resource_rc
import lib.device
import lib.obex_wrapper
from lib.classes import *

ANIMATION_SUPPORT = "QPropertyAnimation" in dir()

class Settings(QDialog,  ui.ui_settings.Ui_Settings):
    def __init__(self, parent, main):
        super(Settings,  self).__init__(parent)

        self.parent = parent
        self.main = main

        self.log = main.log
        self.connection = main.connection
        self.database = main.database
        self.settings = main.settings
        self.helper = main.helper

        self.setupUi(self)
        main.setupButtonBox(self.buttonBox)

        self.connect(self.connection,  SIGNAL("scanStarted"), self.scanStarted)
        self.connect(self.connection,  SIGNAL("foundDevice"), self.scanFoundDevice)
        self.connect(self.connection,  SIGNAL("scanFailed"), self.scanFailed)
        self.connect(self.connection,  SIGNAL("scanCompleted"), self.scanCompleted)
        self.connect(self.addDeviceButton,  SIGNAL("clicked()"),  self.addDevice)
        self.connect(self.removeDeviceButton,  SIGNAL("clicked()"),  self.removeDevice)
        self.connect(self.addContactButton,  SIGNAL("clicked()"),  self.addContact)
        self.connect(self.removeContactButton,  SIGNAL("clicked()"),  self.removeContact)
        self.connect(self.rescanButton,  SIGNAL("clicked()"),  self.scanStart)
        self.connect(self.buttonBox,  SIGNAL("clicked(QAbstractButton *)"),  self.apply)
        self.connect(self.allDevicesList,  SIGNAL("customContextMenuRequested(QPoint)"),  self.showDevicesListContextMenu)
        
        self.sqliteDriverNotFoundLabel.setHidden(QSqlDatabase.drivers().contains("QSQLITE"))
        self.mysqlDriverNotFoundLabel.setHidden(QSqlDatabase.drivers().contains("QMYSQL"))
        
        self.show()

    def show(self):
        self.loadSettings()
        self.scanStart()
        QDialog.show(self)

    def loadSettings(self):
        # Read values
        plugin =  self.settings.setting("general/connectionPlugin")
        pluginList = lib.device.Device(self, self.main).plugins()
        port = self.settings.setting("bluetooth/port")
        lastName = self.settings.setting("bluetooth/lastName")
        showCellnumber = not self.settings.setting("contacts/hideCellnumber")
        sendMessageOnReturn = self.settings.setting("general/sendMessageOnReturn")
        markAsRead = self.settings.setting("messages/markAsRead")
        saveMessages = self.settings.setting("messages/saveAllMessages")
        lastMessages = self.settings.setting("messages/displayLast")
        showPopup = self.settings.setting("popups/show")
        timeoutPopup = self.settings.setting("popups/timeout")
        animatePopup = self.settings.setting("popups/animate")

        minimizeOnClose = self.settings.setting("windows/main/minimizeOnClose")
        if minimizeOnClose == 2:
            quitOnClose = True
        else:
            quitOnClose = False

        ignoreAllContacts = self.settings.setting("contacts/ignoreAll")
        displayIcon = self.settings.setting("contacts/displayIcon")

        db_type = self.settings.setting("database/type")
        if db_type == "sqlite":
            db_type = 0
        else:
            db_type = 1

        mysql_host = self.settings.setting("database/mysql/host")
        mysql_port = self.settings.setting("database/mysql/port")
        mysql_user = self.settings.setting("database/mysql/user")
        mysql_pass = self.settings.setting("database/mysql/pass")
        mysql_database = self.settings.setting("database/mysql/database")

        # Write values
        self.pluginBox.clear()

        for item in pluginList:
            self.pluginBox.addItem(item)

        self.pluginBox.setCurrentIndex(self.pluginBox.findText(plugin))
        self.portBox.setValue(port)

        self.selectedDevicesList.clear()
        for device in self.database.devices():
            item = QListWidgetItem(self.selectedDevicesList)
            item.setText(device.name())
            item.setToolTip(device.bluetoothAddress())

        self.allContactsList.clear()
        self.blockedContactsList.clear()
        if not self.database.contactCount():
            item = QListWidgetItem(self.allContactsList)
            item.setText(self.tr("No contacts available"))
        else:
            for contact in self.database.contacts(True):
                if contact.isIgnored():
                    item = QListWidgetItem(self.blockedContactsList)
                else:
                    item = QListWidgetItem(self.allContactsList)
                
                item.setData(Roles.ContactRole,  QVariant(contact))
                item.setText(contact.name())

        self.allContactsList.sortItems(Qt.AscendingOrder)

        self.showCellnumberBox.setCheckState(self.checkBoxValue(showCellnumber))
        self.sendMessageOnReturnBox.setCheckState(self.checkBoxValue(sendMessageOnReturn))
        self.markAsReadBox.setCheckState(self.checkBoxValue(markAsRead))
        self.saveMessagesBox.setCheckState(self.checkBoxValue(saveMessages))
        self.lastMessagesBox.setValue(lastMessages)
        self.showPopupBox.setCheckState(self.checkBoxValue(showPopup))
        self.timeoutPopupBox.setValue(timeoutPopup)
        self.animatePopupBox.setCheckState(self.checkBoxValue(animatePopup))
        self.quitOnCloseBox.setCheckState(self.checkBoxValue(quitOnClose))
        self.ignoreAllContactsBox.setCheckState(self.checkBoxValue(ignoreAllContacts))
        self.displayIconBox.setCheckState(self.checkBoxValue(displayIcon))
        self.typeBox.setCurrentIndex(db_type)
        self.mysql_hostLine.setText(mysql_host)
        self.mysql_portBox.setValue(mysql_port)
        self.mysql_userLine.setText(mysql_user)
        self.mysql_passLine.setText(mysql_pass)
        self.mysql_databaseLine.setText(mysql_database)

        if showPopup:
            self.popupLabel_1.setEnabled(True)
            self.popupLabel_2.setEnabled(True)
            self.timeoutPopupBox.setEnabled(True)
        
        # Only enable "Animate popups" when there is animation support in PyQt4
        # This is true for Qt versions >= 4.6
        self.animatePopupBox.setEnabled(ANIMATION_SUPPORT)

    def scanStart(self):
        self.connection.scan()

    def scanStarted(self):
        self.log.info(QString("Device scan started..."))
        self.allDevicesList.clear()
        item = QListWidgetItem(self.allDevicesList)
        item.setText(self.tr("Scanning..."))
        item.setIcon(QIcon(":/wait"))

        self.rescanButton.setEnabled(False)

    def scanFoundDevice(self,  addr,  name):
        self.log.info(QString("Found device '%1' with MAC %2").arg(name,  addr))
        item = QListWidgetItem(self.allDevicesList)
        item.setText(name)
        item.setToolTip(addr)

    def scanFailed(self,  message):
        self.log.error(QString("Device scan failed with error: %1").arg(message))
        self.allDevicesList.clear()
        scanFailedItem = QListWidgetItem(self.allDevicesList)
        scanFailedItem.setText(self.tr("Scanning failed: %1").arg(message))
        scanFailedItem.setToolTip(self.tr("Scanning failed: %1").arg(message))
        scanFailedItem.setIcon(QIcon(":/dialog-close"))

        self.rescanButton.setEnabled(True)

    def scanCompleted(self):
        self.log.info(QString("Device scan completed"))
        self.allDevicesList.takeItem(0)
        self.scanStartedItem= None
        
        if self.allDevicesList.count() == 0:
            item = QListWidgetItem(self.allDevicesList)
            item.setText(self.tr("No devices found"))
        
        self.rescanButton.setEnabled(True)

    def showDevicesListContextMenu(self,  pos):
        index = self.allDevicesList.indexAt(pos)

        if not index.isValid():
            return

        item = self.allDevicesList.itemAt(pos)
        
        if not item.toolTip() or not self.connection.isValidBluetoothAddress(str(item.toolTip())):
            return

        menu = QMenu(self)
        device = Device(bluetoothAddress=str(item.toolTip()))
        
        sendPythonAction = QAction(self.tr("Send Python SIS file to device"),  menu)
        sendApplicationAction = QAction(self.tr("Send Application SIS file to device"),  menu)
        
        sendPythonAction.setEnabled(lib.obex_wrapper.FOUND_OBEX)
        sendApplicationAction.setEnabled(lib.obex_wrapper.FOUND_OBEX)
        
        menu.addAction(sendPythonAction)
        menu.addAction(sendApplicationAction)
        
        self.connect(sendPythonAction,  SIGNAL("triggered()"),
                                lambda : self.sendFile(device, self.main.pythonSis,  self.main.pythonSisRegexp))
        self.connect(sendApplicationAction,  SIGNAL("triggered()"),
                                lambda : self.sendFile(device, self.main.applicationSis,  self.main.applicationSisRegexp))
        
        menu.popup(QCursor.pos())

    def sendFile(self,  device,  sis,  regexp):
        self.connection.scanStop()
        self.helper.sendFile(self,  device,  sis,  regexp)

    def addDevice(self):
        try:
            device = self.allDevicesList.currentItem()
            name = device.text()
            addr = device.toolTip()
        except:
            return

        # Only real devices
        if not self.connection.isValidBluetoothAddress(str(addr)):
            return

        # Don't add a device twice
        if self.selectedDevicesList.findItems(name,  Qt.MatchExactly):
            return

        if name:
            item = QListWidgetItem(self.selectedDevicesList)
            item.setText(name)
            item.setToolTip(addr)

    def removeDevice(self):
        try:
            device = self.selectedDevicesList.currentRow()
            self.selectedDevicesList.takeItem(device)
        except:
            return

    def addContact(self):
        try:
            item = self.allContactsList.currentItem()
            contact = item.data(Roles.ContactRole).toPyObject()
            name = contact.name()
        except:
            return

        # Don't add contacts twice
        if self.blockedContactsList.findItems(name,  Qt.MatchExactly):
            return

        contact.ignore()
        tmp = QListWidgetItem(item)
        self.blockedContactsList.addItem(tmp)
    
    def removeContact(self):
        try:
            contact = self.blockedContactsList.currentRow()
            self.blockedContactsList.takeItem(contact)
        except:
            return

    def apply(self,  button):
        if self.buttonBox.buttonRole(button) == QDialogButtonBox.ApplyRole:
            self.saveSettings()

    def accept(self):
        if self.saveSettings():
            self.close()

    def saveSettings(self):
        # Read values
        plugin = str(self.pluginBox.currentText())
        port = int(self.portBox.value())

        self.database.deviceHideAll()
        for row in range(self.selectedDevicesList.count()):
            item = self.selectedDevicesList.item(row)
            device = Device()
            device.setName(str(item.text()))
            device.setBluetoothAddress(str(item.toolTip()))
            self.database.deviceAdd(device)

        self.database.contactUnignoreAll()
        for row in range(self.blockedContactsList.count()):
            item = self.blockedContactsList.item(row)
            contact = item.data(Roles.ContactRole).toPyObject()
            self.database.contactIgnoreUpdate(contact)

        showCellnumber = bool(self.showCellnumberBox.checkState())
        sendMessageOnReturn = bool(self.sendMessageOnReturnBox.checkState())
        markAsRead = bool(self.markAsReadBox.checkState())
        saveMessages = bool(self.saveMessagesBox.checkState())
        lastMessages = int(self.lastMessagesBox.value())
        showPopup = bool(self.showPopupBox.checkState())
        timeoutPopup = int(self.timeoutPopupBox.value())
        animatePopup = bool(self.animatePopupBox.checkState())
        quitOnClose = bool(self.quitOnCloseBox.checkState())
        ignoreAllContacts = bool(self.ignoreAllContactsBox.checkState())
        displayIcon = bool(self.displayIconBox.checkState())
        type = str(self.typeBox.currentText()).lower()
        #sqlite_file = str(QFileInfo(self.settings.fileName()).absoluteDir().absolutePath()) + "/messages.db"
        mysql_host = str(self.mysql_hostLine.text())
        mysql_port = int(self.mysql_portBox.value())
        mysql_user = str(self.mysql_userLine.text())
        mysql_pass = str(self.mysql_passLine.text())
        mysql_database = str(self.mysql_databaseLine.text())

        if quitOnClose:
            minimizeOnClose = 2
        else:
            minimizeOnClose = 1

        # Check database connection
        self.database.close()

        data = dict()
        if type == "sqlite":
           # Use filename from the settings
           #data["filename"] = sqlite_file
           data["filename"] = self.settings.setting("database/sqlite/filename")
        elif type == "mysql":
            if not (mysql_host and mysql_port and mysql_user and mysql_pass and mysql_database):
                QMessageBox.critical(self,  self.tr("MySQL data is incomplete!"), self.tr("You didn't enter all connection informations. If you are unsure please use SQlite."))
                return False

            data["host"] = mysql_host
            data["port"] = mysql_port
            data["user"] = mysql_user
            data["pass"] = mysql_pass
            data["database"] = mysql_database

        self.log.info(QString("Trying to connect to database"))
        ret = self.database.openDatabase(type,  data)

        if ret == False:
            # Connection failed
            # Show error message
            err = self.database.error()

            message = QMessageBox.critical(self,
            self.tr("Error by creating the database!"),
            self.tr("""The creation of the database failed with the following error:\n%1\n
Please mind that the application doesn't work without an database connection.""").arg(err),
            QMessageBox.StandardButtons(\
                    QMessageBox.Ignore | \
                    QMessageBox.Retry),
            QMessageBox.Retry)

            if message == QMessageBox.Retry:
                return False

        # Write values
        self.settings.setSetting("general/connectionPlugin",  plugin)
        self.settings.setSetting("bluetooth/port",  port)
        self.settings.setSetting("contacts/hideCellnumber",  not showCellnumber)
        self.settings.setSetting("general/sendMessageOnReturn",  sendMessageOnReturn)
        self.settings.setSetting("messages/markAsRead",  markAsRead)
        self.settings.setSetting("messages/saveAllMessages",  saveMessages)
        self.settings.setSetting("messages/displayLast",  lastMessages)
        self.settings.setSetting("popups/show",  showPopup)
        self.settings.setSetting("popups/timeout",  timeoutPopup)
        self.settings.setSetting("popups/animate",  animatePopup)
        self.settings.setSetting("windows/main/minimizeOnClose",  minimizeOnClose)
        self.settings.setSetting("contacts/ignoreAll",  ignoreAllContacts)
        self.settings.setSetting("contacts/displayIcon",  displayIcon)
        self.settings.setSetting("database/type",  type)
        
        # Do not replace current filename, the default should be okay
        #self.settings.setSetting("database/sqlite/filename",  sqlite_file)
        self.settings.setSetting("database/mysql/host",  mysql_host)
        self.settings.setSetting("database/mysql/port",  mysql_port)
        self.settings.setSetting("database/mysql/user",  mysql_user)
        self.settings.setSetting("database/mysql/pass",  mysql_pass)
        self.settings.setSetting("database/mysql/database",  mysql_database)

        self.settings.reloadSettings()

        return True

    def checkBoxValue(self,  bool):
        if bool:
            return Qt.Checked
        else:
            return Qt.Unchecked
