# -*- coding: utf-8 -*-

# Copyright (c) 2010 Lukas Hetzenecker <LuHe@gmx.at>

import os
from PyQt4.QtCore import *
from PyQt4.QtGui import *
import lib.export
import ui.ui_export
import window.export_progress
from lib.classes import *

try:
    import matplotlib.figure
    import matplotlib.font_manager
    import matplotlib.backend_bases
except ImportError:
    USE_MATPLOTLIB = False
else:
    USE_MATPLOTLIB= True

class Export(QDialog,  ui.ui_export.Ui_Export):
    def __init__(self, parent, main):
        super(Export,  self).__init__(parent)

        self.parent = parent
        self.main = main

        self.log = main.log
        self.connection = main.connection
        self.database = main.database
        self.settings = main.settings
        self.helper = main.helper

        self.setupUi(self)
        
        if not USE_MATPLOTLIB:
            self.graphBox.setCheckState(Qt.Unchecked)
            self.graphBox.setEnabled(False)
        
        # Complete directory names in export QLineEdit
        self.export = lib.export.Export(self,  main)
        self.completer = QCompleter()
        self.completionModel = QDirModel(self.completer)
        self.completionModel.setFilter(QDir.AllDirs)
        self.completer.setModel(self.completionModel)
        self.exportLine.setCompleter(self.completer)
        self.exportLine.setText(QDir.homePath())

        self.exportButton = self.buttonBox.addButton(self.tr("Export..."),  QDialogButtonBox.AcceptRole)
        main.setupButtonBox(self.buttonBox)

        self.connect(self.selectAllButton,  SIGNAL("clicked()"),  self.selectAllContacts)
        self.connect(self.deselectAllButton,  SIGNAL("clicked()"),  self.deselectAllContacts)
        self.connect(self.openBrowserButton,  SIGNAL("clicked()"),  self.openBrowser)
        self.connect(self.exportLine,  SIGNAL("textChanged(const QString &)"),  self.checkExportButton)
        self.connect(self,  SIGNAL("accepted()"),  self.startExport)

        self.loadContacts()
        self.loadPlugins()
        self.loadPeriods()
        self.loadOrder()
        self.loadGraphformats()
        self.checkExportButton(self.exportLine.text())
        self.show()
    
    def loadContacts(self):
        self.contactsList.clear()
        if not self.database.contactCount():
            item = QListWidgetItem(self.contactsList)
            item.setText(self.tr("No contacts available"))
        else:
            for contact in self.database.contacts(True):
                item = QListWidgetItem(self.contactsList)
                
                item.setData(Roles.ContactRole,  QVariant(contact))
                item.setText(contact.name())
                item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                item.setCheckState(Qt.Unchecked)
            self.contactsList.sortItems(Qt.AscendingOrder)
    
    def loadPlugins(self):
        for format in self.export.formats():
            self.formatBox.addItem(format.format(),  QVariant(format))
    
    def loadPeriods(self):
        self.periodBox.addItem(self.tr("Daily"),  QVariant(Period.Daily))
        self.periodBox.addItem(self.tr("Monthly"),  QVariant(Period.Monthly))
        self.periodBox.addItem(self.tr("Yearly"),  QVariant(Period.Yearly))
        self.periodBox.addItem(self.tr("All messages in one file"),  QVariant(Period.All))

    def loadOrder(self):
        self.orderBox.addItem(self.tr("Ascending"),  QVariant(Order.ASC))
        self.orderBox.addItem(self.tr("Descending"),  QVariant(Order.DESC))

    def loadGraphformats(self):
        self.graphFormatBox.addItem("PNG",  QVariant(GraphFormat.PNG))
        self.graphFormatBox.addItem("SVG",  QVariant(GraphFormat.SVG))

    def selectAllContacts(self):
        for num in range(self.contactsList.count()):
            item = self.contactsList.item(num)
            if item.data(Roles.ContactRole).toPyObject():
                item.setCheckState(Qt.Checked)

    def deselectAllContacts(self):
        for num in range(self.contactsList.count()):
            item = self.contactsList.item(num)
            if item.data(Roles.ContactRole).toPyObject():
                item.setCheckState(Qt.Unchecked)
    
    def openBrowser(self):
        path = QFileDialog.getExistingDirectory(self,  self.tr("Select export directory"))
        if path:
            self.exportLine.setText(path)
    
    def checkExportButton(self,  path):
        if not path:
            self.directoryLabel.setText(self.tr("No export directory selected!"))
            self.exportButton.setEnabled(False)
        elif not QDir(path).exists():
            self.directoryLabel.setText(self.tr("Invalid export directory!"))
            self.exportButton.setEnabled(False)
        elif not os.access(path,  os.W_OK):
            self.directoryLabel.setText(self.tr("Cannot write to export directory!"))
            self.exportButton.setEnabled(False)
        else:
            self.directoryLabel.setText("")
            self.exportButton.setEnabled(True)

    def startExport(self):        
        dlg = window.export_progress.ExportProgress(self,  self.main,  self.export)
        self.plugin = self.formatBox.itemData(self.formatBox.currentIndex()).toPyObject()
        
        exportContacts = None
        if self.filterButton.isChecked():
            contacts = Contacts.Filter
            # Build a list of contacts which should be exported
            exportContacts = list()
            for row in range(self.contactsList.count()):
                contact = self.contactsList.item(row)
                if contact.checkState() == Qt.Checked:
                    exportContacts.append(contact.data(Roles.ContactRole).toPyObject())
        elif self.messageContactsButton.isChecked():
            contacts = Contacts.ContactsWithMessages
        else:
            contacts = Contacts.All
        
        if self.graphBox.isChecked():
            graph = Graph.Yes
        else:
            graph = Graph.No
        
        if self.legendBox.isChecked():
            legend = Legend.Yes
        else:
            legend = Legend.No
        
        if self.thumbnailsBox.isChecked():
            thumbnails = Thumbnails.Yes
        else:
            thumbnails = Thumbnails.No
        
        self.export.start(
                          self.plugin,
                          contacts=contacts, 
                          period=self.periodBox.itemData(self.periodBox.currentIndex()).toPyObject(), 
                          graph=graph, 
                          legend=legend, 
                          thumbnails=thumbnails, 
                          directory=unicode(self.exportLine.text() + u'/'), 
                          order=self.orderBox.itemData(self.orderBox.currentIndex()).toPyObject(),
                          exportContacts=exportContacts, 
                          graphFormat=self.graphFormatBox.itemData(self.graphFormatBox.currentIndex()).toPyObject()
                          )
    
